UNIT XMSBLOCK;

(**********************************************************************

 Store data blocks in XMS memory.
 All blocks must have equal size, block-size must be less than 64 kB.
 Block-Size must be an even number (Limitation of XMS driver) !!

 XMS Memory is being allocated in few large blocks,
 their allocation being controlled by this unit.
 i.e. each XMS block will be filled with many data blocks.
 The number of data blocks per XMS block is limited to about 65530.
 They are counted starting with 1.

 Must be initialized before use with Init (EntrySize, BlockSize)
   EntrySize is size of your data blocks in bytes.
   BlockSize is size of the XMS blocks in KILObytes.
 Fill the memory by adding individual data blocks with add.
 Readout and Modification of data blocks is possible with random access,
   i.e. you can access any arbitrary block with get and replace,
   just specify the block nuber (numbered from 1 upwards).
 Deleting of single data blocks is possible but slow, because
   all blocks after the one being deleted will be moved down
   one place.

**********************************************************************)


INTERFACE

uses
  XMS;

const
  MaxNumXmsBlocks =  30;  { maximum number of XMS blocks  }
                          { caution: also limited by XMS driver }

type
  XmsBlockObj = object
    {--------------------------------}
     EntrySize   : word;      { size of data block in bytes      }
     XmsBlockSize: word;      { size of XMS blocks in kilobytes  }
     DataPerXms  : word;      { number data blocks per XMS block }
     NumEntries  : longint;   { number currently stored data blocks }
     Handle      : array [0..MaxNumXmsBlocks-1] of word;
     tmp         : pointer;   { required by delete }
    {--------------------------------}
     Constructor Init    (_EntrySize:word; var _XmsBlockSize:word; var ok:boolean);

     Procedure   Add     (            var data; var ok:boolean);
     Procedure   Get     (nr:longint; var data; var ok:boolean);
     Procedure   Replace (nr:longint; var data; var ok:boolean);
     Procedure   Delete  (nr:longint;           var ok:boolean);

     Function    GetNumEntries : longint;
     Procedure   Clear;

     Destructor  done;
    {--------------------------------}
  end;


IMPLEMENTATION


const
  MaxStrucSize = 65520;


Constructor XmsBlockObj.Init    (_EntrySize : word; var _XmsBlockSize:word;
                                 var ok : boolean);
var
  i : longint;
begin
  ok := false;
  if maxavail < _EntrySize then exit;

  ok := true;
  getmem (tmp, _EntrySize);

  { reduce size of XMS blocks if it would hold more than 65536 data blocks }
  if longint(_XmsBlockSize) * 1024 div _EntrySize > MaxStrucSize then
     _XmsBlockSize := MaxStrucSize * _EntrySize div 1024;

  EntrySize    := _EntrySize;
  XmsBlockSize := _XmsBlockSize;

  DataPerXms   := longint(XmsBlockSize) * 1024 div EntrySize;
  NumEntries := 0;
end;


Procedure   XmsBlockObj.Add     (var data; var ok:boolean);
var
  n   : longint;
  i,j : word;
begin
  i := NumEntries div DataPerXms;  { XMS block }
  j := NumEntries mod DataPerXms;  { offset in block }

  { if required allocate new XMS block }
  if j = 0 then begin
     ok := false;
     if XmsMaxAvail < XmsBlockSize then exit;
     if i >= MaxNumXmsBlocks then exit;
     XmsError;
     Handle[i] := XMS.GetHandle (XmsBlockSize);
     if XmsError <> 0 then exit;
  end;

  { copy data to XMS }
  n := longint(j) * EntrySize;
  MoveEMB (0, @data, Handle[i], ptr (n div 65536, n mod 65536), EntrySize);

  ok := (XmsError = 0);
  inc (NumEntries);
end;


Procedure   XmsBlockObj.Get     (nr:longint; var data; var ok:boolean);
var
  i,j : word;
  n   : longint;
begin
  ok := false;
  if (nr < 1) or (nr > NumEntries) then exit;

  dec (nr);
  i := nr div DataPerXms;  { XMS block }
  j := nr mod DataPerXms;  { offset in block }

  { retrieve data from XMS }
  n := longint(j) * EntrySize;
  MoveEmb (Handle[i], ptr (n div 65536, n mod 65536), 0, @data, EntrySize);

  ok := (XmsError = 0);
end;


Procedure   XmsBlockObj.Replace (nr:longint; var data; var ok:boolean);
var
  i,j : word;
  n   : longint;
begin
  ok := false;
  if (nr < 1) or (nr > NumEntries) then exit;

  ok := true;
  dec (nr);
  i := nr div DataPerXms;
  j := nr mod DataPerXms;
  n := longint(j) * EntrySize;
  MoveEMB (0, @data, Handle[i], ptr (n div 65536, n mod 65536), EntrySize);
end;


Procedure   XmsBlockObj.Delete  (nr:longint;           var ok:boolean);
var
  n : longint;
  i,j : word;
begin
  ok := false;
  if (nr < 1) or (nr > NumEntries) then exit;
  ok := true;

  { move all higher block down by one element, slow! }
  for n := nr to NumEntries-1 do begin
     Get     (n+1, tmp^, ok); {##}
     Replace (n,   tmp^, ok); {##}
  end;

  { release last element }
  dec (NumEntries);

  { in case the last XMS block is now empty, release it }
  i := NumEntries div DataPerXms;
  j := NumEntries mod DataPerXms;
  if j = 0 then Xms.FreeHandle (Handle[i]);
end;


Function    XmsBlockObj.GetNumEntries : longint;
begin
  GetNumEntries := NumEntries;
end;


Procedure   XmsBlockObj.Clear;
var
  i,n : word;
begin
  n := (NumEntries - 1) div DataPerXms;
  for i := 0 to n do
     Xms.FreeHandle (Handle[i]);
  NumEntries := 0;
end;


Destructor  XmsBlockObj.done;
begin
  Clear;
  FreeMem (Tmp, EntrySize);
end;



END.
