UNIT WAV;

(**********************************************************************

 - read and write header of WAV (sound) files
 - extract data of one channel from stereo wav file

 changes:
 18.12.2001 fix ReadWavInfo for files with more than one data chunk
 ??.09.2004 add WriteWavHeader

**********************************************************************)


INTERFACE

uses
  dos;

type
  riff_header = record
     idriff     : array [1..4] of char;   { 'RIFF' }
     sizeriff   : longint;
     idwave     : array [1..4] of char;   { 'WAVE' }
  end;

  fmt_header = record
     idfmt      : array [1..4] of char;   { 'fmt ' }
     sizefmt    : longint;
     format     : word;     { 1=PCM }
     channels   : word;     { 1=mono, 2=stereo }
     samplerate : longint;  { sample rate in Hz }
     bytepersec : longint;  { average bytes per second }
     samplebytes: word;     { 1 = 8-bit mono,
                              2 = 8-bit stereo or 16-bit mono
                              4 = 16-bit stereo }
     samplebits : word;
  end;

  data_header = record
     iddata     : array [1..4] of char;   { 'data' }
     sizedata   : longint;
  end;


Procedure ReadWavInfo (fn:pathstr;
                       var r : riff_header;
                       var m : fmt_header;
                       var d : data_header;
                       var dataoffset : longint;
                       var err: byte);
{ err:
{ 0=ok, 1=i/o error 2=file too short, 3=RIFF header missing }
{ 4=WAVE header missing, 5=fmt header missing, 6=data header missing }
{ 7=unknown sound format }
{ 8=invalid number of channels (must be 1 or 2) }
{ 9=invalid number of bytes per sample (must be 1,2, or 4) }
{ 10=invalid number of bits per sample (must be 8 or 16) }
{ 11=invalid format }

{ dataoffset is the start of sound data in the file }


Procedure GetWavInfo (fn:pathstr;
                      var samplerate : longint;
                      var stereo, sixteenbit : boolean;
                      var samplebytes : byte;
                      var dataoffset, datasize : longint;
                      var ok : boolean);


procedure WriteWavHeader (var f:file;
                          samplerate:word; sixteenbit, stereo:boolean;
                          databytes:longint;
                          var ok:boolean);


type
  SampleStereo8 = record
              l,r : byte;
           end;
  SampleStereo16 = record
              l,r : integer;
           end;



IMPLEMENTATION


Procedure ReadWavInfo (fn:pathstr;
                       var r : riff_header;
                       var m : fmt_header;
                       var d : data_header;
                       var dataoffset : longint;
                       var err: byte);
label
  leave;
var
  f : file;
  i : word;
  found : boolean;
begin
  err := 0;

  assign (f,fn);
  {$I-}
  reset (f,1);
  if not (ioresult = 0) then begin err := 1; exit; end;

  blockread (f,r,sizeof(r),i);
  if not (ioresult = 0) then err := 1;
  if i < sizeof(r) then err := 2;
  if r.idriff <> 'RIFF' then err := 3;
  if r.idwave <> 'WAVE' then err := 4;
  if err <> 0 then goto leave;

  blockread (f,m,sizeof(m),i);
  if not (ioresult = 0) then err := 1;
  if i < sizeof(m) then err := 2;
  if m.idfmt <> 'fmt ' then err := 5;
  if err <> 0 then goto leave;
  if m.sizefmt > sizeof(m)-8 then
     seek (f,filepos(f)+m.sizefmt-(sizeof(m)-8));

  if m.format <> 1 then err := 7;
  if (m.channels <> 1) and (m.channels <> 2) then err := 8;
  if not (m.samplebytes in [1,2,4]) then err := 9;
  if (m.samplebits <> 8) and (m.samplebits <> 16) then err := 10;
  if err <> 0 then goto leave;

  { repeat-loop added on 18.12.2001 }
  repeat
     blockread (f,d,sizeof(d),i);
     if not (ioresult = 0) then err := 1;
     if i < sizeof(d) then err := 2;
     if err <> 0 then goto leave;
     found := (d.iddata = 'data');
     if not found then
        seek (f,filepos(f)+d.sizedata);
  until found;
  if not found then begin
     err := 6;
     goto leave;
  end;

  i := 1;
  if m.channels   = 2 then i := i * 2;
  if m.samplebits =16 then i := i * 2;
  if i <> m.samplebytes then err := 11;
  if i * m.samplerate <> m.bytepersec then err := 11;

  dataoffset := filepos (f);

leave:
  close (f);
  if (ioresult <> 0) then begin {nothing} end;
  {$I+}
end;


Procedure GetWavInfo (fn:pathstr;
                      var samplerate:longint;
                      var stereo, sixteenbit : boolean;
                      var samplebytes : byte;
                      var dataoffset, datasize : longint;
                      var ok : boolean);
var
  r : riff_header;
  m : fmt_header;
  d : data_header;
  err: byte;
begin
  ReadWavInfo (fn, r,m,d,dataoffset,err);
  ok := (err = 0);
  if not ok then exit;

  samplerate := m.samplerate;
  stereo     := (m.channels = 2);
  sixteenbit := (m.samplebits = 16);
  samplebytes:= m.samplebytes;
  datasize   := d.sizedata;
end;



procedure WriteWavHeader (var f:file;
                          samplerate:word; sixteenbit, stereo:boolean;
                          databytes:longint;
                          var ok:boolean);
{ file must be opened with blocksize 1 and positioned on byte 0 }
var
  channels   : byte;
  samplebits : byte;
  samplebytes: byte;
  riff : riff_header;
  fmt  : fmt_header;
  data : data_header;
begin
  if not sixteenbit
     then samplebits := 8
     else samplebits := 16;
  if not stereo
     then channels := 1
     else channels := 2;
  if not sixteenbit
     then samplebytes := channels
     else samplebytes := channels * 2;

  riff.idriff    := 'RIFF';
  riff.sizeriff  := databytes
                   +sizeof(riff)-sizeof(riff.idriff)-sizeof(riff.sizeriff)
                   +sizeof(data_header)+sizeof(fmt_header);
  riff.idwave    := 'WAVE';
  blockwrite (f, riff, sizeof(riff));

  fmt.idfmt      := 'fmt ';
  fmt.sizefmt    := sizeof(fmt)-sizeof(fmt.idfmt)-sizeof(fmt.sizefmt);
  fmt.format     := 1;
  fmt.channels   := channels;
  fmt.samplerate := samplerate;
  fmt.bytepersec := longint(samplerate) * samplebytes;
  fmt.samplebytes:= samplebytes;
  fmt.samplebits := samplebits;
  blockwrite (f, fmt, sizeof(fmt));

  data.iddata   := 'data';
  data.sizedata := databytes;
  blockwrite (f, data, sizeof(data));

  {$I+}
  ok := (ioresult = 0);
end;



END.

