UNIT TrapInt;

(**********************************************************************

 Intercept Interrupts, allows to use normal pascal procedure as
 interrupt procedure, provides huge stack for interrupt procedure.

 (c) Copyright 1996-2004 by Klaus Hartnegg (www.klaus-hartnegg.de)

 Description
   Comfortable hooking of interrupt procedures, provides big stack

 Advantages
   Borland's Turbo Pascal and Borland Pascal 7 allow to declare
   Interrupt procedures. However such procedures have two big
   drawbacks: they must work with the stack that happens to be
   active in the moment the procedure is called and they can
   disturb 32bit registers that are used in longint arithmetic.
   The stack may be as tiny as 16 bytes. But for example passing a
   string to another pascal procedures requires 256 bytes.
   Passing 3 real numbers requires 15 bytes.

 Use
   Declare a procedure with the same arguments as TrapProcedure below.
   It must be declared FAR like this:
      procedure mytrap (var reg:registers; var abort:boolean); FAR;
   Call TrapInterrupt with interrupt number and the name of your
   interrupt procedure as arguments.
   The procedure may read and change the variable reg and may set abort:=true.
   If abort is set to true, then the original interrupt handler is not called,
   otherwise it is called after your procedure.

   If you want to stop interception, call ReleaseInt.
   If you don't do it, it will automatically be done at
   end of your program by an exit procedure.

 Caveats
   If you use this to intercept hardware interrupts, your procedure
   must do port[$20]:=$20 before it exits with abort=true,
   when using an IRQ 8 or higher, also do port[$A0]:=$20.

 Limitation
   Only one interrupt can be hooked at a time.

 Changes
   30.01.1999     cleanup, don't enable interrupts, added comments
   22.03.2004 fix preserve 32bit registers eax,ebx,ecx,edx
                  because they are used in longint arithmetic
   13.09.2004 fix reset initialized, thus enable re-use

**********************************************************************)


INTERFACE

uses
 dos;

type
  TrapProcedure = procedure (var reg:registers; var abort:boolean); {FAR;}

var
  LostIntCounter : word;

Procedure TrapInterrupt (_IntNr:byte; _IntProc:TrapProcedure);
Procedure ReleaseInterrupt;


IMPLEMENTATION

var
  IntNr   : byte;
  IntProc : TrapProcedure;

var
  OldInt   : pointer;
  OldExit  : pointer;
  SSSave, SPSave,
  CallIP, CallCS  : word;
  initialized: byte;
  active   : byte;
  abort    : boolean;
  PasRegs  : registers;
  PasRegs32: record
               eax,ebx,ecx,edx : longint;
             end;

{ stack for interrupt handler }
const IntrStackSize = 16384;
var   IntrStack : array [1..IntrStackSize] of byte;
const IntrSS : word = Seg(IntrStack[IntrStackSize]);
      IntrSP : word = Ofs(IntrStack[IntrStackSize]);


Procedure IntHandler; FAR; Assembler;
asm
  push  ds                      { save ds on stack }
  push  ax                      { save ax on stack }

  mov   ax, seg @data           { set data segment of Turbo Pascal }
  mov   ds,ax

  cmp   byte ptr [initialized],0{ check if initialization call }
  je    @install

  cmp   byte ptr active,0       { test activity flag }
  jne   @notreentrant           { skip hooked procedure if already active }

  inc   byte ptr active         { set activity flag        }

  pop   [PasRegs.&ax]           { get AX and DS from stack ... }
  pop   [PasRegs.&ds]           { ... and store them           }

  pop   [CallIP]                { get and store stack pointer ... }
  pop   [CallCS]
  pop   [PasRegs.Flags]         { ... mainly to get access to flags }


  cmp   byte ptr [test8086],2   { store 32bit registers if running on 386 }
  jb    @small1

  db    $66
  mov   word ptr [PasRegs32.&eax], ax
  db    $66
  mov   word ptr [PasRegs32.&ebx], bx
  db    $66
  mov   word ptr [PasRegs32.&ecx], cx
  db    $66
  mov   word ptr [PasRegs32.&edx], dx
  @small1:


  mov   [PasRegs.&bx], bx       { store register contents  }
  mov   [PasRegs.&cx], cx
  mov   [PasRegs.&dx], dx
  mov   [PasRegs.&bp], bp
  mov   [PasRegs.&si], si
  mov   [PasRegs.&di], di
  mov   [PasRegs.&es], es

  mov   [SSSave], ss           { store old stack pointer    }
  mov   [SPSave], sp
  mov   ss,[IntrSS]            { set stack for turbo pascal }
  mov   sp,[IntrSP]

  mov   byte ptr abort, 0      { set abort to false }

  { call pascal procedure }
  mov   di,offset PasRegs
  push  ds
  push  di
  mov   di,offset abort
  push  ds
  push  di
  call  dword ptr IntProc

  cli                          { disable interrupts if IntProc enabled them }
  mov   ss,[SSSave]            { restore old stack pointer }
  mov   sp,[SPSave]


  cmp   byte ptr [test8086],2  { restore 32bit registers if running on 386 }
  jb    @small2

  db    $66
  mov   ax, word ptr [PasRegs32.&eax]
  db    $66
  mov   bx, word ptr [PasRegs32.&ebx]
  db    $66
  mov   cx, word ptr [PasRegs32.&ecx]
  db    $66
  mov   dx, word ptr [PasRegs32.&edx]
  @small2:


  mov   es,[PasRegs.&es]       { reload registers from reg variable }
  mov   bp,[PasRegs.&bp]
  mov   di,[PasRegs.&di]
  mov   si,[PasRegs.&si]
  mov   dx,[PasRegs.&dx]
  mov   cx,[PasRegs.&cx]
  mov   bx,[PasRegs.&bx]

  push  [PasRegs.Flags]        { restore old stack contents }
  push  [CallCS]
  push  [CallIP]
  mov   ax,[PasRegs.&ax]

  dec   byte ptr active        { clear activity flag }

  cmp   byte ptr abort,0       { test abort flag     }
  jne   @abort


  push  [PasRegs.&ds]          { restore DS }
  pop   ds
  jmp   dword ptr cs:[@IntAddr]{ call old interrupt procedure }


@abort:
  push  [PasRegs.&ds]          { restore DS }
  pop   ds
  iret                         { return from interrupt }


@notreentrant:
  inc   word ptr LostIntCounter{ note that one interrupt was skipped }
  pop   ax
  pop   ds
  jmp   dword ptr cs:[@IntAddr]{ then call old interrupt handler }


@install:   { write address of original handler into code segment }
  mov   ax, word ptr [OldInt]
  mov   word ptr @intaddr, ax
  mov   ax, word ptr [OldInt+2]
  mov   word ptr @intaddr+2, ax

  inc   byte ptr [initialized]

  pop   ax
  pop   ds

  retf

@intaddr:   dw 0
            dw 0

end;


Procedure TrapInterrupt (_IntNr:byte; _IntProc:TrapProcedure);
begin
  IntNr   := _IntNr;
  IntProc := _IntProc;

  LostIntCounter := 0;

  GetIntVec (IntNr, OldInt);
  initialized := 0;
  IntHandler; { first call does initialization }

  OldExit := ExitProc;
  ExitProc := @ReleaseInterrupt;

  SetIntVec (IntNr, @IntHandler);
end;


Procedure ReleaseInterrupt;
begin
  ExitProc := OldExit;
  SetIntVec (IntNr, OldInt);
end;


END.
