Unit SBMIX;

{ program Mixer of Sound Blaster }


INTERFACE

procedure MixWrite  (reg,value:byte);
function  MixRead   (reg      :byte): byte;

procedure MixerReset;

procedure ACG_On;    { Auto Gain Control off (fixed 20 dB) }
procedure ACG_Off;   { Auto Gain Control off (fixed 20 dB) }

procedure InputLineOnly;
procedure InputMicOnly;


procedure GetParMix (var irq,dma8,dma16:integer);
{ read sound blaster parameters irq, dma and dma16 }
{ undefined values are set to -1 }



(*
  definitions for more user friendly mixer programming
  example: set volume master left to 10
    SetMixReg ('vml',10);     { set volume master left to 10 }
  equivalent to
    SetMixReg ('30.37',10);   { set bits 3-7 of register 30 to 10 }
*)

const
  nMixerRegs = 40;

type
  MixerDefRec = record
    name : string[40];
    keys : string[3];
    regs : string[5];
  end;

const
  MixerDef : array [1..nMixerRegs] of MixerDefRec = (
       (name:'Volume Master Left'    ;keys:'vml';  regs:'30.37'),
       (name:'Volume Master Right'   ;keys:'vmr';  regs:'31.37'),
       (name:'Volume Voice Left'     ;keys:'vvl';  regs:'32.37'),
       (name:'Volume Voice Right'    ;keys:'vvr';  regs:'33.37'),
       (name:'Volume mIdi  Left'     ;keys:'vil';  regs:'34.37'),
       (name:'Volume mIdi  Right'    ;keys:'vir';  regs:'35.37'),
       (name:'Volume CD    Left'     ;keys:'vcl';  regs:'36.37'),
       (name:'Volume CD    Right'    ;keys:'vcr';  regs:'37.37'),
       (name:'Volume Line  Left'     ;keys:'vll';  regs:'38.37'),
       (name:'Volume Line  Right'    ;keys:'vlr';  regs:'39.37'),
       (name:'Volume microPhone'     ;keys:'vp' ;  regs:'3A.37'),
       (name:'Volume Speaker'        ;keys:'vs' ;  regs:'3B.67'),
       (name:'Output Line Left'      ;keys:'oll';  regs:'3C.4'),
       (name:'Output Line Right'     ;keys:'olr';  regs:'3C.3'),
       (name:'Output CD   Left'      ;keys:'ocl';  regs:'3C.2'),
       (name:'Output CD   Right'     ;keys:'ocr';  regs:'3C.1'),
       (name:'Output microPhone'     ;keys:'op' ;  regs:'3C.0'),
       (name:'Left-Input  Midi Left' ;keys:'lml';  regs:'3D.6'),
       (name:'Left-Input  Midi Right';keys:'lmr';  regs:'3D.5'),
       (name:'Left-Input  Line Left' ;keys:'lll';  regs:'3D.4'),
       (name:'Left-Input  Line Right';keys:'llr';  regs:'3D.3'),
       (name:'Left-Input  CD   Left' ;keys:'lcl';  regs:'3D.2'),
       (name:'Left-Input  CD   Right';keys:'lcr';  regs:'3D.1'),
       (name:'Left-Input  MicroPhone';keys:'lp' ;  regs:'3D.0'),
       (name:'Right-Input mIdi Left' ;keys:'ril';  regs:'3E.6'),
       (name:'Right-Input mIdi Right';keys:'rir';  regs:'3E.5'),
       (name:'Right-Input Line Left' ;keys:'rll';  regs:'3E.4'),
       (name:'Right-Input Line Right';keys:'rlr';  regs:'3E.3'),
       (name:'Right-Input CD   Left' ;keys:'rcl';  regs:'3E.2'),
       (name:'Right-Input CD   Right';keys:'rcr';  regs:'3E.1'),
       (name:'Right-Input microPhone';keys:'rp' ;  regs:'3E.0'),
       (name:'Input  Gain  Left'     ;keys:'igl';  regs:'3F.67'),
       (name:'Input  Gain  Right'    ;keys:'igr';  regs:'40.67'),
       (name:'Output Gain  Left'     ;keys:'ogl';  regs:'41.67'),
       (name:'Output Gain  Right'    ;keys:'ogr';  regs:'42.67'),
       (name:'Auto Gain Control'     ;keys:'agc';  regs:'43.0'),
       (name:'Treble Left'           ;keys:'tl' ;  regs:'44.47'),
       (name:'Treble Right'          ;keys:'tr' ;  regs:'45.47'),
       (name:'Bass   Left'           ;keys:'bl' ;  regs:'46.47'),
       (name:'Bass   Right'          ;keys:'br' ;  regs:'47.47')
       );

(*
  Registers 30-35: 0..31  -62dB.. 0dB, default 24  -12 dB
  Registers 36-3A: 0..31  -62dB.. 0dB, default  0  -12 dB
  Register  3B   : 0.. 3  -18dB.. 0dB, Default  0  -18 dB
  Registers 3F-42: 0.. 3    0dB..18dB, default  0    0 dB
  Registers 44-47: 0.. 7  -14dB.. 0dB, 8..15 0dB..14dB, Default 80dB
  Register  43   : 0: auto, 1: fixed 20dB
*)


type
  string5 = string[5];

procedure GetMixRegInfo (regstr:string5; var reg, bit1,bit2,range:byte);
function  GetMixReg     (regstr:string5): byte;
procedure SetMixReg     (regstr:string5; value:byte);


procedure SetDefaultVolume;



IMPLEMENTATION

uses
  sb16, sb16math;


procedure mixwrite (reg,value:byte);
begin
  port[baseport + DSP_Port_MixIndex] := reg;
  port[baseport + DSP_Port_MixData]  := value;
end;

function  mixread (reg:byte) : byte;
begin
  port[baseport + DSP_Port_MixIndex] := reg;
  mixread := port[baseport + DSP_Port_MixData];
end;


procedure MixerReset;
begin
  mixwrite (0,0);
end;


procedure InputLineOnly;
begin
  mixwrite ($3D, mixread($3D) and $80 or $10);
  mixwrite ($3E, mixread($3E) and $80 or $08);
end;

procedure InputMicOnly;
begin
  mixwrite ($3D, mixread($3D) and $80 or 1);
  mixwrite ($3E, mixread($3E) and $80 or 1);
end;


procedure ACG_On;   { Auto Gain Control on }
begin
  mixwrite ($43, mixread($43) and not 1);
end;

procedure ACG_Off;  { Auto Gain Control off (fixed 20 dB) }
begin
  mixwrite ($43, mixread($43) or 1);
end;


procedure GetParMix (var irq,dma8,dma16:integer);
{ undefined: -1 }
var
  i,j : byte;
begin
  irq   := -1;
  dma8  := -1;
  dma16 := -1;

  sbwrite (DSP_Cmd_GetVersion);
  i := sbread;
  j := sbread;
  if i < 4 then exit;

  j := mixread($80);
  case j and $F of
    1 : irq := 2;
    2 : irq := 5;
    4 : irq := 7;
    8 : irq := 10;
  end;

  j := mixread($81);
  if      getbit (j,0) then dma8 := 0
  else if getbit (j,1) then dma8 := 1
  else if getbit (j,3) then dma8 := 3;
  if      getbit (j,5) then dma16 := 5
  else if getbit (j,6) then dma16 := 6
  else if getbit (j,7) then dma16 := 7;
end;


procedure GetMixRegInfo (regstr:string5; var reg, bit1,bit2,range:byte);
var
  c : string[2];
  ok: boolean;
begin
  if length(regstr) < 3 then runerror (255);
  c := copy (regstr,1,2);
  reg := hexval (c,ok); if not ok then runerror (255);
  delete (regstr,1,3);
  if length(regstr)=1 then begin
     bit1 := wordval(regstr,ok); if not ok then runerror (255);
     bit2 := bit1;
  end
  else if length(regstr) = 2 then begin
     bit1 := wordval(regstr[1],ok); if not ok then runerror (255);
     bit2 := wordval(regstr[2],ok); if not ok then runerror (255);
  end
  else runerror (255);
  range := 1 shl (bit2-bit1+1) - 1;
end;


function GetMixReg (regstr:string5) : byte;
var
  reg,bit1,bit2,range : byte;
  i,r : byte;
begin
  GetMixRegInfo (regstr,reg,bit1,bit2,range);
  r := mixread (reg);
  for i := bit2+1 to 7 do clrbitb (r,i);
  GetMixReg := r shr bit1;
end;


procedure SetMixReg (regstr:string5; value:byte);
var
  reg,bit1,bit2,range : byte;
  i,r : byte;
  found : boolean;
begin
  if regstr = '' then exit;
  if regstr[1] in ['a'..'z'] then begin
     found := false;
     i := 1;
     while not found and (i <= nMixerRegs) do begin
        if MixerDef[i].keys = regstr then found := true;
        if not found then inc (i);
     end;
     if not found then exit;
     regstr := MixerDef[i].regs;
  end;

  GetMixRegInfo (regstr,reg,bit1,bit2,range);
  if bit1 = bit2 then begin
     r := mixread (reg);
     if value = 1
        then setbitb (r, bit1)
        else clrbitb (r, bit1);
     mixwrite (reg, r);
  end
  else begin
     value := value shl bit1;
     for i := bit2+1 to 7 do clrbitb (value,i);
     mixwrite (reg,value);
  end;
end;


procedure SetDefaultVolume;
begin
  MixerReset;
  SetMixReg ('vml',  8);  { set Volume Master, 0-31 }
  SetMixReg ('vmr',  8);
  SetMixReg ('vvl', 20);  { set Volume Voice , 0-31 }
  SetMixReg ('vvr', 20);
  SetMixReg ('vil',  0);  { set Volume Midi }
  SetMixReg ('vir',  0);
  SetMixReg ('vcl',  0);  { set Volume CD }
  SetMixReg ('vcr',  0);
  SetMixReg ('vll',  0);  { set Volume Line }
  SetMixReg ('vlr',  0);
end;


END.

