Unit SB16WAV;

(**********************************************************************

 Unit for Turbo Pascal to play a wav-file on a Sound Blaster 16
 (c) 2001,2002,2004 by Klaus Hartnegg, hartnegg@gmx.de

 Use:
    { call the following once at the start of your program }
    if not sb16.ResetDSP then begin
       writeln ('Error: no sound blaster 16 compatible sound card found');
       halt (1);
    end;
    sb16wav.init;

    { do the following for each WAV-file that you want to play }
    sbmix.SetDefaultVolume;
    sb16wav.InitPlayback ('demo.wav', 0, errbyte);
    case errbyte of
      0 : begin {no error} end;
      1 : writeln ('can not read file');
      2 : writeln ('no sound blaster found');
      3 : writeln ('sound card not compatible with sound blaster 16');
      4 : writeln ('can not determine sound card parameters');
      5 : writeln ('not enough memory available');
      6 : writeln ('no XMS memory available');
      7 : writeln ('not enough XMS memory available');
      else writeln ('unknown error');
    end;
    if errbyte <> 0 then halt (1);
    sb16wav.StartPlayback;
    repeat
       sb16wav.fill_buffers;
       { you may add other things here that you want to be done   }
       { while sound is being played, but it must use less time   }
       { than playing the data buffered in XMS memory.            }
       { Buffered are NumXmsBuf * 2 * DmaBufHalfSize bytes        }
       { (one second playback uses samplebytes * samplerate bytes)}
    until sb16wav.ready;
    sb16wav.stop;

    { do this optionally once before the end of your program }
    sb16wav.done;

 Concept:
    The sound card (must be Sound Blaster 16 complatible) is programmed to
    - play in auto-init DMA mode, reading data from a buffer with DMA
    - trigger an interrupt whenever it has played half of the buffer

    The DMA-Buffer is allocated such that it does not cross a 64 kB boundary.

    The interrupt procedure fills new data into the half of the
    buffer that has just been played, while the sound card
    plays data from the other half of the buffer.

    Copying new data to the buffer must be done very fast, it can not rely
    on data coming from hard disk just in time. Thus an additional
    bigger buffer is used. This buffer is located in XMS memory.

    The main program first initializes everything, then it circles in
    an endless loop. In this loop it constantly refills the XMS buffer
    by reading new data from the WAV file.
    The interrupt procedure copies data from the next XMS buffer
    to the most recently used half of the DMA buffer.

 Bugs:
   - Sound can only be stopped when reaching half or end of DmaBuffer
     (set DmaBufHalfSize to small value to minimize this problem)
   - playback of WAV file with other data after end of wav
     will stop too late

 Changes:
 18.10.2001 add start_offset
            fix preserve contents of 16-bit registers in ISR procedure
 22.10.2001 add dmabuf_initialized, automatic call of init on first start
 07.09.2004 add Recording
                renamed playback specific things to include 'Playback'
                moved some stuff to TestBlaster and InitXmsBuf
 13.09.2004 chg use trapint
            fix stop made getintvec instead of setintvec
            chg parts moved to sb16irq
 15.09.2004 mv  TestBlaster moved to sb16.pas

**********************************************************************)


{$DEFINE VERBOSE}
{$DEFINE DEBUG_BUF}    { show xms buffer state }
{ DEFINE DEBUG_IRQ}    { show activity of interrupt and data procedures }

{$DEFINE DEBUG}


INTERFACE

uses
  dos;

const
  NumXmsBuf      = 20;   { suggested range : 10 to 50 }
  DmaBufHalfSize = 8192; { suggested values: 2048, 4096 or 8192 }

const
  DmaBufSize     = 2*DmaBufHalfSize;

var
  BufUnderflow : word;  { number of buffer underflows during playback }
  BufOverflow  : word;  { number of buffer overflows during recording }


procedure init;
procedure done;

{$IFDEF DEBUG}
function  GetDmaBuf : pointer;
{$ENDIF}

procedure InitPlayback (fn:pathstr; start_offset: real; verbose:boolean;
                        var err:byte);
procedure GotoFilePos (pos:longint);
procedure StartPlayback;
function  PlaybackReady : boolean;
procedure fill_buffers;
procedure StopPlayback;

{ start_offset: seconds from start of wav file, normally 0 }


procedure InitRecording  (fn:pathstr;
                          _samplingrate:word; _sixteenbit, _stereo : boolean;
                          verbose:boolean;
                          var err:byte);
procedure StartRecording;
procedure flush_buffers;
procedure StopRecording;



IMPLEMENTATION

uses
  {$IFDEF DEBUG_BUF} crt, {$ENDIF}
  {$IFDEF DEBUG} {$IFNDEF DEBUG_BUF} crt, {$ENDIF} {$ENDIF}
  {$IFDEF VERBOSE} sb16math, {$ENDIF}
  sb16, sb16irq, sbmix, wav,
  irq, trapint,
  dmabuf,
  xmsblock;


Procedure FillZero (var data; size:word);
begin
  FillChar (data, size, 0);
end;


{ unfortunately this unit requires quite a lot of GLOBAL VARIABLES }
{ I don't see a better way to share data with an interrupt procedure }

const
  MaxStrucSize = 65520;

type
  ByteArray1    = array [1..MaxStrucSize] of byte;
  ByteArray1Ptr = ^ByteArray1;
  string3       = string[3];
  string4       = string[4];

var
  { DMA buffer for Sound Blaster }
  dmabuf_initialized : boolean;
  buf          : DmaBufferObj; { DmaBufSize }
  bufptr       : pointer;      { points to start  of DMA buffer (first half)  }
  bufptr2      : pointer;      { points to middle of DMA buffer (second half) }

  { Interrupt handling }
  irqflag      : boolean;   { true if interrupt has occured }
  firsthalf    : boolean;   { denotes which half of DMA buffer is now in use }

  { XMS buffers }
  tmpbuf         : bytearray1ptr;  { DmaBufHalfSize, for FillXms }
  XmsBuf         : XmsBlockObj;
  {$IFDEF DEBUG_BUF}
  XmsBufAvailable: array [1..NumXmsBuf] of boolean;
  {$ENDIF}
  XmsBufHead     : word;  { next to be filled }
  XmsBufTail     : word;  { next to be emptied, buffer is empty if tail = head }
  XmsBufEnd      : word;  { 0 or last with valid data if end reached }
  XmsBufFill     : word;  { number of currently filled buffers }
  MinBufFill     : word;  { minimum number of filled buffers during playback }
  finishing      : byte;  { state variable controlling end of file handling }

  { WAV file }
  wavfile      : file;
  SamplingRate : longint;   { typical values: 11025, 22050, 44100 }
  sixteenbit   : boolean;   { false: sample size 8 bit, true: 16 bit }
  stereo       : boolean;   { false: mono, true: stereo }
  samplebytes  : byte;      { byte per sample }
  recordbytes  : longint;   { filesize in recording mode }

  Playing    : boolean;
  Recording  : boolean;

  {$IFDEF DEBUG_BUF}
  DebugOutputLine : byte;
  {$ENDIF}



procedure init;
begin
  { allocate DMA-buffer }
  { Reason for doing this here and not in start: }
  { if the program plays several WAV files, it should }
  { always use a buffer at the same memory address, }
  { otherwise EMM386 will sometimes spit out the }
  { errorneous error message "DMA mode not supported" }

  if dmabuf_initialized then exit;
  dmabuf_initialized := true;
  buf.init (dmabufsize);
  bufptr  := buf.bufptr;
  bufptr2 := ptr(seg(bufptr^),ofs(bufptr^)+DmaBufHalfSize);
  fillzero (bufptr^, DmaBufSize);
end;


{$IFDEF DEBUG}
function  GetDmaBuf : pointer;
begin
  if dmabuf_initialized
     then GetDmaBuf := buf.BufPtr
     else GetDmaBuf := NIL;
end;
{$ENDIF}



procedure done;
begin
  if not dmabuf_initialized then exit;
  buf.done;
  dmabuf_initialized := false;
end;


{$IFDEF DEBUG_BUF}
Procedure ShowBufferStatus;
var
  i : byte;
begin
  {$IFDEF DEBUG_IRQ}
  mem [$B800:2] := ord('S');
  {$ENDIF}

  gotoxy (1,DebugOutputLine);
  for i := 1 to NumXmsBuf do
     if XmsBufAvailable[i] then write ('*') else write (' ');
  write ('  ');
  write (XmsBufFill:2,'/',NumXmsBuf:2);
  write ('  ',MinBufFill:2);

{ gotoxy (1,DebugOutputLine+1); clreol;
  gotoxy (XmsBufHead,DebugOutputLine+1); write ('H');
  gotoxy (XmsBufTail,DebugOutputLine+1); write ('T'); }

  {$IFDEF DEBUG_IRQ}
  mem [$B800:2] := ord(' ');
  {$ENDIF}
end;
{$ENDIF}



Procedure FillXms (verbose:boolean);
{ check XMS buffers, if any are empty fill them by reading data from wav-file }
var
  i,j : word;
  x,y : real;
  f : word;
  ok: boolean;
  numread : word;
begin
  if XmsBufEnd <> 0 then exit;

  {$IFDEF DEBUG_IRQ}
  mem [$B800:4] := ord('F');
  {$ENDIF}

  repeat
     i := XmsBufHead;
     inc (i);
     if i > NumXmsBuf then i := 1;

     ok := (i <> XmsBufTail);
     if ok and not eof(wavfile) then begin
        {$IFDEF DEBUG_IRQ}
        mem [$B800:6] := ord('R');
        {$ENDIF}
       {pushf;}
       {cli;}
        blockread (wavfile, tmpbuf^, DmaBufHalfSize, numread);
       {popf;}
        {$IFDEF DEBUG_IRQ}
        mem [$B800:6] := ord(' ');
        {$ENDIF}

        inc (XmsBufFill);

        {## hier sollte geprft werden, ob die in der Datei }
        {## angegeben anzahl bytes gelesen wurde }
        {## weil es wav-dateien geben kann, die nach den tondaten }
        {## weitere daten enthalten, dann geht diese erkennung schief }
        {## hier wird nur geprft, ob das Datei-Ende erreicht ist }
        if (numread < DmaBufHalfSize) or eof(wavfile) then begin
           XmsBufEnd := XmsBufHead;
           while numread < DmaBufHalfSize do begin
              inc (numread);
              tmpbuf^[numread] := 0;
           end;
        end;
        if (XmsBufEnd = 0) and eof (wavfile) then XmsBufEnd := i;

        {$IFDEF DEBUG_IRQ}
        mem [$B800:8] := ord('W');
        {$ENDIF}
       {pushf;}
       {cli;}
        XmsBuf.Replace (XmsBufHead, tmpbuf^, ok); {##}
       {popf;}
        {$IFDEF DEBUG_IRQ}
        mem [$B800:8] := ord(' ');
        {$ENDIF}
        {$IFDEF DEBUG_BUF}
        XmsBufAvailable[XmsBufHead] := true;
        {$ENDIF}
        XmsBufHead := i;

        {$IFDEF DEBUG_BUF}
        if verbose then ShowBufferStatus;
        {$ENDIF}
     end;
  until not ok or (XmsBufEnd <> 0);

  {$IFDEF DEBUG_IRQ}
  mem [$B800:4] := ord(' ');
  {$ENDIF}
end;



procedure ISR_PLAYBACK; FAR;
{ called by sound blaster whenever data from half of DMA buffer read }
var
  ok : boolean;
  i  : byte;
begin
  {$IFDEF DEBUG_IRQ}
  mem [$B800:0] := ord('I');
  {$ENDIF}

  irqflag := true;
  firsthalf := not firsthalf;

  if XmsBufTail <> XmsBufHead then begin

     if firsthalf
        then XmsBuf.get (XmsBufTail, bufptr2^, ok)
        else XmsBuf.get (XmsBufTail, bufptr^,  ok);
     if not ok then runerror (255);

     {$IFDEF DEBUG_BUF}
     XmsBufAvailable [XmsBufTail] := false;
     {$ENDIF}
     inc (XmsBufTail);
     dec (XmsBufFill);
     if (XmsBufFill < MinBufFill) and (XmsBufEnd = 0)
        then MinBufFill := XmsBufFill;
     if XmsBufTail > NumXmsBuf then XmsBufTail := 1;
  end
  else
    if finishing < 2 then inc (BufUnderflow);

  if (XmsBufTail = XmsBufEnd) or (finishing > 0) then
     inc (finishing);

  {$IFDEF DEBUG_BUF}
  ShowBufferStatus;
  {$ENDIF}

  {$IFDEF DEBUG_IRQ}
  mem [$B800:0] := ord(' ');
  {$ENDIF}
end;


procedure ISR_RECORDING; FAR;
var
  i : byte;
  ok: boolean;
begin
  irqflag := true;
  firsthalf := not firsthalf;

  { move data from DMA-buffer to XMS-buffer }
  i := XmsBufHead;
  inc (i);
  if i > NumXmsBuf then i := 1;
  if i <> XmsBufTail then begin
     if firsthalf
        then XmsBuf.Replace (XmsBufHead, bufptr2^, ok)
        else XmsBuf.Replace (XmsBufHead, bufptr^,  ok);
     if not ok then runerror (255);
     {$IFDEF DEBUG_BUF}
     XmsBufAvailable[XmsBufHead] := false;
     {$ENDIF}
     XmsBufHead := i;

     inc (XmsBufFill);
  end
  else
     inc (BufOverflow);
end;



var
  OldSbInt : pointer;
  OldExit  : pointer;

procedure MyExit; Far;
begin
  DisableIRQ (sb16.irq);
  (*
  SetIntVec (irq2int[sb16.irq], OldSbInt);
  *)
  ReleaseInterrupt;

  ExitProc := OldExit;
end;


Procedure XmsBufInit (var err:byte);
var
  XmsSize : word;
  i  : byte;
  ok : boolean;
begin
  err := 0;
  firsthalf := true;

  { allocate temp buffer }
  if DmaBufHalfSize > MaxAvail then begin
     err := 1;
     exit;
  end;
  getmem (tmpbuf, DmaBufHalfSize);
  fillzero (tmpbuf^, DmaBufHalfSize);

  { allocate XMS buffers }
  if not xms.isinstalled then begin
     err := 2;
     freemem (tmpbuf, DmaBufHalfSize);
     exit;
  end;
{ xms.initcleanup; }
  XmsSize := (longint(DmaBufHalfSize) * NumXmsBuf + 1) div 1024;
  XmsBuf.init (DmaBufHalfSize, XmsSize, ok);
  if not ok then begin
     err := 3;
     freemem (tmpbuf, DmaBufHalfSize);
     exit;
  end;

  i := 1;
  ok := true;
  while ok and (i <= NumXmsBuf) do begin
     XmsBuf.add (bufptr^, ok);
     {$IFDEF DEBUG_BUF}
     XmsBufAvailable[i] := false;
     {$ENDIF}
     inc (i);
  end;
  if not ok then begin
     err := 3;
     XmsBuf.done;
     freemem (tmpbuf, DmaBufHalfSize);
     exit;
  end;

  XmsBufHead := 1;
  XmsBufTail := 1;
  XmsBufEnd  := 0;
  XmsBufFill := 0;
end;



Procedure XmsBufDone;
begin
  XmsBuf.done;
  freemem (tmpbuf, DmaBufHalfSize);
end;



procedure InitPlayback (fn:pathstr; start_offset: real; verbose:boolean;
                        var err:byte);
var
  dataoffset : longint;
  datasize   : longint;
  samplebytes: byte;

  i   : byte;
  err0: byte;
  ok  : boolean;

begin
  if not dmabuf_initialized then init;

  BufUnderflow := 0;
  err := 0;

  { test file }
  assign (wavfile,fn);
  {$I-}
  reset (wavfile,1);
  ok := (ioresult = 0);
  close (wavfile);
  ok := (ioresult = 0) and ok;
  {$I+}
  if not ok then begin
     err := 1;
     exit;
  end;

  { test sound card }
  TestBlaster (verbose, err0);
  if err0 <> 0 then begin
     case err0 of
       1: err := 2;
       2: err := 3;
       3: err := 4;
       else runerror (255);
     end; {case}
     exit;
  end;

  { read wav file header }
  GetWavInfo (fn, samplingrate, stereo, sixteenbit, samplebytes, dataoffset, datasize, ok);
  if not ok then begin
     err := 1;
     exit;
  end;
  if verbose then begin
     writeln ('filename:    ',fn);
     writeln ('sample rate: ',samplingrate,' Hz');
     write ('sample size: ');
     if not sixteenbit then writeln ('8 bit') else writeln ('16 bit');
     write ('channels:    ');
     if not stereo then writeln ('1 (mono)') else writeln ('2 (stereo)');
  end;


  { init XMS buffer }
  XmsBufInit (err0);
  if err0 <> 0 then begin
     case err0 of
       1 : err := 5;
       2 : err := 6;
       3 : err := 7;
       else runerror (255);
     end; {case}
     exit;
  end;


  { fill XMS buffer }

  {$IFDEF DEBUG_BUF}
  writeln;
  writeln ('data buffers:');
  for i := 1 to 3 do writeln;
  DebugOutputLine := WhereY-3;
  {$ENDIF}

  assign (wavfile,fn);
  reset (wavfile,1);
  seek (wavfile, dataoffset);

  if start_offset <> 0 then
     seek (wavfile, dataoffset + round (start_offset * samplingrate) * samplebytes);

  FillXms (true);

  MinBufFill := XmsBufFill;

  XmsBuf.Get (1, BufPtr^ , ok); {##}
  XmsBuf.Get (2, BufPtr2^, ok); {##}
  {$IFDEF DEBUG_BUF}
  XmsBufAvailable[1] := false;
  XmsBufAvailable[2] := false;
  {$ENDIF}
  dec (XmsBufFill, 2);
  XmsBufTail := 3;
end;



procedure gotofilepos (pos:longint);
begin
  asm
  pushf;
  cli;
  end;

  seek (wavfile, pos);

  asm
  popf;
  end;
end;


procedure StartPlayback;
begin
  if Playing   then StopPlayback;
  if Recording then StopRecording;
  Playing := true;

  irqflag := false;
  finishing := 0;

  { make sure loop ends correct if file is too small for full loop }
  if XmsBufTail >= XmsBufEnd then
     if XmsBufEnd = 1 then finishing := 3
     else if XmsBufEnd = 2 then finishing := 2
     else if XmsBufEnd = 3 then finishing := 1;

  sb16irq.StartPlayback (samplingrate, sixteenbit, stereo,
                         bufptr, dmabufsize, ISR_PLAYBACK);
end;


function  PlaybackReady : boolean;
begin
  PlaybackReady := (finishing >= 3);
end;


procedure fill_buffers;
var
  fill : boolean;
begin
  asm pushf; cli; end;
  fill := irqflag;
  irqflag := false;
  asm popf; end;

  if fill then begin
     irqflag := false;
     FillXms (false);
  end;
  if PlaybackReady then StopPlayback;
end;



procedure StopPlayback;
begin
  if not Playing then exit;
  Playing := false;

  sb16irq.stop;

  XmsBufDone;

  close (wavfile);
end;


procedure InitRecording (fn:pathstr;
                         _samplingrate:word; _sixteenbit, _stereo : boolean;
                         verbose:boolean;
                         var err:byte);
var
  err0 : byte;
  ok   : boolean;
begin
  samplingrate := _samplingrate;
  sixteenbit   := _sixteenbit;
  stereo       := _stereo;

  if not dmabuf_initialized then init;

  BufOverflow := 0;
  err := 0;

  { test sound card }
  TestBlaster (verbose, err0);
  if err0 <> 0 then begin
     case err0 of
       1: err := 2;
       2: err := 3;
       3: err := 4;
       else runerror (255);
     end; {case}
     exit;
  end;


  assign (wavfile,fn);
  {$I-}
  rewrite (wavfile,1);
  {$I+}
  if ioresult <> 0 then begin
     err := 1;
     exit;
  end;

  recordbytes := 0;
  WriteWavHeader (wavfile, samplingrate, sixteenbit, stereo, recordbytes, ok);
  if not ok then begin
     err := 1;
     close (wavfile);
     exit;
  end;


  { init XMS buffer }
  XmsBufInit (err0);
  if err0 <> 0 then begin
     case err0 of
       0 : begin end;
       1 : err := 5;
       2 : err := 6;
       3 : err := 7;
       else runerror (255);
     end; {case}
     exit;
  end;
end;


Procedure StartRecording;
begin
  if Playing   then StopPlayback;
  if Recording then StopRecording;
  Recording := true;

  irqflag := false;

  sb16irq.StartRecording (samplingrate, sixteenbit, stereo,
                          bufptr, dmabufsize, ISR_RECORDING);
end;


procedure flush_buffers;
var
  ok : boolean;
begin
  while XmsBufTail <> XmsBufHead do begin
     XmsBuf.get (XmsBufTail, tmpbuf^, ok);
     if not ok then runerror (255);

     blockwrite (wavfile, tmpbuf^, DmaBufHalfSize);
     inc (recordbytes, DmaBufHalfSize);

     inc (XmsBufTail);
     if XmsBufTail > NumXmsBuf then XmsBufTail := 1;

     dec (XmsBufFill);

     {$IFDEF DEBUG_BUF}
     XmsBufAvailable [XmsBufTail] := true;
     {$ENDIF}
  end;
end;


procedure StopRecording;
var
  ok : boolean;
begin
  if not Recording then exit;
  Recording := false;

  sb16irq.stop;

  flush_buffers;

  XmsBufDone;

  seek (wavfile, 0);
  WriteWavHeader (wavfile, samplingrate, sixteenbit, stereo, recordbytes, ok);
  close (wavfile);
end;



BEGIN
  dmabuf_initialized := false;
  Playing   := false;
  Recording := false;
END.
