Unit sb16irq;

(**********************************************************************

 Interrupt-Handling for playback and recording of sound
 with sound blaster 16 in auto-init-dma mode.

**********************************************************************)


INTERFACE

type
  sbproc = procedure;

procedure StartPlayback (samplingrate:word; sixteenbit, stereo:boolean;
                         dmabufptr:pointer; dmabufsize:word; ISR:sbproc);

procedure StartRecording(samplingrate:word; sixteenbit, stereo:boolean;
                         dmabufptr:pointer; dmabufsize:word; ISR:sbproc);

procedure Stop;


IMPLEMENTATION

uses
  dos,
  sb16, irq, trapint;

var
  _sixteenbit: boolean;
  _ISR       : sbproc;

var
  active : boolean;
  irqflag: boolean;


procedure MyInt (var reg:registers; var abort:boolean); FAR;
var
  dummy : byte;
begin
  _ISR;

  irqflag := true;

  { acknowledge interrupt by reading Read-Buffer Status port }
  if _sixteenbit
     then dummy := port[sb16.baseport+DSP_Port_IrqAck16]
     else dummy := port[sb16.baseport+DSP_Port_IrqAck8];

  ResetInterruptController (sb16.irq);

  abort := true;
end;


var
  OldExit : pointer;

procedure MyExit; FAR;
begin
  ExitProc := OldExit;
  if active then stop;
end;


procedure start (samplingrate:word; sixteenbit, stereo:boolean;
                 playback:boolean;
                 dmabufptr:pointer; dmabufsize:word; ISR:sbproc);
var
  TransferLength : word;
begin
  if active then stop;

  _sixteenbit := sixteenbit;
  _ISR        := ISR;

  if not sixteenbit
     then TransferLength := DmaBufSize
     else TransferLength := DmaBufSize div 2;

  { program DMA controller }
  if playback
     then InitSbDmaRead  (dmabufptr, TransferLength, sixteenbit)
     else InitSbDmaWrite (dmabufptr, TransferLength, sixteenbit);

  { initialize interrupt procedure }
  irqflag := false;
  OldExit := ExitProc;
  ExitProc := @MyExit;
  (*
  GetIntVec (irq2int[sb16.irq], OldSbInt);
  if playback
     then SetIntVec (irq2int[sb16.irq], @ISR_PLAYBACK)
     else SetIntVec (irq2int[sb16.irq], @ISR_RECORDING);
  *)
  if playback
     then TrapInterrupt (irq2int[sb16.irq], MyInt)
     else TrapInterrupt (irq2int[sb16.irq], MyInt);
  EnableIRQ (sb16.irq);

  { set sampling rate }
  if playback
     then sbwrite (DSP_Cmd_SetOutputRate)
     else sbwrite (DSP_Cmd_SetInputRate);
  sbwrite (SamplingRate div 256);
  sbwrite (SamplingRate mod 256);

  { start transfer }
  if playback then
     if sixteenbit
        then sbwrite ($B6)   { 16 bit auto-initialized output }
        else sbwrite ($C6)   {  8 bit auto-initialized output }
  else
     if sixteenbit
        then sbwrite ($BE)   { 16 bit auto-initialized input }
        else sbwrite ($CE);  {  8 bit auto-initialized input }

  if sixteenbit then
     if stereo
        then sbwrite ($30)   { stereo, 16 bit signed }
        else sbwrite ($10)   { mono  , 16 bit signed }
  else
     if stereo
        then sbwrite ($20)   { stereo, 8 bit unsigned }
        else sbwrite ($00);  { mono  , 8 bit unsigned }

  TransferLength := TransferLength div 2 - 1;
  sbwrite (lo(TransferLength));
  sbwrite (hi(TransferLength));

  active := true;
end;


procedure StartPlayback (samplingrate:word; sixteenbit, stereo:boolean;
                         dmabufptr:pointer; dmabufsize:word; ISR:sbproc);
begin
  start (samplingrate, sixteenbit, stereo, true,
         dmabufptr, dmabufsize, ISR);
end;


procedure StartRecording(samplingrate:word; sixteenbit, stereo:boolean;
                         dmabufptr:pointer; dmabufsize:word; ISR:sbproc);
begin
  start (samplingrate, sixteenbit, stereo, false,
         dmabufptr, dmabufsize, ISR);
end;



procedure stop;
begin
  if not active then exit;


  { send exit command, then wait for end of playback of current data block }

  asm pushf; cli; end;
  irqflag := false;

  { disable sound }
  if _sixteenbit
     then sbwrite ($D9)  { exit 16-bit auto-init DMA mode }
     else sbwrite ($DA); { exit  8-bit auto-init DMA mode }

  asm popf; end;

  repeat
  until irqflag;


  { disable interrupt }
  DisableIRQ (sb16.irq);
  (* SetIntVec (irq2int[sb16.irq], OldSbInt); *)
  ReleaseInterrupt;
  ExitProc := OldExit;

  active := false;
  ExitProc := OldExit;
end;



BEGIN
  active := false;
END.
