UNIT SB16;   { Sound Blaster 16 }


INTERFACE


const
  DSP_Port_MixIndex   =  4;
  DSP_Port_MixData    =  5;
  DSP_Port_Reset      =  6;
  DSP_Port_Read       = $A;
  DSP_Port_Write      = $C;
  DSP_Port_WriteStatus= $C;
  DSP_Port_ReadStatus = $E;
  DSP_Port_IrqAck8    = $E;
  DSP_Port_IrqAck16   = $F;

  DSP_Cmd_GetVersion    = $E1;
  DSP_Cmd_SetOutputRate = $41;
  DSP_Cmd_SetInputRate  = $42;

const
  DMA_Port_Mask8  = $A;
  DMA_Port_Mode8  = $B;
  DMA_Port_Clear8 = $C;
  DMA_Port_Mask16 = $D4;
  DMA_Port_Mode16 = $D6;
  DMA_Port_Clear16= $D8;

  DMA_Cmd_Read  = $58;  { read from memory, output to sound blaster }
  DMA_Cmd_Write = $54;  { write to memory, input from sound blaster }

  DMA_Adr_Port : array [0..7] of byte =
               ($00,$02,$04,$06,$C0,$C4,$C8,$CC);
  DMA_Count_Port : array [0..7] of byte =
               ($01,$03,$05,$07,$C2,$C6,$CA,$CE);
  DMA_Page_Port : array [0..7] of byte =
               ($87,$83,$81,$82,$00,$8B,$89,$8A);


var
  baseport : word;
  irq      : byte;
  dma8     : byte;
  dma16    : byte;


Procedure SetParDefault;
Procedure SetPar (_port:word; _irq,_dma8,_dma16 : byte);
Procedure SetParEnv;

procedure sbwrite (data:byte);
function  sbread : byte;

Function  ResetDSP : boolean;
{ false: no sound blaster found }

procedure TestBlaster (verbose:boolean; var err:byte);
{ err: 0=ok, 1=no sb, 2=sb not sb16 compatible, 3=parameter }


Procedure InitSbDma      (bufptr:pointer; transferlength:word; sixteenbit:boolean; cmd:byte);
Procedure InitSbDmaRead  (bufptr:pointer; transferlength:word; sixteenbit:boolean);
Procedure InitSbDmaWrite (bufptr:pointer; transferlength:word; sixteenbit:boolean);


IMPLEMENTATION

uses
  dos, crt,        { dos for getenv('BLASTER'), crt for delay(1) }
  sbmix,
  sb16math;


procedure sbwrite (data:byte);
begin
  repeat until port[baseport + DSP_Port_WriteStatus] and $80 = 0;
  port[baseport + DSP_Port_Write] := data;
end;

function  sbread : byte;
begin
  repeat until port[baseport + DSP_Port_ReadStatus] and $80 <> 0;
  sbread := port[baseport + DSP_Port_Read];
end;



Procedure SetParDefault;
begin
  baseport := $220;
  irq      := 5;
  dma8     := 1;
  dma16    := 5;
end;


Procedure SetPar (_port:word; _irq,_dma8,_dma16 : byte);
begin
  baseport := _port;
  irq      := _irq;
  dma8     := _dma8;
  dma16    := _dma16;
end;


Procedure SetParEnv;
var
  c,d : string;
  i,j : word;
  ok  : boolean;
begin
  c := GetEnv('BLASTER');
  while c <> '' do begin
     i := 1;
     while (i <= length(c)) and (c[i] = ' ') do inc (i);
     j := succ(i);
     while (j <= length(c)) and (c[j] <> ' ') do inc (j);
     d := copy (c,i,j-i);
     delete (c,1,j);

     case d[1] of
     'A' : begin
              delete (d,1,1);
              j := hexval (d,ok);
              if ok then baseport := j;
           end;
     'I' : begin
              delete (d,1,1);
              j := wordval (d,ok);
              if ok then irq := j;
           end;
     'D' : begin
              delete (d,1,1);
              j := wordval (d,ok);
              if ok then dma8 := j;
           end;
     'H' : begin
              delete (d,1,1);
              j := wordval (d,ok);
              if ok then dma16 := j;
           end;
     end; {case}
  end;
end;



Function ResetDSP : boolean;
var
  i,j   : byte;
begin
  port[baseport+DSP_Port_Reset] := 1;
  delay (1);
  port[baseport+DSP_Port_Reset] := 0;
  delay (10);
  i := port [baseport+DSP_Port_ReadStatus];
  j := port [baseport+DSP_Port_Read];

  ResetDSP := (i and $80 <> 0) and (j = $AA);
end;



procedure TestBlaster (verbose:boolean; var err:byte);
{ err: 0=ok, 1=no sb, 2=sb not sb16 compatible, 3=parameter }
var
  DspVersion : byte;
  irqB, dma, dma8B,dma16B  : integer;
  i,j : byte;
begin
  err := 0;

  if verbose then
     writeln ('using SoundBlaster at port $',hex4(sb16.baseport));

  { detect sound card }
  if not ResetDSP then begin
     err := 1;
     exit;
  end;

  { get DSP version number }
  sbwrite (DSP_Cmd_GetVersion);
  DspVersion := sbread;
  j := sbread;
  if verbose then
     writeln ('DSP version: ',DspVersion,'.',j);
  if DspVersion < 4 then begin
     err := 2;
     exit;
  end;

  irqB := -1;
  dma8B := -1;
  dma16B := -1;
  if DspVersion >= 4 then
     GetParMix (irqB,dma8B,dma16B);

  if verbose then begin
     write ('irq  : ',sb16.irq);  if sb16.irq  = irqB   then write (' (verified)'); writeln;
     write ('dma8 : ',sb16.dma8); if sb16.dma8 = dma8B  then write (' (verified)'); writeln;
     write ('dma16: ',sb16.dma16);if sb16.dma16= dma16B then write (' (verified)'); writeln;
     writeln;

     if (irqB <> -1) and (sb16.irq <> irqB) then
        writeln ('error: irq register setting '+wordstr(irqB)+' contradicts environment variable');
     if (dma8B <> -1) and (sb16.dma8 <> dma8B) then
        writeln ('error: dma8 register setting '+wordstr(dma8B)+' contradicts environment variable');
     if (dma16B <> -1) and (sb16.dma16 <> dma16B) then
        writeln ('error: dma16 register setting '+wordstr(dma16B)+' contradicts environment variable');
  end;

  if ((irqB <> -1) and (sb16.irq <> irqB))
  or ((dma8B <> -1) and (sb16.dma8 <> dma8B))
  or ((dma16B <> -1) and (sb16.dma16 <> dma16B)) then begin
     err := 3;
     exit;
  end;
end;






Procedure InitSbDma (bufptr:pointer;
                     TransferLength:word;
                     sixteenbit:boolean;
                     cmd:byte);
var
  LinearAddr     : longint;
  BufOffset      : word;
  dma : byte;
  DMA_Port_Mask, DMA_Port_Clear, DMA_Port_Mode : word;
begin
  LinearAddr := longint(seg(bufptr^))*16 + ofs(bufptr^);
  if sixteenbit then begin
     BufOffset := (LinearAddr div 2) mod 65536;
     dma := dma16;
  end
  else begin
     BufOffset := LinearAddr mod 65536;
     dma := dma8;
  end;

  if sixteenbit then begin
     DMA_Port_Mask := DMA_Port_Mask16;
     DMA_Port_Clear:= DMA_Port_Clear16;
     DMA_Port_Mode := DMA_Port_Mode16;
  end
  else begin
     DMA_Port_Mask := DMA_Port_Mask8;
     DMA_Port_Clear:= DMA_Port_Clear8;
     DMA_Port_Mode := DMA_Port_Mode8;
  end;
  port[DMA_Port_Mask] := 4 + (dma mod 4); { disable dma channel }
  port[DMA_Port_Clear]:= 0;
  port[DMA_Port_Mode] := cmd + (dma mod 4);
  port[DMA_Adr_Port  [dma]] := lo(BufOffset);
  port[DMA_Adr_Port  [dma]] := hi(BufOffset);
  port[DMA_Count_Port[dma]] := lo(TransferLength-1);
  port[DMA_Count_Port[dma]] := hi(TransferLength-1);
  port[DMA_Page_Port [dma]] := LinearAddr div 65536;
  port[DMA_Port_Mask] := (dma mod 4); { enable dma channel }
end;


Procedure InitSbDmaRead (bufptr:pointer;
                         transferlength:word;
                         sixteenbit:boolean);
begin
  InitSbDma (bufptr, transferlength, sixteenbit, DMA_Cmd_Read);
end;


Procedure InitSbDmaWrite (bufptr:pointer;
                          transferlength:word;
                          sixteenbit:boolean);
begin
  InitSbDma (bufptr, transferlength, sixteenbit, DMA_Cmd_Write);
end;



BEGIN
  SetParEnv;
END.
