UNIT IRQ;


INTERFACE

const
  Irq2Int : array [0..15] of byte = (
          { IRQ0  }  $8,  { timer (55ms intervals, 18.2 per second) }
          { IRQ1  }  $9,  { keyboard service required }
          { IRQ2  }  $A,  { slave 8259 or EGA/VGA vertical retrace }
          { IRQ3  }  $B,  { COM2 or COM4 }
          { IRQ4  }  $C,  { COM1 or COM3 }
          { IRQ5  }  $D,  { fixed disk or LPT2 }
          { IRQ6  }  $E,  { floppy disk }
          { IRQ7  }  $F,  { LPT1 }
          { IRQ8  } $70,  { real time clock }
          { IRQ9  } $71,  { software redirected to IRQ2 }
          { IRQ10 } $72,  { reserved }
          { IRQ11 } $73,  { reserved }
          { IRQ12 } $74,  { mouse }
          { IRQ13 } $75,  { numeric coprocessor error }
          { IRQ14 } $76,  { fixed disk controller }
          { IRQ15 } $77); { second fixed disk controller }


procedure EnableIRQ  (irq:byte);
procedure DisableIRQ (irq:byte);
function  IrqEnabled (irq:byte) : boolean;

procedure EnableTimerIrq;
procedure DisableTimerIrq;

procedure HookIRQ    (irq:byte; proc:pointer);
procedure UnkookIRQ  (irq:byte);


const
  PicPort1 = $21;
  PicPort2 = $A1;

procedure ResetInterruptController (irq:byte);
{ must be called before returning from interrupt procedure }
{ otherwise interrupt controller will not trigger next interrupt }


IMPLEMENTATION

uses
  dos;

var
  OldInt   : array [0..15] of pointer;
  OldPort1 : byte;
  OldPort2 : byte;


function GetPicPort (irq:byte) : byte;
begin
  If irq <= 7
     then GetPicPort := PicPort1
     else GetPicPort := PicPort2;
end;


procedure EnableIRQ  (irq:byte);
var
  PicPort : byte;
  mask    : byte;
begin
  PicPort := GetPicPort (irq);
  mask := 1 shl (irq mod 8);
  mask := not mask;

  port[PicPort] := port[PicPort] and mask;
end;


procedure DisableIRQ (irq:byte);
var
  PicPort : byte;
  mask    : byte;
begin
  PicPort := GetPicPort (irq);
  mask := 1 shl (irq mod 8);

  port[PicPort] := port[PicPort] or mask;
end;


function  IrqEnabled (irq:byte) : boolean;
var
  PicPort : byte;
  mask    : byte;
begin
  PicPort := GetPicPort (irq);
  mask := 1 shl (irq mod 8);
  IrqEnabled := ((port[PicPort] and mask) = 0);
end;


procedure EnableTimerIrq;
begin
  EnableIrq (0);
end;


procedure DisableTimerIrq;
begin
  DisableIrq (0);
end;


procedure HookIRQ   (irq:byte; proc:pointer);
var
  int : byte;
begin
  if OldInt[irq] <> NIL then exit;
  int := irq2int [irq];
  GetIntVec (int, OldInt[irq]);
  SetIntVec (int, proc);
end;


procedure UnkookIRQ (irq:byte);
begin
  if OldInt[irq] = NIL then exit;
  SetIntVec (irq2int[irq], OldInt[irq]);
  OldInt[irq] := NIL;
end;


procedure ResetInterruptController (irq:byte);
begin
  if irq >= 8 then port[$A0] := $20;
  port[$20] := $20;
end;


var
  OldExit : pointer;
  i : byte;

procedure MyExit; FAR;
begin
  asm
  pushf
  cli
  end;

  ExitProc := OldExit;
  for i := 0 to 15 do
     if OldInt[i] <> NIL then SetIntVec (irq2int[i], OldInt[i]);
  port [PicPort1] := OldPort1;
  port [PicPort2] := OldPort2;

  asm
  popf
  end;
end;


BEGIN
  OldPort1 := port [PicPort1];
  OldPort2 := port [PicPort2];
  for i := 0 to 15 do OldInt[i] := NIL;
  OldExit := ExitProc;
  ExitProc := @MyExit;
END.

