
// ---------------------------------------------------------------------
//
// PictUtl.c - Picture Viewer - QuickTime for Windows
//
//             Version 1.0
//
//             (c) Copyright 1988-1994 Apple Computer, Inc. All Rights Reserved.
//
// ---------------------------------------------------------------------



// Includes
// --------
   #include <Windows.H>               // Required by Windows
   #include <qtw.h>                   // Interface to QuickTime
   #include <qtole.h>                 // Interface to qtole dll
   #include <string.h>                // Standard "C"

   #include "common.h"                // Interface to common.c

   #include "viewer.h"                // Interface to other *.c files
   #include "viewer.hr"               // Defines used in *.rc files
   #include "picture.h"               // Interface to picture window
                                      // child window processing

// Constants
// ---------
   #define  OPTIONSPROP   "PictOptionsProp"


// typedefs
// --------
   typedef struct _tagWORKOPTIONS
      {QTOLE_OPTIONSPICTURE      SaveOptions;
       LPQTOLE_OPTIONSPICTURE   lpOptions;
       HLOCAL                   hmem;
      } WORKOPTIONS;
   typedef WORKOPTIONS NEAR * NPWORKOPTIONS;


// Exported callback functions
// ----------------------------
   BOOL __export CALLBACK GetOptionsDlgProc     (HWND, UINT, WPARAM, LPARAM);

// Internal Function Declarations
// ------------------------------
   static VOID    NEAR    SetDlgForOptions       (HWND, LPQTOLE_OPTIONSPICTURE);
   static VOID    NEAR    GetCurrentDlgSettings  (HWND, LPQTOLE_OPTIONSPICTURE);
   static VOID    NEAR    SaveOptionsAsDefault   (LPQTOLE_OPTIONSPICTURE);


// Function: ViewerGetOptions - Opens the options dialog
// --------------------------------------------------------------------
// Parameters: HWND                    hwndParent     HWND of dialog parent
//             LPQTOLE_OPTIONSPICTURE  lpOptions      -> options struct
//
// Returns:    LONG                    0L if OK
// --------------------------------------------------------------------
   BOOL FAR ViewerGetOptions( HWND hwndParent, LPQTOLE_OPTIONSPICTURE lpOptions )

     {BOOL         bChangedOptions = FALSE;
      DLGPROC      lpDlgProc;

      if( lpDlgProc = (DLGPROC) MakeProcInstance
                  ( (FARPROC) GetOptionsDlgProc, ViewerQueryInstance()))
         {bChangedOptions = (BOOL) DialogBoxParam( ViewerQueryResources(),
                                 MAKEINTRESOURCE( VIEWER_DLG_OPTIONS ),
                                  hwndParent? hwndParent: GetActiveWindow(),
                                                lpDlgProc, (LPARAM) lpOptions );
          FreeProcInstance( (FARPROC) lpDlgProc );

             // Null hwndParent indicates that dlg was started by qtole message
             // with no open picture. In this case tell QTOle.dll that dialog has
             // been closed. Must do this even if options are not changed and
             // must return the same pointer to lpOptions as input argument to
             // this function. When qtole opens dialog for open picture, normal
             // closing will updata options
          if( !hwndParent )
              QTOLE_ClosedOptionsDlg( ViewerQueryOleData(),
                               (LPQTOLE_OPTIONS) lpOptions, bChangedOptions );
         }
      else
         {CommonTellUser( ViewerQueryResources(),
                                 VIEWER_STRING_NOMEMORY, NULL, MB_OK );
         }

      return bChangedOptions;
     }


// Function: GetOptionsDlgProc - Get options dialog proc
// --------------------------------------------------------------------
// Parameters: As required by Microsoft Windows
//
// Returns:    As required by Microsoft Windows
// --------------------------------------------------------------------
   BOOL __export CALLBACK GetOptionsDlgProc
                ( HWND hdlg, UINT message, WPARAM wParam, LPARAM lParam )

     {NPWORKOPTIONS           pwoWorkOptions;
      HLOCAL                  hmem;
      BOOL                    bOptionsChanged;
      BOOL                    bChecked;
      LPQTOLE_OPTIONSPICTURE  lpOptions;


      switch( message )
         {case WM_INITDIALOG:
             lpOptions = (LPQTOLE_OPTIONSPICTURE) lParam;

             hmem = LocalAlloc( LHND, sizeof( WORKOPTIONS ));
             pwoWorkOptions = (NPWORKOPTIONS) LocalLock( hmem );
             pwoWorkOptions->hmem = hmem;
             pwoWorkOptions->SaveOptions = *lpOptions;
             pwoWorkOptions->lpOptions   = lpOptions;

             SetProp( hdlg, OPTIONSPROP, (HANDLE) pwoWorkOptions );

             SetDlgForOptions( hdlg, pwoWorkOptions->lpOptions );

             SetDlgItemText( hdlg, EDIT_OPTIONS_CAPTION,
                                    pwoWorkOptions->lpOptions->szCaption );
             return TRUE;

          case WM_COMMAND:
             switch( wParam  )
                {case IDOK:
                     pwoWorkOptions = (NPWORKOPTIONS) GetProp( hdlg, OPTIONSPROP );
                     GetCurrentDlgSettings( hdlg, pwoWorkOptions->lpOptions );

                     bOptionsChanged
                         = (_fmemcmp( pwoWorkOptions->lpOptions,
                                       &pwoWorkOptions->SaveOptions,
                                         sizeof( QTOLE_OPTIONSPICTURE )) != 0 );

                     hmem = pwoWorkOptions->hmem;
                     LocalUnlock( hmem );
                     LocalFree( hmem );
                     RemoveProp( hdlg, OPTIONSPROP );

                     EndDialog( hdlg, bOptionsChanged );
                     return TRUE;

                 case IDCANCEL:
                     pwoWorkOptions = (NPWORKOPTIONS) GetProp( hdlg, OPTIONSPROP );
                     hmem = pwoWorkOptions->hmem;
                     LocalUnlock( hmem );
                     LocalFree( hmem );
                     RemoveProp( hdlg, OPTIONSPROP );

                     EndDialog( hdlg, FALSE  );
                     return TRUE;

                 case EDIT_OPTIONS_SHOWPC:
                     bChecked = IsDlgButtonChecked( hdlg, EDIT_OPTIONS_SHOWPC );
                     EnableWindow( GetDlgItem( hdlg, EDIT_OPTIONS_CAPTION_TEXT ),
                                                                        bChecked );
                     EnableWindow( GetDlgItem( hdlg, EDIT_OPTIONS_CAPTION ),
                                                                        bChecked );
                     return TRUE;

                 case EDIT_OPTIONS_COPYICON:
                     bChecked = IsDlgButtonChecked( hdlg, EDIT_OPTIONS_COPYICON );
                     EnableWindow( GetDlgItem( hdlg,
                                           EDIT_OPTIONS_SHOWTITLEBAR ), bChecked );
                     if( !bChecked )
                         CheckDlgButton( hdlg, EDIT_OPTIONS_SHOWTITLEBAR, TRUE );
                     return TRUE;

                 case EDIT_OPTIONS_SAVEASDEF:
                     pwoWorkOptions = (NPWORKOPTIONS) GetProp( hdlg, OPTIONSPROP );
                     GetCurrentDlgSettings( hdlg, pwoWorkOptions->lpOptions );
                     SaveOptionsAsDefault( pwoWorkOptions->lpOptions );
                     return TRUE;

                 case EDIT_OPTIONS_RESTRDEF:
                     pwoWorkOptions = (NPWORKOPTIONS) GetProp( hdlg, OPTIONSPROP );
                     ViewerGetDefaultOptions( pwoWorkOptions->lpOptions );
                     SetDlgForOptions( hdlg, pwoWorkOptions->lpOptions );
                     return FALSE;

                 default:
                     return FALSE;
                }
             return TRUE;

          default:
             return FALSE;
         }

      return FALSE;
     }


// Function: QTOLEServerCallBack - Server callback used by qtole.dll to
//                                 send commands to the server
// --------------------------------------------------------------------
// Parameters: UINT                      uMessage
//             WPARAM                    wParam
//             LPARAM                    lParam
//             LPQTOLE_OPTIONSPICTURE    lpOptions
//
//
// Returns:    QTOLE_ERR        QTOLE_OK if OK
// --------------------------------------------------------------------
   QTOLE_ERR __export CALLBACK QTOLEServerCallBack
                      ( UINT uMessage, WPARAM wParam, LPARAM lParam,
                                            LPQTOLE_OPTIONSPICTURE lpOptions )

     {HWND           hwndPicture;
      NPPICTUREDATA  pPictureData;
      RECT           rcPicture;
      RECT           rcClient;
      RECT           rcIntersect;


      switch( uMessage )
         {case QTOLE_MSG_OPENOBJECT:
                 // Open a picture window
              SendMessage( ViewerQueryFrameWindow(),
                                            WM_VIEWER_CMDLINE, 0, lParam );
              hwndPicture = (HWND) SendMessage( ViewerQueryClientWindow(),
                                                     WM_MDIGETACTIVE, 0, 0L );

              if( hwndPicture &&
                       ( pPictureData = (NPPICTUREDATA) GetWindowWord( hwndPicture, 0 )))
                 {pPictureData->qtoleOptions = *lpOptions;
                  pPictureData->qtoleOptions.hwndObject = hwndPicture;
                  pPictureData->qtoleOptions.phPicture  = pPictureData->phPicture;
                  UpdatePictForOptions( hwndPicture, pPictureData, TRUE );
                 }

              break;

          case QTOLE_MSG_SHOWOBJECT:
              hwndPicture = (HWND) wParam;

              ShowWindow( ViewerQueryFrameWindow(), SW_SHOWNORMAL );

              GetWindowRect( hwndPicture, &rcPicture );
              MapWindowPoints( HWND_DESKTOP,
                             ViewerQueryClientWindow(), (LPPOINT) &rcPicture, 2 );
              GetClientRect( ViewerQueryClientWindow(), &rcClient );
              if( !IntersectRect( &rcIntersect, &rcClient, &rcPicture ))
                 {MoveWindow( hwndPicture, 0, 0,
                                         rcPicture.right - rcPicture.left,
                                         rcPicture.bottom - rcPicture.top, TRUE );
                 }

              SendMessage( ViewerQueryClientWindow(),
                                             WM_MDIACTIVATE, wParam, 0L );

              if( pPictureData = (NPPICTUREDATA) GetWindowWord( hwndPicture, 0 ))
                 {pPictureData->qtoleOptions = *lpOptions;
                  pPictureData->qtoleOptions.hwndObject = hwndPicture;
                  pPictureData->qtoleOptions.phPicture  = pPictureData->phPicture;
                  UpdatePictForOptions( hwndPicture, pPictureData, TRUE );
                 }

              break;

          case QTOLE_MSG_OPENOPTIONSDLG:
             // Post message because we are not allowed to open a dialog
             // inside an OLE method, in this case DoVerb which sends this message.
             // In fact, opening the dialog directly seems to work but this
             // is probably just lucky coincidence and can't be counted on.
             // If wParam == hwndPicture is NULL, post to frame wnd because we don't
             // open a movie wnd. If wParam != NULL, post message to hwndPicture to
             // open the dialog

              if( NULL != ( hwndPicture = (HWND) wParam ))
                 {PostMessage( hwndPicture, WM_COMMAND, VIEWER_EDIT_OPTIONS, 0L );
                 }
              else
                 {PostMessage( ViewerQueryFrameWindow(),
                                 WM_VIEWER_OLE_OPTIONSDLG, 0, (LPARAM) lpOptions );
                 }

              break;

          case QTOLE_MSG_PLAYOBJECT:
              PostMessage( ViewerQueryFrameWindow(),
                                         WM_VIEWER_OLE_PLAYOBJECT, 0, lParam );
              break;

          case QTOLE_MSG_FILEOPEN:
              PostMessage( ViewerQueryFrameWindow(),
                                           WM_COMMAND, VIEWER_FILE_OPEN, 0L );
              break;

          default:
              return QTOLE_GEN_ERROR;
         }

      return QTOLE_OK;
     }


// Function: ViewerGetDefaultOptions - Gets the default options
// --------------------------------------------------------------------
// Parameters: LPQTOLE_OPTIONSPICTURE      lpOptions
//
// Returns:    VOID
// --------------------------------------------------------------------
   VOID FAR ViewerGetDefaultOptions( LPQTOLE_OPTIONSPICTURE lpOptions )

     {char      szSection[32];
      char      szEntry[64];

     // Initial defaults
   #define SHOWPC_DEF         TRUE
   #define DRAWFRAME_DEF      TRUE
   #define USEPALETTE_DEF     TRUE
   #define COPYICON_DEF       FALSE
   #define SHOWTITLE_DEF      TRUE
   #define ZOOMHALF_DEF       FALSE
   #define ZOOMNORMAL_DEF     TRUE
   #define ZOOMDOUBLE_DEF     FALSE

      LoadString( ViewerQueryResources(),
                    VIEWER_STRING_OPTIONS_NAME, szSection, sizeof( szSection ));

      if( !szSection[0] || !LoadString( ViewerQueryResources(),
                    VIEWER_STRING_OPTIONS_SHOWPC, szEntry, sizeof( szEntry )))
          lpOptions->bShowPictureControls = SHOWPC_DEF;
      else
          lpOptions->bShowPictureControls = (BOOL)
                   GetPrivateProfileInt( szSection, szEntry, SHOWPC_DEF, QTW_PROFILE );

      if( !szSection[0] || !LoadString( ViewerQueryResources(),
                      VIEWER_STRING_OPTIONS_DRAWFRAME, szEntry, sizeof( szEntry )))
          lpOptions->bDrawWindowFrame = DRAWFRAME_DEF;
      else
          lpOptions->bDrawWindowFrame = (BOOL)
                GetPrivateProfileInt( szSection, szEntry, DRAWFRAME_DEF, QTW_PROFILE );

      if( !szSection[0] || !LoadString( ViewerQueryResources(),
                    VIEWER_STRING_OPTIONS_USEPALETTE, szEntry, sizeof( szEntry )))
          lpOptions->bUsePicturePalette = USEPALETTE_DEF;
      else
          lpOptions->bUsePicturePalette = (BOOL)
               GetPrivateProfileInt( szSection, szEntry, USEPALETTE_DEF, QTW_PROFILE );


      if( !szSection[0] || !LoadString( ViewerQueryResources(),
                    VIEWER_STRING_OPTIONS_COPYICON, szEntry, sizeof( szEntry )))
          lpOptions->bCopyIcon = COPYICON_DEF;
      else
          lpOptions->bCopyIcon = (BOOL)
                 GetPrivateProfileInt( szSection, szEntry, COPYICON_DEF, QTW_PROFILE );

      if( !szSection[0] || !LoadString( ViewerQueryResources(),
                    VIEWER_STRING_OPTIONS_SHOWTITLE, szEntry, sizeof( szEntry )))
          lpOptions->bShowTitleBar = SHOWTITLE_DEF;
      else
          lpOptions->bShowTitleBar = (BOOL)
                GetPrivateProfileInt( szSection, szEntry, SHOWTITLE_DEF, QTW_PROFILE );


      if( !szSection[0] || !LoadString( ViewerQueryResources(),
                   VIEWER_STRING_OPTIONS_ZOOMHALF, szEntry, sizeof( szEntry )))
          lpOptions->bZoomHalf = ZOOMHALF_DEF;
      else
          lpOptions->bZoomHalf = (BOOL)
                GetPrivateProfileInt( szSection, szEntry, ZOOMHALF_DEF, QTW_PROFILE );

      if( !szSection[0] || !LoadString( ViewerQueryResources(),
                   VIEWER_STRING_OPTIONS_ZOOMNORMAL, szEntry, sizeof( szEntry )))
          lpOptions->bZoomNormal = ZOOMNORMAL_DEF;
      else
          lpOptions->bZoomNormal = (BOOL)
               GetPrivateProfileInt( szSection, szEntry, ZOOMNORMAL_DEF, QTW_PROFILE );

      if( !szSection[0] || !LoadString( ViewerQueryResources(),
                   VIEWER_STRING_OPTIONS_ZOOMDOUBLE, szEntry, sizeof( szEntry )))
          lpOptions->bZoomDouble = ZOOMDOUBLE_DEF;
      else
          lpOptions->bZoomDouble = (BOOL)
               GetPrivateProfileInt( szSection, szEntry, ZOOMDOUBLE_DEF, QTW_PROFILE );

         // Now make sure no more than one option is on
         // Give priority to normal
      if( lpOptions->bZoomNormal )
         {lpOptions->bZoomHalf   = FALSE;
          lpOptions->bZoomDouble = FALSE;
         }
      else if( lpOptions->bZoomDouble )
         {lpOptions->bZoomHalf = FALSE;
         }

      return;
     }


// Function: SaveOptionsAsDefault - Caches the current options as the defaults
//                                  in qtw.ini
// --------------------------------------------------------------------
// Parameters: LPQTOLE_OPTIONSPICTURE      lpOptions
//
// Returns:    VOID
// --------------------------------------------------------------------
   static VOID NEAR SaveOptionsAsDefault( LPQTOLE_OPTIONSPICTURE lpOptions )

     {char      szSection[32];
      char      szEntry[64];

  #define szONE     "1"
  #define szZERO    "0"

      if( !LoadString( ViewerQueryResources(),
                    VIEWER_STRING_OPTIONS_NAME, szSection, sizeof( szSection )))
          return;

      if( LoadString( ViewerQueryResources(),
                    VIEWER_STRING_OPTIONS_SHOWPC, szEntry, sizeof( szEntry )))
          WritePrivateProfileString( szSection, szEntry,
                    lpOptions->bShowPictureControls? szONE: szZERO, QTW_PROFILE );

      if( LoadString( ViewerQueryResources(),
                    VIEWER_STRING_OPTIONS_DRAWFRAME, szEntry, sizeof( szEntry )))
          WritePrivateProfileString( szSection, szEntry,
                         lpOptions->bDrawWindowFrame? szONE: szZERO, QTW_PROFILE );

      if( LoadString( ViewerQueryResources(),
                    VIEWER_STRING_OPTIONS_USEPALETTE, szEntry, sizeof( szEntry )))
          WritePrivateProfileString( szSection, szEntry,
                         lpOptions->bUsePicturePalette? szONE: szZERO, QTW_PROFILE );

      if( LoadString( ViewerQueryResources(),
                    VIEWER_STRING_OPTIONS_COPYICON, szEntry, sizeof( szEntry )))
          WritePrivateProfileString( szSection, szEntry,
                           lpOptions->bCopyIcon? szONE: szZERO, QTW_PROFILE );

      if( LoadString( ViewerQueryResources(),
                    VIEWER_STRING_OPTIONS_SHOWTITLE, szEntry, sizeof( szEntry )))
          WritePrivateProfileString( szSection, szEntry,
                           lpOptions->bShowTitleBar? szONE: szZERO, QTW_PROFILE );

      if( LoadString( ViewerQueryResources(),
                   VIEWER_STRING_OPTIONS_ZOOMHALF, szEntry, sizeof( szEntry )))
          WritePrivateProfileString( szSection, szEntry,
                            lpOptions->bZoomHalf? szONE: szZERO, QTW_PROFILE );

      if( LoadString( ViewerQueryResources(),
                   VIEWER_STRING_OPTIONS_ZOOMNORMAL, szEntry, sizeof( szEntry )))
          WritePrivateProfileString( szSection, szEntry,
                           lpOptions->bZoomNormal? szONE: szZERO, QTW_PROFILE );

      if( LoadString( ViewerQueryResources(),
                   VIEWER_STRING_OPTIONS_ZOOMDOUBLE, szEntry, sizeof( szEntry )))
          WritePrivateProfileString( szSection, szEntry,
                           lpOptions->bZoomDouble? szONE: szZERO, QTW_PROFILE );

      return;
     }


// Function: SetDlgForOptions - Sets the options dialog controls according
//                              to current options
// --------------------------------------------------------------------
// Parameters: HWND                      hdlg
//             LPQTOLE_OPTIONSPICTURE    lpOptions
//
//
// Returns:    VOID
// --------------------------------------------------------------------
   static VOID NEAR SetDlgForOptions( HWND hdlg, LPQTOLE_OPTIONSPICTURE lpOptions )

     {int       idButton;

      CheckDlgButton( hdlg, EDIT_OPTIONS_SHOWPC,
                                         lpOptions->bShowPictureControls );

      EnableWindow( GetDlgItem( hdlg, EDIT_OPTIONS_CAPTION_TEXT ),
                                         lpOptions->bShowPictureControls );
      EnableWindow( GetDlgItem( hdlg, EDIT_OPTIONS_CAPTION ),
                                         lpOptions->bShowPictureControls );

      CheckDlgButton( hdlg, EDIT_OPTIONS_DRAWFRAME,
                                         lpOptions->bDrawWindowFrame );
      CheckDlgButton( hdlg, EDIT_OPTIONS_USEPALETTE,
                                         lpOptions->bUsePicturePalette );

      CheckDlgButton( hdlg, EDIT_OPTIONS_COPYICON,
                                         lpOptions->bCopyIcon );
      CheckDlgButton( hdlg, EDIT_OPTIONS_SHOWTITLEBAR,
                                         lpOptions->bShowTitleBar );
      if( !lpOptions->bCopyIcon )
          EnableWindow( GetDlgItem( hdlg, EDIT_OPTIONS_SHOWTITLEBAR ), FALSE );


      if( lpOptions->bZoomHalf )
          idButton = EDIT_OPTIONS_ZOOMHALF;
      else if( lpOptions->bZoomNormal )
          idButton = EDIT_OPTIONS_ZOOMNORMAL;
      else if( lpOptions->bZoomDouble )
          idButton = EDIT_OPTIONS_ZOOMDOUBLE;
      else
          idButton = EDIT_OPTIONS_ZOOMCURRENT;

      CheckRadioButton( hdlg, EDIT_OPTIONS_ZOOMCURRENT,
                                   EDIT_OPTIONS_ZOOMDOUBLE, idButton );

      return;
     }

// Function: GetCurrentDlgSettings - Fills the options struct with the current state
//                                   of the dialog controls
// --------------------------------------------------------------------
// Parameters: HWND                      hdlg
//             LPQTOLE_OPTIONSPICTURE    lpOptions
//
//
// Returns:    VOID
// --------------------------------------------------------------------
   static VOID NEAR GetCurrentDlgSettings
                              ( HWND hdlg, LPQTOLE_OPTIONSPICTURE lpOptions )

     {lpOptions->bShowPictureControls =
                             IsDlgButtonChecked( hdlg, EDIT_OPTIONS_SHOWPC );
      GetDlgItemText( hdlg, EDIT_OPTIONS_CAPTION,
                                       lpOptions->szCaption, MAX_TEXT_LEN );

      lpOptions->bDrawWindowFrame =
                            IsDlgButtonChecked( hdlg, EDIT_OPTIONS_DRAWFRAME );
      lpOptions->bUsePicturePalette =
                            IsDlgButtonChecked( hdlg, EDIT_OPTIONS_USEPALETTE );

      lpOptions->bCopyIcon = IsDlgButtonChecked( hdlg, EDIT_OPTIONS_COPYICON );

      lpOptions->bShowTitleBar =
                      IsDlgButtonChecked( hdlg, EDIT_OPTIONS_SHOWTITLEBAR );

      lpOptions->bZoomHalf   = IsDlgButtonChecked( hdlg, EDIT_OPTIONS_ZOOMHALF );
      lpOptions->bZoomNormal = IsDlgButtonChecked( hdlg, EDIT_OPTIONS_ZOOMNORMAL );
      lpOptions->bZoomDouble = IsDlgButtonChecked( hdlg, EDIT_OPTIONS_ZOOMDOUBLE );

      return;
     }


