
// ---------------------------------------------------------------------
//
// MovieUtl.c - Movie Player - QuickTime for Windows
//
//              Version 1.0
//
//              (c) Copyright 1988-1994 Apple Computer, Inc. All Rights Reserved.
//
// ---------------------------------------------------------------------



// Includes
// --------
#include <Windows.H> // Required by Windows
#include <qtw.h>     // Interface to QuickTime
#include <qtole.h>   // Interface to qtole dll
#include <string.h>  // Standard "C"

#include "common.h" // Interface to common.c

#include "player.h"  // Interface to other *.c files
#include "player.hr" // Defines used in *.rc files

// Constants
// ---------
#define  OPTIONSPROP   "MovieOptionsProp"


// typedefs
// --------
typedef struct _tagWORKOPTIONS {
    QTOLE_OPTIONSMOVIE      SaveOptions;
    LPQTOLE_OPTIONSMOVIE    lpOptions;
    HLOCAL                  hmem;
   } WORKOPTIONS;
typedef WORKOPTIONS NEAR * NPWORKOPTIONS;


// Exported callback functions
// ----------------------------
BOOL __export CALLBACK GetOptionsDlgProc    (HWND, UINT, WPARAM, LPARAM);

// Internal Function Declarations
// ------------------------------
static VOID    NEAR    SetDlgForOptions       (HWND, LPQTOLE_OPTIONSMOVIE);
static VOID    NEAR    GetCurrentDlgSettings  (HWND, LPQTOLE_OPTIONSMOVIE);
static VOID    NEAR    SaveOptionsAsDefault   (LPQTOLE_OPTIONSMOVIE);


// Function: PlayerGetOptions - Opens the options dialog
// --------------------------------------------------------------------
// Parameters: HWND                    hwndParent     HWND of dialog parent
//             LPQTOLE_OPTIONSMOVIE    lpOptions      -> options struct
//
// Returns:    LONG                    0L if OK
// --------------------------------------------------------------------
BOOL FAR PlayerGetOptions( HWND hwndParent, LPQTOLE_OPTIONSMOVIE lpOptions )

{
    BOOL         bChangedOptions = FALSE;
    DLGPROC      lpDlgProc;

    if( lpDlgProc = (DLGPROC) MakeProcInstance
        ( (FARPROC) GetOptionsDlgProc, PlayerQueryInstance())) {
        bChangedOptions = (BOOL) DialogBoxParam( PlayerQueryResources(),
            MAKEINTRESOURCE( PLAYER_DLG_OPTIONS ),
            hwndParent? hwndParent: GetActiveWindow(),
            lpDlgProc, (LPARAM) lpOptions );
        FreeProcInstance( (FARPROC) lpDlgProc );

        // Null hwndParent indicates that dlg was started by qtole message
        // with no open movie. In this case tell QTOle.dll that dialog has
        // been closed. Must do this even if options are not changed and
        // must return the same pointer to lpOptions as input argument to
        // this function. When qtole opens dialog for open movie, normal
        // closing will updata options
        if( !hwndParent )
            QTOLE_ClosedOptionsDlg( PlayerQueryOleData(),
            (LPQTOLE_OPTIONS) lpOptions, bChangedOptions );
    }
    else {
        CommonTellUser( PlayerQueryResources(),
            PLAYER_STRING_NOMEMORY, NULL, MB_OK );
    }

    return bChangedOptions;
}


// Function: GetOptionsDlgProc - Get options dialog proc
// --------------------------------------------------------------------
// Parameters: As required by Microsoft Windows
//
// Returns:    As required by Microsoft Windows
// --------------------------------------------------------------------
BOOL __export CALLBACK GetOptionsDlgProc
    ( HWND hdlg, UINT message, WPARAM wParam, LPARAM lParam )

{
    NPWORKOPTIONS           pwoWorkOptions;
    HLOCAL                  hmem;
    BOOL                    bOptionsChanged;
    BOOL                    bChecked;
    LPQTOLE_OPTIONSMOVIE    lpOptions;

    switch( message ) {
        case WM_INITDIALOG:
            lpOptions = (LPQTOLE_OPTIONSMOVIE) lParam;

            hmem = LocalAlloc( LHND, sizeof( WORKOPTIONS ));
            pwoWorkOptions = (NPWORKOPTIONS) LocalLock( hmem );
            pwoWorkOptions->hmem = hmem;
            pwoWorkOptions->SaveOptions = *lpOptions;
            pwoWorkOptions->lpOptions   = lpOptions;

            SetProp( hdlg, OPTIONSPROP, (HANDLE) pwoWorkOptions );

            SetDlgForOptions( hdlg, pwoWorkOptions->lpOptions );

            SetDlgItemText( hdlg, EDIT_OPTIONS_CAPTION,
                pwoWorkOptions->lpOptions->szCaption );

            return TRUE;

        case WM_COMMAND:
            switch( wParam  ) {
                case IDOK:
                    pwoWorkOptions = (NPWORKOPTIONS) GetProp( hdlg, OPTIONSPROP );
                    GetCurrentDlgSettings( hdlg, pwoWorkOptions->lpOptions );

                    bOptionsChanged
                        = (_fmemcmp( pwoWorkOptions->lpOptions,
                        &pwoWorkOptions->SaveOptions,
                        sizeof( QTOLE_OPTIONSMOVIE )) != 0 );
                    hmem = pwoWorkOptions->hmem;
                    LocalUnlock( hmem );
                    LocalFree( hmem );
                    RemoveProp( hdlg, OPTIONSPROP );

                    EndDialog( hdlg, bOptionsChanged );
                    return TRUE;

                case IDCANCEL:
                    pwoWorkOptions = (NPWORKOPTIONS) GetProp( hdlg, OPTIONSPROP );
                    hmem = pwoWorkOptions->hmem;
                    LocalUnlock( hmem );
                    LocalFree( hmem );
                    RemoveProp( hdlg, OPTIONSPROP );

                    EndDialog( hdlg, FALSE  );
                    return TRUE;

                case EDIT_OPTIONS_SHOWMC:
                    bChecked = IsDlgButtonChecked( hdlg, EDIT_OPTIONS_SHOWMC );
                    EnableWindow( GetDlgItem( hdlg, EDIT_OPTIONS_CAPTION_TEXT ),
                        bChecked );
                    EnableWindow( GetDlgItem( hdlg, EDIT_OPTIONS_CAPTION ),
                        bChecked );
                    return TRUE;

                case EDIT_OPTIONS_COPYICON:
                    bChecked = IsDlgButtonChecked( hdlg, EDIT_OPTIONS_COPYICON );
                    EnableWindow( GetDlgItem( hdlg,
                        EDIT_OPTIONS_SHOWTITLEBAR ), bChecked );
                    if( !bChecked )
                        CheckDlgButton( hdlg, EDIT_OPTIONS_SHOWTITLEBAR, TRUE );
                    return TRUE;

                case EDIT_OPTIONS_SAVEASDEF:
                    pwoWorkOptions = (NPWORKOPTIONS) GetProp( hdlg, OPTIONSPROP );
                    GetCurrentDlgSettings( hdlg, pwoWorkOptions->lpOptions );
                    SaveOptionsAsDefault( pwoWorkOptions->lpOptions );
                    return TRUE;

                case EDIT_OPTIONS_RESTRDEF:
                    pwoWorkOptions = (NPWORKOPTIONS) GetProp( hdlg, OPTIONSPROP );
                    PlayerGetDefaultOptions( pwoWorkOptions->lpOptions );
                    SetDlgForOptions( hdlg, pwoWorkOptions->lpOptions );
                    return FALSE;

                default:
                    return FALSE;
            }
            return TRUE;

        default:
            return FALSE;
    }

    return FALSE;
}


// Function: QTOLEServerCallBack - Server callback used by qtole.dll to
//                                 send commands to the server
// --------------------------------------------------------------------
// Parameters: UINT                      uMessage
//             WPARAM                    wParam
//             LPARAM                    lParam
//             LPQTOLE_OPTIONSMOVIE      lpOptions
//
//
// Returns:    QTOLE_ERR        QTOLE_OK if OK
// --------------------------------------------------------------------
QTOLE_ERR __export CALLBACK QTOLEServerCallBack
    ( UINT uMessage, WPARAM wParam, LPARAM lParam,
    LPQTOLE_OPTIONSMOVIE lpOptions )

{
    HWND           hwndMovie;
    NPMOVIEDATA    pMovieData;
    RECT           rcMovie;
    RECT           rcClient;
    RECT           rcIntersect;


    switch( uMessage ) {
        case QTOLE_MSG_OPENOBJECT:
            // Open a movie window
            SendMessage( PlayerQueryFrameWindow(),
                WM_PLAYER_CMDLINE, 0, lParam );
            hwndMovie = (HWND) SendMessage( PlayerQueryClientWindow(),
                WM_MDIGETACTIVE, 0, 0L );

            if( hwndMovie &&
                ( pMovieData = (NPMOVIEDATA) GetWindowWord( hwndMovie, 0 ))) {
                pMovieData->qtoleOptions = *lpOptions;
                pMovieData->qtoleOptions.hwndObject = hwndMovie;
                pMovieData->qtoleOptions.mMovie     = pMovieData->mMovie;
                UpdateMovieForOptions( hwndMovie, pMovieData, TRUE );
            }

            break;

        case QTOLE_MSG_SHOWOBJECT:
            hwndMovie = (HWND) wParam;

            ShowWindow( PlayerQueryFrameWindow(), SW_SHOWNORMAL );

            GetWindowRect( hwndMovie, &rcMovie );
            MapWindowPoints( HWND_DESKTOP,
                PlayerQueryClientWindow(), (LPPOINT) &rcMovie, 2 );
            GetClientRect( PlayerQueryClientWindow(), &rcClient );
            if( !IntersectRect( &rcIntersect, &rcClient, &rcMovie )) {
                MoveWindow( hwndMovie, 0, 0,
                    rcMovie.right - rcMovie.left,
                    rcMovie.bottom - rcMovie.top, TRUE );
            }

            SendMessage( PlayerQueryClientWindow(),
                WM_MDIACTIVATE, wParam, 0L );

            if( pMovieData = (NPMOVIEDATA) GetWindowWord( hwndMovie, 0 )) {
                pMovieData->qtoleOptions = *lpOptions;
                pMovieData->qtoleOptions.hwndObject = hwndMovie;
                pMovieData->qtoleOptions.mMovie     = pMovieData->mMovie;
                UpdateMovieForOptions( hwndMovie, pMovieData, TRUE );
            }

            break;

        case QTOLE_MSG_OPENOPTIONSDLG:
            // Post message because we are not allowed to open a dialog
            // inside an OLE method, in this case DoVerb which sends this message.
            // In fact, opening the dialog directly seems to work but this
            // is probably just lucky coincidence and can't be counted on.
            // If wParam == hwndMovie is NULL, post to frame wnd because we don't
            // open a movie wnd. If wParam != NULL, post message to hwndMovie to
            // open the dialog

            if( NULL != ( hwndMovie = (HWND) wParam )) {
                PostMessage( hwndMovie, WM_COMMAND, PLAYER_EDIT_OPTIONS, 0L );
            }
            else {
                PostMessage( PlayerQueryFrameWindow(),
                    WM_PLAYER_OLE_OPTIONSDLG, 0, (LPARAM) lpOptions );
            }

            break;

        case QTOLE_MSG_PLAYOBJECT:
            PostMessage( PlayerQueryFrameWindow(),
                WM_PLAYER_OLE_PLAYOBJECT, 0, lParam );
            break;

        case QTOLE_MSG_FILEOPEN:
            PostMessage( PlayerQueryFrameWindow(),
                WM_COMMAND, PLAYER_FILE_OPEN, 0L );
            break;

        default:
            return QTOLE_GEN_ERROR;
    }

    return QTOLE_OK;
}


// Function: PlayerGetDefaultOptions - Gets the default options
// --------------------------------------------------------------------
// Parameters: LPQTOLE_OPTIONSMOVIE      lpOptions
//
// Returns:    VOID
// --------------------------------------------------------------------
VOID FAR PlayerGetDefaultOptions( LPQTOLE_OPTIONSMOVIE lpOptions )

{
    char      szSection[32];
    char      szEntry[64];

    // Initial defaults
    #define SHOWMC_DEF         TRUE
    #define SELONLY_DEF        FALSE
    #define DRAWFRAME_DEF      TRUE
    #define USEPALETTE_DEF     TRUE
    #define COPYICON_DEF       FALSE
    #define SHOWTITLE_DEF      TRUE
    #define CURRENTFRAME_DEF   TRUE
    #define LOOP_DEF           FALSE
    #define LOOPPALIND_DEF     FALSE
    #define SIZEHALF_DEF       FALSE
    #define SIZENORMAL_DEF     TRUE
    #define SIZEDOUBLE_DEF     FALSE

    LoadString( PlayerQueryResources(),
        PLAYER_STRING_OPTIONS_NAME, szSection, sizeof( szSection ));

    if( !szSection[0] || !LoadString( PlayerQueryResources(),
        PLAYER_STRING_OPTIONS_SHOWMC, szEntry, sizeof( szEntry )))
        lpOptions->bShowMovieController = SHOWMC_DEF;
    else
        lpOptions->bShowMovieController = (BOOL)
        GetPrivateProfileInt( szSection, szEntry, SHOWMC_DEF, QTW_PROFILE );

    if( !szSection[0] || !LoadString( PlayerQueryResources(),
        PLAYER_STRING_OPTIONS_PLAYSELONLY, szEntry, sizeof( szEntry )))
        lpOptions->bPlaySelectionOnly = SELONLY_DEF;
    else
        lpOptions->bPlaySelectionOnly = (BOOL)
        GetPrivateProfileInt( szSection, szEntry, SELONLY_DEF, QTW_PROFILE );

    if( lpOptions->bSoundOnlyMovie )
        lpOptions->bDrawWindowFrame = FALSE;
    else if( !szSection[0] || !LoadString( PlayerQueryResources(),
        PLAYER_STRING_OPTIONS_DRAWFRAME, szEntry, sizeof( szEntry )))
        lpOptions->bDrawWindowFrame = DRAWFRAME_DEF;
    else
        lpOptions->bDrawWindowFrame = (BOOL)
        GetPrivateProfileInt( szSection, szEntry, DRAWFRAME_DEF, QTW_PROFILE );

    if( lpOptions->bSoundOnlyMovie )
        lpOptions->bUseMoviePalette = FALSE;
    else if( !szSection[0] || !LoadString( PlayerQueryResources(),
        PLAYER_STRING_OPTIONS_USEPALETTE, szEntry, sizeof( szEntry )))
        lpOptions->bUseMoviePalette = USEPALETTE_DEF;
    else
        lpOptions->bUseMoviePalette = (BOOL)
        GetPrivateProfileInt( szSection, szEntry, USEPALETTE_DEF, QTW_PROFILE );


    if( lpOptions->bSoundOnlyMovie )
        lpOptions->bCopyIcon = TRUE;
    else if( !szSection[0] || !LoadString( PlayerQueryResources(),
        PLAYER_STRING_OPTIONS_COPYICON, szEntry, sizeof( szEntry )))
        lpOptions->bCopyIcon = COPYICON_DEF;
    else
        lpOptions->bCopyIcon = (BOOL)
        GetPrivateProfileInt( szSection, szEntry, COPYICON_DEF, QTW_PROFILE );


    if( !szSection[0] || !LoadString( PlayerQueryResources(),
        PLAYER_STRING_OPTIONS_SHOWTITLE, szEntry, sizeof( szEntry )))
        lpOptions->bShowTitleBar = SHOWTITLE_DEF;
    else
        lpOptions->bShowTitleBar = (BOOL)
        GetPrivateProfileInt( szSection, szEntry, SHOWTITLE_DEF, QTW_PROFILE );


    if( lpOptions->bSoundOnlyMovie )                     // This actually doesn't do anything
        lpOptions->bCopyCurrentFrame = CURRENTFRAME_DEF; // for sound only movies
    else if( !szSection[0] || !LoadString( PlayerQueryResources(),
        PLAYER_STRING_OPTIONS_CURRENTFRAME, szEntry, sizeof( szEntry )))
        lpOptions->bCopyCurrentFrame = CURRENTFRAME_DEF;
    else
        lpOptions->bCopyCurrentFrame = (BOOL)
        GetPrivateProfileInt( szSection, szEntry, CURRENTFRAME_DEF, QTW_PROFILE );


    if( !szSection[0] || !LoadString( PlayerQueryResources(),
        PLAYER_STRING_OPTIONS_LOOP, szEntry, sizeof( szEntry )))
        lpOptions->bLoop = LOOP_DEF;
    else
        lpOptions->bLoop = (BOOL)
        GetPrivateProfileInt( szSection, szEntry, LOOP_DEF, QTW_PROFILE );

    if( !szSection[0] || !LoadString( PlayerQueryResources(),
        PLAYER_STRING_OPTIONS_LOOPPALIND, szEntry, sizeof( szEntry )))
        lpOptions->bLoopPalindrome = LOOPPALIND_DEF;
    else
        lpOptions->bLoopPalindrome = (BOOL)
        GetPrivateProfileInt( szSection, szEntry, LOOPPALIND_DEF, QTW_PROFILE );

    // Now make sure only one option is on
    if( lpOptions->bLoop ) {
        lpOptions->bLoopPalindrome = FALSE;
    }

    if( !szSection[0] || !LoadString( PlayerQueryResources(),
        PLAYER_STRING_OPTIONS_SIZEHALF, szEntry, sizeof( szEntry )))
        lpOptions->bSizeHalf = SIZEHALF_DEF;
    else
        lpOptions->bSizeHalf = (BOOL)
        GetPrivateProfileInt( szSection, szEntry, SIZEHALF_DEF, QTW_PROFILE );

    if( !szSection[0] || !LoadString( PlayerQueryResources(),
        PLAYER_STRING_OPTIONS_SIZENORMAL, szEntry, sizeof( szEntry )))
        lpOptions->bSizeNormal = SIZENORMAL_DEF;
    else
        lpOptions->bSizeNormal = (BOOL)
        GetPrivateProfileInt( szSection, szEntry, SIZENORMAL_DEF, QTW_PROFILE );

    if( !szSection[0] || !LoadString( PlayerQueryResources(),
        PLAYER_STRING_OPTIONS_SIZEDOUBLE, szEntry, sizeof( szEntry )))
        lpOptions->bSizeDouble = SIZEDOUBLE_DEF;
    else
        lpOptions->bSizeDouble = (BOOL)
        GetPrivateProfileInt( szSection, szEntry, SIZEDOUBLE_DEF, QTW_PROFILE );

    // Now make sure no more than one option is on
    // Give priority to normal
    if( lpOptions->bSizeNormal ) {
        lpOptions->bSizeHalf   = FALSE;
        lpOptions->bSizeDouble = FALSE;
    }
    else if( lpOptions->bSizeDouble ) {
        lpOptions->bSizeHalf = FALSE;
    }

    return;
}


// Function: SaveOptionsAsDefault - Caches the current options as the defaults
//                                  in qtw.ini
// --------------------------------------------------------------------
// Parameters: LPQTOLE_OPTIONSMOVIE      lpOptions
//
// Returns:    VOID
// --------------------------------------------------------------------
static VOID NEAR SaveOptionsAsDefault( LPQTOLE_OPTIONSMOVIE lpOptions )

{
    char      szSection[32];
    char      szEntry[64];

    #define szONE     "1"
    #define szZERO    "0"

    if( !LoadString( PlayerQueryResources(),
        PLAYER_STRING_OPTIONS_NAME, szSection, sizeof( szSection )))
        return;

    if( LoadString( PlayerQueryResources(),
        PLAYER_STRING_OPTIONS_SHOWMC, szEntry, sizeof( szEntry )))
        WritePrivateProfileString( szSection, szEntry,
        lpOptions->bShowMovieController? szONE: szZERO, QTW_PROFILE );

    if( LoadString( PlayerQueryResources(),
        PLAYER_STRING_OPTIONS_PLAYSELONLY, szEntry, sizeof( szEntry )))
        WritePrivateProfileString( szSection, szEntry,
        lpOptions->bPlaySelectionOnly? szONE: szZERO, QTW_PROFILE );

    if( !lpOptions->bSoundOnlyMovie ) {
        if( LoadString( PlayerQueryResources(),
            PLAYER_STRING_OPTIONS_DRAWFRAME, szEntry, sizeof( szEntry )))
            WritePrivateProfileString( szSection, szEntry,
            lpOptions->bDrawWindowFrame? szONE: szZERO, QTW_PROFILE );

        if( LoadString( PlayerQueryResources(),
            PLAYER_STRING_OPTIONS_USEPALETTE, szEntry, sizeof( szEntry )))
            WritePrivateProfileString( szSection, szEntry,
            lpOptions->bUseMoviePalette? szONE: szZERO, QTW_PROFILE );

        if( LoadString( PlayerQueryResources(),
            PLAYER_STRING_OPTIONS_COPYICON, szEntry, sizeof( szEntry )))
            WritePrivateProfileString( szSection, szEntry,
            lpOptions->bCopyIcon? szONE: szZERO, QTW_PROFILE );
    }

    if( LoadString( PlayerQueryResources(),
        PLAYER_STRING_OPTIONS_SHOWTITLE, szEntry, sizeof( szEntry )))
        WritePrivateProfileString( szSection, szEntry,
        lpOptions->bShowTitleBar? szONE: szZERO, QTW_PROFILE );

    if( !lpOptions->bSoundOnlyMovie &&
        LoadString( PlayerQueryResources(),
        PLAYER_STRING_OPTIONS_CURRENTFRAME, szEntry, sizeof( szEntry )))
        WritePrivateProfileString( szSection, szEntry,
        lpOptions->bCopyCurrentFrame? szONE: szZERO, QTW_PROFILE );


    if( LoadString( PlayerQueryResources(),
        PLAYER_STRING_OPTIONS_LOOP, szEntry, sizeof( szEntry )))
        WritePrivateProfileString( szSection, szEntry,
        lpOptions->bLoop? szONE: szZERO, QTW_PROFILE );

    if( LoadString( PlayerQueryResources(),
        PLAYER_STRING_OPTIONS_LOOPPALIND, szEntry, sizeof( szEntry )))
        WritePrivateProfileString( szSection, szEntry,
        lpOptions->bLoopPalindrome? szONE: szZERO, QTW_PROFILE );

    if( !lpOptions->bSoundOnlyMovie ) {
        if( LoadString( PlayerQueryResources(),
            PLAYER_STRING_OPTIONS_SIZEHALF, szEntry, sizeof( szEntry )))
            WritePrivateProfileString( szSection, szEntry,
            lpOptions->bSizeHalf? szONE: szZERO, QTW_PROFILE );

        if( LoadString( PlayerQueryResources(),
            PLAYER_STRING_OPTIONS_SIZENORMAL, szEntry, sizeof( szEntry )))
            WritePrivateProfileString( szSection, szEntry,
            lpOptions->bSizeNormal? szONE: szZERO, QTW_PROFILE );

        if( LoadString( PlayerQueryResources(),
            PLAYER_STRING_OPTIONS_SIZEDOUBLE, szEntry, sizeof( szEntry )))
            WritePrivateProfileString( szSection, szEntry,
            lpOptions->bSizeDouble? szONE: szZERO, QTW_PROFILE );
    }

    return;
}


// Function: SetDlgForOptions - Sets the options dialog controls according
//                              to current options
// --------------------------------------------------------------------
// Parameters: HWND                      hdlg
//             LPQTOLE_OPTIONSMOVIE      lpOptions
//
//
// Returns:    VOID
// --------------------------------------------------------------------
static VOID NEAR SetDlgForOptions( HWND hdlg, LPQTOLE_OPTIONSMOVIE lpOptions )

{
    BOOL      bCopyIcon;
    int       idButton;


    CheckDlgButton( hdlg, EDIT_OPTIONS_SHOWMC,
        lpOptions->bShowMovieController );

    EnableWindow( GetDlgItem( hdlg, EDIT_OPTIONS_CAPTION_TEXT ),
        lpOptions->bShowMovieController );
    EnableWindow( GetDlgItem( hdlg, EDIT_OPTIONS_CAPTION ),
        lpOptions->bShowMovieController );

    CheckDlgButton( hdlg, EDIT_OPTIONS_PLAYSELONLY,
        lpOptions->bPlaySelectionOnly );
    if( !lpOptions->bSoundOnlyMovie ) {
        CheckDlgButton( hdlg, EDIT_OPTIONS_DRAWFRAME,
            lpOptions->bDrawWindowFrame );
        CheckDlgButton( hdlg, EDIT_OPTIONS_USEPALETTE,
            lpOptions->bUseMoviePalette );
    }
    else {
        EnableWindow( GetDlgItem( hdlg, EDIT_OPTIONS_DRAWFRAME ), FALSE );
        EnableWindow( GetDlgItem( hdlg, EDIT_OPTIONS_USEPALETTE ), FALSE );
    }


    if( !lpOptions->bSoundOnlyMovie ) {
        bCopyIcon = lpOptions->bCopyIcon;
    }
    else {
        bCopyIcon = TRUE;
        EnableWindow( GetDlgItem( hdlg, EDIT_OPTIONS_COPYICON ), FALSE );
    }

    CheckDlgButton( hdlg, EDIT_OPTIONS_COPYICON, bCopyIcon );
    CheckDlgButton( hdlg, EDIT_OPTIONS_SHOWTITLEBAR, lpOptions->bShowTitleBar );
    if( !bCopyIcon )
        EnableWindow( GetDlgItem( hdlg, EDIT_OPTIONS_SHOWTITLEBAR ), FALSE );


    if( !lpOptions->bSoundOnlyMovie ) {
        if( lpOptions->bCopyCurrentFrame )
            idButton = EDIT_OPTIONS_COPYCURRENT;
        else
            idButton = EDIT_OPTIONS_COPYPOSTER;

        CheckRadioButton( hdlg, EDIT_OPTIONS_COPYCURRENT,
            EDIT_OPTIONS_COPYPOSTER, idButton );
    }
    else {
        EnableWindow( GetDlgItem( hdlg, EDIT_OPTIONS_COPYCURRENT ), FALSE );
        EnableWindow( GetDlgItem( hdlg, EDIT_OPTIONS_COPYPOSTER ),  FALSE );
    }

    if( lpOptions->bLoop )
        idButton = EDIT_OPTIONS_LOOP;
    else if( lpOptions->bLoopPalindrome )
        idButton = EDIT_OPTIONS_PALINDROME;
    else
        idButton = EDIT_OPTIONS_STOPATEND;

    CheckRadioButton( hdlg, EDIT_OPTIONS_STOPATEND,
        EDIT_OPTIONS_PALINDROME, idButton );

    if( !lpOptions->bSoundOnlyMovie ) {
        if( lpOptions->bSizeHalf )
            idButton = EDIT_OPTIONS_SIZEHALF;
        else if( lpOptions->bSizeNormal )
            idButton = EDIT_OPTIONS_SIZENORMAL;
        else if( lpOptions->bSizeDouble )
            idButton = EDIT_OPTIONS_SIZEDOUBLE;
        else
            idButton = EDIT_OPTIONS_SIZECURRENT;

        CheckRadioButton( hdlg, EDIT_OPTIONS_SIZECURRENT,
            EDIT_OPTIONS_SIZEDOUBLE, idButton );
    }
    else {
        EnableWindow( GetDlgItem( hdlg, EDIT_OPTIONS_SIZECURRENT ), FALSE );
        EnableWindow( GetDlgItem( hdlg, EDIT_OPTIONS_SIZEHALF ),    FALSE );
        EnableWindow( GetDlgItem( hdlg, EDIT_OPTIONS_SIZENORMAL ),  FALSE );
        EnableWindow( GetDlgItem( hdlg, EDIT_OPTIONS_SIZEDOUBLE ),  FALSE );
        CheckRadioButton( hdlg, EDIT_OPTIONS_SIZECURRENT,
            EDIT_OPTIONS_SIZEDOUBLE, EDIT_OPTIONS_SIZECURRENT );
    }

    return;
}


// Function: GetCurrentDlgSettings - Fills the options struct with the current state
//                                   of the dialog controls
// --------------------------------------------------------------------
// Parameters: HWND                      hdlg
//             LPQTOLE_OPTIONSMOVIE      lpOptions
//
//
// Returns:    VOID
// --------------------------------------------------------------------
static VOID NEAR GetCurrentDlgSettings( HWND hdlg, LPQTOLE_OPTIONSMOVIE lpOptions )

{
    lpOptions->bShowMovieController =
        IsDlgButtonChecked( hdlg, EDIT_OPTIONS_SHOWMC );
    GetDlgItemText( hdlg, EDIT_OPTIONS_CAPTION,
        lpOptions->szCaption, MAX_TEXT_LEN );
    lpOptions->bPlaySelectionOnly =
        IsDlgButtonChecked( hdlg, EDIT_OPTIONS_PLAYSELONLY );

    if( !lpOptions->bSoundOnlyMovie ) {
        lpOptions->bDrawWindowFrame =
            IsDlgButtonChecked( hdlg, EDIT_OPTIONS_DRAWFRAME );
        lpOptions->bUseMoviePalette =
            IsDlgButtonChecked( hdlg, EDIT_OPTIONS_USEPALETTE );

        lpOptions->bCopyIcon =
            IsDlgButtonChecked( hdlg, EDIT_OPTIONS_COPYICON );
        lpOptions->bShowTitleBar =
            IsDlgButtonChecked( hdlg, EDIT_OPTIONS_SHOWTITLEBAR );
    }
    else {
        lpOptions->bDrawWindowFrame = FALSE;
        lpOptions->bUseMoviePalette = FALSE;
        lpOptions->bCopyIcon        = TRUE;
        lpOptions->bShowTitleBar    = IsDlgButtonChecked( hdlg,
            EDIT_OPTIONS_SHOWTITLEBAR );
    }

    lpOptions->bLoop = IsDlgButtonChecked( hdlg, EDIT_OPTIONS_LOOP );
    lpOptions->bLoopPalindrome =
        IsDlgButtonChecked( hdlg, EDIT_OPTIONS_PALINDROME );

    if( !lpOptions->bSoundOnlyMovie ) {
        lpOptions->bSizeHalf =
            IsDlgButtonChecked( hdlg, EDIT_OPTIONS_SIZEHALF );
        lpOptions->bSizeNormal =
            IsDlgButtonChecked( hdlg, EDIT_OPTIONS_SIZENORMAL );
        lpOptions->bSizeDouble =
            IsDlgButtonChecked( hdlg, EDIT_OPTIONS_SIZEDOUBLE );

        lpOptions->bCopyCurrentFrame =
            IsDlgButtonChecked( hdlg, EDIT_OPTIONS_COPYCURRENT );
    }
    else {
        lpOptions->bSizeHalf         = FALSE;
        lpOptions->bSizeNormal       = FALSE;
        lpOptions->bSizeDouble       = FALSE;
        lpOptions->bCopyCurrentFrame = TRUE;
    }

    return;
}
