/*
********************************************************************************
**
** File: datah.cpp
**
** Description:
**
**  Set tab stops to 4.
**
**  Data Handler component for QuickTime for Windows.
**
** Routines:
**
**  Routines enclosed in [brackets] exist, but are unsupported.
**
**  DataHOpen();                        - component manager open call
**  DataHClose();                       - component manager close call
**  DataHCanDo();                       - component manager cando call
**  DataHVersion();                     - component manager version call
**  DataHGetData();                     - immediate data read
**  [DataHPutData();]                   - data write
**  [DataHFlushData();]                 - flush write buffers
**  [DataHOpenForWrite();]              - open for write access
**  [DataHCloseForWrite();]             - close for write access
**  DataHOpenForRead();                 - open for read access
**  DataHCloseForRead();                - close for read access
**  DataHSetDataRef();                  - set data reference
**  DataHGetDataRef();                  - get data reference
**  DataHCompareDataRef();              - compare data references
**  DataHTask();                        - provide background time
**  DataHScheduleData();                - schedule advance read
**  DataHFinishData();                  - complete scheduled reads
**  DataHFlushCache();                  - flush cache buffers
**  DataHResolveDataRef();              - resolve data reference
**  DataHGetFileSize();                 - return file size
**  DataHCanUseDataRef();               - check if data ref can be used
**  DataHGetVolumeList();               - return list of volumes supported
**  [DataHWrite();]                     - write data
**  [DataHPreextend();]                 - extend file
**  [DataHSetFileSize();]               - set file size
**  DataHGetFreeSpace();                - get device free space
**  [DataHCreateFile();]                - create file
**  DataHGetPreferredBlockSize();       - get preferred block size
**  DataHGetDeviceIndex();              - get unique device index
**  DataHGetScheduleAheadTime();        - get preferred advance read time
**  DataHPlaybackHints();               - provide data ref playback hints
**  DataHSetOSFileReference();          - set mmioH as data reference
**  DataHGetOSFileReference();          - get references from SetOSFile...
**
********************************************************************************
*/

// Windows header files
#include <windows.h>
#include <windowsx.h>
#include <mmsystem.h>

// dos headers
#include <direct.h>
#include <dos.h>

// Compiler header files
#include <string.h>
#include <ctype.h>
#include <stdlib.h>
#include <malloc.h>
#include <io.h>

// Application header files
#define INTERNAL_DHLR
#include "datahp.h"

// four character type

#define ostypeOTHR QTFOURCC('o','t','h','r')

// macros
#define DATAHPARM(x,y) GetPrivateProfileInt("Data Handler",x,y,QTW_PROFILE)

// prototypes for external functions
DWORD _cdecl DataHEntry(void);

// Global data
ComponentDescription cdTable =              // one component in this DLL
  { ostypeDHLR                              // ostypeComponentType
  , ostypeHNDL                              // ostypeComponentSubType
  , ostypeOTHR                              // ostypeComponentManufacturer
  , 0                                       // dwComponentFlags
  , 0                                       // dwComponentFlagsMask
  , ( ComponentRoutine) DataHEntry          // crEntryPoint
  , 0                                       // hrsrcName
  , 0                                       // hrsrcInfo
  , 0                                       // hrsrcIcon
  } ;

// EJECT PAGE 
/*
********************************************************************************
**
** Name: DataHOpen()
**
** Description:
**
**  Opens an instance of the component.
**
**  The general data handler initialization is done here, so that any memory
**  used will not be allocated until an instance of the data handler is actually
**  opened.
**
********************************************************************************
*/
ComponentResult _cdecl DataHOpen( STKOFF_CMP so, ComponentInstance ci)
{
    void far *storageH, far *globalH;
    DataHInstanceStoragePtr storage;
    DataHGlobalStoragePtr globals;
    
    // allocate the cross-instance globals
    globalH = (void far *)GetComponentRefcon(ci);
    if(globalH == NULL)
    {
        // allocate global storage
        globalH = (void far *)GlobalAlloc(GMEM_ZEROINIT, sizeof(DataHGlobalStorage));
        if(globalH == NULL)
            return(insufficientMemory);

        // set the refcon so that we know we have been initialized
        SetComponentRefcon(ci, (long)globalH);
    }
        
    globals = (DataHGlobalStoragePtr)GlobalLock((HGLOBAL)LOWORD(globalH));
    if(globals == NULL)
    {
        GlobalFree((HGLOBAL)LOWORD(globalH));
        return(insufficientMemory);
    }

    // if first instance ever
    if (!globals->mmioInitialized) {
        globals->mmioInitialized = TRUE;    // mmio requires no initialization
    }

    // allocate instance storage, all members zero init.
    storageH = (void far *)GlobalAlloc(GMEM_ZEROINIT, sizeof(DataHInstanceStorage));
    if(storageH == NULL)
        return(insufficientMemory);
    storage = (DataHInstanceStoragePtr)GlobalLock((HGLOBAL)LOWORD(storageH));
    if(storage == NULL)
    {
        GlobalUnlock((HGLOBAL)LOWORD(globalH));
        GlobalFree((HGLOBAL)LOWORD(storageH));
        return(insufficientMemory);
    }

    // init storage fields
    storage->ci = ci;
    storage->mmioH = mmioINVALID;       // invalid HMMIO handle indicates non-open session
        
    // set storage for this component
    SetComponentInstanceStorage(ci, (LPVOID)storageH);
    
    // done
    GlobalUnlock((HGLOBAL)LOWORD(storageH));
    GlobalUnlock((HGLOBAL)LOWORD(globalH));
    return(noErr);
}

// EJECT PAGE 
/*
********************************************************************************
**
** Name: DataHClose()
**
** Description:
**
**  Closes an instance of the component.
**
********************************************************************************
*/
ComponentResult _cdecl DataHClose( STKOFF_CMP so, ComponentInstance ci)  
{
    void far *storageH, far *globalH;
    DataHInstanceStoragePtr storage;

    // locate instance storage
    storageH = GetComponentInstanceStorage(ci);
    if(storageH != NULL)
    {
        storage = (DataHInstanceStoragePtr)GlobalLock((HGLOBAL)LOWORD(storageH));
        if(storage != NULL)
        {
            // close the mmio session if open
            if (storage->mmioH != mmioINVALID) {
                mmioClose(storage->mmioH,0);
            }
            // release memory allocated for filename
            if(storage->fileName)
                GlobalFree((HGLOBAL)storage->fileName);
            
            // release memory for instance storage
            GlobalUnlock((HGLOBAL)LOWORD(storageH));
            GlobalFree((HGLOBAL)LOWORD(storageH));
        }
    }
    
    // release global storage if this is the last instance of the component
    if(CountComponentInstances(ci) == 1)
    {
        globalH = (void far *)GetComponentRefcon(ci);
        if(globalH)
        {
            // do a global release of i/o subsystem here
            
            GlobalFree((HGLOBAL)LOWORD(globalH));
            SetComponentRefcon(ci,NULL);
        }
    }   
    return(noErr);
}

// EJECT PAGE 
/*
********************************************************************************
**
** Name: DataHCanDo()
**
** Description:
**
**  Returns TRUE if a call is supported.
**
********************************************************************************
*/
ComponentResult _cdecl DataHCanDo( STKOFF_CMP so, long lFunctionSelector)
{
    switch (lFunctionSelector)
    {
        /* standard component manager calls */
        case kDataVersionSelector:
        case kDataCanDoSelector:
        case kDataCloseSelector:
        case kDataOpenSelector:
        
        /* data handler calls */
        case kDataGetDataSelector:
        case kDataOpenForReadSelector:
        case kDataCloseForReadSelector:
        case kDataSetDatRefSelector:
        case kDataGetDataRefSelector:
        case kDataCompareDataRefSelector:
        case kDataTaskSelector:
        case kDataScheduleDataSelector:
        case kDataFinishDataSelector:
        case kDataFlushCacheSelector:
        case kDataResolveDataRefSelector:
        case kDataGetFileSizeSelector:
        case kDataCanUseDataRefSelector:
        case kDataGetVolumeListSelector:
        case kDataPlaybackHintsSelector:
        case kDataSetOSFileReferenceSelector:
        case kDataGetOSFileReferenceSelector:
            return(TRUE);
            break;
        default:
            return(FALSE);
            break;
    }
    
    return(FALSE);
}

// EJECT PAGE 
/*
********************************************************************************
**
** Name: DataHVersion()
**
** Description:
**
**  Returns version number of the component.
**
********************************************************************************
*/
ComponentResult _cdecl DataHVersion( STKOFF_CMP so, ComponentInstance ci)
{
    return(kDataHVersion);
}

// EJECT PAGE 
/*
********************************************************************************
**
** Name: DataHGetData()
**
** Description:
**
**  Synchronous data read.
**
********************************************************************************
*/
ComponentResult _cdecl DataHGetData (DHLR_FPARM1
    Handle h,       // handle to destination of data
    long hOffset,   // offset into handle to place data
    long offset,    // offset within file of data to read
    long size)      // amount of data to read
{
    void FAR *storageH = instanceStorage;
    DataHInstanceStoragePtr storage;
    Ptr dataPtr;
    OSErr oserr = noErr;

    // lock the storage
    storage = (DataHInstanceStoragePtr)GlobalLock((HGLOBAL)LOWORD(storageH));
    if(storage == NULL)
        return(insufficientMemory);

    // verify that we have an open file
    if(storage->mmioH  == mmioINVALID)
    {
        GlobalUnlock((HGLOBAL)LOWORD(storageH));
        return(dataNotOpenForRead);
    }

    // check dest
    if (hOffset + size > GetHandleSize(h)) {
        GlobalUnlock((HGLOBAL)LOWORD(storageH));
        return(insufficientMemory);
    }

    // lock and deref the user handle
    HLock(h);
    dataPtr = (Ptr)DereferenceHandle(h);
    if(dataPtr == NULL)
    {
        HUnlock(h);
        GlobalUnlock((HGLOBAL)LOWORD(storageH));
        return(insufficientMemory);
    }

    // build the pointer to the destination area
    dataPtr += hOffset;

    if (storage->mmioH != mmioINVALID) {
        if (offset != mmioSeek(storage->mmioH,offset,SEEK_SET)) {
            oserr = readErr;
        } else {
            if (size != mmioRead(storage->mmioH,(LPSTR)dataPtr,size)) {               
                oserr = readErr;
            }
        }
    }
    
    // done
    HUnlock(h);

    GlobalUnlock((HGLOBAL)LOWORD(storageH));
    return(oserr);
}

// EJECT PAGE 
/*
********************************************************************************
**
** Name: DataHPutData()
**
** Description:
**
**  Synchronous write.  Not supported.
**
********************************************************************************
*/
ComponentResult _cdecl DataHPutData (DHLR_FPARM1
    Handle h,
    long hOffset,
    long *offset,
    long size)
{
    return(badComponentSelector);
}

// EJECT PAGE 
/*
********************************************************************************
**
** Name: DataHFlushData()
**
** Description:
**
**  Flush unwritten data.  Not supported.
**
********************************************************************************
*/
ComponentResult _cdecl DataHFlushData (DHLR_FPARM2)
{
    return(badComponentSelector);
}

// EJECT PAGE 
/*
********************************************************************************
**
** Name: DataHOpenForWrite()
**
** Description:
**
**  Open data reference for write access.  Not supported.
**
********************************************************************************
*/
ComponentResult _cdecl DataHOpenForWrite (DHLR_FPARM2)
{
    return(badComponentSelector);
}

// EJECT PAGE 
/*
********************************************************************************
**
** Name: DataHCloseForWrite()
**
** Description:
**
**  Close data reference that has been opened for write access.  Not supported.
**
********************************************************************************
*/
ComponentResult _cdecl DataHCloseForWrite (DHLR_FPARM2)
{
    return(badComponentSelector);
}

// EJECT PAGE 
/*
********************************************************************************
**
** Name: DataHOpenForRead()
**
** Description:
**
**  Open data reference for read access.
**
********************************************************************************
*/
ComponentResult _cdecl DataHOpenForRead (DHLR_FPARM2)
{
    void far *storageH = instanceStorage;
    DataHInstanceStoragePtr storage;
    char FAR *fileName;
    OSErr oserr = noErr;
    
    storage = (DataHInstanceStoragePtr)GlobalLock((HGLOBAL)LOWORD(storageH));
    if(storage == NULL)
        return(insufficientMemory);

    // must have a data reference
    if(storage->fileName == NULL)
    {
        GlobalUnlock((HGLOBAL)LOWORD(storageH));
        return(dataNoDataRef);
    }
    
    // dereference the handle to the file name
    fileName = (char FAR *)GlobalLock(storage->fileName);
    if(fileName == NULL)
    {
        GlobalUnlock((HGLOBAL)LOWORD(storageH));
        return(invalidDataRef);     
    }
    
    // open the movie file
    storage->mmioH = mmioOpen(fileName, NULL, MMIO_READ | MMIO_DENYWRITE);
    if (storage->mmioH == mmioINVALID)
    {
        oserr = invalidDataRef;
    }
    
    // unlock handles
    GlobalUnlock(storage->fileName);
    GlobalUnlock((HGLOBAL)LOWORD(storageH));
    
    return(oserr);
}

// EJECT PAGE 
/*
********************************************************************************
**
** Name: DataHCloseForRead()
**
** Description:
**
**  Close data reference that has been opened for read access.
**
********************************************************************************
*/
ComponentResult _cdecl DataHCloseForRead (DHLR_FPARM2)
{
    void far *storageH = instanceStorage;
    DataHInstanceStoragePtr storage;
    OSErr oserr = noErr;
    
    // locate instance storage
    storage = (DataHInstanceStoragePtr)GlobalLock((HGLOBAL)LOWORD(storageH));
    if(storage == NULL)
        return(insufficientMemory);

    // close the file
    if (storage->mmioH != mmioINVALID) {
        mmioClose(storage->mmioH,0);
        storage->mmioH = mmioINVALID;
        if (storage->fileName) {
            GlobalFree(storage->fileName);
            storage->fileName = NULL;
        }
    } else {
       oserr = dataNotOpenForRead;
    }
        
    GlobalUnlock((HGLOBAL)LOWORD(storageH));
    return(oserr);
}

// EJECT PAGE 
/*
********************************************************************************
**
** Name: DataHSetDataRef()
**
** Description:
**
**  Set data reference for this component instance.  In this mmio sample for QTW
**  the data reference is the file path.  The input data reference is assumed to
**  be a locked Handle.
**
**  this sample implementation assumes 0x00 terminated strings
**
********************************************************************************
*/
ComponentResult _cdecl DataHSetDataRef (DHLR_FPARM1
    Handle dataRef)
{
    char far *strIn;
    char far *myStr;    
    int len;
    HLOCAL mem;
    void far *storageH = instanceStorage;
    DataHInstanceStoragePtr storage;

    // locate instance storage
    storage = (DataHInstanceStoragePtr)GlobalLock((HGLOBAL)LOWORD(storageH));
    if(storage == NULL)
        return(insufficientMemory);
        
    // release any existing allocated memory
    // some data handler implementations may want to close the i/o system
    // when the data ref changes, then prep for the new dataref
    if(storage->fileName)
    {
        GlobalFree(storage->fileName);
        storage->fileName = NULL;
    }
    
    // deref the path
    strIn = (char far *)DereferenceHandle(dataRef);
    if(strIn == NULL)
    {
        GlobalUnlock((HGLOBAL)LOWORD(storageH));
        return(invalidUserDataHandle);
    }
    len = _fstrlen(strIn);
    
    // allocate the memory, allow extra for ASCIIZ
    mem = GlobalAlloc(GMEM_ZEROINIT, len+1);
    if(mem == NULL)
    {
        GlobalUnlock((HGLOBAL)LOWORD(storageH));
        return(insufficientMemory);
    }
        
    // copy data
    myStr = (char FAR *)GlobalLock(mem);
    if (myStr == NULL) {
        GlobalUnlock((HGLOBAL)LOWORD(storageH));
        return(insufficientMemory);
    }
    
    while (*myStr++ = *strIn++) /* empty body */;
    GlobalUnlock(mem);
        
    // store handle
    storage->fileName = mem;
    
    // done
    GlobalUnlock((HGLOBAL)LOWORD(storageH));
    return(noErr);
}

// EJECT PAGE 
/*
********************************************************************************
**
** Name: DataHGetDataRef()
**
** Description:
**
**  Return data reference for this component instance.
**
** this sample implementation assumes 0x00 terminated strings
**
********************************************************************************
*/
ComponentResult _cdecl DataHGetDataRef (DHLR_FPARM1
    Handle *dataRef)
{
    char far *strOut;
    char far *myStr;    
    int len;
    Handle memH;
    void far *storageH = instanceStorage;
    DataHInstanceStoragePtr storage;
    
    if (dataRef == NULL) {
        return(paramErr);
    }

    // locate instance storage
    storage = (DataHInstanceStoragePtr)GlobalLock((HGLOBAL)LOWORD(storageH));
    if(storage == NULL)
        return(insufficientMemory);
    
    // deref the path
    myStr = (char far *)GlobalLock(storage->fileName);
    if (myStr == NULL) {
        GlobalUnlock((HGLOBAL)LOWORD(storageH));
        return(insufficientMemory);
    }
    
    len = _fstrlen(myStr);
    
    // allocate the memory, allow for asciiz
    memH = NewHandle(len+1);
    if(memH == NULL)
    {
        GlobalUnlock(storage->fileName);
        GlobalUnlock((HGLOBAL)LOWORD(storageH));
        return(insufficientMemory);
    }
        
    // copy data
    HLock(memH);
    strOut = (char far *)DereferenceHandle(memH);
    if(strOut == NULL)
    {
        GlobalUnlock(storage->fileName);
        GlobalUnlock((HGLOBAL)LOWORD(storageH));
        return(insufficientMemory);
    }
    while (*strOut++ = *myStr++) /* empty body */;
    HUnlock(memH);
        
    // store handle
    *dataRef = memH;
    
    // done
    GlobalUnlock((HGLOBAL)LOWORD(storageH));
    return(noErr);
}

// EJECT PAGE 
/*
********************************************************************************
**
** Name: DataHCompareDataRef()
**
** Description:
**
**  Compare provided data reference with the one established for this
**  component instance.
**
** this sample implementation tests for bitwise equality of 0x00 terminated strings
**
********************************************************************************
*/
ComponentResult _cdecl DataHCompareDataRef (DHLR_FPARM1
    Handle dataRef,
    Boolean *equal)
{
    char far *inStr;
    char far *myStr;    
    int myLen, inLen;
    void far *storageH = instanceStorage;
    DataHInstanceStoragePtr storage;
    
    if (equal == NULL) {
        return(paramErr);
    }

    // locate instance storage
    storage = (DataHInstanceStoragePtr)GlobalLock((HGLOBAL)LOWORD(storageH));
    if(storage == NULL)
        return(insufficientMemory);
    
    // deref the paths
    myStr = (char FAR *)GlobalLock(storage->fileName);
    if(myStr == NULL)
    {
        GlobalUnlock((HGLOBAL)LOWORD(storageH));
        return(insufficientMemory);
    }
    myLen = _fstrlen(myStr);
    
    HLock(dataRef);
    inStr = (char far *)DereferenceHandle(dataRef);
    if(inStr == NULL)
    {
        HUnlock(dataRef);
        GlobalUnlock(storage->fileName);
        GlobalUnlock((HGLOBAL)LOWORD(storageH));
        return(invalidUserDataHandle);
    }
    inLen = _fstrlen(inStr);
    
    // assume equal
    *equal = TRUE;
    
    // check lengths
    if(myLen != inLen)
    {
        *equal = FALSE;
    }
    else
    {
        // lengths are same, so check contents
        for(int i = 0; i < myLen; i++)
        {
            if(toupper(*myStr) != toupper(*inStr))
            {
                *equal = FALSE;
                break;
            }
            myStr++;
            inStr++;
        }
    }
            
    // done
    HUnlock(dataRef);
    GlobalUnlock(storage->fileName);
    GlobalUnlock((HGLOBAL)LOWORD(storageH));
    return(noErr);
}

// EJECT PAGE 
/*
********************************************************************************
**
** Name: DataHTask()
**
** Description:
**
**  Provides time slices for the data handler to perform background operations.
**
********************************************************************************
*/
ComponentResult _cdecl DataHTask (DHLR_FPARM2)
{
    void far *storageH = instanceStorage;
    DataHInstanceStoragePtr storage;
    OSErr err = noErr;
    
    // locate instance storage
    storage = (DataHInstanceStoragePtr)GlobalLock((HGLOBAL)LOWORD(storageH));
    if(storage == NULL)
        return(insufficientMemory);
        
    // do appropriate background processing here.
    // do not use synchronous (blocking) i/o calls if at all avoidable

    // done
    GlobalUnlock((HGLOBAL)LOWORD(storageH));
    return(err);
}

// EJECT PAGE 
/*
********************************************************************************
**
** Name: DataHScheduleData()
**
** Description:
**
**  Async or synchronous read operation.
**  completion routine not call under all error conditions in this implementation
**
********************************************************************************
*/
ComponentResult _cdecl DataHScheduleData (DHLR_FPARM1
    Ptr placeToPutDataPtr,
    long fileOffset,
    long dataSize,
    long refCon,
    DataHSchedulePtr scheduleRec,
    DHCompleteProc completionRtn)
{
    void far *storageH = instanceStorage;
    DataHInstanceStoragePtr storage;
    OSErr oserr = noErr;
    
    // lock instance storage
    storage = (DataHInstanceStoragePtr)GlobalLock((HGLOBAL)LOWORD(storageH));
    if(storage == NULL)
        return(insufficientMemory);

    // verify that we have an open file
    if(storage->mmioH  == mmioINVALID)
    {
        GlobalUnlock((HGLOBAL)LOWORD(storageH));
        return(dataNotOpenForRead);
    }

    // read the data right now...synchronous read
    if(scheduleRec == NULL)
    {
        if (storage->mmioH != mmioINVALID) {
            if (fileOffset == mmioSeek(storage->mmioH,fileOffset,SEEK_SET)) {
                if (dataSize != mmioRead(storage->mmioH, (char far *)placeToPutDataPtr, dataSize)) {
                    oserr = readErr;
                }
            } else {
                oserr = readErr;
            }
        }
        
        // do the callback
        if(completionRtn)
            (*completionRtn)(placeToPutDataPtr, refCon, oserr);
    }
    else // the request is asynchronous
    {
        // add to queue and initiate at appropriate time
        /* null body */;
    }

    // done
    GlobalUnlock((HGLOBAL)LOWORD(storageH));
    return(oserr);
}

// EJECT PAGE 
/*
********************************************************************************
**
** Name: DataHFinishData()
**
** Description:
**
**  Complete specified async read requests.
**
********************************************************************************
*/
ComponentResult _cdecl DataHFinishData (DHLR_FPARM1
    Ptr placeToPutDataPtr,
    Boolean cancel )
{
    return(badComponentSelector);
}

// EJECT PAGE 
/*
********************************************************************************
**
** Name: DataHFlushCache()
**
** Description:
**
**  Flush read caches.
**
********************************************************************************
*/
ComponentResult _cdecl DataHFlushCache (DHLR_FPARM2)
{
    return(noErr);
}

// EJECT PAGE 
/*
********************************************************************************
**
** Name: DataHResolveDataRef()
**
** Description:
**
**  Resolves a data reference.  No operation is performed, as a data reference
**  under this sample datahandler for QTW is a path name.
**
********************************************************************************
*/
ComponentResult _cdecl DataHResolveDataRef (DHLR_FPARM1
    Handle dataRef,
    Boolean *wasChanged,
    Boolean userInterfaceAllowed)
{
    if (wasChanged != NULL) {
        *wasChanged = FALSE;
    }
    return(noErr);
}

// EJECT PAGE 
/*
********************************************************************************
**
** Name: DataHGetFileSize()
**
** Description:
**
**  Return size of data reference.  The data reference must already be open
**  for this call to work.
**
********************************************************************************
*/
ComponentResult _cdecl DataHGetFileSize (DHLR_FPARM1
    long *fileSize)
{
    void far *storageH = instanceStorage;
    DataHInstanceStoragePtr storage;
    LONG lCurrentPos;
    OSErr oserr = noErr;
    
    if (fileSize == NULL) {
        return paramErr;
    }
    
    // locate instance storage
    storage = (DataHInstanceStoragePtr)GlobalLock((HGLOBAL)LOWORD(storageH));
    if(storage == NULL)
        return(insufficientMemory);

    // file must be open
    // verify that we have an open file
    if(storage->mmioH  != mmioINVALID) {
        // save current pos
        lCurrentPos = mmioSeek(storage->mmioH,0,SEEK_CUR);
        if (lCurrentPos >= 0) {
            // get ending pos
            *fileSize = mmioSeek(storage->mmioH,0,SEEK_END);
            if (*fileSize >= 0) {
                // reset to prior position
                if (mmioSeek(storage->mmioH,lCurrentPos,SEEK_SET) < 0) {
                    oserr = readErr;
                }
            } else {
                oserr = readErr;
            }
        } else {
            oserr = readErr;
        }
    } else {
        oserr = dataNotOpenForRead;
    }
    
    // done
    GlobalUnlock((HGLOBAL)LOWORD(storageH));
    return(oserr);
}

// EJECT PAGE 
/*
********************************************************************************
**
** Name: DataHCanUseDataRef()
**
** Description:
**
**  Return flags indicating the ability for the data handler to access the
**  data reference.
**
**  Currently only reading of files is supported.
**
********************************************************************************
*/
ComponentResult _cdecl DataHCanUseDataRef (DHLR_FPARM1
    Handle dataRef,
    DataHUseFlags *useFlags)
{
    if (useFlags != NULL) {
        *useFlags = kDataHCanRead;
    }
    return(noErr);
}

// EJECT PAGE 
/*
********************************************************************************
**
** Name: DataHGetVolumeList()
**
** Description:
**
**  Return a list of volumes supported by this data handler.
**
********************************************************************************
*/
ComponentResult _cdecl DataHGetVolumeList (DHLR_FPARM1
    DataHVolumeList *volumeList)
{
    return(badComponentSelector);
}

// EJECT PAGE 
/*
********************************************************************************
**
** Name: DataHWrite()
**
** Description:
**
**  Write data to data reference.  Not supported.
**
********************************************************************************
*/
ComponentResult _cdecl DataHWrite (DHLR_FPARM1
    Ptr data,
    long offset,
    long size,
    DHCompleteProc completion,
    long refcon)
{
    return(badComponentSelector);
}

// EJECT PAGE 
/*
********************************************************************************
**
** Name: DataHPreextend()
**
** Description:
**
**  Preextend the data reference.  Not supported.
**
********************************************************************************
*/
ComponentResult _cdecl DataHPreextend (DHLR_FPARM1
    long maxToAdd,
    long *spaceAdded)
{
    return(badComponentSelector);
}

// EJECT PAGE 
/*
********************************************************************************
**
** Name: DataHSetFileSize()
**
** Description:
**
**  Change file size of data reference.  Not supported.
**
********************************************************************************
*/
ComponentResult _cdecl DataHSetFileSize (DHLR_FPARM1
    long fileSize)
{
    return(badComponentSelector);
}

// EJECT PAGE 
/*
********************************************************************************
**
** Name: DataHGetFreeSpace()
**
** Description:
**
**  Return amount of free space on the device holding the data reference.
**  Not supported.
**
********************************************************************************
*/
ComponentResult _cdecl DataHGetFreeSpace (DHLR_FPARM1
    unsigned long *freeSize)
{
    void far *storageH = instanceStorage;
    DataHInstanceStoragePtr storage;
    char FAR *fileName;
    int deviceIndex;
    OSErr err;
    
    // locate instance storage
    storage = (DataHInstanceStoragePtr)GlobalLock((HGLOBAL)LOWORD(storageH));
    if(storage == NULL)
        return(insufficientMemory);

    // must have a data reference
    if(storage->fileName == NULL)
    {
        GlobalUnlock((HGLOBAL)LOWORD(storageH));
        return(dataNoDataRef);
    }
    
    // dereference the handle to the file name
    fileName = (char FAR *)GlobalLock(storage->fileName);
    if(fileName == NULL)
    {
        GlobalUnlock((HGLOBAL)LOWORD(storageH));
        return(invalidDataRef);     
    }
    
    // get the index, use the drive letter.  If the caller didn't
    // specify a drive letter, then get the current drive.  The
    // index is 0 for A:, 1 for B:, ...
    if(fileName[1] != ':')
        deviceIndex = _getdrive();
    else
        deviceIndex = tolower(fileName[0])-'a' +1;
    
    // get the free space on the device
    _diskfree_t *pfreeSpace = (_diskfree_t *)malloc(sizeof _diskfree_t);
    if (pfreeSpace != NULL) {
        if(_dos_getdiskfree(deviceIndex, pfreeSpace))
        {
            err = couldNotResolveDataRef;
        }
        else
        {
            // calculate the free space from the dos info returned
            if (freeSize != NULL) {
                *freeSize = pfreeSpace->avail_clusters * pfreeSpace->sectors_per_cluster * pfreeSpace->bytes_per_sector;
            }
            err = noErr;    
        }
        free(pfreeSpace);
    } else {
        err = insufficientMemory;
    }
                
    // unlock handles
    GlobalUnlock(storage->fileName);
    GlobalUnlock((HGLOBAL)LOWORD(storageH));
    
    // done
    return(err);
}

// EJECT PAGE 
/*
********************************************************************************
**
** Name: DataHCreateFile()
**
** Description:
**
**  Create a file corresponding to the data reference.  Not supported.
**
********************************************************************************
*/
ComponentResult _cdecl DataHCreateFile (DHLR_FPARM1
    OSType creator,
    Boolean deleteExisting)
{
    return(badComponentSelector);
}

// EJECT PAGE 
/*
********************************************************************************
**
** Name: DataHGetPreferredBlockSize()
**
** Description:
**
**  Return the block size, in bytes, the data handler prefers to work with.
**
********************************************************************************
*/
ComponentResult _cdecl DataHGetPreferredBlockSize (DHLR_FPARM1
    long *blockSize)
{
    // we are happiest with blocks of this size
    if (blockSize != NULL) {
        *blockSize = MMIO_DEFAULTBUFFER;
    }
    return(noErr);
}

// EJECT PAGE 
/*
********************************************************************************
**
** Name: DataHGetDeviceIndex()
**
** Description:
**
**  Return a unique identifier for the device the data reference resides on.
**
********************************************************************************
*/
ComponentResult _cdecl DataHGetDeviceIndex (DHLR_FPARM1
    long *deviceIndex)
{
    void far *storageH = instanceStorage;
    DataHInstanceStoragePtr storage;
    char far *fileName;
    
    storage = (DataHInstanceStoragePtr)GlobalLock((HGLOBAL)LOWORD(storageH));
    if(storage == NULL)
        return(insufficientMemory);

    // must have a data reference
    if(storage->fileName == NULL)
    {
        GlobalUnlock((HGLOBAL)LOWORD(storageH));
        return(dataNoDataRef);
    }
    
    // dereference the handle to the file name
    fileName = (char far *)GlobalLock(storage->fileName);
    if(fileName == NULL)
    {
        GlobalUnlock((HGLOBAL)LOWORD(storageH));
        return(invalidDataRef);     
    }
    
    // get the index, use the drive letter.  If the caller didn't
    // specify a drive letter, then get the current drive.  The
    // index is 1 for A:, 2 for B:, ...
    if (deviceIndex != NULL) {
        if(fileName[1] != ':')
            *deviceIndex = _getdrive();
        else
            *deviceIndex = (long)(tolower(fileName[0])-'a'+1);
    }            
    // unlock handles
    GlobalUnlock(storage->fileName);
    GlobalUnlock((HGLOBAL)LOWORD(storageH));
    
    // done
    return(noErr);
}

// EJECT PAGE 
/*
********************************************************************************
**
** Name: DataHGetScheduleAheadTime()
**
** Description:
**
**  Return schedule ahead time that the data handler prefers.  Currently
**  an arbitrary value is returned.
**
********************************************************************************
*/
ComponentResult _cdecl DataHGetScheduleAheadTime (DHLR_FPARM1
    long *millisecs)
{
    // 1/2 second, close approximation to old sound media read ahead
    if (millisecs != NULL) {
        *millisecs = 500;
    }
    return(noErr);
}

// EJECT PAGE 
/*
********************************************************************************
**
** Name: DataHPlaybackHints()
**
** Description:
**
**  Provides hints about the data reference being accessed.  This function
**  may be called at any time, even during movie playback if the user has
**  made edits to the movie.
**
********************************************************************************
*/
ComponentResult _cdecl DataHPlaybackHints (DHLR_FPARM1
    long flags,
    unsigned long minFileOffset,
    unsigned long maxFileOffset,
    long bytesPerSecond)
{
    void far *storageH = instanceStorage;
    DataHInstanceStoragePtr storage;

    // locate instance storage
    storage = (DataHInstanceStoragePtr)GlobalLock((HGLOBAL)LOWORD(storageH));
    if(storage == NULL)
        return(insufficientMemory);

    // store playback hints
    storage->minFileOffset = minFileOffset;
    storage->maxFileOffset = maxFileOffset;
    storage->bytesPerSecond = bytesPerSecond;
    
    // done 
    GlobalUnlock((HGLOBAL)LOWORD(storageH));
    return(noErr);
}

// EJECT PAGE 
/*
********************************************************************************
**
** Name: DataHSetOSFileReference()
**
** Description:
**
**  Set the file reference directly to an already open file.  This call
**  exists because NewMovieFromDataFork() is only given an HFILE to work with,
**  and MS-Windows can't backup to the filename from just the HFILE
**  without using lots of undocumented stuff.
**
********************************************************************************
*/
ComponentResult _cdecl DataHSetOSFileReference (DHLR_FPARM1
    long fileRef,
    long flags)
{
    void far *storageH = instanceStorage;
    DataHInstanceStoragePtr storage;
    OSErr oserr = noErr;
    MMIOINFO info;
    int newFile;

    // locate instance storage
    storage = (DataHInstanceStoragePtr)GlobalLock((HGLOBAL)LOWORD(storageH));
    if(storage == NULL)
        return(insufficientMemory);

    // make sure we are not already open via the regular data reference
    if(storage->mmioH != mmioINVALID) {
        GlobalUnlock((HGLOBAL)LOWORD(storageH));
        return(invalidDataRef);
    }
    
    newFile = _dup((int)fileRef);   // note that HFILE and C runtime file handle are polymorphic
    
    // assign the file reference
    // cast from an HFILE to an HMMIO
    storage->lFlags = flags;
    _fmemset (&info, 0, sizeof info);
    info.adwInfo[0] = (DWORD)newFile;
    storage->mmioH = mmioOpen (NULL, &info, MMIO_READ | MMIO_DENYWRITE);

    if(storage->mmioH == mmioINVALID) {
        oserr = invalidDataRef;
    }
    
    // done 
    GlobalUnlock((HGLOBAL)LOWORD(storageH));
    return(oserr);
}

// EJECT PAGE 
/*
********************************************************************************
**
** Name: DataHGetOSFileReference()
**
** Description:
**
**  Returns the file reference set by SetOSFileReference();
**
********************************************************************************
*/
ComponentResult _cdecl DataHGetOSFileReference (DHLR_FPARM1
    long far *fileRef,
    long far *flags)
{
    void far *storageH = instanceStorage;
    DataHInstanceStoragePtr storage;
    OSErr oserr = noErr;

    if ((fileRef == NULL) || (flags == NULL)) {
        return(paramErr);
    }

    // locate instance storage
    storage = (DataHInstanceStoragePtr)GlobalLock((HGLOBAL)LOWORD(storageH));
    if(storage == NULL)
        return(insufficientMemory);

    if (storage->mmioH != mmioINVALID) {
        *fileRef = (int)storage->mmioH;
        *flags = storage->lFlags;
    } else {
        oserr = invalidDataRef;
    }
    // done 
    GlobalUnlock((HGLOBAL)LOWORD(storageH));
    return(oserr);
}
