
// ---------------------------------------------------------------------
//
// EACHPICT.C   - QuickTime for Windows
//
//                Version 1.0
//
//                (c) Copyright 1988-1994 Apple Computer, Inc. All Rights Reserved.
//
// ---------------------------------------------------------------------


#include <windows.h>
#include <mmsystem.h>
#include <dlgs.h>
#include <commdlg.h>
#include <qtw.h>
#include <dos.h>
#include <direct.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys\types.h>
#include <sys\stat.h>

#define ID_TIMER 1
#define SHOWTIME 2500

typedef UINT (CALLBACK * OPENDLGHOOKPROC) (HWND, UINT, WPARAM, LPARAM);

long FAR PASCAL __export WndProc (HWND, UINT, WPARAM, LPARAM);
UINT CALLBACK __export OpenDlgHookProc (HWND, UINT, WPARAM, LPARAM);
static VOID NEAR MyListAllPictures (char*, char*);
static VOID NEAR MyDrawNextPicture  (VOID);
static BOOL NEAR MySoundEffect  (VOID);

static HINSTANCE hInstance;
static HPALETTE hpalPicture;
static PicFile pfPicture;
static PicHandle phPicture;
static FILE* hList;
static HWND hWnd;
static RECT rcDraw;
static char szAppName[] = "EACHPICT";


// WinMain
// ---------------------------------------------------------------------
int PASCAL WinMain (HINSTANCE hInst, HINSTANCE hPrevInstance,
   LPSTR lpszCmdParam, int nCmdShow)
{
    MSG msg;
    WNDCLASS wndclass;
    char szDrive[_MAX_DRIVE];
    char szDir[_MAX_DIR];
    char szFileTitle[_MAX_PATH];
    char szParm[_MAX_PATH];
    char* pszTemp;
    OPENFILENAME ofn;

    hInstance = hInst;

    // If parameter is missing, use the standard Open Dialog
    // to get the root directory

    if (lstrlen (lpszCmdParam) == 0) {
        _getcwd (szDir, sizeof (szDir));
        szParm[0] = '\0';
        memset (&ofn, 0, sizeof(OPENFILENAME));
        ofn.lStructSize = sizeof(OPENFILENAME);
        ofn.hwndOwner = 0;
        ofn.lpstrFilter = "Pictures\0all\0\0";
        ofn.nFilterIndex = 1;
        ofn.lpstrFile = szParm;
        ofn.nMaxFile = sizeof(szParm);
        ofn.lpstrFileTitle = szFileTitle;
        ofn.nMaxFileTitle = sizeof(szFileTitle);
        ofn.lpstrInitialDir = szDir;
        ofn.lpstrTitle = "Draw Each Picture";
        ofn.Flags = OFN_HIDEREADONLY | OFN_ENABLEHOOK | OFN_ENABLETEMPLATE;
        ofn.hInstance = hInstance;
        ofn.lpTemplateName = "CustomDirectorySelect";
        ofn.lpfnHook = (OPENDLGHOOKPROC) MakeProcInstance ((FARPROC) OpenDlgHookProc, hInstance);
        if (GetOpenFileName (&ofn) == 0)
            return 0;
        FreeProcInstance ((FARPROC) ofn.lpfnHook);
        _strupr (szParm);
    }

    // Otherwise, treat the command line parameter as the
    // root directory. We append a dummy file name to assist in the
    // subsequent parsing

    else {
        lstrcpy (szParm, lpszCmdParam);
        strcat (szParm, "\\crap");
        _strupr (szParm);
    }

    // Establish links to QuickTime for Windows

    if (QTInitialize (NULL)) {
        MessageBox (NULL, "QTInitialize failure", szAppName, MB_OK);
        return 0;
    }

    // Register and create main window

    if (!hPrevInstance) {
        wndclass.style = CS_DBLCLKS | CS_HREDRAW | CS_VREDRAW;
        wndclass.lpfnWndProc = WndProc;
        wndclass.cbClsExtra = 0;
        wndclass.cbWndExtra = 0;
        wndclass.hInstance = hInstance;
        wndclass.hIcon = LoadIcon (NULL,IDI_APPLICATION);
        wndclass.hCursor = LoadCursor (NULL, IDC_ARROW);
        wndclass.hbrBackground = (HBRUSH) (COLOR_WINDOW + 1);
        wndclass.lpszMenuName = NULL;
        wndclass.lpszClassName = szAppName;
        if (!RegisterClass (&wndclass)) {
            MessageBox (NULL, "RegisterClass failure", szAppName, MB_OK);
            return 0;
        }
    }

    hWnd = CreateWindow(szAppName, szAppName, WS_CAPTION | WS_SYSMENU |
        WS_CLIPCHILDREN | WS_OVERLAPPED, CW_USEDEFAULT, CW_USEDEFAULT,
        CW_USEDEFAULT, CW_USEDEFAULT, NULL, NULL, hInstance, NULL);

    if (hWnd == NULL) {
        MessageBox (NULL, "CreateWindow failure", szAppName, MB_OK);
        return 0;
    }

    // Make a list of all the pictures

    pszTemp = _tempnam ("c:\\", "each");
    hList = fopen (pszTemp, "w+");
    _splitpath (szParm, szDrive, szDir, NULL, NULL);
    if (strlen (szDrive) > 0)
        _chdrive ((int) (szDrive[0] - '@'));
    if (strlen (szDir) > 1)
        szDir[strlen(szDir) - 1] = 0;
    MyListAllPictures (szDir, ".PIC");
    MyListAllPictures (szDir, ".JPG");

    // Open the list of pictures and process the first in the list

    rewind (hList);
    MyDrawNextPicture ();

    // Make the main window visible

    ShowWindow (hWnd, nCmdShow);
    UpdateWindow (hWnd);

    // Draw the picture

    while (GetMessage (&msg, NULL, 0, 0)) {
        TranslateMessage (&msg);
        DispatchMessage (&msg);
    }

    // Close and delete list of pictures

    fclose (hList);
    remove (pszTemp);
    free (pszTemp);

    // Free any old palette

    if (hpalPicture)
        DeleteObject (hpalPicture);

    // Free any old picture

    if (phPicture)
        DisposePicture (phPicture);

    // Cut the connections to QuickTime for Windows

    QTTerminate ();

    // Return to Windows

    return msg.wParam;
}


// WndProc
// ---------------------------------------------------------------------
long FAR PASCAL __export WndProc (HWND hWnd, UINT message, WPARAM wParam,
   LPARAM lParam)
{
    PAINTSTRUCT ps;
    HPALETTE hpalOld;
    ImageDescription id;
    DIBHandle dh;
    OSErr oserr;

    // Process the windows message

    switch (message) {

        // Paint this picture

        case WM_PAINT:
            if (!BeginPaint (hWnd, &ps))
                return 0;
            if (phPicture) {
                if (hpalPicture) {
                    hpalOld = SelectPalette (ps.hdc, hpalPicture, 0);
                    RealizePalette (ps.hdc);
                    SelectPalette (ps.hdc, hpalOld, 0);
                }
                if ((dh = PictureToDIB (phPicture)) == NULL)
                    DebugOutput (DBF_WARNING, "PictureToDIB failure");
                else GlobalFree ((HGLOBAL) dh);
                if ((oserr = GetPictureInfo (phPicture, &id)) != noErr)
                    DebugOutput (DBF_WARNING, "GetPictureInfo failure %ld", oserr);
                if ((oserr = DrawPicture (ps.hdc, phPicture, &rcDraw, NULL)) != noErr)
                    DebugOutput (DBF_WARNING, "DrawPicture failure %ld", oserr);
                if ((oserr = GetPictureInfo (phPicture, &id)) != noErr)
                    DebugOutput (DBF_WARNING, "GetPictureInfo failure %ld", oserr);
            }
            EndPaint (hWnd, &ps);
            return 0;

        // Paint the next picture

        case WM_TIMER:
            KillTimer (hWnd, ID_TIMER);
            MyDrawNextPicture ();
            return 0;

        // All done!

        case WM_DESTROY:
            PostQuitMessage (0);
            return 0;

    }

    // Return to Windows

    return DefWindowProc (hWnd, message, wParam, lParam);
}


// OpenDlgHookProc
// --------------------------------------------------------------------
UINT CALLBACK __export OpenDlgHookProc (HWND hDlg, UINT message,
                                        WPARAM wParam, LPARAM lParam )

{
    return FALSE;
}


// MyListAllPictures
// ---------------------------------------------------------------------
static VOID NEAR MyListAllPictures (char* pszDir, char* pszFilter)

{
    char szPicture[512];
    struct find_t c_file;
    struct _stat st;

    _chdir (pszDir);

    if (_dos_findfirst ("*.*", _A_NORMAL | _A_SUBDIR, &c_file) == 0) {
        do {
            if (c_file.name[0] != '.') {
                if (c_file.attrib & _A_SUBDIR) {
                    MyListAllPictures (c_file.name, pszFilter);
                }
                else {
                    if (strstr (c_file.name, pszFilter) != NULL) {
                        _getcwd (szPicture, sizeof (szPicture));
                        if (strlen (szPicture) > 3)
                            strcat (szPicture, "\\");
                        strcat (szPicture, c_file.name);
                        _stat (szPicture, &st);
                        if (st.st_size < 3000000L) {
                            fputs (szPicture, hList);
                            fputc ('\n', hList);
                        }
                    }
                }
            }
        } while (_dos_findnext (&c_file) == 0);
    }

    _chdir ("..");

}


// MyDrawNextPicture
// ---------------------------------------------------------------------
static VOID NEAR MyDrawNextPicture (VOID)

{
    RECT rcDesktop, rcPicture;
    OpenCPicParams ocp;
    char szPicture[512];
    char szTitle[_MAX_PATH];
    char szExt[_MAX_EXT];
    OFSTRUCT of;
    HFILE hFile;
    LONG lSize;
    HGLOBAL hPict;
    LPBYTE hpPict;
    OSErr oserr;

    // Free any old palette

    if (hpalPicture)
        DeleteObject (hpalPicture);

    // Free any old picture

    if (phPicture) {
        DisposePicture (phPicture);
        phPicture = 0;
    }

    // Instantiate the next picture

    if (fgets (szPicture, sizeof (szPicture), hList) == NULL) {
        PostQuitMessage (0);
    }
    szPicture[strlen(szPicture) - 1] = 0;
    _splitpath (szPicture, NULL, NULL, szTitle, szExt);
    _strlwr (szTitle);
    _strlwr (szExt);
    SetWindowText (hWnd, "");
    DebugOutput (DBF_WARNING, "Drawing %s", szPicture);

    if ((oserr = OpenPictureFile (szPicture, &pfPicture, OF_READ | OF_SHARE_COMPAT)) != noErr) {
        DebugOutput (DBF_WARNING, "OpenPictureFile failure %ld", oserr);
        goto NextPicture;
    }

    // Make the frame just big enough for the picture and center it
    // in the desktop

    GetPictureFileHeader (pfPicture, &rcPicture, &ocp);
    OffsetRect (&rcPicture, -rcPicture.left, -rcPicture.top);
    rcDraw = rcPicture;
    AdjustWindowRect (&rcPicture, WS_CAPTION | WS_SYSMENU | WS_OVERLAPPED, FALSE);
    OffsetRect (&rcPicture, -rcPicture.left, -rcPicture.top);
    GetWindowRect (GetDesktopWindow (), &rcDesktop);
    SetWindowPos (hWnd, 0,
        max ((rcDesktop.right - rcPicture.right) / 2, 0),
        max ((rcDesktop.bottom - rcPicture.bottom) / 2, 0),
        rcPicture.right, rcPicture.bottom,
        SWP_NOZORDER | SWP_NOACTIVATE);

    // Sound effect

    MySoundEffect ();

    // Extract picture from file the hard way for PIC files

    if (lstrcmp (szExt, ".pic") == 0) {
        if ((hFile = OpenFile (szPicture, &of, OF_READ | OF_SHARE_COMPAT)) != NULL) {
            lSize = _llseek (hFile, 0, 2) - 512;
            if ((hPict = GlobalAlloc (GHND, lSize)) != NULL) {
                hpPict = (LPBYTE) GlobalLock (hPict);
                _llseek (hFile, 512, 0);
                _hread (hFile, hpPict, lSize);
                _lclose (hFile);
                GlobalUnlock (hPict);
                phPicture = (PicHandle) hPict;
            }
            else DebugOutput (DBF_WARNING, "GlobalAlloc failure for PICT");
        }
        else DebugOutput (DBF_WARNING, "OpenFile failure for PICT");
    }

    // Extract picture from file the easy way for non-PIC files

    else if ((phPicture = GetPictureFromFile (pfPicture)) == NULL)
        DebugOutput (DBF_WARNING, "GetPictureFromFile failure %ld", oserr);

    // We're done with the picture file

    ClosePictureFile (pfPicture);

    // Extract palette from picture

    hpalPicture = GetPicturePalette (phPicture);

    // Force a paint to draw the picture

    SetWindowText (hWnd, szTitle);
    InvalidateRect (hWnd, NULL, TRUE);
    UpdateWindow (hWnd);

    // We'll open the next picture after a while

    NextPicture: SetTimer (hWnd, ID_TIMER, SHOWTIME, NULL);

}


// MySoundEffect
// ---------------------------------------------------------------------
static BOOL NEAR MySoundEffect (VOID)

{
    HRSRC hResInfo;
    HGLOBAL hRes;
    LPSTR lpRes;
    BOOL bRtn;

    // Find the WAVE resource

    if ((hResInfo = FindResource (hInstance, "SoundEffect", "WAVE")) == NULL)
        return FALSE;

    // Load the WAVE resource

    if ((hRes = LoadResource (hInstance, hResInfo)) == NULL)
        return FALSE;

    // Lock the WAVE resource and play it

    if ((lpRes = LockResource (hRes)) != NULL) {
        bRtn = sndPlaySound (lpRes, SND_MEMORY | SND_SYNC | SND_NODEFAULT);
        UnlockResource (hRes);
    }
    else bRtn = FALSE;

    // Free the WAVE resource and return

    FreeResource (hRes);
    return bRtn;

}
