
// ---------------------------------------------------------------------
//
// EACHMOOV.C   - QuickTime for Windows
//
//                Version 1.0
//
//                (c) Copyright 1988-1994 Apple Computer, Inc. All Rights Reserved.
//
// ---------------------------------------------------------------------


#include <windows.h>
#include <dlgs.h>
#include <commdlg.h>
#include <qtw.h>
#include <dos.h>
#include <direct.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#define ID_TIMER 1
//
// Function declarations.
//
typedef UINT (CALLBACK * OPENDLGHOOKPROC) (HWND, UINT, WPARAM, LPARAM);

long FAR PASCAL __export WndProc (HWND, UINT, WPARAM, LPARAM);
UINT CALLBACK __export OpenDlgHookProc (HWND, UINT, WPARAM, LPARAM);
static VOID NEAR MyListAllMovies (char*, char*);
static VOID NEAR MyPlayNextMovie  (VOID);
static VOID NEAR MyProcessTimer( HWND);
//
// Global variables.
//
static MovieFile mfMovie;
static Movie mMovie;
static MovieController mcController;
static FILE* hList;
static HWND hWnd;
static char szAppName[] = "EACHMOOV";

// WinMain
// ---------------------------------------------------------------------
int PASCAL WinMain (HINSTANCE hInstance, HINSTANCE hPrevInstance,
   LPSTR lpszCmdParam, int nCmdShow)
{
    MSG msg;
    WNDCLASS wndclass;
    char szDrive[_MAX_DRIVE];
    char szDir[_MAX_DIR];
    char szFileTitle[_MAX_PATH];
    char szParm[_MAX_PATH];
    char* pszTemp;
    OPENFILENAME ofn;

    // If parameter is missing, use the standard Open Dialog
    // to get the root directory

    if (lstrlen (lpszCmdParam) == 0) {
        _getcwd (szDir, sizeof (szDir));
        szParm[0] = '\0';
        memset (&ofn, 0, sizeof(OPENFILENAME));
        ofn.lStructSize = sizeof(OPENFILENAME);
        ofn.hwndOwner = 0;
        ofn.lpstrFilter = "Movies\0all\0\0";
        ofn.nFilterIndex = 1;
        ofn.lpstrFile = szParm;
        ofn.nMaxFile = sizeof(szParm);
        ofn.lpstrFileTitle = szFileTitle;
        ofn.nMaxFileTitle = sizeof(szFileTitle);
        ofn.lpstrInitialDir = szDir;
        ofn.lpstrTitle = "Play Each Movie";
        ofn.Flags = OFN_HIDEREADONLY | OFN_ENABLEHOOK | OFN_ENABLETEMPLATE;
        ofn.hInstance = hInstance;
        ofn.lpTemplateName = "CustomDirectorySelect";
        ofn.lpfnHook = (OPENDLGHOOKPROC) MakeProcInstance ((FARPROC) OpenDlgHookProc, hInstance);
        if (GetOpenFileName (&ofn) == 0)
            return 0;
        FreeProcInstance ((FARPROC) ofn.lpfnHook);
        _strupr (szParm);
    }

    // Otherwise, treat the command line parameter as the
    // root directory. We append a dummy file name to assist in the
    // subsequent parsing

    else {
        lstrcpy (szParm, lpszCmdParam);
        strcat (szParm, "\\crap");
        _strupr (szParm);
    }

    // Establish links to QuickTime for Windows

    if (QTInitialize (NULL)) {
        MessageBox (NULL, "QTInitialize failure", szAppName, MB_OK);
        return 0;
    }

    // Allocate memory required for playing movies

    if (EnterMovies ()) {
        MessageBox (NULL, "EnterMovies failure", szAppName, MB_OK);
        return 0;
    }

    // Register and create main window

    if (!hPrevInstance) {
        wndclass.style = CS_DBLCLKS | CS_HREDRAW | CS_VREDRAW;
        wndclass.lpfnWndProc = WndProc;
        wndclass.cbClsExtra = 0;
        wndclass.cbWndExtra = 0;
        wndclass.hInstance = hInstance;
        wndclass.hIcon = LoadIcon (NULL,IDI_APPLICATION);
        wndclass.hCursor = LoadCursor (NULL, IDC_ARROW);
        wndclass.hbrBackground = (HBRUSH) (COLOR_WINDOW + 1);
        wndclass.lpszMenuName = NULL;
        wndclass.lpszClassName = szAppName;
        if (!RegisterClass (&wndclass)) {
            MessageBox (NULL, "RegisterClass failure", szAppName, MB_OK);
            return 0;
        }
    }

    hWnd = CreateWindow(szAppName, szAppName, WS_CAPTION | WS_SYSMENU |
        WS_CLIPCHILDREN | WS_OVERLAPPED, CW_USEDEFAULT, CW_USEDEFAULT,
        CW_USEDEFAULT, CW_USEDEFAULT, NULL, NULL, hInstance, NULL);

    if (hWnd == NULL) {
        MessageBox (NULL, "CreateWindow failure", szAppName, MB_OK);
        return 0;
    }

    // Make a list of all the movies

    pszTemp = _tempnam ("c:\\", "each");
    hList = fopen (pszTemp, "w+");
    _splitpath (szParm, szDrive, szDir, NULL, NULL);
    if (strlen (szDrive) > 0)
        _chdrive ((int) (szDrive[0] - '@'));
    if (strlen (szDir) > 1)
        szDir[strlen(szDir) - 1] = 0;
    MyListAllMovies (szDir, ".MOV");

    // Open the list of movies and process the first in the list

    rewind (hList);
    MyPlayNextMovie ();

    // Make the main window visible

    ShowWindow (hWnd, nCmdShow);
    UpdateWindow (hWnd);

    // Play the movie

    while (GetMessage (&msg, NULL, 0, 0)) {
        TranslateMessage (&msg);
        DispatchMessage (&msg);
    }

    // Close and delete list of movies

    fclose (hList);
    remove (pszTemp);
    free (pszTemp);

    // Destroy any old movie controller

    if (mcController)
        DisposeMovieController (mcController);

    // Destroy any old movie

    if (mMovie)
        DisposeMovie (mMovie);

    // Cut the connections to QuickTime for Windows

    ExitMovies ();
    QTTerminate ();

    // Return to Windows

    return msg.wParam;
}


// WndProc
// ---------------------------------------------------------------------
long FAR PASCAL __export WndProc (HWND hWnd, UINT message, WPARAM wParam,
   LPARAM lParam)
{
    PAINTSTRUCT ps;

    // Drive the movie controller

    MCIsPlayerMessage (mcController, hWnd, message, wParam, lParam);

    // Process the windows message

    switch (message) {

        // In case we have to paint this movie

        case WM_PAINT:
            if (!BeginPaint (hWnd, &ps))
                return 0;
            EndPaint (hWnd, &ps);
            return 0;

        // Play next movie

        case WM_TIMER:
            MyProcessTimer( hWnd);

            return 0;

        // All done!

        case WM_DESTROY:
            PostQuitMessage (0);
            return 0;

    }

    // Return to Windows

    return DefWindowProc (hWnd, message, wParam, lParam);
}


// OpenDlgHookProc
// --------------------------------------------------------------------
UINT CALLBACK __export OpenDlgHookProc (HWND hDlg, UINT message,
                                        WPARAM wParam, LPARAM lParam )

{
    return FALSE;
}


// MyListAllMovies
// ---------------------------------------------------------------------
static VOID NEAR MyListAllMovies (char* pszDir, char* pszFilter)

{
    char szMovie[512];
    struct find_t c_file;

    _chdir (pszDir);

    if (_dos_findfirst ("*.*", _A_NORMAL | _A_SUBDIR, &c_file) == 0) {
        do {
            if (c_file.name[0] != '.') {
                if (c_file.attrib & _A_SUBDIR) {
                    MyListAllMovies (c_file.name, pszFilter);
                }
                else {
                    if (strstr (c_file.name, pszFilter) != NULL) {
                        _getcwd (szMovie, sizeof (szMovie));
                        if (strlen (szMovie) > 3)
                            strcat (szMovie, "\\");
                        strcat (szMovie, c_file.name);
                        fputs (szMovie, hList);
                        fputc ('\n', hList);
                    }
                }
            }
        } while (_dos_findnext (&c_file) == 0);
    }

    _chdir ("..");

}


// MyPlayNextMovie
// ---------------------------------------------------------------------
static VOID NEAR MyPlayNextMovie (VOID)

{
    RECT rcDesktop, rcMovie;
    char szMovie[512];
    char szTitle[_MAX_PATH];
    OSErr oserr;
    PicHandle ph;
    DIBHandle dh;

    // Destroy any old movie controller

    if (mcController)
        DisposeMovieController (mcController);

    // Destroy any old movie

    if (mMovie)
        DisposeMovie (mMovie);

    // Instantiate the next movie

    if (fgets (szMovie, sizeof (szMovie), hList) == NULL) {
        PostQuitMessage (0);
        return;
    }
    szMovie[strlen(szMovie) - 1] = 0;
    _splitpath (szMovie, NULL, NULL, szTitle, NULL);
    _strlwr (szTitle);
    DebugOutput (DBF_WARNING, "Playing %s", szMovie);

    if ((oserr = OpenMovieFile (szMovie, &mfMovie, OF_READ)) != noErr) {
        DebugOutput (DBF_WARNING, "OpenMovieFile failure %ld", oserr);
        goto NextMovie;
    }

    oserr = NewMovieFromFile (&mMovie, mfMovie, NULL, NULL, 0, NULL);
    CloseMovieFile (mfMovie);
    if (oserr != noErr) {
        DebugOutput (DBF_WARNING, "NewMovieFromFile failure %ld", oserr);
        goto NextMovie;
    }

    // Instantiate the movie controller

    GetMovieBox (mMovie, &rcMovie);
    OffsetRect(&rcMovie, -rcMovie.left, -rcMovie.top);
    mcController = NewMovieController (mMovie, &rcMovie,
        mcTopLeftMovie + mcScaleMovieToFit, hWnd);

    // Eliminate the grow box

    SetRectEmpty (&rcMovie);
    MCDoAction (mcController, mcActionSetGrowBoxBounds, &rcMovie);

    // Make the frame just big enough for the movie and center it
    // in the desktop

    MCGetControllerBoundsRect (mcController, &rcMovie);

    AdjustWindowRect (&rcMovie, WS_CAPTION | WS_OVERLAPPED, FALSE);
    OffsetRect(&rcMovie, -rcMovie.left, -rcMovie.top);

    GetWindowRect (GetDesktopWindow (), &rcDesktop);
    SetWindowPos (hWnd, 0,
        max ((rcDesktop.right - rcMovie.right) / 2, 0),
        max ((rcDesktop.bottom - rcMovie.bottom) / 2, 0),
        rcMovie.right, rcMovie.bottom,
        SWP_NOZORDER | SWP_NOACTIVATE);
    SetWindowText (hWnd, szTitle);

    // Assert the use of the palette, if any and enable the keyboard intf.

    MCDoAction (mcController, mcActionSetFlags, (LPVOID) mcFlagsUseWindowPalette);
    MCDoAction (mcController, mcActionSetKeysEnabled, (LPVOID) TRUE);

    // Make the movie active

    SetMovieActive (mMovie, TRUE);
    PrerollMovie (mMovie, 0, MAKELFIXED (1, 0));

    // We take a few seconds to test the GetMoviePict, PicturetoDIB combination 
    
    ph = NULL, dh = NULL;
    #ifdef _DEBUG
        if ((ph = GetMoviePict (mMovie, 0)) != NULL) {
            if ((dh = PictureToDIB (ph)) != NULL) {
                oserr = GetMoviesError ();
                DebugOutput (DBF_WARNING, "PictureToDIB failure %ld", oserr);
            }
        }
        else {
            oserr = GetMoviesError ();
            DebugOutput (DBF_WARNING, "GetMoviePict failure %ld", oserr);
        }
        if (ph != NULL)
            DisposePicture (ph);
        if (dh != NULL)
            GlobalFree (dh);
    #endif
     
    // Start playing

    MCDoAction (mcController, mcActionPlay, (LPVOID) MAKELFIXED (1, 0));

    // Set a 2.5 second timer which is when we will check up on the
    // movie

    NextMovie: SetTimer (hWnd, ID_TIMER, 2500, NULL);
}


// MyProcessTimer
// ---------------------------------------------------------------------
static VOID NEAR MyProcessTimer( HWND hWnd)
{
    TimeValue tvEnd;
    TimeValue tvNow;
    TimeScale tsMovie;
    DWORD dwTimeLeft;

    // Remove any old timer

    KillTimer (hWnd, ID_TIMER);

    // Check to see if we are at the end of the movie by getting
    // the duration and the current time.

    if ( mMovie) {
        tvEnd = GetMovieDuration( mMovie);
        tvNow = MCGetCurrentTime( mcController, NULL);

        // Movie is not done yet. Compute the time left in milliseconds
        // and set another timer based on that.

        if ( tvEnd - tvNow > 0) {
            tsMovie = GetMovieTimeScale( mMovie);
            dwTimeLeft = ((DWORD) tvEnd - tvNow) * 1000 / tsMovie;
            if ( dwTimeLeft > 2500)
                SetTimer (hWnd, ID_TIMER, 2500, NULL);
            else SetTimer( hWnd, ID_TIMER, (WORD) dwTimeLeft + 2000, NULL);
        }

        // Fire up the next movie if this one is complete

        else MyPlayNextMovie ();

    }

}
