
// ---------------------------------------------------------------------
//
// BROWSER.C    - QuickTime for Windows
//
//                Version 1.0
//
//                (c) Copyright 1988-1994 Apple Computer, Inc. All Rights Reserved.
//
// ---------------------------------------------------------------------


#define OEMRESOURCE
#include <windows.h>
#include <qtw.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <common.h>
#include "browser.hr"


// Constants
// ---------
#define MAX_THUMBNAILS 64
#define THUMBNAIL_INDENT 20
#define THUMBNAIL_INDENT_BOTTOM 40
#define THUMBNAIL_HEIGHT 60
#define THUMBNAIL_WIDTH 80
#define BEVEL_OUTDENT 6
#define DESC_TEXT_HEIGHT 12
#define REPEAT_CLICK 1
#define STRIP_SHIFT 2


// Functions
// ---------
long FAR PASCAL __export WndProc (HWND, UINT, WPARAM, LPARAM);
static BOOL NEAR MyBuildThumbNails (HWND, LPSTR);
static BOOL NEAR MyBuildThumbNailPoster (HWND, int);
static LONG NEAR MyClickDown (HWND, int, int);
static LONG NEAR MyClickUp (HWND, int, int);
static LONG NEAR MyDoubleClick (HWND, int, int);
static LONG NEAR MyPaintBrowser (HWND);
static BOOL NEAR MyShowMovie (HWND, int);
static LONG NEAR MySizeBrowser (HWND, int, int);


// Globals
// -------
static char szAppName[] = "BROWSER";
static char szWaiting[] = "Initializing Browser...";
static char szMovie[MAX_THUMBNAILS][_MAX_FNAME];
static char szDesc[MAX_THUMBNAILS][COMMON_STRING_MAX];
static Movie mMovie;
static MovieController mcController;
static PicHandle phPoster[MAX_THUMBNAILS];
static RECT rcMovie, rcThumbNail, rcThumb, rcTopStrip, rcBotStrip, rcRArrow, rcLArrow, rcDesc;
static int iStrip, iThumbNail, cThumbNails, cThumbsShown;
static HINSTANCE hInstance;
static HBITMAP hbmRArrow, hbmLArrow, hbmRArrowD, hbmLArrowD, hbmStrip;
static BOOL bInLArrow, bInRArrow, bInThumbNail;


// WinMain
// ---------------------------------------------------------------------
int PASCAL WinMain (HINSTANCE hInst, HINSTANCE hPrevInstance,
                    LPSTR lpszCmdLine, int nCmdShow)
{
    HWND hWnd;
    RECT rcDesktop, rcBrowser;
    MSG msg;
    WNDCLASS wc;

    hInstance = hInst;

    // Register our special class

    if (!hPrevInstance) {
        wc.style = CS_DBLCLKS | CS_HREDRAW | CS_VREDRAW;
        wc.lpfnWndProc = WndProc;
        wc.cbClsExtra = 0;
        wc.cbWndExtra = DLGWINDOWEXTRA;
        wc.hInstance = hInstance;
        wc.hIcon = LoadIcon (hInstance, szAppName);
        wc.hCursor = LoadCursor (NULL, IDC_ARROW);
        wc.hbrBackground = (HBRUSH) GetStockObject (LTGRAY_BRUSH);
        wc.lpszMenuName = NULL;
        wc.lpszClassName = szAppName;
        RegisterClass (&wc);
    }

    // Load some bitmaps

    hbmRArrow  = LoadBitmap ( hInstance, "Rbtnup");
    hbmLArrow  = LoadBitmap ( hInstance, "Lbtnup");
    hbmRArrowD = LoadBitmap ( hInstance, "Rbtndn");
    hbmLArrowD = LoadBitmap ( hInstance, "Lbtndn");
    hbmStrip   = LoadBitmap (hInstance, "Strip");

    // Create the dialog

    hWnd = CreateDialog (hInstance, szAppName, 0, NULL);
    GetWindowRect (hWnd, &rcBrowser);
    GetWindowRect (GetDesktopWindow (), &rcDesktop);
    SetWindowPos (hWnd, 0,
        rcDesktop.left + (((rcDesktop.right - rcDesktop.left)
        - (rcBrowser.right - rcBrowser.left)) / 2),
        rcDesktop.top + (((rcDesktop.bottom - rcDesktop.top)
        - (rcBrowser.bottom - rcBrowser.top)) / 2),
        rcBrowser.right - rcBrowser.left,
        rcBrowser.bottom - rcBrowser.top, SWP_NOZORDER);

    // Establish links to QuickTime for Windows

    if (QTInitialize (NULL)) {
        MessageBox (NULL, "QTInitialize failure", szAppName, MB_OK);
        return 0;
    }

    // Allocate memory required for playing movies

    if (EnterMovies ()) {
        MessageBox (NULL, "EnterMovies failure", szAppName, MB_OK);
        return 0;
    }

    // Create a movie controller

    mcController = NewMovieController (NULL, &rcBrowser, mcNotVisible, hWnd);

    // Display the dialog

    ShowWindow (hWnd, nCmdShow);

    // Build posters for each thumbnail

    if (! MyBuildThumbNails (hWnd, lpszCmdLine)) {
        return 0;
    }

    // Show the first movie

    MyShowMovie (hWnd, 0);

    // Process all Windows messages

    while (GetMessage (&msg, NULL, 0, 0)) {
        TranslateMessage (&msg);
        DispatchMessage (&msg);
    }

    // Cut links to QuickTime

    DisposeMovieController (mcController);
    DisposeMovie (mMovie);

    for (iThumbNail = 0; iThumbNail < cThumbNails; iThumbNail++) {
        DisposePicture (phPoster[iThumbNail]);
    }

    ExitMovies ();
    QTTerminate ();

    // Delete the Windows bitmaps we used

    DeleteObject (hbmRArrow);
    DeleteObject (hbmLArrow);
    DeleteObject (hbmRArrowD);
    DeleteObject (hbmLArrowD);
    DeleteObject (hbmStrip);

    // Return to Windows

    return msg.wParam;

}


// WndProc
// ---------------------------------------------------------------------
long FAR PASCAL __export WndProc (HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{

    // Drive the movie controller

    MCIsPlayerMessage (mcController, hWnd, message, wParam, lParam);

    // Act on the message

    switch (message) {

        // All Done!

        case WM_DESTROY:
            PostQuitMessage (0);
            return 0;

        // Background erase

        case WM_ERASEBKGND:
            return (bInLArrow || bInRArrow)? TRUE : DefWindowProc (hWnd, message, wParam, lParam);

        // Double click

        case WM_LBUTTONDBLCLK:
            return MyDoubleClick (hWnd, (int) LOWORD (lParam), (int) HIWORD (lParam));

        // Click down

        case WM_LBUTTONDOWN:
            return MyClickDown (hWnd, (int) LOWORD (lParam), (int) HIWORD (lParam));

        // Click up

        case WM_LBUTTONUP:
            return MyClickUp (hWnd, (int) LOWORD (lParam), (int) HIWORD (lParam));

        // Paint

        case WM_PAINT:
            return MyPaintBrowser (hWnd);

        // Resize

        case WM_SIZE:
            return MySizeBrowser (hWnd, (int) LOWORD (lParam), (int) HIWORD (lParam));

        // Timer (repeat click)

        case WM_TIMER:
            return MyClickDown (hWnd, 0, 0);

    }

    // Return to Windows

    return DefWindowProc (hWnd, message, wParam, lParam);

}


// MyBuildThumbNails
// ---------------------------------------------------------------------
static BOOL NEAR MyBuildThumbNails (HWND hWnd, LPSTR lpszCmdLine)
{
    FILE* hList;
    HDC hdc;
    int iNameLen;
    DWORD dw;
    RECT rcClient;
    char szList[_MAX_FNAME];
    char szBuffer[COMMON_STRING_MAX];
    char* p;

    // Get name of browser list

    lstrcpy (szList, lpszCmdLine);
    if (strlen (szList) == 0) {
        GetModuleFileName (hInstance, szList, sizeof (szList));
        p = strrchr (szList, '.');
        *p = 0;
        strcat (szList, ".lst");
    }

    // Attempt to open browser list

    if ((hList = fopen (szList, "r")) == NULL) {
        MessageBox (0, "Nothing to do!", szAppName, MB_ICONEXCLAMATION);
        return FALSE;
    }

    // Issue "waiting" message

    hdc = GetDC (hWnd);
    SetBkMode (hdc, TRANSPARENT);
    dw = GetTextExtent (hdc, szWaiting, strlen (szWaiting));
    GetClientRect (hWnd, &rcClient);
    TextOut (hdc, rcClient.left + ((rcClient.right - rcClient.left - LOWORD (dw)) / 2),
        rcClient.top + ((rcClient.bottom - rcClient.top - HIWORD (dw)) / 2),
        szWaiting, strlen (szWaiting));
    ReleaseDC (hWnd, hdc);

    // Parse browser list

    memset (szBuffer, 0, sizeof (szBuffer));
    while (fgets (szBuffer, sizeof (szBuffer), hList) != NULL) {

        // Eliminate trailing new line character

        iNameLen = strlen( szBuffer);
        if ((iNameLen == sizeof( szBuffer)) ||
            (szBuffer[iNameLen-1] == 0x0A))
            szBuffer[strlen (szBuffer) - 1] = 0;
        else szBuffer[iNameLen] = 0;

        // Only process so many

        if (cThumbNails == MAX_THUMBNAILS)
            break;

        // Extract movie name and build its bitmap

        p = strchr (szBuffer, ' ');
        *p = 0;
        strcpy (szMovie[cThumbNails], szBuffer);
        if (! MyBuildThumbNailPoster (hWnd, cThumbNails))
            continue;

        // Extract description (use movie name if missing)

        p = szBuffer + strlen (szBuffer) + 1;
        while (p) {
            if ((*p != ' ') && (*p != '\t')) break;
            ++p;
        }
        if (strlen (p) > 0)
            strcpy (szDesc[cThumbNails], p);
        else strcpy (szDesc[cThumbNails], szMovie[cThumbNails]);

        // Keep going until we've found them all

        memset (szBuffer, 0, sizeof (szBuffer));
        cThumbNails++;
    }

    // All done!

    fclose (hList);
    InvalidateRect (hWnd, NULL, TRUE);
    return TRUE;

}


// MyBuildThumbNailPoster
// ---------------------------------------------------------------------
static BOOL NEAR MyBuildThumbNailPoster (HWND hWnd, int iThumbNail)
{
    MovieFile mfMovie;
    Movie mMovie;

    // Instantiate the movie

    if (OpenMovieFile (szMovie[iThumbNail], &mfMovie, OF_READ) != noErr) {
        return FALSE;
    }
    NewMovieFromFile (&mMovie, mfMovie, NULL, NULL, 0, NULL);
    CloseMovieFile (mfMovie);

    // Get its poster pict

    phPoster[iThumbNail] = GetMoviePosterPict (mMovie);
    DisposeMovie (mMovie);

    // Return to caller

    return (phPoster[iThumbNail] != NULL);
}


// MyClickDown
// ---------------------------------------------------------------------
static LONG NEAR MyClickDown (HWND hWnd, int x, int y)
{
    POINT pt;
    LOGFONT lf;
    HFONT hFont, hFontOld;
    HBITMAP hbmOld;
    HDC hdc, hdcMem;
    DWORD dw;
    UINT uiDelay;
    int i;

    // Get location of click

    pt.x = x, pt.y = y;

    // Get ready to paint

    hdc = GetDC (hWnd);
    SetBkMode (hdc, TRANSPARENT);
    hdcMem = CreateCompatibleDC (hdc);

    // Did we click the left arrow?

    if ((iThumbNail > 0) && (PtInRect (&rcLArrow, pt) || bInLArrow)) {
        bInLArrow = TRUE;
        hbmOld = SelectObject (hdcMem, hbmLArrowD);
        BitBlt (hdc, rcLArrow.left, rcLArrow.top,
            (rcLArrow.right - rcLArrow.left),
            (rcLArrow.bottom - rcLArrow.top),
            hdcMem, 0, 0, SRCCOPY);
        SelectObject (hdcMem, hbmOld);
        iThumbNail = max (0, iThumbNail - 1);
        iStrip -= STRIP_SHIFT;
        InvalidateRect (hWnd, &rcRArrow, FALSE);
        InvalidateRect (hWnd, &rcTopStrip, FALSE);
        InvalidateRect (hWnd, &rcThumbNail, FALSE);
        InvalidateRect (hWnd, &rcBotStrip, FALSE);
        UpdateWindow (hWnd);
        SystemParametersInfo (SPI_GETKEYBOARDDELAY, 0, &uiDelay, 0);
        SetTimer (hWnd, REPEAT_CLICK, uiDelay, NULL);
    }

    // Did we click the right arrow?

    if (((iThumbNail + cThumbsShown) < cThumbNails)
        && (PtInRect (&rcRArrow, pt) || bInRArrow)) {
        bInRArrow = TRUE;
        hbmOld = SelectObject (hdcMem, hbmRArrowD);
        BitBlt (hdc, rcRArrow.left, rcRArrow.top,
            (rcRArrow.right - rcRArrow.left),
            (rcRArrow.bottom - rcRArrow.top),
            hdcMem, 0, 0, SRCCOPY);
        SelectObject (hdcMem, hbmOld);
        iThumbNail = min (cThumbNails - cThumbsShown, iThumbNail + 1);
        iStrip += STRIP_SHIFT;
        InvalidateRect (hWnd, &rcLArrow, FALSE);
        InvalidateRect (hWnd, &rcTopStrip, FALSE);
        InvalidateRect (hWnd, &rcThumbNail, FALSE);
        InvalidateRect (hWnd, &rcBotStrip, FALSE);
        UpdateWindow (hWnd);
        SystemParametersInfo (SPI_GETKEYBOARDDELAY, 0, &uiDelay, 0);
        SetTimer (hWnd, REPEAT_CLICK, uiDelay, NULL);
    }

    // Did we click the thumbnails?

    else if (PtInRect (&rcThumbNail, pt)) {
        i = (x - rcThumbNail.left) / THUMBNAIL_WIDTH;
        if ((i + iThumbNail) >= cThumbNails)
            MessageBeep (MB_ICONEXCLAMATION);
        else {
            bInThumbNail = TRUE;
            rcThumb = rcThumbNail;
            rcThumb.left += (i * THUMBNAIL_WIDTH);
            rcThumb.right = min (rcThumb.left + THUMBNAIL_WIDTH, rcThumbNail.right);
            InvertRect (hdc, &rcThumb);
            memset (&lf, 0, sizeof (lf));
            lf.lfHeight = MulDiv (DESC_TEXT_HEIGHT, GetDeviceCaps (hdc, LOGPIXELSY), 72);
            lf.lfWeight = FW_LIGHT;
            lf.lfOutPrecision = OUT_TT_ONLY_PRECIS;
            lf.lfPitchAndFamily = FF_SWISS;
            strcpy (lf.lfFaceName, "Arial");
            hFont = CreateFontIndirect (&lf);
            hFontOld = SelectObject (hdc, hFont);
            dw = GetTextExtent (hdc, szDesc[iThumbNail+i], strlen (szDesc[iThumbNail+i]));
            TextOut (hdc, rcDesc.left + ((rcDesc.right - rcDesc.left - LOWORD (dw)) / 2),
                rcDesc.top + ((rcDesc.bottom - rcDesc.top - HIWORD (dw)) / 2),
                szDesc[iThumbNail+i], strlen (szDesc[iThumbNail+i]));
            SelectObject (hdc, hFontOld);
            DeleteObject (hFont);
        }
    }

    // Done painting

    DeleteDC (hdcMem);
    ReleaseDC (hWnd, hdc);

    // Capture the mouse

    SetCapture (hWnd);

    // Return to Windows

    return 0;

}


// MyClickUp
// ---------------------------------------------------------------------
static LONG NEAR MyClickUp (HWND hWnd, int x, int y)
{
    HDC hdc;

    // Get ready to paint

    hdc = GetDC (hWnd);

    // Did we click an arrow?

    if (bInLArrow || bInRArrow) {
        bInLArrow = FALSE;
        bInRArrow = FALSE;
        InvalidateRect (hWnd, &rcLArrow, TRUE);
        InvalidateRect (hWnd, &rcRArrow, TRUE);
        UpdateWindow (hWnd);
        KillTimer (hWnd, REPEAT_CLICK);
    }

    // Did we click the thumbnails?

    else if (bInThumbNail) {
        bInThumbNail = FALSE;
        InvertRect (hdc, &rcThumb);
        InvalidateRect (hWnd, &rcDesc, TRUE);
    }

    // Done painting

    ReleaseDC (hWnd, hdc);

    // Release the mouse capture

    ReleaseCapture ();

    // Return to Windows

    return 0;

}


// MyDoubleClick
// ---------------------------------------------------------------------
static LONG NEAR MyDoubleClick (HWND hWnd, int x, int y)
{
    POINT pt;
    int i;

    // Get location of click

    pt.x = x, pt.y = y;

    // Did we click the thumbnails?

    if (PtInRect (&rcThumbNail, pt)) {
        i = (x - rcThumbNail.left) / THUMBNAIL_WIDTH;
        if ((i + iThumbNail) < cThumbNails)
            MyShowMovie (hWnd, iThumbNail + i);
    }

    // Return to Windows

    return 0;

}


// MyPaintBrowser
// ---------------------------------------------------------------------
static LONG NEAR MyPaintBrowser (HWND hWnd)
{
    RECT rcPoster;
    HDC hdcMem;
    PAINTSTRUCT ps;
    HPEN hpenLight, hpenShadow, hpenOld;
    HBITMAP hbmOld;
    BITMAP bm;
    int i;

    // Create the pens we'll need to draw the bevels

    hpenLight = CreatePen (PS_SOLID, 1, RGB (255, 255, 255));
    hpenShadow = CreatePen (PS_SOLID, 1, RGB (128, 128, 128));

    // Prepare to paint

    if (BeginPaint (hWnd, &ps) == 0)
        return 0;

    // Draw bevel around movie

    hpenOld = SelectObject (ps.hdc, hpenLight);
    MoveTo (ps.hdc, rcMovie.left - BEVEL_OUTDENT, rcMovie.bottom + BEVEL_OUTDENT);
    LineTo (ps.hdc, rcMovie.left - BEVEL_OUTDENT, rcMovie.top - BEVEL_OUTDENT);
    LineTo (ps.hdc, rcMovie.right + BEVEL_OUTDENT, rcMovie.top - BEVEL_OUTDENT);
    MoveTo (ps.hdc, rcMovie.left - BEVEL_OUTDENT + 1, rcMovie.bottom + BEVEL_OUTDENT - 1);
    LineTo (ps.hdc, rcMovie.left - BEVEL_OUTDENT + 1, rcMovie.top - BEVEL_OUTDENT + 1);
    LineTo (ps.hdc, rcMovie.right + BEVEL_OUTDENT - 1, rcMovie.top - BEVEL_OUTDENT + 1);
    MoveTo (ps.hdc, rcMovie.left - 1, rcMovie.bottom);
    LineTo (ps.hdc, rcMovie.right, rcMovie.bottom);
    LineTo (ps.hdc, rcMovie.right, rcMovie.top - 1);
    SelectObject (ps.hdc, hpenOld);

    hpenOld = SelectObject (ps.hdc, hpenShadow);
    MoveTo (ps.hdc, rcMovie.left - BEVEL_OUTDENT, rcMovie.bottom + BEVEL_OUTDENT);
    LineTo (ps.hdc, rcMovie.right + BEVEL_OUTDENT, rcMovie.bottom + BEVEL_OUTDENT);
    LineTo (ps.hdc, rcMovie.right + BEVEL_OUTDENT, rcMovie.top - BEVEL_OUTDENT);
    MoveTo (ps.hdc, rcMovie.left - BEVEL_OUTDENT + 1, rcMovie.bottom + BEVEL_OUTDENT - 1);
    LineTo (ps.hdc, rcMovie.right + BEVEL_OUTDENT - 1, rcMovie.bottom + BEVEL_OUTDENT - 1);
    LineTo (ps.hdc, rcMovie.right + BEVEL_OUTDENT - 1, rcMovie.top - BEVEL_OUTDENT + 1);
    MoveTo (ps.hdc, rcMovie.left - 1, rcMovie.bottom - 1);
    LineTo (ps.hdc, rcMovie.left - 1, rcMovie.top - 1);
    LineTo (ps.hdc, rcMovie.right, rcMovie.top - 1);
    SelectObject (ps.hdc, hpenOld);

    // Draw bevel around thumbnails

    hpenOld = SelectObject (ps.hdc, hpenLight);
    MoveTo (ps.hdc, rcThumbNail.left - BEVEL_OUTDENT, rcBotStrip.bottom + BEVEL_OUTDENT);
    LineTo (ps.hdc, rcThumbNail.left - BEVEL_OUTDENT, rcTopStrip.top - BEVEL_OUTDENT);
    LineTo (ps.hdc, rcThumbNail.right + BEVEL_OUTDENT, rcTopStrip.top - BEVEL_OUTDENT);
    MoveTo (ps.hdc, rcThumbNail.left - BEVEL_OUTDENT + 1, rcBotStrip.bottom + BEVEL_OUTDENT - 1);
    LineTo (ps.hdc, rcThumbNail.left - BEVEL_OUTDENT + 1, rcTopStrip.top - BEVEL_OUTDENT + 1);
    LineTo (ps.hdc, rcThumbNail.right + BEVEL_OUTDENT - 1, rcTopStrip.top - BEVEL_OUTDENT + 1);
    SelectObject (ps.hdc, hpenOld);

    hpenOld = SelectObject (ps.hdc, hpenShadow);
    MoveTo (ps.hdc, rcThumbNail.left - BEVEL_OUTDENT, rcBotStrip.bottom + BEVEL_OUTDENT);
    LineTo (ps.hdc, rcThumbNail.right + BEVEL_OUTDENT, rcBotStrip.bottom + BEVEL_OUTDENT);
    LineTo (ps.hdc, rcThumbNail.right + BEVEL_OUTDENT, rcTopStrip.top - BEVEL_OUTDENT);
    MoveTo (ps.hdc, rcThumbNail.left - BEVEL_OUTDENT + 1, rcBotStrip.bottom + BEVEL_OUTDENT - 1);
    LineTo (ps.hdc, rcThumbNail.right + BEVEL_OUTDENT - 1, rcBotStrip.bottom + BEVEL_OUTDENT - 1);
    LineTo (ps.hdc, rcThumbNail.right + BEVEL_OUTDENT - 1, rcTopStrip.top - BEVEL_OUTDENT + 1);
    SelectObject (ps.hdc, hpenOld);

    // Bitblt the arrows

    hdcMem = CreateCompatibleDC (ps.hdc);
    if (iThumbNail > 0) {
        hbmOld = SelectObject (hdcMem, hbmLArrow);
        BitBlt (ps.hdc, rcLArrow.left, rcLArrow.top,
            (rcLArrow.right - rcLArrow.left),
            (rcLArrow.bottom - rcLArrow.top),
            hdcMem, 0, 0, SRCCOPY);
        SelectObject (ps.hdc, hbmOld);
    }
    if ((iThumbNail + cThumbsShown) < cThumbNails) {
        hbmOld = SelectObject (hdcMem, hbmRArrow);
        BitBlt (ps.hdc, rcRArrow.left, rcRArrow.top,
            (rcRArrow.right - rcRArrow.left),
            (rcRArrow.bottom - rcRArrow.top),
            hdcMem, 0, 0, SRCCOPY);
        SelectObject (ps.hdc, hbmOld);
    }
    DeleteDC (hdcMem);

    // Clip out poster area to help drawing logic

    GetObject (hbmStrip, sizeof (bm), &bm);
    IntersectClipRect (ps.hdc,
        rcThumbNail.left, rcThumbNail.top - bm.bmHeight,
        rcThumbNail.right, rcThumbNail.bottom + bm.bmHeight);

    // Draw each poster in the thumbnail area

    rcPoster = rcThumbNail;
    for (i = iThumbNail; i < cThumbNails; i++) {
        rcPoster.right = rcPoster.left + THUMBNAIL_WIDTH;
        DrawPicture (ps.hdc, phPoster[i], &rcPoster, NULL);
        if ((rcPoster.left += THUMBNAIL_WIDTH) >= rcThumbNail.right)
            break;
    }

    // Blank out any unused poster slots

    rcPoster.right = rcThumbNail.right;
    FillRect (ps.hdc, &rcPoster, (HBRUSH) GetStockObject (WHITE_BRUSH));

    // Bitblt the film strips

    hdcMem = CreateCompatibleDC (ps.hdc);
    hbmOld = SelectObject (hdcMem, hbmStrip);
    BitBlt (ps.hdc, rcTopStrip.left, rcTopStrip.top,
        (rcTopStrip.right - rcTopStrip.left),
        (rcTopStrip.bottom - rcTopStrip.top),
        hdcMem, iStrip, 0, SRCCOPY);
    BitBlt (ps.hdc, rcBotStrip.left, rcBotStrip.top,
        (rcBotStrip.right - rcBotStrip.left),
        (rcBotStrip.bottom - rcBotStrip.top),
        hdcMem, iStrip, 0, SRCCOPY);
    SelectObject (hdcMem, hbmOld);
    DeleteDC (hdcMem);

    // Done painting, return to Windows

    EndPaint (hWnd, &ps);
    DeleteObject (hpenLight);
    DeleteObject (hpenShadow);
    return 0;

}


// MyShowMovie
// ---------------------------------------------------------------------
static BOOL NEAR MyShowMovie (HWND hWnd, int iThumbNail)
{
    MovieFile mfMovie;
    POINT ptMovie;
    RECT rcClient, rcBrowser, rcWash;
    int xDelta, yDelta, iDepth;
    BOOL bDouble;
    HDC hdc;

    // Dispose of any prior movie

    if (mMovie) {
        MCRemoveMovie (mcController);
        DisposeMovie (mMovie);
    }

    // Instantiate the movie

    if (OpenMovieFile (szMovie[iThumbNail], &mfMovie, OF_READ) != noErr) {
        return FALSE;
    }
    NewMovieFromFile (&mMovie, mfMovie, NULL, NULL, 0, NULL);
    CloseMovieFile (mfMovie);

    // Calculate where to put the controller

    for (bDouble = TRUE;;) {
        GetMovieBox (mMovie, &rcMovie);
        GetClientRect (hWnd, &rcClient);
        rcClient.bottom = rcTopStrip.top - (2 * THUMBNAIL_INDENT);
        OffsetRect (&rcMovie, -rcMovie.left, -rcMovie.top);
        hdc = GetDC (hWnd);
        iDepth = GetDeviceCaps (hdc, BITSPIXEL) * GetDeviceCaps (hdc, PLANES);
        ReleaseDC (hWnd, hdc);
        if (bDouble && (iDepth > 8))
            rcMovie.right *= 2, rcMovie.bottom *= 2;
        OffsetRect (&rcMovie, (rcClient.right - rcMovie.right) / 2,
            (rcClient.bottom - rcMovie.bottom) / 2);
        xDelta = (rcClient.left + THUMBNAIL_INDENT) - rcMovie.left;
        yDelta = (rcClient.top + THUMBNAIL_INDENT) - rcMovie.top;
        GetWindowRect (hWnd, &rcBrowser);
        if (xDelta > 0)
            rcBrowser.left -= xDelta, rcBrowser.right += xDelta;
        else if (yDelta > 0)
            rcBrowser.top -= yDelta, rcBrowser.bottom += yDelta;
        else break;
        if ((rcBrowser.left < 0) || (rcBrowser.top < 0)) {
            bDouble = FALSE;
            continue;
        }
        SetWindowPos (hWnd, 0, rcBrowser.left, rcBrowser.top,
            rcBrowser.right - rcBrowser.left,
            rcBrowser.bottom - rcBrowser.top, SWP_NOZORDER | SWP_SHOWWINDOW);
    }

    // Associate the movie and the controller

    ptMovie.x = rcMovie.left;
    ptMovie.y = rcMovie.top;
    MCSetMovie (mcController, mMovie, hWnd, ptMovie);

    // Reposition the controller

    MCPositionController (mcController, &rcMovie,
        NULL, mcTopLeftMovie + mcScaleMovieToFit);
    MCSetVisible (mcController, TRUE);
    MCGetControllerBoundsRect (mcController, &rcMovie);
    SetRectEmpty (&rcClient);
    MCDoAction (mcController, mcActionSetGrowBoxBounds, (LPVOID) &rcClient);

    // Action!

    SetMovieActive (mMovie, TRUE);
    GetClientRect (hWnd, &rcWash);
    rcWash.bottom = rcDesc.bottom;
    InvalidateRect (hWnd, &rcWash, TRUE);
    MCDoAction (mcController, mcActionSetFlags, (LPVOID) mcFlagsUseWindowPalette);
    MCDoAction (mcController, mcActionSetKeysEnabled, (LPVOID) TRUE);

    // Update the window title

    SetWindowText (hWnd, szDesc[iThumbNail]);

    // Return to caller

    return TRUE;
}


// MySizeBrowser
// ---------------------------------------------------------------------
static LONG NEAR MySizeBrowser (HWND hWnd, int cx, int cy)
{
    BITMAP bm;

    // Calculate the location of the left scroll arrow

    GetObject (hbmLArrow, sizeof (bm), &bm);
    rcLArrow.left = bm.bmWidth / 2;
    rcLArrow.right = rcLArrow.left + bm.bmWidth;
    rcLArrow.top = (cy - (THUMBNAIL_HEIGHT + THUMBNAIL_INDENT_BOTTOM))
        + ((THUMBNAIL_HEIGHT - bm.bmHeight) / 2);
    rcLArrow.bottom = rcLArrow.top + bm.bmHeight;

    // Calculate the location of the right scroll arrow

    rcRArrow.left = cx - ((3 * bm.bmWidth) / 2);
    rcRArrow.right = rcRArrow.left + bm.bmWidth;
    rcRArrow.top = (cy - (THUMBNAIL_HEIGHT + THUMBNAIL_INDENT_BOTTOM))
        + ((THUMBNAIL_HEIGHT - bm.bmHeight) / 2);
    rcRArrow.bottom = rcRArrow.top + bm.bmHeight;

    // Calculate thumb nail area

    rcThumbNail.left = rcLArrow.right + (bm.bmWidth / 4) + BEVEL_OUTDENT;
    rcThumbNail.right = rcRArrow.left - (bm.bmWidth / 4) - BEVEL_OUTDENT;
    rcThumbNail.top = cy - (THUMBNAIL_HEIGHT + THUMBNAIL_INDENT_BOTTOM);
    rcThumbNail.bottom = cy - THUMBNAIL_INDENT_BOTTOM;
    cThumbsShown = (rcThumbNail.right - rcThumbNail.left) / THUMBNAIL_WIDTH;

    // Calculate top and bottom film strip areas

    GetObject (hbmStrip, sizeof (bm), &bm);
    rcTopStrip = rcThumbNail;
    rcTopStrip.bottom = rcTopStrip.top;
    rcTopStrip.top -= bm.bmHeight;
    rcBotStrip = rcThumbNail;
    rcBotStrip.top = rcBotStrip.bottom;
    rcBotStrip.bottom += bm.bmHeight;

    // Calculate description area

    rcDesc = rcTopStrip;
    rcDesc.bottom = rcTopStrip.top;
    rcDesc.top = rcDesc.bottom - (3 * DESC_TEXT_HEIGHT);

    // Return to Windows

    return 0;

}
