
ns_register_proc GET  /NS/GetNewTableForm GetNewTableForm
ns_register_proc POST /NS/GetNewTableForm GetNewTableForm
ns_register_proc GET  /NS/GetDropTableForm GetDropTableForm
ns_register_proc POST /NS/VerifyDropTable VerifyDropTable
ns_register_proc POST /NS/DropTable DropTable
ns_register_proc POST /NS/CreateTable CreateTable
ns_register_proc GET  /NS/GetExtendTableForm GetExtendTableForm
ns_register_proc POST /NS/GetRealExtendTableForm GetRealExtendTableForm
ns_register_proc POST /NS/ExtendTable ExtendTable


#
# The maximum number of allowed columns.
#
set NS(maxColumns) 10

#
# The list of reserved words from Appendix C of the Illustra User's Manual.
#
set NS(reservedWords) [list \
	abort absolute add after aggregate alert alerter all after and \
	any archive arch_store arg as asc at authorization backward before \
	begin between binary binding bit boolean both by cascade cast char \
	character char_length character_length check close collate collection \
	column commit copy create createdb current_date current_time \
	current_timestamp current_user cursor database date desc decimal \
	declare delete desc distinct do double drop dump end escape exists \
	external extract fetch first float for forward from function grant \
	group having ignore in index indexable insensitive insert instead \
	int integer interval into is key language last leading like link \
	listen local micopy move next none nonulls not null numeric of \
	on only open operator option or order overlaps poll position \
	precision prior privileges public read real recover ref relative \
	rename restore restrict return returns revode rollback rule \
	schema scroll select session_user set smallint some stdin stdout \
	store substring system_user table time timestamp to trailing \
	transaction trim type unary under union unique update usage user \
	using vacuum values varchar variant varying version view virtual \
	where with zone]

proc GetNewTableForm {conn ignore} {
    global NS
    
    if {[ns_conn urlc $conn] == 2} {
	set form [ns_conn form $conn]
	if {$form != ""} {
	    set maxcolumns [ns_set get $form maxColumns]
	    if {$maxcolumns == ""} {
		set maxcolumns $NS(maxColumns)
	    }
	} else {
	    set maxcolumns $NS(maxColumns)
	}
    } elseif {[ns_conn urlc $conn] == 3} {
	set maxcolumns [lindex [ns_conn urlv $conn] 2]
    }
    
    # List of allowed Illustra types.
    foreach type [list text integer real boolean date time timestamp] {
	append types "<OPTION>$type"
    }

    set page \
	    "<html><head><title>New Table</title></head><body>
<IMG SRC=\"[ns_info location]/NS/Asset/NewTableFormHeader.gif\"><BR>\n
To create a new table you must:\n
<ul>
<li>Name the table.
<li>Provide a short description for the table.
<li>Decide if the table should show up on the list of searchable tables.
</ul>
In addition, for each column in the new table (up to $maxcolumns
columns are allowed), you need to:
<ul>
<li>Name the column.
<li>Select the column type.
<li>Provide a short description of the column.
<li>Optionally, select one or more of:
<ul>
<li>not null:  Null values are not allowed for this column.
<li>unique:  All values must be unique (also implies not null).
<li>index:  An index should be created on this column
</ul>
</ul>
When finished, press the <strong>Create Table</strong> button
to create the new table.
<BR><HR>
<form action=/NS/CreateTable method=post>
<pre>Table Name:           <input name=Table size=20>
Table Description:    <input name=Description size=40>
Table is Searchable:  <input type=checkbox name=Searchable checked>"

    for {set n 1} {$n <= $maxcolumns} {incr n} {
	append page \
		"<BR><HR>
<b>Column $n</b>:
Name:  <input name=Col$n.Name size=20>  Type:  <select name=Col$n.Type>$types</select>
Description: <input name=Col$n.Description size=40>
not null:  <input type=checkbox name=Col$n.NotNull>  unique:  <input type=checkbox name=Col$n.Unique>  index:  <input type=checkbox name=Col$n.Index>
	"
    }
    append page \
	    "<input type=submit value=\"Create Table\"></pre></form></body></html>"

    ns_return $conn 200 text/html $page
}


proc CreateTable {conn ignore} {
    global NS
    
    #
    # Sanity checks.
    #
    
    set form [ns_conn form $conn]
    set db [ns_conn db $conn]
    
    if {$form == ""} {
	return [ns_returnerror $conn 500 "No table name specified"]
    }
    set basetable [ns_set get $form Table]
    
    if {$basetable == ""} {
	return [ns_returnerror $conn 500 "No table name specified"]
    }
    if {[ns_set get $form Col1.Name] == ""} {
	return [ns_returnerror $conn 500 "No column name(s) specified"]
    }

    #
    # Initialize the word mangler.
    #
    MangleInit

    #
    # Initallize variables to hold the SQL statements.
    #
    set ismeta f
    set searchable f

    set table [Mangle $basetable 0]
    set n 0
    while {[ns_table exists $table]} {
	set table $basetable$n
	incr n
	if {$n > 10} {
	    return [ns_returnerror $conn 500 "Too many tables starting with $basetable"]
	}
    }
    # Hack:  Forget the table name so like-named columns aren't mangled.
    set NS(knownWords) ""

    if {[ns_set get $form "Searchable"]!=""} {
	set searchable t
    }

    set create "create table $table (\n"
    set first 1
    foreach col [ns_set split $form] {
	if [string match Col* [ns_set name $col]] {
	    set name [Mangle [ns_set get $col Name]]
	    set type [ns_set get $col Type]

	    if {$name == ""} {
		break
	    }

	    if {$name == "ns_url" && $type == "text"} {
		set ismeta t
	    }
	    if {!$first} {append create ",\n"}
	    append create "    $name $type"
	    if {[ns_set get $col Unique] != ""} {
		append create " not null unique"
	    } elseif {[ns_set get $col NotNull] != ""} {
		append create " not null"
	    }
	    if {[ns_set get $col Index] != ""} {
		append indices "
create index ${table}_by_$name on $table using btree($name);\n"
	    }
	    set description [ns_set get $col Description]
	    if {$description == ""} {
		lappend cinfo \
"insert into ns_columns (column_table,column_name)
values ('$table','$name');"
	    } else {
		lappend cinfo \
"insert into ns_columns (column_table,column_name,column_description)
values ('$table','$name','[enquote $description]');"
            }
	    set first 0
	}
    }
    append create "\n);\n"
    
    set tinfo_cols "table_name, table_issearchable, table_ismeta"
    set tinfo_vals "'$table', '$searchable', '$ismeta'"

    set description [ns_set get $form Description]
    
    if {$description != ""} {
	append tinfo_cols ", table_description"
	append tinfo_vals ", '[enquote $description]'"
    }
    set tinfo "insert into ns_tables ($tinfo_cols)\n\tvalues ($tinfo_vals);"

    if [catch {
	ns_db dml $db $create
	if [info exists indices] {
	    ns_db dml $db $indices
	}
	ns_db dml $db $tinfo
	if [info exists cinfo] {
	    # Get around alerter bug - multiple transactions.
	    foreach info $cinfo {
		ns_db dml $db $info
	    }
	}
    } errMsg] {
	return [ns_returnerror $conn 500 "Could not create new table:  $errMsg"]
    }

    set user [ns_conn authuser $conn]

    if {$user != "" && $user != "nsadmin"} {
	if {[ns_checkpasswd $user [ns_conn authpassword $conn]]} {
	    ns_db dml [ns_conn db $conn] "insert into ns_permissions(permission_method,permission_url,permission_user,permission_group,permission_user_ok,permission_group_ok,permission_world_ok) values ('POST','/NS/InsertRow/$table','$user','public','t','t','t');"
	    ns_db dml [ns_conn db $conn] "insert into ns_permissions(permission_method,permission_url,permission_user,permission_group,permission_user_ok,permission_group_ok,permission_world_ok) values ('POST','/NS/SearchQBF/$table','$user','public','t','f','f');"
	    ns_db dml [ns_conn db $conn] "insert into ns_permissions(permission_method,permission_url,permission_user,permission_group,permission_user_ok,permission_group_ok,permission_world_ok) values ('GET','/NS/SearchQBF/$table','$user','public','t','f','f');"
	    ns_db dml [ns_conn db $conn] "insert into ns_permissions(permission_method,permission_url,permission_user,permission_group,permission_user_ok,permission_group_ok,permission_world_ok) values ('POST','/NS/DeleteRow/$table','$user','public','t','f','f');"
	    ns_db dml [ns_conn db $conn] "insert into ns_permissions(permission_method,permission_url,permission_user,permission_group,permission_user_ok,permission_group_ok,permission_world_ok) values ('POST','/NS/UpdateRow/$table','$user','public','t','f','f');"
	}
    }
    set page "<html><head><title>New Table Created</title></head>
<body><h1>New Table $table Created</h1>
Your new table has been created and you can now
<a href=/NS/GetEntryForm/$table>enter new data.</a>"

    #
    # If any names where mangled, warn the user that you created
    # the table anyway and give them a chance to drop the table.
    #
    if [info exists NS(mangledWords)] {
	append page \
		"<p><strong>Notice:</strong>
The table and/or column names in your new table
had to be altered to avoid conflicts with existing tables or
reserved words.  Please review the following list of altered names.
If the new names are acceptable, you need do nothing.
Otherwise, press the \"Drop Altered Table\"
button to drop the altered table and try again.
<dl compact>"
        foreach word $NS(mangledWords) {
	    append page "
	    <dt>$NS(mangledWordOrig.$word)<dd>$word"
	}
	append page "
</dl><form action=/NS/VerifyDropTable method=post>
<input type=hidden name=Table value=\"$table\">
<input type=submit value=\"Drop Altered Table\">"
    }

    append page "</body></html>"
    ns_return $conn 200 text/html $page
}



proc GetDropTableForm {conn ignore} {

    set db [ns_conn db $conn]
    set tableselect "<SELECT NAME=Table>"
    set row [ns_db select $db "select table_name from ns_tables order by table_name;"]
    while {[ns_db getrow $db $row]} {
	set name [ns_set get $row table_name]
	if {![string match ns_* $name]} {
	    append tableselect "<OPTION>$name\n"
	}
    }
    append tableselect "</SELECT>"

    ns_return $conn 200 text/html \
	    "<html><head>
<title>Drop Table</title></head><body>
<IMG SRC=\"[ns_info location]/NS/Asset/DropTable.gif\"><BR>
<form action=/NS/VerifyDropTable method=post>
Table to Drop:        $tableselect
<input type=submit value=\"Drop\"></form></body></html>"
}


proc VerifyDropTable {conn ignore} {

    set form [ns_conn form $conn]

    if {$form == ""} {
	return [ns_returnerror $conn 500 "No table name specified"]
    }

    set table [string trim [ns_set get $form Table]]
    if {$table == ""} {
	return [ns_returnerror $conn 500 "No table name specified"]
    }
    if ![ns_table exists $table] {
	return [ns_returnerror $conn 500 "$table does not exist."]
    }
    if [string match ns_* $table] {
	return [ns_returnerror $conn 500 "$table is a Naviserver system table and can not be dropped."]
    }

    ns_return $conn 200 text/html \
	    "<html><head>
<title>Verify Drop Table</title></head><body>
<h1>Verify Drop of an Existing Table</h1>
<form action=/NS/DropTable method=post>
<input type=hidden name=Table value=$table>
Are you sure you want to drop the <i>$table</i> table with the following SQL:
<pre>
[dropSql $table]
</pre>
<b>All present and historical data will be permanently lost!</b>
<input type=submit value=\"Yes, Really Drop\">
</form></body></html>"
}

proc dropSql table {
    return "delete from ns_columns where column_table = '$table';
delete from ns_tables where table_name = '$table';
drop table $table;"
}


proc DropTable {conn ignore} {
    set form [ns_conn form $conn]

    if {$form == ""} {
	return [ns_returnerror $conn 500 "No table name specified"]
    }

    set table [string trim [ns_set get $form Table]]
    if {$table == ""} {
	return [ns_returnerror $conn 500 "No table name specified"]
    }
    if ![ns_table exists $table] {
	return [ns_returnerror $conn 500 "$table does not exist."]
    }
    if [string match ns_* $table] {
	return [ns_returnerror $conn 500 "$table is a Naviserver system table and can not be dropped."]
    }
    if [catch {ns_db dml [ns_conn db $conn] [dropSql $table]}] {
	return [ns_returnerror $conn 500 "Could not drop table."]
    }
    ns_returnnotice $conn 200 "Table $table dropped."
}

proc MangleInit {} {
    global NS

    set NS(knownWords) ""
    if [info exists NS(mangledWords)] {
	foreach word $NS(mangledWords) {
	    catch {unset NS(mangledWordOrig.$word)}
	}
	unset NS(mangledWords)
    }
}


proc Mangle {word {remember 1}} {
    global NS

    # Save the original word.
    set orig $word

    # Strip punctuation and change space, tab and dash to underscore.
    regsub -all {[^a-zA-Z0-9_ -]} $word "" word
    regsub -all {[- 	]+} $word _ word

    # Check for used and reserved words.
    set n 0
    set base $word
    while {1} {
	set loword [string tolower $word]
	if {([lsearch $NS(knownWords) $word] == -1)
	   && ([lsearch $NS(reservedWords) $loword] == -1)} {
	    break;
	}
	set word $base$n
	incr n
    }

    # Remember mangled words.
    if {$word != $orig} {
	lappend NS(mangledWords) $word
	set NS(mangledWordOrig.$word) $orig
    }

    # Remember known words.
    if $remember {
	lappend NS(knownWords) $word
    }
    return $word
}

proc enquote {string} {
    regsub "'" $string "''" retval
    return $retval
}

proc GetExtendTableForm {conn ignore} {
    global NS
    ns_return $conn 200 text/html \
	    "<html><head>
<title>Extend Table</title></head><body>
<H1>Extend Table</H1>
<form action=/NS/GetRealExtendTableForm method=post>
Table to Extend: <input name=Table size=20><BR>
Maximum number of colums to add: <input name=maxColumns size=5 value=5><BR>
<input type=submit value=\"Extend\"></form></body></html>"
}

#
# Table name and max columns come in on form data
#

proc GetRealExtendTableForm {conn ignore} {
    global NS
    
    set form [ns_conn form $conn]
    if {$form == ""} {
	return [ns_returnerror $conn 500 "No table name specified"]
    }
    set table [ns_set get $form Table]

    if ![ns_table exists $table] {
	return [ns_returnerror $conn 500 "Table $table does not exist"]
    }

    set maxcolumns [ns_set get $form maxColumns]
    if {$maxcolumns == ""} {
	set maxcolumns $NS(maxColumns)
    }
    
    # List of allowed Illustra types.
    foreach type [list text integer real boolean date time timestamp] {
	append types "<OPTION>$type"
    }

    set page \
	    "<html><head><title>Extend Table</title></head><body>
<H1>Extend Table</H1>
<BR>\n
To extend table <b>$table</b> you must do the following for each column you want to add (up to $maxcolumns columns are allowed):
<ul>
<li>Name the column.
<li>Select the column type.
<li>Provide a short description of the column.
<li>Optionally, check the <b>index</b> box to create an index on this column
</ul>
When finished, press the <b>Alter Table</b> button
to extend the table.
Table <B>$table</B> currently has this structure:<PRE>"
    set size [ns_column count $table]
    for {set i 0} {$i < $size} {incr i} {
	append page "<code>Column [expr $i+1]: <b>[ns_column name $table $i]</b> is type [ns_column valuebyindex $table $i column_type]</code><BR>\n"
    }
    append page "</PRE>\n"
    append page "<form action=/NS/ExtendTable/$table method=post>"
    for {set n 1} {$n <= $maxcolumns} {incr n} {
	append page \
		"<HR><PRE>
<b>New Column [expr $n+$size]</b>:
Name:  <input name=Col$n.Name size=20>  Type:  <select name=Col$n.Type>$types</select>
Description: <input name=Col$n.Description size=40>
"
# not null:  <input type=checkbox name=Col$n.NotNull>  unique:  <input type=checkbox name=Col$n.Unique>  index:  <input type=checkbox name=Col$n.Index>
append page "index:  <input type=checkbox name=Col$n.Index>
"
    }
    append page \
	    "<input type=submit value=\"Extend Table\"></pre></form></body></html>"

    ns_return $conn 200 text/html $page
}

proc ExtendTable {conn ignore} {
    global NS
    
    #
    # Sanity checks.
    #
    
    set form [ns_conn form $conn]
    set db [ns_conn db $conn]
    
    if {[ns_conn urlc $conn] < 3} {
	return [ns_returnerror $conn 500 "No table name specified"]
    }
    set table [lindex [ns_conn urlv $conn] 2]

    if {$table == ""} {
	return [ns_returnerror $conn 500 "No table name specified"]
    }

    if ![ns_table exists $table] {
	return [ns_returnerror $conn 500 "Table $table does not exist."]
    }
    
    if {$form == ""} {
	return [ns_returnerror $conn 500 "No column name(s) specified"]
    }
    if {[ns_set get $form Col1.Name] == ""} {
	return [ns_returnerror $conn 500 "No column name(s) specified"]
    }

    #
    # Initialize the word mangler.
    #
    MangleInit

    # Put in the existing column names.

    set size [ns_column count $table]
    for {set i 0} {$i < $size} {incr i} {
	lappend NS(knownWords) [ns_column name $table $i]
    }

    #
    # Initallize variables to hold the SQL statements.
    #
    set ismeta f
    set searchable f

    foreach col [ns_set split $form] {
	if [string match Col* [ns_set name $col]] {
	    set name [Mangle [ns_set get $col Name]]
	    set type [ns_set get $col Type]

	    if {$name == ""} {
		break
	    }

	    if {$name == "ns_url" && $type == "text"} {
		set ismeta t
	    }
	    append alter "alter table $table add column $name $type"
	    if {[ns_set get $col Unique] != ""} {
		append alter " not null unique"
	    } elseif {[ns_set get $col NotNull] != ""} {
		append alter " not null"
	    }
	    append alter ";\n"
	    if {[ns_set get $col Index] != ""} {
		append indices "
create index ${table}_by_$name on $table using btree($name);\n"
	    }
	    set description [ns_set get $col Description]
	    if {$description != ""} {
		lappend cinfo "
		insert into ns_columns (column_table,column_name,column_description)
		values ('$table','$name','[enquote $description]');"
	    }
	}
    }

    if [catch {
	ns_db dml $db $alter
	if [info exists indices] {
	    ns_db dml $db $indices
	}
	if [info exists cinfo] {
	    # Get around alerter bug - multiple transactions.
	    foreach info $cinfo {
		ns_db dml $db $info
	    }
	} else {
	    ns_db dml $db "alert ns_tables_alert;\n"
	}
    } errMsg] {
	return [ns_returnerror $conn 500 "Could not extend table:  $errMsg"]
    }

    set page "<html><head><title>Table $table has been extended</title></head>
<body><h1>Table $table has been extended</h1>
Your table has been modified and you can now
<a href=/NS/GetEntryForm/$table>enter new data.</a>"

    #
    # If any names where mangled, warn the user that you modified
    # the table anyway and give them a chance to drop the table.
    #
    if [info exists NS(mangledWords)] {
	append page \
		"<p><strong>Notice:</strong>
Some of your new column names
had to be altered to avoid conflicts with existing names or
reserved words.  Please review the following list of altered names.
If the new names are acceptable, you need do nothing.
Otherwise, press the \"Drop Altered Table\"
button to drop the altered table and try again.  All existing data will be lost if you do this!!
<dl compact>"
        foreach word $NS(mangledWords) {
	    append page "
	    <dt>$NS(mangledWordOrig.$word)<dd>$word"
	}
	append page "
</dl><form action=/NS/VerifyDropTable method=post>
<input type=hidden name=Table value=\"$table\">
<input type=submit value=\"Drop Altered Table\">"
    }

    append page "</body></html>"
    ns_return $conn 200 text/html $page
}
