--
-- $Id: nsbuild.sql,v 1.22 1995/06/16 18:20:41 jimbo Exp $
--

begin transaction;

create table ns_schema_info (info_key text, info_value text);
create index ns_schema_info_index on ns_schema_info using btree(info_key);
insert into ns_schema_info values
('Version', '$Id: nsbuild.sql,v 1.22 1995/06/16 18:20:41 jimbo Exp $');

create alerter ns_tables_alert (mechanism = 'callback');
create alerter ns_users_alert (mechanism = 'callback');
create alerter ns_permissions_alert (mechanism = 'callback');

create function ns_table_exists(text) returns boolean
as return (exists (select 1 from tables where table_name = $1));

create function ns_column_exists(text,text) returns boolean
as return (exists (select 1 from columns,tables where
column_name = $2 and column_member = table_type and table_name = $1));

create function ns_table_ismeta(text) returns boolean
as return (exists (select 1 from columns,tables
where column_name = 'ns_url' and column_type::text = 'text'
and column_member = table_type and table_name = $1
and table_name <> 'ns_pages'));

create table ns_tables (
	table_name text not null primary key,
	table_issearchable boolean default 'f',
	table_maxorderby int default 1,
	table_ismeta boolean default 'f',
	table_inserturl text,
	table_description text,
	check(ns_table_exists(table_name) and
	table_ismeta = ns_table_ismeta(table_name))
);
create index ns_tables_index on ns_tables using btree(table_name);

create rule ns_tables_insert on insert to ns_tables
do alert ns_tables_alert;
create rule ns_tables_delete on delete to ns_tables
do alert ns_tables_alert;
create rule ns_tables_update on update to ns_tables
do alert ns_tables_alert;

create table ns_columns (
	column_table 		text not null,
	column_name		text not null,
	column_description	text,
	column_htmltag_type	text,
	column_htmltag_data	text,
	unique(column_table, column_name),
	check(ns_column_exists(column_table,column_name))
);
create index ns_columns_index on ns_columns
	using btree(column_table, column_name);

create rule ns_columns_insert on insert to ns_columns
do alert ns_tables_alert;
create rule ns_columns_delete on delete to ns_columns
do alert ns_tables_alert;
create rule ns_columns_update on update to ns_columns
do alert ns_tables_alert;

create table ns_groups (
	group_name text not null primary key,
	group_description text,
	group_data1 text,
	group_data2 text,
	group_data3 text
);
create index ns_groups_index on ns_groups using btree(group_name);
insert into ns_groups (group_name, group_description)
	values ('public', 'Public Group');
insert into ns_groups (group_name, group_description)
	values ('users', 'Authorized Users');
insert into ns_groups (group_name, group_description)
	values ('system', 'System Users');

create rule ns_groups_insert on insert to ns_groups
do alert ns_users_alert;
create rule ns_groups_delete on delete to ns_groups
do alert ns_users_alert;
create rule ns_groups_update on update to ns_groups
do alert ns_users_alert;

create table ns_users (
	user_name text not null primary key,
	user_fullname text,
	user_passwd text,
	user_defgroup text not null references ns_groups,
	user_data1 text,
	user_data2 text,
	user_data3 text
);
create index ns_users_index on ns_users using btree(user_name);
insert into ns_users (user_name, user_fullname, user_passwd, user_defgroup)
	values ('public', 'Public Access User', NULL, 'public');
insert into ns_users (user_name, user_fullname, user_passwd, user_defgroup)
	values ('nsadmin', 'Super User', 'XQN6fv4OawO/w', 'system');

create rule ns_users_insert on insert to ns_users
do alert ns_users_alert;
create rule ns_users_delete on delete to ns_users
do alert ns_users_alert;
create rule ns_users_update on update to ns_users
do alert ns_users_alert;

create table ns_groups2users (
	group_name text not null references ns_groups,
	user_name text not null references ns_users,
	unique(group_name,user_name)
);
create index ns_groups2users_user_index on ns_groups2users
	using btree(user_name);
create index ns_groups2users_group_index on ns_groups2users
	using btree(group_name);

create rule ns_groups2users_insert on insert to ns_groups2users
do alert ns_users_alert;
create rule ns_groups2users_delete on delete to ns_groups2users
do alert ns_users_alert;
create rule ns_groups2users_update on update to ns_groups2users
do alert ns_users_alert;


create table ns_methods (
	method_name text not null primary key,
	method_alias text,
	check(method_alias = upper(method_alias))
);
create index ns_methods_index on ns_methods using btree(method_name);
insert into ns_methods values ('BROWSE', 'GET');
insert into ns_methods values ('HEAD', 'GET');
insert into ns_methods values ('DELETE', 'PUT');
insert into ns_methods values ('MKDIR', 'PUT');

create rule ns_methods_insert on insert to ns_methods
do alert ns_permissions_alert;
create rule ns_methods_delete on delete to ns_methods
do alert ns_permissions_alert;
create rule ns_methods_update on update to ns_methods
do alert ns_permissions_alert;


--
-- Function to check if an URL is a normalized relative URL.
--
create function ns_normalized_url(text)
returns boolean as return $1 = '/' or (
	    $1     like '/%'
	and $1 not like '%/'
 	and $1 not like '%//%'
	and $1 not like '%/../%'
	and $1 not like '%/..'
	and $1 not like '%/./%'
	and $1 not like '%/.'
);

create table ns_permissions (
	permission_method text not null,
	permission_url text not null,
	permission_user text references ns_users,
	permission_group text references ns_groups,
	permission_user_ok boolean not null,
	permission_group_ok boolean not null,
	permission_world_ok boolean not null,
	unique(permission_url, permission_method),
	check(ns_normalized_url(permission_url) and
		permission_method = upper(permission_method))
);
create index ns_permissions_index on ns_permissions
	using btree(permission_url, permission_method);

create rule ns_permissions_insert on insert to ns_permissions
do alert ns_permissions_alert;
create rule ns_permissions_delete on delete to ns_permissions
do alert ns_permissions_alert;
create rule ns_permissions_update on update to ns_permissions
do alert ns_permissions_alert;

insert into ns_permissions
values ('GET','/','nsadmin','system','t','t','t');
insert into ns_permissions
values ('PUT','/','nsadmin','system','t','t','f');
insert into ns_permissions
values ('POST','/','nsadmin','system','t','t','f');


--
-- By default, all the /NS URL's have the same permissions as
-- the / URL's except we do not allow GET /NS/SearchQBF by
-- default, and we allow PUT /NS/SetPerm and /NS/ChangePassword
-- which do their own permission checking.
--
insert into ns_permissions
values ('GET','/NS/GetTcl','nsadmin','system','t','t','f');
insert into ns_permissions
values ('GET','/NS/TclInit','nsadmin','system','t','t','f');
insert into ns_permissions
values ('GET','/NS/SearchQBF','nsadmin','system','t','t','f');
insert into ns_permissions
values ('GET','/NS/GetUpdateForm','nsadmin','system','t','t','f');
insert into ns_permissions
values ('POST','/NS/SetPerm','nsadmin','system','t','t','t');
insert into ns_permissions
values ('POST','/NS/ChangePassword','nsadmin','system','t','t','t');


--
-- Make Archive and ArchiveVersions wide open because
-- the op procedures do secondary checking on the base urls.
--
insert into ns_permissions values (
'GET','/NS/ArchiveVersions','nsadmin','system','t','t','t');
insert into ns_permissions values (
'GET','/NS/Archive','nsadmin','system','t','t','t');

create table ns_prices (
	price_method	text not null,
	price_url	text not null,
	price_amount	int default 0,
	unique(price_method, price_url),
	check(ns_normalized_url(price_url) and
		price_method = upper(price_method))
);

create rule ns_prices_insert on insert to ns_prices
do alert ns_permissions_alert;
create rule ns_prices_delete on delete to ns_prices
do alert ns_permissions_alert;
create rule ns_prices_update on update to ns_prices
do alert ns_permissions_alert;

--
-- By default, nothing is charged.
--
--insert into ns_prices values ('GET','/', 0);
--insert into ns_prices values ('POST','/', 0);
--insert into ns_prices values ('PUT','/', 0);


create table ns_charges (
	charge_user	text references ns_users,
	charge_method	text not null,
	charge_url	text not null,
	charge_price	int not null,
	charge_when	timestamp default current_timestamp
);


create function ns_user_charges(text) returns int
as select unique sum(charge_price) from ns_charges
where charge_user = $1;


create view ns_total_charges as select user_name total_user,
	ns_user_charges(user_name) total_charge from ns_users;


create table ns_pages (
	ns_url		text not null primary key,
	ns_title	text,
	ns_body		doc not null,
	check(ns_url not like '/%' or ns_normalized_url(ns_url))
);
create index ns_pages_by_ns_body on ns_pages
	using dtree(ns_body text_ops);
create index ns_pages_index on ns_pages
	using btree(ns_url);


insert into ns_tables (table_name, table_description, table_issearchable)
values ('ns_pages', 'Naviserver Collection of all pages.','t');

insert into ns_tables (table_name, table_description, table_issearchable)
values ('ns_users', 'Naviserver Authorized Users','t');

insert into ns_tables (table_name, table_description, table_issearchable)
values ('ns_groups', 'Naviserver User Groups','t');

insert into ns_tables (table_name, table_description, table_issearchable)
values ('ns_groups2users', 'Naviserver Users to Groups Map','t');

insert into ns_tables (table_name, table_description, table_issearchable)
values ('ns_tables', 'Naviserver Table of Tables','t');

insert into ns_tables (table_name, table_description, table_issearchable)
values ('ns_permissions', 'Naviserver Relative URL Permissions','t');

insert into ns_tables (table_name, table_description, table_issearchable)
values ('ns_prices', 'Naviserver Relative URL Charges','t');

insert into ns_tables (table_name, table_description, table_issearchable)
values ('ns_methods', 'Naviserver HTTP Method Aliases','t');

insert into ns_tables (table_name, table_description, table_issearchable)
values ('ns_charges', 'Naviserver Log of Running Charges','t');

insert into ns_tables (table_name, table_description, table_issearchable)
values ('ns_total_charges', 'Naviserver View of User Charge Totals','t');

insert into ns_tables (table_name, table_description, table_issearchable)
values ('ns_columns', 'Naviserver Table Columns Meta Data','t');

insert into ns_columns (column_table, column_name, 
			column_htmltag_type, column_htmltag_data)
values ('ns_permissions','permission_user',
	'SELECTSQL','select user_name from ns_users order by user_name;');

insert into ns_columns (column_table, column_name, 
			column_htmltag_type, column_htmltag_data)
values ('ns_permissions','permission_group',
	'SELECTSQL','select group_name from ns_groups order by group_name;');

insert into ns_columns (column_table, column_name, 
			column_htmltag_type, column_htmltag_data)
values ('ns_permissions','permission_method',
	'SELECT','GET PUT POST');

insert into ns_columns (column_table, column_name, 
			column_htmltag_type, column_htmltag_data)
values ('ns_columns','column_htmltag_type',
	'SELECTORNULL','SELECTSQL SELECTSQLORNULL SELECT SELECTORNULL RADIO RADIOORNULL TEXTAREA INPUT AUTOINCR');

insert into ns_columns (column_table, column_name, 
			column_htmltag_type, column_htmltag_data)
values ('ns_columns','column_table',
	'SELECTSQL','select table_name from ns_tables order by table_name;');

insert into ns_columns (column_table, column_name, 
			column_htmltag_type, column_htmltag_data)
values ('ns_groups2users','user_name',
	'SELECTSQL','select user_name from ns_users order by user_name;');

insert into ns_columns (column_table, column_name, 
			column_htmltag_type, column_htmltag_data)
values ('ns_groups2users','group_name',
	'SELECTSQL','select group_name from ns_groups order by group_name;');

insert into ns_columns (column_table, column_name, 
			column_htmltag_type, column_htmltag_data)
values ('ns_prices','price_method', 'SELECT','GET PUT POST');

insert into ns_columns (column_table, column_name, 
			column_htmltag_type, column_htmltag_data)
values ('ns_charges','charge_user',
	'SELECTSQL','select user_name from ns_users order by user_name;');

insert into ns_columns (column_table, column_name, 
			column_htmltag_type, column_htmltag_data)
values ('ns_total_charges','total_user',
	'SELECTSQL','select user_name from ns_users order by user_name;');

insert into ns_columns (column_table, column_name, column_description)
values ('ns_permissions','permission_url','The permission URL should begin with a slash and contain no trailing slash,
as in "/mydir" or "/mydir/home.html"');

insert into ns_columns (column_table, column_name, column_description)
values ('ns_prices','price_url','The price URL should begin with a slash and contain no trailing slash, 
as in "/mydir" or "/mydir/home.html"');

create table ns_archives (
	archive_url text not null primary key,
	archive_lo large_object,
	check(ns_normalized_url(archive_url))
);
create index ns_archives_index on ns_archives using btree(archive_url);


--
-- Privilages for nsadmin.
-- 
grant select on table tables to nsadmin;
grant select on table columns to nsadmin;
grant all on table ns_schema_info to nsadmin;
grant all on table ns_tables to nsadmin;
grant all on table ns_columns to nsadmin;
grant all on table ns_users to nsadmin;
grant all on table ns_groups to nsadmin;
grant all on table ns_groups2users to nsadmin;
grant all on table ns_prices to nsadmin;
grant all on table ns_permissions to nsadmin;
grant all on table ns_methods to nsadmin;
grant all on table ns_charges to nsadmin;
grant all on table ns_pages to nsadmin;
grant all on table ns_archives to nsadmin;
grant select on table ns_total_charges to nsadmin;

grant usage on alerter ns_tables_alert to nsadmin;
grant usage on alerter ns_users_alert to nsadmin;
grant usage on alerter ns_permissions_alert to nsadmin;

grant usage on function ns_table_exists(text) to nsadmin;
grant usage on function ns_column_exists(text,text) to nsadmin;
grant usage on function ns_table_ismeta(text) to nsadmin;
grant usage on function ns_user_charges(text) to nsadmin;

end transaction;
