/*****************************************************************************
** FILES.PRG
**
** Clip-2-VO conversion sample
** functions to generate, open, browse, and close the sample database files
**
** Copyright 1995 SofDesign International, Inc.
** All rights reserved
**
*****************************************************************************/

#include "inkey.ch"
#include "sample.ch"


#define  nGaugeRow   10
#define  nGaugeCol   27

STATIC   aEditList   := {}             // records currently being edited



/*****************************************************************************
** OpenFiles() --> NIL
**
** This function opens the files and creates them, if necessary.
**
*****************************************************************************/
FUNCTION OpenFiles()


   // Open the files.  Create them if they don't exist.
   IF FILE("CUSTOMER.DBF") .AND. FILE("INVOICES.DBF") .AND. ;
         FILE("PAYMENTS.DBF")

      // They're all there.
      USE Invoices NEW EXCLUSIVE

      USE Payments NEW EXCLUSIVE

      USE Customer NEW EXCLUSIVE

   ELSE

      // Generate some pseudorandom records.  This will also force new
      // index files to be built.
      GenData()   // Reminds me of the old NetWare days!

   END IF // FILE("CUSTOMER.DBF") .AND. FILE("INVOICES.DBF") .A...

   // Open the index files.  Create them if they don't exist.
   IF FILE("CUSTOMER" + INDEXEXT()) .AND. ;
         FILE("INVOICES" + INDEXEXT()) .AND. ;
         FILE("PAYMENTS" + INDEXEXT())

      SELECT Invoices
      SET INDEX TO Invoices

      SELECT Payments
      SET INDEX TO Payments

      SELECT Customer
      SET INDEX TO Customer

   ELSE

      GenData()   // Reminds me of the old NetWare days!

   END IF // FILE("CUSTOMER" + INDEXEXT()) .AND. ...

   RETURN(NIL)
   // END OpenFiles()



/*****************************************************************************
** CloseFiles() --> NIL
**
** This function closes the files.
**
*****************************************************************************/
FUNCTION CloseFiles()


   // Close the files.
   SELECT Invoices
   CLOSE

   SELECT Payments
   CLOSE

   SELECT Customer
   CLOSE

   RETURN(NIL)
   // END CloseFiles()



/*****************************************************************************
** STATIC GenData() --> NIL
**
** This function generates 100 pseudorandom customer records.  It displays
** a progress gauge in a window and, through the use of pw():yield(), allows
** the user to cancel the generation.  It then builds an index for the
** generated data, also displaying a progress gauge, but not allowing the
** user to cancel the operation.
**
*****************************************************************************/
STATIC FUNCTION GenData()

   LOCAL GetList  := {}

   LOCAL lQuit    := .F.

   LOCAL i, ;
         j, ;
         nLastRec, ;
         nPrefix


   IF .NOT. (FILE("CUSTOMER.DBF") .AND. FILE("INVOICES.DBF") .AND. ;
         FILE("PAYMENTS.DBF"))

      // Initialize the progress gauge.
      Gauge(0, 100, "Creating databases...")

      // Create the files.
      DBCREATE("INVOICES.DBF", ;
         {  { "CUSTNMBR",  "N",  5, 0 }, ;
            { "INVNMBR",   "N",  5, 0 }, ;
            { "AMOUNT",    "N", 15, 2 } })

      DBCREATE("PAYMENTS.DBF", ;
         {  { "CUSTNMBR",  "N",  5, 0 }, ;
            { "PAYMENT",   "D",  8, 0 }, ;
            { "AMOUNT",    "N", 15, 2 } })

      DBCREATE("CUSTOMER.DBF", ;
         {  { "CUSTNMBR",  "N",  5, 0 }, ;
            { "ACTIVE",    "L",  1, 0 }, ;
            { "COMPANY",   "C", 35, 0 }, ;
            { "DIVISION",  "C", 15, 0 }, ;
            { "CONTACT",   "C", 35, 0 }, ;
            { "SALUTE",    "N",  1, 0 }, ;
            { "TITLE",     "C", 10, 0 }, ;
            { "ADDR1",     "C", 35, 0 }, ;
            { "ADDR2",     "C", 35, 0 }, ;
            { "CITY",      "C", 25, 0 }, ;
            { "STATE",     "C", 20, 0 }, ;
            { "COUNTRY",   "C", 22, 0 }, ;
            { "ZIP",       "C", 10, 0 }, ;
            { "HOMEPHONE", "N", 10, 0 }, ;
            { "WORKPHONE", "N", 10, 0 }, ;
            { "CARPHONE",  "N", 10, 0 }, ;
            { "FAXPHONE",  "N", 10, 0 }, ;
            { "MAXCREDIT", "N", 15, 2 }, ;
            { "LASTPMT",   "N", 15, 2 }, ;
            { "TERMS",     "N",  2, 0 }, ;
            { "DISCOUNT",  "N",  2, 0 }, ;
            { "NOTES",     "M", 10, 0 } })

      // Open the files.
      USE Invoices NEW EXCLUSIVE

      USE Payments NEW EXCLUSIVE

      USE Customer NEW EXCLUSIVE

      // Add 100 pseudorandom customer records.
      i  := 1
      WHILE (i <= 100) .AND. (.NOT. lQuit)

         // Update the progress gauge.
         Gauge(i, 100)

         // Add a record.
         APPEND BLANK
         REPLACE Customer->CustNmbr    WITH NewNumber(10000, 99999)

         // Add 1 to 15 pseudorandom invoice records.
         SELECT Invoices
         FOR j := 1 TO NewNumber(1, 15)
            APPEND BLANK
            REPLACE Invoices->CustNmbr WITH Customer->CustNmbr
            REPLACE Invoices->InvNmbr  WITH NewNumber(10000, 99999)
            REPLACE Invoices->Amount   WITH NewNumber(100, 9999)
         NEXT j

         // Add 1 to 15 pseudorandom payment records.
         SELECT Payments
         FOR j := 1 TO NewNumber(1, 15)
            APPEND BLANK
            REPLACE Payments->CustNmbr WITH Customer->CustNmbr

            REPLACE Payments->Payment  WITH CTOD(STRZERO(NewNumber(1, ;
               12), 2) + "/" + STRZERO(NewNumber(1, 28), 2) + "/" + ;
               STRZERO(NewNumber(90, 92), 2))
            REPLACE Payments->Amount   WITH NewNumber(100, 9999)
         NEXT j

         SELECT Customer
         REPLACE Customer->Active      WITH (NewNumber(1, 2) == 1)
         REPLACE Customer->Company     WITH NewCompany()
         REPLACE Customer->Division    WITH NewDivision()
         REPLACE Customer->Contact     WITH NewName()
         REPLACE Customer->Salute      WITH NewNumber(1, 3)
         REPLACE Customer->Title       WITH NewTitle()
         REPLACE Customer->Addr1       WITH NewAddress()
         REPLACE Customer->Addr2       WITH IIF(NewNumber(1, 4) == 1, ;
            "deliver to " + NewAddress(), "")
         REPLACE Customer->City        WITH NewCity()
         REPLACE Customer->State       WITH NewState()
         REPLACE Customer->Country     WITH "USA"
         REPLACE Customer->Zip         WITH LTRIM(STR(NewNumber(10000, ;
            99999)))
         nPrefix  := NewNumber(100, 999) * 10000000
         REPLACE Customer->HomePhone   WITH nPrefix + NewNumber(1000000, ;
            9999999)
         REPLACE Customer->WorkPhone   WITH nPrefix + NewNumber(1000000, ;
            9999999)
         REPLACE Customer->CarPhone    WITH nPrefix + NewNumber(1000000, ;
            9999999)
         REPLACE Customer->FAXPhone    WITH nPrefix + NewNumber(1000000, ;
            9999999)
         REPLACE Customer->MaxCredit   WITH NewNumber(10, NewNumber(15, ;
            1000)) * 1000
         REPLACE Customer->LastPmt     WITH NewNumber(500, 2000)
         REPLACE Customer->Terms       WITH IIF(NewNumber(1,2) == 1 , ;
            10 , IIF(NewNumber(1,2)==1, 30 , 90 ) )
         REPLACE Customer->Discount    WITH NewNumber(0, 33)

         // Increment the record count.
         ++i

      END WHILE // (i <= 100) .AND. (.NOT. lQuit)

   END IF // .NOT. (FILE("CUSTOMER.DBF") .AND. FILE("INVOICES.DBF"...

   // Initialize the progress gauge.
   nLastRec := Customer->(LASTREC())
   Gauge(0, nLastRec, "Indexing CUSTOMER.DBF..." )

   SELECT Customer
   DBCREATEINDEX("CUSTOMER" + INDEXEXT(), "(CustNmbr)", ;
      { || Gauge(RECNO(), nLastRec), Customer->CustNmbr }, .F.)

   // Initialize the progress gauge.
   nLastRec := Invoices->(LASTREC())
   Gauge(0, nLastRec, "Indexing INVOICE.DBF..." )

   // Build the indexes, updating the progress gauge.  The yield() calls
   // are only necessary if you want timer, timeout, idle, and background
   // processing to occur while the indexes are being built.  Removing the
   // yield() calls speeds up the indexing process a great deal.
   SELECT Invoices
   DBCREATEINDEX("INVOICES" + INDEXEXT(), "(CustNmbr)", ;
      { || Gauge(RECNO(), nLastRec), Invoices->CustNmbr }, .F.)

   // Initialize the progress gauge.
   nLastRec := Payments->(LASTREC())
   Gauge(0, nLastRec, "Indexing PAYMENTS.DBF..." )

   SELECT Payments
   DBCREATEINDEX("PAYMENTS" + INDEXEXT(), "(CustNmbr)", ;
      { || Gauge(RECNO(), nLastRec), Payments->CustNmbr }, .F.)

   SELECT Customer

   RETURN(NIL)
   // END GenData()



/*****************************************************************************
** STATIC Gauge( nCurrent, nMax ) --> NIL
**
** This displays a progress indicator
**
*****************************************************************************/
STATIC FUNCTION Gauge( nCurrent, nMax , cMsg )

   local cSaveColor

   DISPBEGIN()

   IF nCurrent == 0
      @ nGaugeRow, nGaugeCol, nGaugeRow + 4 , nGaugeCol + 26 ;
         BOX "͸Գ "
      IF ( cMsg != nil )
         @ nGaugeRow + 01, nGaugeCol + 1 SAY SPACE( 25 )
         @ nGaugeRow + 01, nGaugeCol + 1 + INT((26-LEN(cMsg))/2) SAY cMsg
      END IF
      @ nGaugeRow + 02, nGaugeCol + 03 SAY "Ĵ"
      @ nGaugeRow + 03, nGaugeCol + 03 SAY "0   25   50   75   100"
   END IF // nCurrent == 0

   IF nCurrent <= nMax
      cSaveColor := SETCOLOR( "N/G+" )
      @ nGaugeRow + 02, nGaugeCol + 03 SAY SPACE(INT((nCurrent / nMax) * 21)) COLOR "N/W"
      SETCOLOR( cSaveColor )
   END IF // nCurrent <= nMax

   DISPEND()

   RETURN(NIL)
   // END Gauge( nCurrent, nMax )





/*****************************************************************************
** AddEditList( nRecNo ) --> lSuccess
**
** This adds the specified record number to the list of records currently
** being edited.  If the record is already being edited by another
** procedure, .F. is returned.
**
*****************************************************************************/
FUNCTION AddEditList( nRecNo )

   LOCAL nPos  := ASCAN(aEditList, nRecNo)


   IF nPos == 0
      AADD(aEditList, nRecNo)
   END IF // nPos == 0

   RETURN(nPos == 0)
   // END AddEditList( nRecNo )



/*****************************************************************************
** DelEditList( nRecNo ) --> lSuccess
**
** This removes the specified record number from the list of records
** currently being edited.  If the record is not being edited by any
** procedure, .F. is returned.
**
*****************************************************************************/
FUNCTION DelEditList( nRecNo )

   LOCAL nPos  := ASCAN(aEditList, nRecNo)


   IF nPos != 0
      ADEL(aEditList, nPos)
      ASIZE(aEditList, LEN(aEditList) - 1)
   END IF // nPos != 0

   RETURN(nPos != 0)
   // END DelEditList( nRecNo )

#define RND_DIVISOR                    10000000
#define RND_FACTOR                     3236713



/*****************************************************************************
** NewAddress() --> cAddress
**
** Returns a pseudorandom address.
**
*****************************************************************************/
FUNCTION NewAddress()

   STATIC   aDirs    := { "", " N. ", " S. ", " E. ", " W. " }, ;
            aLabels  := { "St.", "Rd.", "Ln.", "Ave.", "Blvd.", "Dr." }, ;
            aStreets := { "First ", "Second ", "Third ", "Fourth ", ;
               "Fifth ", "Sixth ", "Seventh ", "Eighth ", "Ninth ", ;
               "Tenth ", "Oak ", "Elm ", "Maple ", "Pine ", "Aspen ", ;
               "Willow ", "Crabapple ", "Birch ", "Palm ", "Hickory ", ;
               "Mecury ", "Venus ", "Earth ", "Mars ", "Jupiter ", ;
               "Saturn ", "Uranus ", "Neptune ", "Pluto " }


   RETURN(LTRIM(STR(NewNumber(100, 20000))) + aDirs[NewNumber(1, 5)] + ;
      aStreets[NewNumber(1, 29)] + aLabels[NewNumber(1, 6)])
   // END NewAddress()



/*****************************************************************************
** NewCity() --> cCity
**
** Returns a pseudorandom city name.
**
*****************************************************************************/
FUNCTION NewCity()

   STATIC   aCities  := { "Walnut Creek", "Montgomery", "St. Ann", ;
               "Riverside", "Rose Hill", "West Lake", "Liverpool", ;
               "Hartford", "Norfolk", "Port Hope", "Ellenwood", ;
               "Overland", "Jefferson", "Rockville", "Sweet Springs", ;
               "New Villa", "Webster", "Uniontown", "Northwood", ;
               "Portland", "Lebanon", "Reading", "Cookeville", ;
               "Greenville", "Church Hill", "Alexandria", "Liberty", ;
               "Madison", "Richland", "Lynnwood", "Anderson", ;
               "Irvington" }


   RETURN(aCities[NewNumber(1, 32)])
   // END NewCity()



/*****************************************************************************
** NewCompany() --> cCompany
**
** Returns a pseudorandom company name.
**
*****************************************************************************/
FUNCTION NewCompany()

   STATIC   aPre     := { "Specialty ", "Custom ", "General ", "Master ", ;
               "Professional ", "Applied ", "Strategic ", "Resource ", ;
               "Cheap ", "Expensive ", "Amateur ", "Fast ", "Slow " , ;
               "A-One" }, ;
            aLabels  := { "", ", Inc.", ", Ltd.", " Int'l., Inc.", ;
               ", UnLtd.", " Corp.", " Int'l." }, ;
            aTypes   := { "Industries", "Programming", "Services", ;
               "Software", "Hardware", "Repair", "Cleaning", "Furniture", ;
               "Publishing", "Systems", "Communications", "Consulting", ;
               "Associates", "Instruments", "Management", "Design", ;
               "Technologies", "Group", "Development", "Enterprises", ;
               "Sciences", "Homemakers", "Childcare", "Landscaping", ;
               "Electronics" }


   RETURN(aPre[NewNumber(1, 14)] + aTypes[NewNumber(1, 25)] + ;
      aLabels[NewNumber(1, 7)])
   // END NewCompany()



/*****************************************************************************
** NewName() --> cContact
**
** Returns a pseudorandom personal name.
**
*****************************************************************************/
FUNCTION NewName()

   STATIC   aNames   := { "Appleby", "Barnes", "Christian", "Dolson", ;
               "Engles", "Fritch", "Garvey", "Holt", "Innervy", "Jenkins", ;
               "Kleese", "Lambert", "Masterson", "Nelms", "Oppenheimer", ;
               "Peterson", "Quesa", "Richards", "Smith", "Thompson", ;
               "Updike", "Varol", "Waters", "Xavier", "Young", ;
               "Zimmerman" }


   RETURN(CHR(NewNumber(65, 90)) + "." + CHR(NewNumber(65, 90)) + ". " + ;
      aNames[NewNumber(1, 26)])
   // END NewName()



/*****************************************************************************
** NewNumber( nLRange, nURange ) --> nNumber
**
** This function is a variation on a pseudorandom number generator found
** in the NanForum Toolkit.
**
*****************************************************************************/
FUNCTION NewNumber( nLRange, nURange )

   STATIC nSeed := nil

   // init seed
   if ( nSeed == nil )
      nSeed := SECONDS()
   endif

   RETURN(INT((nURange - nLRange + 1) * ((nSeed := MOD(nSeed * ;
      RND_FACTOR + 1, RND_DIVISOR)) / RND_DIVISOR)) + nLRange)
   // END NewNumber( nLRange, nURange )



/*****************************************************************************
** NewState() --> cState
**
** Returns a pseudorandom state name.
**
*****************************************************************************/
FUNCTION NewState()

   STATIC   aStates  := { "Alabama", "Alaska", "Arizona", "Arkansas", ;
               "California", "Colorado", "Connecticut", "Delaware", ;
               "Florida", "Georgia", "Hawaii", "Idaho", ;
               "Illinois", "Indiana", "Iowa", "Kansas", ;
               "Kentucky", "Louisiana", "Maine", "Maryland", ;
               "Massachusetts", "Michigan", "Minnesota", "Mississippi", ;
               "Missouri", "Montanna", "Nebraska", "Nevada", ;
               "New Hampshire", "New Jersey", "New Mexico", "New York", ;
               "North Carolina", "North Dakota", "Ohio", "Oklahoma", ;
               "Oregon", "Pennsylvania", "Rhode Island", "South Carolina", ;
               "South Dakota", "Tennessee", "Texas", "Utah", ;
               "Vermont", "Virginia", "Washington", "West Virginia", ;
               "Wisconsin", "Wyoming" }


   RETURN(aStates[NewNumber(1, 50)])
   // END NewState()


/*****************************************************************************
** NewDivision() --> cDivision
**
** Returns a pseudorandom division name.
**
*****************************************************************************/
FUNCTION NewDivision()

   STATIC   aDivisions := { "", "Electronics", ;
               "Maintenance", "Engineering", "Administration", ;
               "Development" , "Human Resources" , "Shipping" }

   RETURN(aDivisions[NewNumber(1, 8)])
   // END NewDivision()



/*****************************************************************************
** NewTitle() --> cTitle
**
** Returns a pseudorandom title.
**
*****************************************************************************/
FUNCTION NewTitle()

   STATIC   aTitles := { "", "V.P.", "Pres.", ;
            "C.E.O.", "C.I.O.", "C.F.O.", "Director" }

   RETURN(aTitles[NewNumber(1, 7)])
   // END NewTitle()



