; Copyright (c) 1988 Borland International.  All Rights Reserved.
;
; General permission to re-distribute all or part of this script is granted,
; provided that this statement, including the above copyright notice, is not
; removed.  You may add your own copyright notice to secure copyright
; protection for new matter that you add to this script, but Borland
; International will not support, nor assume any legal responsibility for,
; material added or changes made to this script.
;
; Revs.: MJP 2/29/88, DCY 12/18/88
; ****************************************************************************
;  NAME: SetOrdrPrompt
; EVENT: Table Arrive
; TABLE: Orders
; NOTES: SetOrdrPrompt customizes the prompt for editing within the Orders
;        table.  DoWait will call it whenever a user enters the Orders table.
;        Since many of our procedures explicitly move to the Orders table, we
;        call the Record Arrive procedure within SetOrdrPrompt to increase the
;        efficiency of our code and to ensure that we always call it.
; ****************************************************************************
Proc SetOrdrPrompt()

   Prompt "Entering order information into Orders table.",
          "Press [F1] for help or [F2] to return to Invoice table."
   InitOrdrRec()
   TKHoldCanvas = True

Endproc
Writelib DemoLib SetOrdrPrompt
Release Procs SetOrdrPrompt

; ****************************************************************************
;  NAME: InitOrdrRec
; EVENT: Record Arrive
; TABLE: Orders
;  FORM: F
; NOTES: The Orders table is linked to the Invoice table via its Invoice No.
;        field.  In order to create a one-to-many link, i.e., have several
;        order records for each invoice record, we maintain a secondary index
;        field, the Key field.  InitOrdrRec fills the Key field of the Orders
;        table with the next available number.  InitOrdrRec also assigns a
;        value to the CRecTotal variable used by ChngOrdrRec to determine the
;        total for the current order record.
; ****************************************************************************
Proc InitOrdrRec()

   RefreshCanvas()
   If IsBlank([Key])          ;Is the Key field blank?
      Then [Key] = NOrderRec  ;Give it next order number
           NOrderRec = NOrderRec + 1
   Endif

   If IsBlank([Unit Price]) or IsBlank([Quantity])
      Then CRecTotal = 0
      Else CRecTotal = Round([Unit Price] * [Quantity],2)
   Endif

Endproc
Writelib DemoLib InitOrdrRec
Release Procs InitOrdrRec

; ****************************************************************************
;  NAME: ChngOrdrRec
; EVENT: Record Depart
; TABLE: Orders
;  FORM: F
; NOTES: ChngOrdrRec illustrates many uses of a record depart procedure.  For
;        example, not only does it update running calculations for the
;        current invoice record, it also controls a user's movements
;        through order records.  If a user moves upward while on the first
;        record, ChngOrdrRec moves the user to the Method of Payment field on
;        the first page.  An attempt to move downward while on the last
;        order record moves the user to the Ship Via field of the Invoice
;        table.  Additionally, ChngOrdrRec only allows a user to leave the
;        current record if both the Item No. and Quantity fields are either
;        empty or filled.
; ****************************************************************************
Proc ChngOrdrRec()

  ;This first SWITCH handles abnormal attempts to leave an order record.  A
  ; user is only allowed to leave a half-filled record by pressing Del or
  ; Undo.
   Switch
      Case TKChar = TKDel:
         RefreshCanvas()
         Message "Recalculating totals..."
         Del
         NOrderRec = NOrderRec + 1
         DownImage   ; Update total stored in invoice record
         [Subtotal] = Round([Subtotal] - CRecTotal,2)
         CalcTotals()
         UpImage
         ArriveRecord() ;Arriving in new record, set TKAccept False
      Case TKChar = TKUndo: ;Need to update summary info after an undo
         UpdtAftrUndo()
      Case not IsValid() ;Only allow user to leave field if both fields are
           or (IsBlank([Item]) and not IsBlank([Quantity]))  ; empty or both
           or (not IsBlank([Item]) and IsBlank([Quantity])): ; are filled
         TKAccept = False   ;Reject user's attempt to leave the current record
         TKMessage = "Please enter both an Item No. and Quantity for this order record"
   Endswitch

  ;This SWITCH performs special record exit processing.  It deletes blank
  ; order records and recalculates invoice summary fields for records whose
  ; totals change after entry.

   Switch
      Case TKAccept = False:  ;Cases above which set TKAccept False have
         Return               ; completed special handling.
      Case IsBlank([Quantity]):
         Echo Off
         Del        ;Delete empty record and reinsert a blank record
         If AtLast()  ; without rejecting key.
            Then Down         ; Allow Paradox to process movement key.
            Else If RecNo() <> 1
                    Then Ins
                 Endif
         Endif
         ;NOTE: Since we require a user to fill both the Item No. and Quantity
         ;      fields in order to leave an order record, we know that if the
         ;      Quantity field is empty the Item No. field must also be empty.
      Case Round([Quantity] * [Unit Price],2) <> CRecTotal: ;Has current total
         RefreshCanvas()                           ; changed since arrival?
         [Amount] = Round([Quantity] * [Unit Price],2)
         CRecTotal = Round(CRecTotal - [Amount],2)
         Message "Recalculating totals..."         ;Update running total
         DownImage
         [Subtotal] = Round([Subtotal] - CRecTotal,2)
         CalcTotals()
         UpImage
   Endswitch

  ;This final SWITCH redirects movement for special exit cases.  An attempt
  ; to move upward from the first order record takes the user to the first
  ; page of the Invoice table.  Moving downward (but not with [Right] or
  ; [Enter]) from the last order record takes a user to the Ship Via field on
  ; the current page.

   Switch
      Case RecNo() = 1 and TKChar = TKUp: ;Moving to first page
         Moveto [Invoice->Method of Payment]
         ArriveTable()  ;ArriveTable sets TKAccept False
      Case (AtLast() and TKChar = TKDown)  ;Moving down to Ship Via field?
        or TKChar = TKUpImage:             ;Moving to Carriers table?
         DownImage         ;Redirect movement to Invoice table...
         ArriveTable()
   Endswitch
Endproc
Writelib DemoLib ChngOrdrRec
Release Procs ChngOrdrRec

; ****************************************************************************
;  NAME: SelectItem
; EVENT: N/A
; FIELD: Item
; TABLE: Orders
;  FORM: F
; NOTES: SelectItem invokes the LookupSelect procedure so that we can
;        determine the key a user presses to exit lookup help.  We do this so
;        that we can automatically move the user to the next field if
;        appropriate.
; ****************************************************************************
Proc SelectItem()

   Help                ;Enter lookup help, show items in stock
   LookUpSelect()      ;Begin lookup help interaction

   If Retval = TKDo_It!     ;User pressed Do-It!
      Then Do_It!           ; Exit lookup help, reassign pending key to
           TKChar = TKEnter ; Enter to move user to next field
      Else Esc              ;User pressed Esc, exit lookup help
           TKAccept = False
   Endif

Endproc
Writelib DemoLib SelectItem
Release Procs SelectItem

; ****************************************************************************
;  NAME: OrdrSpclKey
; EVENT: Special key
; TABLE: Orders
; NOTES: OrdrSpclKey handles special keys a user presses while within the
;        Orders table.
; ****************************************************************************
Proc OrdrSpclKey()

   Switch
      Case TKChar = TKHelp:
         If TKFieldNum = 4     ;Help for the Item field or Quantity field?
            Then SelectItem()
            Else Echo Off
                 Style Attribute SysColor(9)
                 OrdersBox()   ;Show general help box
                 Style
                 SyncCursor
                 TKHoldCanvas = True    ;Hold canvas until a keypress
                 TKAccept = False ;Don't let Paradox display its own help
         Endif
      Case TKChar = TKDo_It!: ;Finished adding orders, move to Invoice table
         TKChar = TKDownImage  ;Reassign pending key to DownImage
      Case TKChar = -46:      ;[Alt][C], commentary key--
         ToggleCommentary()   ; Need to toggle the commentary state
   Endswitch
Endproc
Writelib DemoLib OrdrSpclKey
Release Procs OrdrSpclKey

; ****************************************************************************
;  NAME: UpdtAftrUndo
; EVENT: N/A
; NOTES: UpdtAftrUndo is called by ChngOrdrRec when a user presses Undo while
;        editing order records.  When a user presses Undo within the Orders
;        table, we have no way of knowing what changes will be undone.  Thus
;        we must re-scan the linked records to update the invoice record
;        subtotal.  UpdtAftrUndo examines the message window after doing an
;        Undo for a user and continues to do Undos until a deleted record is
;        resinserted or the current, unposted record is deleted.
; ****************************************************************************
Proc UpdtAftrUndo()

   RefreshCanvas()
   CRecNo = RecNo()   ;Save cursor position within table
   CRowNo = RowNo()
   Undo ;
   If Search("Resinserted record",Window()) <> 0
      Then Undo   ;Doing undo three Undos will guarantee that change to a
           Undo   ; record will be undone and the record will be deleted.
   Endif
   CImgTotl = 0
   Scan For Not IsBlank([Amount])
      CImgTotl = CImgTotl + [Amount]  ;Recalculate total
   Endscan
   DownImage
   [Subtotal] = Round(CImgTotl,2)
   CalcTotals()
   UpImage
   If RecNo() < NImageRecords()
      Then SetRecordPosition CRecNo CRowNo  ;Restore cursor position
   Endif
   ArriveRecord()   ;ArriveRecord will set TKAccept False

Endproc
Writelib DemoLib UpdtAftrUndo
Release Procs UpdtAftrUndo

; ****************************************************************************
;  NAME: CalcTotals
; EVENT: N/A
; NOTES: CalcTotals computes the absolute Discount, Tax, and Total fields for
;        the current invoice record.
; ****************************************************************************
Proc CalcTotals()

   [Discount] = Round([Subtotal] * [Discount %] / 100,2)
   [Tax] = Round( ([Subtotal] - [Discount]) * [Tax %] / 100,2 )
   [Total] = Round([Subtotal] - [Discount] + [Tax] + [Shipping],2)

Endproc
Writelib DemoLib CalcTotals
Release Procs CalcTotals
