; Copyright (c) 1988 Borland International.  All Rights Reserved.
;
; General permission to re-distribute all or part of this script is granted,
; provided that this statement, including the above copyright notice, is not
; removed.  You may add your own copyright notice to secure copyright
; protection for new matter that you add to this script, but Borland
; International will not support, nor assume any legal responsibility for,
; material added or changes made to this script.
;
; Revs.: MJP 2/29/88, DCY 12/17/88
; ****************************************************************************
;  NAME: SetCustPrompt
; EVENT: Table Arrive
; TABLE: Cust
;  FORM: F
; NOTES: SetCustPrompt customizes the system prompt and sets image rights for
;        the Cust table depending upon whether a user is allowed to edit
;        customer information.  A user is not allowed to edit existing
;        customer records.  SetCustPrompt first calls LockNewCust to test for
;        competition for the new customer number.
;        The variable NewCust specifies whether the current customer was newly
;        added to the Cust table, or is an existing customer record.  The
;        variable CustLink specifies whether a customer record is currently
;        linked to the current invoice record.
; ****************************************************************************
Proc SetCustPrompt()

   If NewCust and not CustLink  ;New customer, allow user to edit record and
      Then LockNewCust()        ; display appropriate prompt
           ImageRights
           Prompt "Entering customer information into Cust table.",
                  "Press [F2] to add order information, or [Esc] to cancel."
           Readlib SDir()+"DoW8Demo" NormalizePhone,LeaveFullField,FillPhoneNo
           DelCustLink = True
      Else If not NewCust   ;If customer already exists, don't allow user to
              Then ImageRights ReadOnly ; edit information
           Endif
           Prompt "Viewing existing customer information in Cust table.",
                  "Press [F3] or [F4] to return to Invoice table."
   Endif
   TKHoldCanvas = True
   Readlib SDir()+"DoW8Demo" EditCustRec,CustSpclKey

Endproc
Writelib DemoLib SetCustPrompt
Release Procs SetCustPrompt

; ****************************************************************************
;  NAME: LockNewCust
; EVENT: N/A
; TABLE: Cust
;  FORM: F
; NOTES: If a user enters a new Customer No., the GetCustNo procedure (see
;        INVDEMO.SC) moves the user to the Cust table to enter customer data
;        for the new customer.  The table arrival procedure for the Cust
;        table, SetCustPrompt, then calls this procedure to check if another
;        user has entered the same Customer No. BETWEEN the time GetCustNo
;        determined that the new customer number did not exist and when the
;        user is actually about to enter customer data.  Without this check,
;        more than one user on a network could enter the same new customer
;        number simultaneously, and the second user to post a customer record
;        would be entering a duplicate key into Cust.
; ****************************************************************************
Proc LockNewCust()

   LockRecord     ;Attempt to post and lock the customer record.  If we can't,
   If not Retval  ; another user must have already done so.
      Then NewCust = False    ;We're now linked to an existing customer record
           CustLink = True    ;A customer record is linked to the current rec.
           Moveto [Invoice->Customer No.] ;Return to Customer No. field of
           ArriveTable()            ; Invoice table, issue relevant message...
           TKMessage = "User "+ErrorUser()+
                     " has just entered a new customer with this Customer No."
   Endif

Endproc
Writelib DemoLib LockNewCust
Release Procs LockNewCust

; ****************************************************************************
;  NAME: EditCustRec
; EVENT: Depart Record
; TABLE: Cust
;  FORM: F
; NOTES: Because DoWait calls record depart procedures before calling table
;        depart procedures, we let EditCustRec process attempts to leave the
;        Cust table.  Note that DoWait only calls table depart procedures
;        when a user presses [F3] or [F4] (UpImage or DownImage).  Thus if we
;        didn't trap [F3] and [F4] at the record level, we would need to
;        designate them as Special keys and handle them in a Special key
;        procedure.  DoWait calls Special key procedures before calling field,
;        record, and table level procedures.
; ****************************************************************************
Proc EditCustRec()

   TKAccept = False   ;We're only allowing a few keys to exit the Cust table.
   Switch             ;  Thus we'll reject any keys we don't process specially
      Case NewCust and DelCustLink ;[F2] is only valid for new customer entry
       and (TKChar = TKDo_It! or (TKFieldNum = 10 and TKChar = 13)):
         RefreshCanvas()
         CtrlHome
         For X From 1 to 7    ;Verify that all fields have data, display a
             If IsBlank([])   ; message for fields which are empty
                Then TKMessage = "Please enter the customer's "
                               + Lower(Field())
                     If ColNo() = 6 or ColNo() = 7
                        Then TKMessage = TKMessage + " of residence"
                     Endif
                     NewField() ;Moved explicitly to a new field, alert DoWait
                     Return
             Endif
             Right
          Endfor
          If IsBlank([])
             Then TKMessage = "Please enter a standard discount rate for this customer"
                  NewField()
                  Return
          Endif
          UnlockRecord
          CustLink = True   ;All fields are filled, we have valid cust data
          DelCustLink = False ;Don't allow user to delete linked customer now
          Moveto [Invoice->Discount %];Copy the default discount rate into the
          [] = [Cust->Discount %]     ; discount rate for the current invoice
          If [Cust->State] = "CA"     ;If the customer is not a CA resident,
             Then [Tax %] = TaxRate   ; doesn't need to pay sales tax
             Else [Tax %] = 0
          Endif
          CalcTotals()
          Enter                       ;Move to [Method of Payment] field
          ArriveTable()      ;Inform DoWait we've just entered a new table
         ; NOTE: ArriveTable calls NewField, thus we don't need to call
         ;       ArriveField or NewField.  However, had we assigned an arrival
         ;       procedure to the [Method of Payment] field we would have
         ;       needed to call either ArriveField or our arrival procedure
         ;       to invoke it.
      Case CustLink and TKChar = TKDownImage or TKChar = TKUpImage:
         If not DelCustLink
            Then TKAccept = True  ;Accept the attempt to move to Invoice table
         Endif
         ; If NewCust is False, we're viewing an existing customer record.  In
         ;  view-only mode, [F3] and [F4] are enabled.
      Case TKChar = TKUndo :
         UpdtAftrUndo()
      Otherwise :  ;Reject with emphasis!
         Beep
   Endswitch

Endproc
Writelib DemoLib EditCustRec
Release Procs EditCustRec

; ****************************************************************************
;  NAME: LeaveFullField()
; EVENT: Keystroke
; TABLE: Cust
;  FORM: F
; NOTES: LeaveFullField determines whether the pending key (in TKChar) will
;        complete the State or Zip Code fields.  This kind of procedure
;        generally works well for alphanumeric fields in which we expect a
;        user to enter fixed-length data.  See the LeaveFullDate procedure in
;        INVDEMO.SC for a procedure that works for date fields.  See also the
;        FillPhoneNo procedure below for a more complex example of a keystroke
;        procedure.  Note that because DoWait calls keystroke procedures when
;        a user presses "regular" keys, we can be sure that we'll only
;        keypress visible characters, not movement or control characters.
; ****************************************************************************
Proc LeaveFullField()

   If NewCust
      Then Switch
              Case TKFieldNum = 7 : ;[State] field is complete when it
                 MaxLen = 2         ; contains two characters
              Case TKFieldNum = 8 : ;[Zip Code] field requires 5 characters
                 MaxLen = 5
           Endswitch
           Keypress TKChar          ;Type the character
           If Len(FieldStr()) = MaxLen and IsValid()
              Then TKChar = TKEnter  ;Field is full, move to next field
              Else TKAccept = False  ;We've already keypressed the key, we
           Endif                     ; don't want DoWait to press it again
   Endif

Endproc
Writelib DemoLib LeaveFullField
Release Procs LeaveFullField

; ****************************************************************************
;  NAME: FillPhoneNo
; EVENT: Keystroke
; FIELD: Telephone
; TABLE: Cust
;  FORM: F
; NOTES: FillPhoneNo serves two purposes.  First, it enhances the PICTURE
;        validity check by allowing after-the-fact reformatting of data in the
;        field.  Second, it automatically moves the user out of the field if
;        the phone number completes a valid pattern.
; ****************************************************************************
Proc FillPhoneNo()

   Switch
      Case not NewCust : ;Disable field formatting if user is in view mode
      Case TKChar < 48 or TKChar > 57 :  ;Make sure the pending key is a digit
      Case Len([]) = 8 and                       ;If there is a digit pending
           Substr([],1,1) <> "(" and             ; and the partial data looks
           Substr([],8,1) <> "-" :               ; like ###-####, then we want
         [] = Substr([],1,7)+"-"+Substr([],8,1)  ; to change it to ###-###-#
      Case Len([]) = 12 or Len([]) = 11 and
           Substr([],4,1) = "-" :       ;Field complete?
         Keypress TKChar   ;Accept the character which will complete the field
         TKChar = TKEnter  ;Instruct DoWait to press the [Enter] character,
   Endswitch               ; moving the cursor to the next field
  ;In this procedure we could have easily explicitly moved the user to the
  ; [Discount %] field (the next field).  But by allowing DoWait to act upon
  ; the key, we don't need to call ArriveField or NewField.

Endproc
Writelib DemoLib FillPhoneNo
Release Procs FillPhoneNo

; ****************************************************************************
;  NAME: NormalizePhone
; EVENT: Good Depart (field level)
; FIELD: Telephone
; TABLE: Cust
;  FORM: F
; NOTES: A user can enter telephone numbers in any one of three formats;
;        (###)###-####, ###-###-####, or ###-####.  NormalizePhone converts
;        telephone numbers to a standard (###)###-#### format.
;        NormalizePhone illustrates how using a Paradox's validity checks can
;        simplify your custom procedures.  For example, at first glance, the
;        checks NormalizePhone performs may seem inadequate.  The fact that a
;        number is 8 characters long does not guarantee that it is in the form
;        ###-####.  (It could be (###)###.)  However, we have defined this
;        "picture" for the [Telephone] field: (###)###-####,###-###{#,-####}.
;        Since DoWait calls NormalizePhone only if the user leaves valid data
;        in the field (NormalizePhone is a Good Depart procedure), the field
;        must look like ###-#### if it only contains 8 characters.
; ****************************************************************************
Proc NormalizePhone()

   Switch
      Case Len([]) = 12 :                              ;###-###-####
         [] = "("+Substr([],1,3)+")"+Substr([],5,8)  ;Reformat the area code
      Case Len([]) = 8 :                               ;###-####
         [] = AreaCode+[]   ;AreaCode is defined to be "(###)"
   Endswitch

Endproc
Writelib DemoLib NormalizePhone
Release Procs NormalizePhone

; ****************************************************************************
;  NAME: CustSpclKey
; EVENT: Special key
; TABLE: Cust
; NOTES: CustSpclKey handles special keys a user presses while within the
;        Cust table.
; ****************************************************************************
Proc CustSpclKey()

   Switch
      Case TKChar = -46:      ;[Alt][C], commentary key--
         ToggleCommentary()   ; Need to toggle the commentary state
      Case DelCustLink and TKChar = TKEsc :  ;User wants to cancel new cust
         Del           ;Delete the customer record
         Moveto [Invoice->Customer No.]
         ReSyncKey      ;Update Cust table link
         [Discount %] = BlankNum()  ;Reset discount and tax rates
         [Tax %] = BlankNum()
         CustLink = False     ;We no longer have a linked customer record
         DelCustLink = False
         ArriveTable()    ;See note above
   Endswitch
Endproc
Writelib DemoLib CustSpclKey
Release Procs CustSpclKey
