#ifndef RSQLSYS_H
#define RSQLSYS_H
#include "caldefs.h"

/*-------------------------------------------------------------------------
    rsqlsys.h  -  RSQL system constant and type definitions.
                  (Common to both server, rdbc & sddl utilities)

    Copyright (c) 1992-1994 Raima Corporation.  All Rights Reserved.
-------------------------------------------------------------------------*/

/* size of error info buffer */
#define INFOLEN       80

/* maximum length of column heading expression string */
#define XSTRLEN      128

/* maximum length of identifier (syscat dependent value) */
#define NAMELEN       33

/* maximum length of a string literal */
#define MAXSTRING    256

/* maximum length of a statement */
#define MAXSTMT     4096

/* maximum number of sort columns */
#define MAXSORTCOLS   24

#define UNEXECUTED    -2
#define INDETERMINATE -1

#ifndef DPL_DT_STRING
#define DPL_DT_STRING (DPL_DT_CHAR | DPL_DT_MOD_ARRAY)
#endif

#define ERR_DRIVER  0                /* ERRSTAT.origin values */
#define ERR_SOURCE  1                /* required for client error processing */

typedef struct error_status
{
    short                code;          /* rsqlcode (index into rsql_errtab */
    char                 origin;        /* Where error originated */
    char                 info[INFOLEN]; /* buffer for supplementary error info */
    struct error_status *next;          /* next error in list */
} ERRSTAT;

/* linked list of literal values */
typedef struct val_list {
    struct val_list   *next;         /* next literal value in list */
    VALUE              val;          /* literal value container */
    short              len;          /* length of value */
    unsigned char      prec;         /* precision if SQL_DECIMAL */
    unsigned char      scale;        /* scale if SQL_DECIMAL */
    short              from_select;  /* was len set from a select stmt? */
} VAL_LIST;

/* identifier linked list */
typedef struct id_list {
    struct id_list    *next;
    char               id[NAMELEN];  /* identifier string */
    char               qual[NAMELEN];/* qualifier string */
    char              *str;          /* associated string (e.g., comment) */
    char               type;         /* sort type or unused */
} ID_LIST;

/* where clause expression tree data structures */
typedef struct x_ptr {
    short              type;         /* X_NODE, X_LEAF, or X_LIST */
    short              data_type;    /* expr result's data type */
    short              slen;         /* string length if data_type == SQL_CHAR */
    union {
        struct x_node *node;         /* EXPR_NODE ptr */
        struct x_leaf *leaf;         /* EXPR_LEAF ptr */
        struct x_list *list;         /* EXPR_LIST ptr */
    } p;
} EXPR_PTR;

typedef struct x_list {
    short              item_count;   /* number of list entries */
    struct x_list     *next;         /* next expr_list entry */
    EXPR_PTR           expr;         /* ptr to expression */
} EXPR_LIST;

typedef struct x_leaf {
    short              op;           /* operation code */
    struct x_node     *parent;       /* parent node pointer */
    VALUE              val;          /* associated value */
    EXPR_PTR           expr;         /* associated expression(s) */
} EXPR_LEAF;

typedef struct x_node {
    short              op;           /* operation code */
    struct x_node     *parent;       /* parent node pointer */
    EXPR_PTR           left;         /* left expression (expr_leaf or expr_node) */
    EXPR_PTR           right;        /* right expression (expr_leaf or expr_node) */
} EXPR_NODE;

/* string type token for YACC parser */
typedef struct strtok {
    short              num;          /* token number */
    short              code;         /* token code */
    char               id[NAMELEN];  /* identifier string */
    char               qual[NAMELEN];/* qualifier string */
    short              item_count;   /* number of list (e.g., id_list) items */
    union {
        char          *str;          /* string literal */
        ID_LIST       *ids;          /* identifier linked list */
        VAL_LIST      *vals;         /* literal values linked list */
        long           comment;      /* database address of comment */
        struct xpath  *path;         /* pointer to xnode path */
    } pt;
} STRTOK;

/* number type token for parser */
typedef struct {
    short              num;          /* short value or token number */
    long               lnum;         /* long value */
    double             dnum;         /* double value */
} NUMTOK;

/* datetime type token for parser */
typedef struct dattok {
    CAL_DATE           date;
    CAL_TIME           time;
} DATTOK;

/* destination or source token for insert */
typedef struct instok {
    STRTOK  name;                    /* name of table or IEF */
    STRTOK  args;                    /* id list or IEF args */
} INSTOK;

/* token for column data type */
typedef struct col_type {
    short           type;
    unsigned char   prec;
    unsigned char   scale;
} COL_TYPE;

/* token for column's definition */
typedef struct col_defn {
    char               name[NAMELEN];
    short              data_type;
    short              size[3];
    unsigned char      prec;
    unsigned char      scale;
    short              col_type;
    char              *comment;
    VALUE              deflt_val;
    short              fmt_width;
    short              fmt_no;
    short              fmt_pos;
    struct col_defn   *next;
} COL_DEFN;

/* token for sort list on index */
typedef struct sort_defn {
    char               name[NAMELEN];
    char               order;
    struct sort_defn  *next;
} SORT_DEFN;

/* scalar function declaration table */
typedef struct scalar_fcns {
    char    *name;              /* function name */
    short    noargs;            /* number of arguments */
    short    rtype;             /* result type (0=>same as atype[0][0]) */
    short    rlen;              /* result string length code */
    short    atypes[4];         /* valid arg types for each arg (max of 4 args) */
} SCALAR_FCNS;

/* c_data type field declaration */
typedef struct {
    char               name[NAMELEN];  /* RDM field name */
    char               unsgnd;         /* unsigned type flag */
    char               type;           /* RDM data type */
    short              offset;         /* offset to start of field in struct */
    short              eltsz[MAXDIMS]; /* size of each array element */
    short              dims[MAXDIMS];  /* specified array dimensions */
} C_DATA_FIELD;

typedef struct {
    short              noflds;       /* number of c_data fields */
    C_DATA_FIELD       flds[1];      /* start of c_data fields */
} C_DATA_BLOB;

/* atypes are encoded as follows:

    SQL_NUMERIC   => any numeric type
    SQL_FLOAT     => SQL_REAL or SQL_DOUBLE
    SQL_CHAR      => SQL_CHAR or SQL_VARCHAR
    SQL_INTEGER   => SQL_SMALLINT or SQL_INTEGER
    SQL_DATE      => SQL_DATE or SQL_TIMESTAMP
    SQL_TIME      => SQL_TIME or SQL_TIMESTAMP
*/

/* public_access and sysright.access bitmaps */
#define SEC_NO_RIGHTS  ((char) 0x00)
#define SEC_SELECT     ((char) 0x01)
#define SEC_DELETE     ((char) 0x02)
#define SEC_INSERT     ((char) 0x04)
#define SEC_UPDATE     ((char) 0x08)
#define SEC_REFERENCE  ((char) 0x10)
#define SEC_EXEC       ((char) 0x20)
#define SEC_GRANT      ((char) 0x40)
#define SEC_ALL_RIGHTS ((char) 0xFF)

/* command bitmaps */
#define SEC_NO_CMD_RIGHTS    ((char) 0x00)
#define SEC_CMD_DATABASE     ((char) 0x01)
#define SEC_CMD_VIEW         ((char) 0x02)
#define SEC_CMD_INSERT       ((char) 0x04)
#define SEC_CMD_UPDATE       ((char) 0x08)
#define SEC_CMD_DELETE       ((char) 0x10)
#define SEC_CMD_LOCK         ((char) 0x20)
#define SEC_CMD_PROCEDURE    ((char) 0x40)
#define SEC_CMD_GRANT        ((char) 0x80)
#define SEC_ALL_CMD_RIGHTS   ((char) 0xFF)

/* SecType defines for secObjSetup() and sysrights.type */
#define SEC_TYPE_COLUMN             0
#define SEC_TYPE_ALIAS              1
#define SEC_TYPE_TABLE              2
#define SEC_TYPE_VIEW               3
#define SEC_TYPE_DATABASE           4
#define SEC_TYPE_USER               5

/* OpType defines for secObjSetup() */
#define SEC_GRANT_ONE           0
#define SEC_GRANT_ALL           1
#define SEC_REVOKE_ONE          2
#define SEC_REVOKE_ALL          3

/* column type definitions */
#define CT_C_DATA   -1
#define CT_NONE      0
#define CT_NOTNULL   1
#define CT_UNIQUE    2
#define CT_PRIMARY   3
#define CT_ALIAS     4

/* expression pointer types */
#define X_NULL             0
#define X_NODE             1
#define X_LEAF             2
#define X_LIST             3

/* compilation phase codes */
#define PH_DEFAULT         0x0000
#define PH_CRPROC          0x0001
#define PH_DATE            0x0002
#define PH_TIME            0x0004 
#define PH_AT              0x0008
#define PH_FRACTION        0x0010
#define PH_REPORT          0x0020
#define PH_SELECT          0x0040
#define PH_TEMPORARY       0x0080
#define PH_SDDLP           0x0100

/* binary arithmetic operation operand type codes
   code = (xtype1 - 1)*E_SCALE + xtype2 - 1
*/

enum expr_base_types {
    E_SHORT=0,
    E_LONG,
    E_FLOAT,
    E_DOUBLE,
    E_BCD,
    E_DATE,
    E_SCALE 
};

/*lint -e641 */
enum expr_eval_types {
    E_SH_SH = E_SCALE*E_SHORT,      /* short     +- short     */
    E_SH_LO,                        /* short     +- long      */
    E_SH_FL,                        /* short     +- float     */
    E_SH_DO,                        /* short     +- double    */
    E_SH_BC,                        /* short     +- decimal   */
    E_SH_DA,                        /* short     +  date      */
    E_LO_SH = E_SCALE*E_LONG,       /* long      +- short     */
    E_LO_LO,                        /* long      +- long      */
    E_LO_FL,                        /* long      +- float     */
    E_LO_DO,                        /* long      +- double    */
    E_LO_BC,                        /* long      +- decimal   */
    E_LO_DA,                        /* long      +  date      */
    E_FL_SH = E_SCALE*E_FLOAT,      /* float     +- short     */
    E_FL_LO,                        /* float     +- long      */
    E_FL_FL,                        /* float     +- float     */
    E_FL_DO,                        /* float     +- double    */
    E_FL_BC,                        /* float     +- decimal   */
    E_DO_SH = E_SCALE*E_DOUBLE,     /* double    +- short     */
    E_DO_LO,                        /* double    +- long      */
    E_DO_FL,                        /* double    +- float     */
    E_DO_DO,                        /* double    +- double    */
    E_DO_BC,                        /* double    +- decimal   */
    E_BC_SH = E_SCALE*E_BCD,        /* decimal   +- short     */
    E_BC_LO,                        /* decimal   +- long      */
    E_BC_FL,                        /* decimal   +- float     */
    E_BC_DO,                        /* decimal   +- double    */
    E_BC_BC,                        /* decimal   +- decimal   */
    E_DA_SH = E_SCALE*E_DATE,       /* date      +- short     */
    E_DA_LO,                        /* date      +- long      */
    E_DA_DA = E_DA_SH+E_DATE        /* date      -  date      */
};
/*lint +e641 */

/********** expression op code definitions:
   even number codes => no data (i.e., emit op code only)
   odd  number codes => data (always short except OP_VALUE, OP_USER_..)(
*/
#define OP_CLASS 0xF0

/* arith ops: 0x0_ series */
#define OP_ARITH        0x00
#define OP_ADD          0x02
#define OP_SUB          0x04
#define OP_MULT         0x06
#define OP_DIV          0x08
#define OP_CHGSIGN      0x0A
#define OP_CAT          0x0C

/* aggregate calc fcns: 0x1_ series */
#define OP_CALC         0x10
#define OP_SUM          0x11
#define OP_SUM_DIST     0x13
#define OP_AVG          0x15
#define OP_AVG_DIST     0x17
#define OP_COUNT        0x19
#define OP_COUNT_DIST   0x1B
#define OP_MIN          0x1D
#define OP_MAX          0x1F

/* functions: 0x2_ series */
#define OP_FUNCTION     0x20
#define OP_SCALAR       0x21
#define OP_IF           0x22
#define OP_CONVERT      0x23
#define OP_CONVERT_FMT  0x24
#define OP_USER         0x25
#define OP_USER_DIST    0x27
#define OP_C_DATA       0x29

/* relational ops: 0x3_ series */
#define OP_RELOPS       0x30
#define OP_EQ           0x30
#define OP_NE           0x32
#define OP_LT           0x34
#define OP_LE           0x36
#define OP_GT           0x38
#define OP_GE           0x3A
#define OP_STAREQ       0x3C

/* special comparison ops: 0x4_ series */
#define OP_COMPOPS      0x40
#define OP_IS           0x40
#define OP_BETWEEN      0x42
#define OP_IN           0x43
#define OP_LIKE         0x44
#define OP_SUBQUERY     0x45
#define OP_NOTIS        0x46
#define OP_NOTBETWEEN   0x48
#define OP_NOTIN        0x49
#define OP_NOTLIKE      0x4A
#define OP_EXISTS       0x4C
#define OP_NOTEXISTS    0x4E

/* conditional expression ops: 0x6_ series */
#define OP_COND         0x60
#define OP_WHERE_BEGIN  0x60
#define OP_HAVING_BEGIN 0x62
#define OP_AND          0x64
#define OP_SKIPIF_0     0x65
#define OP_OR           0x66
#define OP_SKIPIF_1     0x67
#define OP_NOT          0x6A
#define OP_CHECK        0x6C
#define OP_ENDCHECK     0x6E

/* data ops: 0x7_ series */
#define OP_DATA         0x70
#define OP_READ         0x71
#define OP_STORE        0x73
#define OP_VALUE        0x75
#define OP_ARRAYREF     0x77
#define OP_UPDATE       0x7B
#define OP_FETCH        0x7D
#define OP_PROCVAR      0x7F

/* end of expression */
#define OP_END          0x80

/* builtin scalar function codes */
#define FN_ABS           1
#define FN_ACOS          2
#define FN_ASCII         3
#define FN_ASIN          4
#define FN_ATAN          5
#define FN_ATAN2         6
#define FN_CEILING       7
#define FN_CHAR          8
#define FN_CONCAT        9
#define FN_CONVERT      10
#define FN_COS          11
#define FN_COT          12
#define FN_CURDATE      13
#define FN_CURTIME      14
#define FN_DAYOFMONTH   15
#define FN_DAYOFWEEK    16
#define FN_DAYOFYEAR    17
#define FN_EXP          18
#define FN_FLOOR        19
#define FN_HOUR         20
#define FN_IFNULL       21
#define FN_INSERT       22
#define FN_LCASE        23
#define FN_LEFT         24
#define FN_LENGTH       25
#define FN_LOCATE       26
#define FN_LOG          27
#define FN_LTRIM        28
#define FN_MINUTE       29
#define FN_MOD          30
#define FN_MONTH        31
#define FN_NOW          32
#define FN_PI           33
#define FN_QUARTER      34
#define FN_RAND         35
#define FN_REPEAT       36
#define FN_REPLACE      37
#define FN_RIGHT        38
#define FN_RTRIM        39
#define FN_SECOND       40
#define FN_SIGN         41
#define FN_SIN          42
#define FN_SQRT         43
#define FN_SUBSTRING    44
#define FN_TAN          45
#define FN_UCASE        46
#define FN_WEEK         47
#define FN_YEAR         48
#define FN_DATABASE     49
#define FN_USER         50
#define FN_AGE          51

/* SQL0SetMode() arguments */
#define RD_REPEATABLE   1
#define TX_ISOLATION    2

#define RD_REPEATABLE_ON   1
#define RD_REPEATABLE_OFF  2
#define TX_ISOLATION_ON    3
#define TX_ISOLATION_OFF   4


enum sql_server_functions {
    SQL0ALLOCSTMT = 0,
    SQL0DESCRIBEBREAK,
    SQL0DESCRIBECALC,
    SQL0DESCRIBEREPORT,
    SQL0DESCRIBECOL,
    SQL0ERRORINFO,
    SQL0EXECUTE,
    SQL0EXTENDEDTRANSACT,
    SQL0FETCH,
    SQL0FREESTMT,
    SQL0GETCONNECTOPTION,
    SQL0GETCURSORNAME,
    SQL0MORERESULTS,
    SQL0PREPARE,
    SQL0ROWCOUNT,
    SQL0SETCONNECTOPTION,
    SQL0SETCURSORNAME,
    SQL0TRANSACT,
    SQL0TRANSACTIVE,
    SQL0SETMODE,
    SQL0GETMODE,
    SQL0DBACTIVE,
    SQL0DBLIST,
    SQL0DBHANDLE,
    SQL0ROWID,
    SQL0ROWDBA,
    SQL0ROWIDTODBA,
    SQL0DBATOROWID,
    SQL0TRANSACTSTATUS,
    SQL0DESCRIBECDATA,
    SQL0CDATADPL
};

/* ======================================================================
    Reset syscat cache tables
*/
void REXTERNAL ResetSyscatCache(
   char      *auth)
;

/* ======================================================================
    Call A UDF check function
*/
short REXTERNAL SQLUDFCheck (
    RDM_SESS      hSess,
    void         *name,
    short         noargs,
    VALUE        *args,
    VALUE        *result,
    short        *len)
;

#endif /* RSQLSYS_H */
/*
$Header:   P:/PAT/INCLUDE/VCS/RSQLSYS.H_V   1.48   19 May 1994 12:29:08   RANDY  $
*/
