                       F i l e    I n f o r m a t i o n

* DESCRIPTION
Program to setup and manage data files with format similar to laboratory
notebook. Requires: Turbo Pascal 4.0. Author: William Nuttle.
Version 1.0. 1985/86 TUG O'Wards entry. Converted to version 4.0.

* ASSOCIATED FILES
NOTEBOOK.PAS
ASCII.DOC

* KEYWORDS
PASCAL 4.0 PROGRAM DATA FILE

==========================================================================
}
{$R-}    {Range checking off}
{$B+}    {Boolean complete evaluation on}
{$S+}    {Stack checking on}
{$I+}    {I/O checking on}
{$N-}    {No numeric coprocessor}
{$M 65500,16384,655360} {Turbo 3 default stack and heap}

program notebook;

{******************************************************************
   Program to setup and manage data files with format similar to
   a laboratory notebook.  All routines in turbo-pascal, running
   on an IBM PC.  Direct all comments and suggestions to author:

                       WILLIAM NUTTLE
                       RM 48-420
                       M I T
                       CAMBRIDGE, MA  02139

  version 1.0   November 1985
  ************************************************************** }


Uses
  Crt,
  Turbo3;

const
     maxline=40;               { sets max number of lines on a page}
     add: boolean= false;

type
    cellcontent=(header,date,time,real_,integer_,char_);
    note=(new,changed,deleteflag,format,active);
    descriptor=set of note;

{ descriptor is a set of characteristics which describe the data on a
  line in the notebook:
           new - a flag information just added
           changed - a flag to indicate that the line has been modified
           deleteflag - a flag used to mask a line that has been deleted but
                not overwritten.  Deleted information is not removed from the
                file until the editor is exited, and then only at the
                discretion of the user.
           format - a flag used on the first two pages of a notebook file
                where the page format and editor prompt information is
                stored.
           active - a flage used to indicate that a line is written on as
                opposed to blank, no data.
 }
    modyyr=array[1..3] of integer;
    hrmin=array[1..2] of integer;
    longstring=string[16];
    shortstring=string[8];

{ cell is the basic unit of data.  A line in the notebook is a cell record.
  A notebook contains more cells than lines containing data.  The extra cells
  are used to store page format information and to delimit pages.
}
    cell=record
               attributes:descriptor;
               case contains:cellcontent of
                    header  :(pagenum:integer);
                    date    :(dateval:modyyr);
                    time    :(timeval:hrmin);
                    real_   :(realval:real);
                    integer_:(intval:integer);
                    char_   :(text:longstring);
          end;
    book=file of cell;
    page=array[0..maxline] of cell;

{ The editor accesses lines in the notebook taking them a page at a time.  A
  page is any number of lines (cells) up to maxline.  Every page in the
  notebook has the same number of lines.
  }
    lineid=array[1..maxline] of longstring;
    linetype=array[1..maxline] of cellcontent;

var
   fname:longstring;
   line:cell;
   note_book:book;
   prompt:lineid;

{ prompt is as array of user-specified editor prompts used in the screen
  display of a notebook page.  prompt is stored in the notebook on the second
  page of the file.
}
   contents:linetype;

{ contents is an array of user-specified data types, one for each line of a
  page, that specifies the format of a page.  contents is stored in the
  notebook on the first page in the file.
}
   pagesize, maxpage:integer;
   currentpage:page;
   command: char;
   resp: shortstring;

{end of definition part}

{***************************************************************************}
procedure beep;

{procedure to produce a 1000 hz tone for 0.2 seconds}
{****************************************************************************}

begin
sound(2000);
delay(50);
nosound;
end; {beep}


{***************************************************************************}
procedure initialize(var note_book:book);

{ procedure to open new notebook file and set up format pages}

{***************************************************************************}

const
     validtype: set of char=['D','T','R','I','C'];
     blanks: longstring= '                ';

var
   line:cell;
   linenum, pagesize: integer;
   prompt:lineid;
   contents:linetype;
   resp: shortstring;
   invalid:boolean;

begin
     rewrite(note_book);
     writeln; writeln('  new file assumed.  provide format below:'); beep;
     writeln; write('  lines per page = '); readln(pagesize);
     writeln; writeln('  for each line, enter line contents (Date, Time, Real, Integer,');
              writeln('      or Char) and an editor prompt');
     for linenum:=1 to pagesize do
         begin
         writeln; write('    line ',linenum,': ');
         invalid:= false;
         repeat
               write(' contents? '); if invalid then beep;
               read(resp);
               resp[1]:=upcase(resp[1]);
               invalid:= true;
         until resp[1] in validtype;
         case resp[1] of
              'D':contents[linenum]:=date;
              'T':contents[linenum]:=time;
              'R':contents[linenum]:=real_;
              'I':contents[linenum]:=integer_;
              'C':contents[linenum]:=char_;
         end; {case}
         writeln; write('             prompt? ');
         readln(prompt[linenum]); prompt[linenum]:= prompt[linenum] + blanks;
    end; {for}

    { now have all format infomation, write format pages}
    { write pagesize and cell contents to first page}

    with line do
         begin
         contains:=header;
         pagenum:=pagesize;
         attributes:=[format];
    end; {with}
    write(note_book,line);
    for linenum:=1 to pagesize do
        begin
        with line do
             begin
             attributes:=[];
             contains:=contents[linenum];
        end; {with}
        write(note_book,line);
    end; {for}

    { write number of pages and editor prompts to second page}

    with line do
         begin
         contains:=header;
         pagenum:=0;
         attributes:=[format];
    end; {with}
    write(note_book,line);
    for linenum:=1 to pagesize do
        begin
        with line do
             begin
             contains:=char_;
             text:=prompt[linenum];
             attributes:=[];
        end; {with}
        write(note_book,line);
    end; {for}
    {reset file pointer}
    seek(note_book,0);
end; {initialize}

{***********************************************************************}
procedure setup( var note_book:book;
                 var fname:longstring;
                 var prompt:lineid;
                 var contents:linetype;
                 var pagesize,maxpage:integer);

{ procedure to set up notebook file for editing }
{***********************************************************************}

var
   new:boolean;
   line:cell;
   linenum:integer;

begin
lowvideo;
writeln; writeln('  NOTEBOOK EDITOR');
writeln; write('  enter file name - ');
readln(fname);
assign(note_book,fname);
{$I-} reset(note_book) {$I+};
new:= not(ioresult=0);

if new then initialize(note_book);

{get page size, read line formats }

read(note_book, line);
pagesize:=line.pagenum;
for linenum:=1 to pagesize do
    begin
    read(note_book, line);
    contents[linenum]:= line.contains;
end; {for}

{ get number of pages, read editor prompts }

read(note_book,line);
maxpage:=line.pagenum;
for linenum:=1 to pagesize do
    begin
    read(note_book, line);
    prompt[linenum]:=line.text;
end; {for}

{ current position is at header of first page }

end; {setup}

{********************************************************************}
procedure newpage( var note_book:book;
                   var maxpage:integer;
                       contents:linetype;
                       pagesize:integer);

{procedure to append blank page to notebook}

{*******************************************************************}

var
   line:cell;
   linenum:integer;

begin

{write header of new page}

maxpage:=maxpage+1;
line.contains:=header;
line.pagenum:=maxpage;
line.attributes:=[new];
write(note_book,line);

{write lines of new page}

for linenum:= 1 to pagesize do
    begin
    line.attributes:=[];
    line.contains:=contents[linenum];
    write(note_book,line);
end; {for}

{reset file pointer to header of current page}

seek(note_book,(filepos(note_book)-(pagesize+1)));

read(note_book,line);
if not(line.contains=header) then writeln('position error');
seek(note_book,(filepos(note_book)-1));

end; {newpage}


{******************************************************************}
procedure addpage( var note_book: book;
                       fname: longstring;
                   var currentpage: page;
                       contents: linetype;
                       prompt: lineid;
                       pagesize: integer;
                   var maxpage: integer);   forward;

{ procedure to add a page to end of file }
{*******************************************************************}


{******************************************************************}
procedure getpage( var note_book:book;
                   var currentpage:page;
                   var maxpage:integer;
                       contents:linetype;
                       pagesize:integer);

{ procedure to get page that begins at current file pointer position}
{*******************************************************************}

var
   line:cell;
   linenum:integer;

begin

if eof(note_book) then addpage(note_book, fname, currentpage, contents,
                      prompt, pagesize, maxpage);
read( note_book, line);
if not(line.contains = header) then write(' position error');
seek( note_book, (filepos(note_book) -1));
for linenum:= 0 to pagesize do read(note_book, currentpage[linenum]);

{reset file pointer to header of current page}

seek(note_book,(filepos(note_book)-(pagesize+1)));

end; {getpage}

{***********************************************************************}
procedure writeline( line:cell;
                     prompt:longstring;
                     linenum, position: integer);

{ procedure to write line at (position) on screen }
{***********************************************************************}

begin
gotoxy(1,(2*position-1)); insline;
write(' ',linenum:2,'  ',prompt,': ');
if (active in line.attributes) and not(deleteflag in line.attributes) then
case line.contains of
     header:write('error, header');
     date:write(line.dateval[1],'/',line.dateval[2],'/',line.dateval[3]);
     time:write(line.timeval[1],':',line.timeval[2]);
     real_:write(line.realval);
     integer_:write(line.intval);
     char_:write(line.text);
end; {case}
gotoxy(22,wherey+1); insline;
end; {writeline}

{***********************************************************************}
procedure display( filename:longstring;
                   currentpage:page;
                   contents:linetype;
                   prompt:lineid;
                   pagesize, maxpage:integer);

{ procedure to set up screen display of current page }
{***********************************************************************}

var
   linenum: integer;

begin

{ write screen heading }

clrscr; lowvideo;
gotoxy(14,1); write('notebook: ',filename);
with currentpage[0] do
     begin
     gotoxy(11,2); write('page number: ',pagenum,'/',maxpage);
     gotoxy(11,3); write('page status: ');
                  if(new in attributes) then write('new ');
                  if(changed in attributes) then write('changed ');
                  if(deleteflag in attributes) then write('deleted ');
     end; {with}
gotoxy(13,4); write('edit mode: page');
normvideo;
gotoxy(41,2); write('U'); lowvideo; write('p'); normvideo;
gotoxy(41,3); write('E'); lowvideo; write('dit'); normvideo;
gotoxy(41,4); write('A'); lowvideo; write('dd page'); normvideo;
gotoxy(51,2); write('D'); lowvideo; write('own'); normvideo;
gotoxy(51,3); write('Q'); lowvideo; write('uit'); normvideo;
gotoxy(51,4); write('K'); lowvideo; write('ill'); normvideo;
gotoxy(61,2); write('G'); lowvideo; write('oto'); normvideo;
gotoxy(61,3); write('R'); lowvideo; write('estore');
gotoxy(28,4);

{ heading written, write first 10 lines of page }

window(11,6,80,25);
for linenum:=1 to 10 do
    if linenum <= pagesize then writeline( currentpage[linenum],
                                          prompt[linenum], linenum, linenum);
window(1,1,80,25);

gotoxy(28,4);
end; {display}

{****************************************************************************}
procedure update( var note_book: book;
                      currentpage: page;
                      pagesize: integer);

{procedure to write current page into notebook}
{****************************************************************************}

var
   linenum: integer;

begin
for linenum:= 0 to pagesize do write(note_book, currentpage[linenum]);
end; {update}

{****************************************************************************}
procedure shut( var note_book: book;
                    pagesize, maxpage: integer);

{ procedure to update maxpage in format pages and close notebook file }
{****************************************************************************}

var
   line: cell;

begin
seek( note_book, (pagesize+1));
read (note_book, line);
seek( note_book, (filepos( note_book)-1));
line.pagenum:= maxpage;
write( note_book, line);
close(note_book);
end; {shut}


{****************************************************************************}
procedure cull( var note_book: book;
                    fname: longstring);

{ procedure to rewrite notebook file excluding "deleted" information
  dum.nbk contains unmodified notebook file.}
{****************************************************************************}

var
   line: cell;
   currentpage: page;
   newbook: book;
   linenum, pagesize, maxpage: integer;
   contents: linetype;

begin
assign( newbook,'dum.nbk');

{ write over previous 'dum.nbk'}

rewrite(newbook); close(newbook); erase(newbook);
rename(note_book,'dum.nbk');
assign(newbook, fname);
reset(note_book); rewrite(newbook);

{ begin transfer of data to newbook }

read(note_book,line);
pagesize:= line.pagenum;
maxpage:= -1;
seek(note_book, 0);
getpage(note_book, currentpage, maxpage, contents, pagesize);
update( newbook, currentpage, pagesize);
seek(note_book, (filepos(note_book)+pagesize+1));
repeat
      getpage( note_book, currentpage, maxpage, contents, pagesize);
      seek( note_book, (filepos(note_book)+pagesize+1));
      if not( deleteflag in currentpage[0].attributes) then
         begin
         for linenum:= 1 to pagesize do
             if deleteflag in currentpage[linenum].attributes then
                currentpage[linenum].attributes:=
                  currentpage[linenum].attributes -[deleteflag, active];
         maxpage:= maxpage+1;
         if maxpage > 0 then currentpage[0].pagenum:= maxpage;
         update( newbook, currentpage, pagesize);
      end; {if}
until eof(note_book);
shut(newbook, pagesize, maxpage);
close(note_book);
end; {cull}

{****************************************************************************}
procedure pageedit( var command: char);

{ procedure to get command for pageeditor }
{*****************************************************************************}

const
     validcommand: set of char= ['U','E','C','D','Q','K','G','R','A'];

var
   invalid: boolean;

begin
gotoxy(29,4);
invalid:= false;
repeat
      write ('?'); if invalid then beep;
      repeat until keypressed;
      read( kbd, command);
      command:= upcase( command);
      invalid:= true;
until command in validcommand;
end; {pageedit}



{****************************************************************************}
procedure move( var note_book: book;
                    pagesize,page: integer);

{ procedure to move notebook file pointer to (page) header}
{*****************************************************************************}

begin
seek( note_book, ((page+1)*(pagesize+1)));
end; {move}


{****************************************************************************}
procedure gopage( var note_book: book;
                      fname: longstring;
                  var currentpage: page;
                      contents: linetype;
                      prompt: lineid;
                      pagesize: integer;
                  var maxpage: integer;
                      page: integer);

{ procedure to goto (page) }
{*****************************************************************************}

begin
update( note_book, currentpage, pagesize);
if page > maxpage then page:= maxpage;
if page <= 1 then move( note_book, pagesize, 1)
   else move( note_book, pagesize, page);
getpage( note_book, currentpage, maxpage, contents, pagesize);
display( fname, currentpage, contents, prompt, pagesize, maxpage);
end; {gopage}

{****************************************************************************}
procedure uppage( var note_book: book;
                      fname: longstring;
                  var currentpage: page;
                      contents: linetype;
                      prompt: lineid;
                      pagesize: integer;
                  var maxpage: integer);

{ procedure to move to previous page }
{****************************************************************************}

var
   newpage: integer;

begin
newpage:= currentpage[0].pagenum - 1;
gopage( note_book, fname, currentpage, contents, prompt, pagesize, maxpage,
        newpage);
end; {uppage}

{****************************************************************************}
procedure downpage( var note_book   : book;
                        fname       : longstring;
                    var currentpage : page;
                        contents    : linetype;
                        prompt      : lineid;
                        pagesize    : integer;
                    var maxpage     : integer);

{ procedure to move to next page }
{****************************************************************************}

var
   newpage: integer;

begin
newpage:= currentpage[0].pagenum +1;
gopage( note_book, fname, currentpage, contents, prompt, pagesize, maxpage,
        newpage);
end; {downpage}

{************************************************************************}
procedure parse( var st1, st2: longstring);

{ procedure to read first group of non-blank char from st1 to st2
  char's are deleted from st1 as they are transfered}
{************************************************************************}

const
        blank : char = ' ';

begin
st2 := '';
while ((st1[1] = blank) and not(length(st1) = 0)) do delete(st1,1,1);
while (not(st1[1] = blank) and not(length(st1) = 0)) do
      begin
      st2:= st2+ st1[1];
      delete(st1,1,1);
end; {while}
end; {parse}

{****************************************************************************}
procedure gotopage( var note_book: book;
                        fname: longstring;
                    var currentpage: page;
                        contents: linetype;
                        prompt: lineid;
                        pagesize: integer;
                    var maxpage: integer);

{ procedure to get target page number and move to that page}
{****************************************************************************}

var
   resp, cvar: longstring;
   page, error: integer;

begin
repeat
      write('n?'); beep;
      read(resp);
      if length(resp)= 0 then str(currentpage[0].pagenum, resp);
      parse( resp, cvar);
      val( cvar, page, error);
until error = 0;
gopage( note_book, fname, currentpage, contents, prompt, pagesize,
        maxpage, page);
end; {gotopage}


{**************************************************************************}
procedure scrollup(     currentpage: page;
                        prompt: lineid;
                    var top, bottom: integer;
                        pagesize: integer);

{ procedure to scroll up five lines on the screen}
{***************************************************************************}

var
   position, linenum: integer;

begin
gotoxy( 1, 1);
for position:= 1 to 10 do delline;
top:= top + 5; bottom:= bottom +5;
for position:= 6 to 10 do
    begin
    linenum:= top + position -1;
    if linenum<= pagesize then
       writeline( currentpage[linenum], prompt[linenum], linenum, position);
end; {for}
end; {scrollup}

{***************************************************************************}
procedure scrolldown(    currentpage: page;
                         prompt: lineid;
                     var top, bottom: integer;
                         pagesize: integer);

{ procedure to scroll down five lines on the screen }
{**************************************************************************}

var
   position, linenum: integer;

begin
gotoxy( 1, 1);
top:= top- 5; bottom:= bottom- 5;
for position:= 1 to 5 do
    begin
    linenum:= top + position-1;
    if linenum <= pagesize then
       writeline( currentpage[linenum], prompt[linenum], linenum, position);
end; {for}
end; {scrolldown}

{***************************************************************************}
procedure goline(    currentpage: page;
                     prompt: lineid;
                 var linenum, top, bottom: integer;
                     pagesize: integer);

{ procedure to move control to target line number}
{***************************************************************************}

begin
if linenum < 1 then linenum:=1;
if linenum > pagesize then linenum:= pagesize;
while linenum < top do scrolldown( currentpage, prompt, top, bottom, pagesize);
while linenum > bottom do scrollup( currentpage, prompt, top, bottom,pagesize);
gotoxy( 23,(2*(linenum- top)+ 2));
end; { goline}


{****************************************************************************}
procedure upline( var currentpage: page;
                      prompt: lineid;
                  var linenum, top, bottom: integer;
                      pagesize: integer);

{ procedure to move to previous line }
{****************************************************************************}

begin
linenum:= linenum-1;
goline( currentpage, prompt, linenum, top, bottom, pagesize);
end; {upline}

{*****************************************************************************}
procedure downline( var currentpage: page;
                        prompt: lineid;
                    var linenum, top, bottom: integer;
                        pagesize: integer);

{ procedure to move to succeeding line }
{****************************************************************************}

begin
linenum:= linenum+ 1;
goline( currentpage, prompt, linenum, top, bottom, pagesize);
end; { downline}


{***************************************************************************}
procedure gotoline( var command: longstring;
                        currentpage: page;
                        prompt: lineid;
                    var linenum, top, bottom: integer;
                        pagesize: integer);

{ procedure to receive input target line and move there }
{***************************************************************************}

var
   cvar: longstring;
   tmp, error: integer;

begin
tmp:= linenum;

{ get line number}

cvar:= ' ';
parse (command, cvar);
val(cvar, linenum, error);
while error <> 0 do
      begin
      gotoxy( 23, wherey); clreol;
      write(' n?'); beep;
      read( command);
      if length( command) = 0 then
         begin
         linenum:= tmp;
         error:=0;
         end
      else
          begin
          parse( command, cvar);
          val( cvar, linenum, error);
          end;

end; {while}
goline( currentpage, prompt, linenum, top, bottom, pagesize);
end; { gotoline}

{****************************************************************************}
procedure deleteline( var currentpage: page;
                          prompt: lineid;
                          linenum, top: integer);

{ procedure to delete current line}
{****************************************************************************}

begin

if not ( deleteflag in currentpage[linenum].attributes) then
currentpage[0].attributes:= currentpage[0].attributes + [changed];

currentpage[linenum].attributes:= currentpage[linenum].attributes + [deleteflag];
gotoxy( 1, wherey -1); delline; delline;
writeline( currentpage[linenum], prompt[linenum], linenum, (linenum -top +1));
end; { deleteline }

{****************************************************************************}
procedure restoreline( var currentpage: page;
                           prompt: lineid;
                           linenum, top: integer);

{ procedure to restore deleted line }
{****************************************************************************}

begin

if deleteflag in currentpage[linenum].attributes then
currentpage[0].attributes:= currentpage[0].attributes + [changed];

currentpage[linenum].attributes:= currentpage[linenum].attributes -[deleteflag];
gotoxy( 1, wherey -1); delline; delline;
writeline( currentpage[linenum], prompt[linenum], linenum, (linenum -top +1));
end; { restoreline}

{****************************************************************************}
procedure changeline( var currentpage: page;
                          prompt: lineid;
                          linenum, top: integer;
                          remainder: longstring);

{ procedure to change contents of a line }
{****************************************************************************}

var
   change: boolean;
   scratch: array[1..3] of integer;
   cvar: longstring;
   index, error, terror: integer;

begin
change:= false;
terror:= 0;
if length( remainder)= 0 then
   begin
   write(' change to? '); beep;
   read( remainder);
   end; {if}

if length( remainder) <>0 then
   with currentpage[linenum] do
   case contains of
        char_: begin
               text:= remainder;
               change:= true;
               attributes:= attributes+ [active];
               end;

     integer_: begin
               parse( remainder, cvar);
               val( cvar, intval, error);
               if error <>0 then begin write( '  format error, integer '); beep;
               end
               else begin
                    change:= true;
                    attributes:= attributes+ [active];
                    end;
               end;

        real_: begin
               parse( remainder, cvar);
               val( cvar, realval, error);
               if error <> 0 then begin write( '  format error, real '); beep;
               end
               else begin
                    change:= true;
                    attributes:= attributes+ [active];
                    end;
               end;

        time : begin
               for index:= 1 to 2 do
                   begin
                   if length( remainder) <>0 then
                      begin
                      parse( remainder, cvar);
                      if length(cvar) = 0 then terror:= 1;
                      val( cvar, scratch[index], error);
                      terror:= terror+ error;
                      end
               else terror:= 1;
               end; {for}
               if terror = 0 then
                  begin
                  for index:= 1 to 2 do
                  timeval[ index]:= scratch[ index];
                  change:= true;
                  end
               else begin write('  format error, time '); beep; end;
               end; { time}

        date : begin
               for index:= 1 to 3 do
               begin
               if length( remainder) <>0 then
                  begin
                  parse( remainder, cvar);
                  if length( cvar) = 0 then terror:=1;
                  val( cvar, scratch[ index], error);
                  terror:= terror + error;
                  end
               else terror:=1;
               end; {for}
               if terror = 0 then
                  begin
                  for index:= 1 to 3 do dateval[ index]:= scratch[ index];
                  change:= true;
                  end
               else begin write('  format error, date '); beep; end;
               end;

end; {with}
if change then
   begin

   if not( new in currentpage[0].attributes) then
      currentpage[0].attributes:= currentpage[0].attributes +[changed];

   currentpage[linenum].attributes:= currentpage[linenum].attributes+[active];
   gotoxy( 1, wherey -1); delline; delline;
   writeline( currentpage[linenum], prompt[linenum], linenum, (linenum -top +1));
   end;

end; {changeline}


{****************************************************************************}
procedure lineedit( var command: char;
                    var remainder: longstring);

{ procedure to read keyboard input to line editor }
{****************************************************************************}

const
     validcommand: set of char = ['U','D','G','K','R','C','Q','A'];

var
   invalid:boolean;

begin
invalid:= false;
repeat
gotoxy(23, wherey); clreol;
write( ' ?'); if invalid then beep;
read( remainder);
if length (remainder) = 0 then command:= 'D'
else if remainder[1] = '/' then  begin
     delete (remainder, 1, 1);
     if length (remainder) <>0 then begin
        command:= remainder[1];
        delete (remainder, 1, 1);
        end
     else command:= 'Z';
     end
else begin
     command:= 'A';
end; {if}
command:= upcase(command);
invalid:= true;
until command in validcommand;
gotoxy( 23,wherey); clreol;
end; { lineedit}


{****************************************************************************}
procedure editline( var note_book: book;
                        fname: longstring;
                    var currentpage: page;
                        contents: linetype;
                        prompt: lineid;
                        pagesize, maxpage: integer;
                        add: boolean);

{ procedure to edit the contents of a page line by line }
{*****************************************************************************}

var
   top, bottom, linenum: integer;
   remainder: longstring;
   command: char;

begin
gotoxy(13,4); write('edit mode: '); if add then write( 'add ');write('line  ');

{ write line editor commands }

window( 41,2,80,4); normvideo;
gotoxy(1,1); insline; write('/U'); lowvideo; write('p'); normvideo;
gotoxy(11,1); write('/D'); lowvideo; write('own'); normvideo;
gotoxy(21,1); write('/K'); lowvideo; write('ill'); normvideo;
gotoxy(1,2); insline; write('/G'); lowvideo; write('oto'); normvideo;
gotoxy(11,2); write('/Q'); lowvideo; write('uit'); normvideo;
gotoxy(21,2); write('/R'); lowvideo; write('estore'); normvideo;
gotoxy(1,3); insline;
write('/C'); lowvideo; write('hange line...direct input in add mode');
window(11, 6, 80, 25);
top:= 1; bottom:= 10;
linenum:=1;
gotoxy( 23, 2);
repeat

{execute line editor commands}

      lineedit( command, remainder);
      case command of
           'U': upline( currentpage, prompt, linenum, top, bottom, pagesize);
           'D': downline( currentpage, prompt, linenum, top, bottom, pagesize);
           'G': gotoline( remainder, currentpage, prompt, linenum, top, bottom,
                          pagesize);
           'K': deleteline( currentpage, prompt, linenum, top);
           'R': restoreline( currentpage, prompt, linenum, top);
           'C': changeline(currentpage, prompt, linenum, top, remainder);
           'A': if add then begin
                   changeline(currentpage, prompt, linenum, top,
                                       remainder);
                   downline( currentpage, prompt, linenum, top, bottom, pagesize);
                   end
                else beep;
     end; {case}

{return to page editor}

until command ='Q';
window( 1, 1, 80, 25);
update( note_book, currentpage, pagesize);
seek( note_book, (filepos(note_book) -(pagesize+1)));
getpage( note_book, currentpage, maxpage, contents, pagesize);
display( fname, currentpage, contents, prompt,pagesize, maxpage);
end; {editline}


{*******************************************************************}
procedure addpage;

{procedure to add a page to end of file}
{*******************************************************************}

const
     add: boolean= true;

begin
seek( note_book, filesize(note_book));
newpage( note_book, maxpage, contents, pagesize);
getpage( note_book, currentpage, maxpage, contents, pagesize);
display( fname, currentpage, contents, prompt, pagesize, maxpage);
gotoxy(13,4); write( 'edit mode: line');
editline( note_book, fname, currentpage, contents, prompt, pagesize, maxpage,
          add);
end; {addpage}

{****************************************************************************}
procedure deletepage( var note_book: book;
                          fname: longstring;
                      var currentpage: page;
                          contents: linetype;
                          prompt: lineid;
                          pagesize, maxpage: integer);

{ procedure to set delete flag in current page attributes}
{****************************************************************************}

begin
currentpage[0].attributes:= currentpage[0].attributes + [deleteflag];
update( note_book, currentpage, pagesize);
seek( note_book, (filepos(note_book)- (pagesize+ 1)));
getpage( note_book, currentpage, maxpage, contents, pagesize);
display( fname, currentpage, contents, prompt, pagesize, maxpage);
end; {deletepage}

{***************************************************************************}
procedure restorepage( var note_book: book;
                           fname: longstring;
                       var currentpage: page;
                           contents: linetype;
                           prompt: lineid;
                           pagesize, maxpage: integer);

{ procedure to restore deleted pages }
{***************************************************************************}

begin
currentpage[0].attributes:= currentpage[0].attributes - [deleteflag];
update( note_book, currentpage, pagesize);
seek( note_book, (filepos( note_book) -(pagesize +1)));
getpage( note_book, currentpage, maxpage, contents, pagesize);
display( fname, currentpage, contents, prompt, pagesize, maxpage);
end; {restorepage}


{************************** main program ************************************}

begin

beep;

setup( note_book, fname, prompt, contents, pagesize, maxpage);

getpage( note_book, currentpage, maxpage, contents, pagesize);

display( fname, currentpage, contents, prompt, pagesize, maxpage);

repeat
      pageedit( command);
      case command of
           'U': uppage( note_book, fname, currentpage, contents, prompt,
                        pagesize, maxpage);
           'E': editline( note_book, fname, currentpage, contents, prompt,
                          pagesize, maxpage, add);
           'D': downpage( note_book, fname, currentpage, contents, prompt,
                          pagesize, maxpage);
           'K': deletepage( note_book, fname, currentpage, contents, prompt,
                            pagesize, maxpage);
           'G': gotopage( note_book, fname, currentpage, contents, prompt,
                          pagesize, maxpage);
           'R': restorepage( note_book, fname, currentpage, contents, prompt,
                             pagesize, maxpage);
           'A': addpage( note_book, fname, currentpage, contents, prompt,
                         pagesize, maxpage);
      end; {case}
until command = 'Q';  {quit}

update( note_book, currentpage, pagesize);

clrscr;

shut( note_book, pagesize, maxpage);

writeln; write(' Remove deleted lines and pages from file? '); beep;
read( resp);
if upcase( resp[1]) = 'Y' then cull( note_book, fname);
clrscr;
end.


