{ SCRNSTUF.PAS }

UNIT ScrnStuf;

{
Description:  PC-specific video routines

Author:       Don Taylor and Keith Harris
Date:         6/14/87
Last revised: 02/03/1989  14:37:51
Application:  IBM PC and compatibles

}


{--------------------}
     INTERFACE
{--------------------}

USES
 Crt,          { Built-in video routines                                     }
 Dos,          { Built-in DOS interface routines                             }
 FastStuf;     { TUG's direct screen-writing routines                        }

TYPE  { Public Data Types }
 CursorType      = (Single, Double, Half, Block, None);
 ScrnModeType    = (MonoMode, ColorMode);

{ The ordinal value of AdapterTypes NoDisplay through MCGAColor correspond to
  the Display Combination Codes as assigned by IBM. The ordinal values of the
  Hercules AdapterTypes do NOT correspond to a DCC. They are merely added for
  clarity. - Keith A. Harris }
                                            { DCC }
 AdapterType     = (NoDisplay,              { $00 }
		    MDA,                    { $01 }
                    CGA,                    { $02 }
                    DCC3,                   { Reserved by IBM }
                    EGAColor,               { $04 }
                    EGAMono,                { $05 }
                    PGA,                    { $06 }
                    VGAMono,                { $07 }
                    VGAColor,               { $08 }
                    DCC9,                   { Reserved by IBM }
                    MCGADigitalColor,       { $0A }
                    MCGAMono,               { $0B }
                    MCGAColor,              { $0C }
                    HercMono,
                    HercPlus,
                    HercInColor,
                    Unknown);               { $FF }


CONST
 ScrnColorSeg    = $B800;    { Segment of color video memory                 }
 ScrnMonoSeg     = $B000;    { Segment of monochrome video memory            }

 ScrnNoErr       = 0;        { No error detected                             }
 ScrnNoMemErr    = 1;        { Not enough memory available                   }
 ScrnBadPtrErr   = 2;        { Pointer value invalid                         }
 ScrnBadModeErr  = 3;        { Undefined video mode                          }

FUNCTION  ScrnResult : BYTE;                   { Returns error result value  }

FUNCTION  ScrnAdapter   : AdapterType;         { Type of video adapter       }
FUNCTION  ScrnMode      : ScrnModeType;        { Current video mode          }
FUNCTION  ScrnSeg       : WORD;                { Segment of video memory     }
FUNCTION  ScrnMaxColumn : BYTE;                { Last column on screen       }
FUNCTION  ScrnMaxRow    : BYTE;                { Last row on screen          }

PROCEDURE GetCursor(VAR Cursor : CursorType);  { Read current cursor type    }
PROCEDURE SetCursor(Cursor : CursorType);      { Select type of cursor       }
PROCEDURE SaveCursorPosn(VAR x,y : BYTE);      { Save cursor's position      }
PROCEDURE RestoreCursorPosn(x,y : BYTE);       { Restore cursor's position   }

PROCEDURE GetScreenBuf(VAR SPtr : POINTER);    { Reserve screen buffer       }
PROCEDURE FreeScreenBuf(VAR SPtr : POINTER);   { Release screen buffer       }
PROCEDURE SaveScreen(VAR SPtr : POINTER);      { Save whole screen to memory }
PROCEDURE RestoreScreen(SPtr : POINTER);       { Get screen from memory      }

PROCEDURE GetDisplayStatus( VAR x    : BYTE;        { Cursor x position      }
                            VAR y    : BYTE;        { Cursor y position      }
                            VAR c    : CursorType;  { Cursor shape           }
                            VAR a    : BYTE;        { Text attribute         }
                            VAR wmin : WORD;        { Window minimum         }
                            VAR wmax : WORD);       { Window maximum         }

PROCEDURE SetDisplayStatus( x        : BYTE;        { Cursor x position      }
                            y        : BYTE;        { Cursor y position      }
                            c        : CursorType;  { Cursor shape           }
                            a        : BYTE;        { Text attribute         }
                            wmin     : WORD;        { Window minimum         }
                            wmax     : WORD);       { Window maximum         }

PROCEDURE AbsGOTOXY(x, y : BYTE);   { GOTOXY absolute (ignore window limit)  }

PROCEDURE BlinkOn;                  { Set blinking mode                      }
PROCEDURE BlinkOff;                 { Set non-blinking mode                  }

PROCEDURE ScrollDown( NumLines : BYTE;  { Number of lines to scroll          }
                      ULx      : BYTE;  { Upper left column number           }
                      ULy      : BYTE;  { Upper left row number              }
                      LRx      : BYTE;  { Lower right column number          }
                      LRy      : BYTE;  { Lower right row number             }
                      DAttr    : BYTE); { Display attribute                  }

PROCEDURE ScrollUp  ( NumLines : BYTE;  { Number of lines to scroll          }
                      ULx      : BYTE;  { Upper left column number           }
                      ULy      : BYTE;  { Upper left row number              }
                      LRx      : BYTE;  { Lower right column number          }
                      LRy      : BYTE;  { Lower right row number             }
                      DAttr    : BYTE); { Display attribute                  }


{--------------------}
   IMPLEMENTATION
{--------------------}

VAR  { Private Variables }
 SCursShape  : WORD;            { Holds cursor shape during screen saves     }
 SAdapter    : AdapterType;     { Type of video adapter                      }
 SCursor     : CursorType;      { Type of cursor                             }
 SErr        : BYTE;            { Error result value                         }

 SNumCol : INTEGER ABSOLUTE $0000:$044A;  { Max number of display columns    }
 SNumRow : BYTE    ABSOLUTE $0000:$0484;  { Max number of display lines      }


{--------------------}

FUNCTION ScrnResult : BYTE;

BEGIN
 ScrnResult := SErr;
 SErr       := ScrnNoErr
END; { ScrnResult }


{--------------------}

FUNCTION ScrnMaxColumn : BYTE;

BEGIN
 SErr          := ScrnNoErr;
 ScrnMaxColumn := SNumCol
END; { ScrnMaxColumn }


{--------------------}

FUNCTION ScrnMaxRow : BYTE;

BEGIN
 SErr       := ScrnNoErr;
 ScrnMaxRow := SNumRow + 1
END; { ScrnMaxColumn }


{--------------------}

FUNCTION ScrnMode : ScrnModeType;

CONST
 IntrCall    = 16;
 ServiceCall = 15;

VAR
 Result  : INTEGER;
 ModeRec : Registers;

BEGIN
 WITH ModeRec DO
  BEGIN
   AH := ServiceCall;
   INTR(IntrCall, ModeRec);
   Result := AL;
   CASE Result OF
    0..6   : ScrnMode := ColorMode;
    7      : ScrnMode := MonoMode;
   ELSE
    SErr := ScrnBadModeErr
   END { CASE }
  END { WITH }
END;  { ScrnMode }


{--------------------}

FUNCTION ScrnAdapter : AdapterType;
{ Written by Keith A. Harris }

VAR
  Regs      : Dos.Registers;
  LoopCount : INTEGER;
  GoodHits  : BYTE;
  TestBit   : BYTE;

BEGIN { Check for the lastest standard }
 SErr := ScrnNoErr;
 WITH Regs DO
  BEGIN
    AX := $1A00;  { Return Display Combination Code }
    INTR($10,Regs);

    IF (AL = $1A) AND (NOT (AdapterType(BL) IN [MDA, DCC3, DCC9, UnKnown]))
     THEN ScrnAdapter := AdapterType(BL)
     ELSE BEGIN { DCC call is not supported.  No PS/2 hardware/BIOS present }
           AX := $1200;   { Video SubSystem Configuration (Alternate Select) }
           BX := $0010;   { Return video configuration }
           INTR($10,Regs);

           IF BL <> $10 { Valid call }
            THEN BEGIN
	          IF (BH = 0)
                   THEN ScrnAdapter := EGAColor
	           ELSE ScrnAdapter := EGAMono
                 END
            ELSE BEGIN { No Ega hardware/bios - Call equipment service }
	          INTR($11,Regs);
	          IF (((AL AND $30) SHR $04) = $03)
                   THEN BEGIN { Monochrome card: Is it Hercules? }
	                 TestBit := Port[$03BA] AND $80;
	                 LoopCount := 8000;
	                 GoodHits := 0;
	                 REPEAT
	                  IF (TestBit <> (Port[$03BA] AND $80))
                           THEN INC(GoodHits);
	                  DEC(LoopCount);
	                 UNTIL (GoodHits >= 10) OR (LoopCount = 0);
	                 IF GoodHits >= 10
                          THEN BEGIN { Which Hercules card is this? }
    { Bits 4, 5 & 6 mask }      TestBit := Port[$03BA] AND $70;
	                        CASE TestBit of
    { Bit 4 mask         }       $10 : ScrnAdapter := HercPlus;
    { Bits 4 & 6 mask    }	 $5A : ScrnAdapter := HercInColor
	                        ELSE ScrnAdapter := HercMono
	                        END; { Case }

	                       END { which Hercules card }
	                  ELSE ScrnAdapter := MDA;
	                END { Mono card detected }
	           ELSE ScrnAdapter := CGA;
                 END { Equipment determination service }
          END { DCC Call - Look for EGA }
  END { WITH }
END; { ScrnAdapter }


{--------------------}

FUNCTION ScrnSeg : WORD;

BEGIN
 SErr    := ScrnNoErr;
 IF ScrnMode = ColorMode
  THEN ScrnSeg := ScrnColorSeg
  ELSE ScrnSeg := ScrnMonoSeg
END; { ScrnSeg }


{--------------------}

PROCEDURE GetCursor(VAR Cursor : CursorType);

CONST
 IntrCall    = 16;
 ServiceCall =  3;

VAR
 FirstLine : INTEGER;
 LastLine  : INTEGER;
 CursorRec : Registers;

BEGIN
 SErr := ScrnNoErr;
 WITH CursorRec DO
  BEGIN
   AH := ServiceCall;
   INTR(IntrCall, CursorRec);
   FirstLine := CH;
   LastLine  := CL
  END; {WITH}

 CASE FirstLine OF
  0  : Cursor := Block;
  4  : Cursor := Half;
  6  : Cursor := Double;
  7  : IF LastLine = 7
        THEN Cursor := Single
        ELSE Cursor := Half;
  12 : Cursor := Double;
  13 : Cursor := Single;
  32 : Cursor := None
 END { CASE }
END; { GetCursor }


{--------------------}

PROCEDURE SetCursor(Cursor : CursorType);

CONST
 IntrCall    = 16;
 ServiceCall =  1;

VAR
 FirstLine : INTEGER;
 LastLine  : INTEGER;
 CursorRec : Registers;

BEGIN
 SErr := ScrnNoErr;
 CASE Cursor OF
  Single : IF ScrnMode = ColorMode
            THEN BEGIN
                  FirstLine := 7;
                  LastLine  := 7
                 END
            ELSE BEGIN
                  FirstLine := 13;
                  LastLine  := 13
                 END;
  Double : IF ScrnMode = ColorMode
            THEN BEGIN
                  FirstLine := 6;
                  LastLine  := 7
                 END
            ELSE BEGIN
                  FirstLine := 12;
                  LastLine  := 13
                 END;
  Half   : IF ScrnMode = ColorMode
              THEN BEGIN
                  FirstLine := 4;
                  LastLine  := 7
                 END
            ELSE BEGIN
                  FirstLine :=  7;
                  LastLine  := 13
                 END;
  Block  : IF ScrnMode = ColorMode
              THEN BEGIN
                  FirstLine := 0;
                  LastLine  := 7
                 END
            ELSE BEGIN
                  FirstLine :=  0;
                  LastLine  := 13
                 END;
  None   : BEGIN
            FirstLine := 32;
            LastLine  := 32
           END;
 END; { CASE }

 WITH CursorRec DO
  BEGIN
   AH := ServiceCall;
   CH := FirstLine;
   CL := LastLine;
   INTR(IntrCall, CursorRec)
  END { WITH }
END;  { SetCursor }


{--------------------}

PROCEDURE AbsGOTOXY(x, y : BYTE);

CONST
 IntrCall    = 16;   { ROM Video BIOS call         }
 ServiceCall =  2;   { Set cursor position service }

VAR
 CurDOSRec : Registers;

BEGIN
 SErr := ScrnNoErr;
 WITH CurDOSRec DO
  BEGIN
   AH := ServiceCall;
   BH := 0;      { Base video page          }
   DH := y - 1;  { 0,0 is upper left corner }
   DL := x - 1
  END; { WITH }
 INTR(IntrCall, CurDOSRec)
END; { AbsGOTOXY }


{--------------------}

PROCEDURE SaveCursorPosn(VAR x,y : BYTE);

BEGIN
 SErr := ScrnNoErr;
 x    := WhereX;
 y    := WhereY
END; { SaveCursorPosn }


{--------------------}

PROCEDURE RestoreCursorPosn(x,y : BYTE);

BEGIN
 SErr := ScrnNoErr;
 GOTOXY(x,y)
END; { RestoreCursorPosn }


{--------------------}

PROCEDURE GetDisplayStatus( VAR x    : BYTE;        { Cursor x position  }
                            VAR y    : BYTE;        { Cursor y position  }
                            VAR c    : CursorType;  { Cursor shape       }
                            VAR a    : BYTE;        { Text attribute     }
                            VAR wmin : WORD;        { Window minimum     }
                            VAR wmax : WORD);       { Window maximum     }

BEGIN
 SaveCursorPosn(x,y);  { Also reinitializes SErr }
 GetCursor(c);
 a    := TextAttr;
 wmin := WindMin;
 wmax := WindMax
END; { SaveDisplayStatus }


{--------------------}

PROCEDURE SetDisplayStatus( x    : BYTE;            { Cursor x position  }
                            y    : BYTE;            { Cursor y position  }
                            c    : CursorType;      { Cursor shape       }
                            a    : BYTE;            { Text attribute     }
                            wmin : WORD;            { Window minimum     }
                            wmax : WORD);           { Window maximum     }

BEGIN
 Window(LO(wmin) + 1, HI(wmin) + 1, LO(wmax) + 1, HI(wmax) + 1);
 RestoreCursorPosn(x,y); { Also reinitializes SErr }
 SetCursor(c);
 TextAttr := a
END; { RestoreDisplayStatus }


{--------------------}

PROCEDURE GetScreenBuf(VAR SPtr : POINTER);

VAR
 ScreenSize : WORD;

BEGIN
 SPtr       := NIL;
 SErr       := ScrnNoErr;
 ScreenSize := ScrnMaxRow * ScrnMaxColumn * 2;

 IF MaxAvail > ScreenSize
  THEN GETMEM(SPtr, ScreenSize)
  ELSE SErr := ScrnNoMemErr
END; { GetScreenBuf }


{--------------------}

PROCEDURE FreeScreenBuf(VAR SPtr : POINTER);

BEGIN
 SErr := ScrnNoErr;
 IF SPtr <> NIL
  THEN BEGIN
        FREEMEM(SPtr, ScrnMaxRow * ScrnMaxColumn * 2);
        SPtr := NIL
       END
  ELSE SErr := ScrnBadPtrErr
END; { FreeScreenBuf }


{--------------------}

PROCEDURE SaveScreen(VAR SPtr : POINTER);

VAR
 ScreenLoc  : POINTER;
 SScrnSeg   : WORD;

BEGIN
 SErr     := ScrnNoErr;
 SScrnSeg := ScrnSeg;

 IF SPtr <> NIL
  THEN BEGIN
        ScreenLoc := PTR(SScrnSeg, 0);
        CopyFromScreen(ScreenLoc^, SPtr^, ScrnMaxRow * ScrnMaxColumn * 2)
       END
  ELSE SErr := ScrnBadPtrErr
END; { SaveScreen }


{--------------------}

PROCEDURE RestoreScreen(SPtr : POINTER);

VAR
 ScreenLoc : POINTER;
 SScrnSeg  : WORD;

BEGIN
 SErr     := ScrnNoErr;
 SScrnSeg := ScrnSeg;

 IF SPtr <> NIL
  THEN BEGIN
        ScreenLoc := PTR(SScrnSeg, 0);
        CopyToScreen(SPtr^, ScreenLoc^, ScrnMaxRow * ScrnMaxColumn * 2)
       END
  ELSE SErr := ScrnBadPtrErr
END; { RestoreScreen }


{--------------------}

PROCEDURE BlinkOn;

BEGIN
 SErr     := ScrnNoErr;
 TextAttr := TextAttr OR $80
END; { BlinkOn }


{--------------------}

PROCEDURE BlinkOff;

BEGIN
 SErr     := ScrnNoErr;
 TextAttr := TextAttr AND $7F
END; { BlinkOff }


{--------------------}

PROCEDURE ScrollDown( NumLines : BYTE;  { Number of lines to scroll }
                      ULx      : BYTE;  { Upper left column number  }
                      ULy      : BYTE;  { Upper left row number     }
                      LRx      : BYTE;  { Lower right column number }
                      LRy      : BYTE;  { Lower right row number    }
                      DAttr    : BYTE); { Display attribute         }

CONST
 IntrCall    = 16;   { ROM Video BIOS call        }
 ServiceCall =  7;   { Scroll window down service }

VAR
 SDDOSRec : Registers;

BEGIN
 SErr := ScrnNoErr;
 WITH SDDOSRec DO
  BEGIN
   AH := ServiceCall;
   AL := NumLines;
   CH := ULy - 1;
   CL := ULx - 1;
   DH := LRy - 1;
   DL := LRx - 1;
   BH := DAttr
  END; { WITH }
 INTR(IntrCall, SDDOSRec)
END;  { ScrollDown }


{--------------------}

PROCEDURE ScrollUp  ( NumLines : BYTE;  { Number of lines to scroll }
                      ULx      : BYTE;  { Upper left column number  }
                      ULy      : BYTE;  { Upper left row number     }
                      LRx      : BYTE;  { Lower right column number }
                      LRy      : BYTE;  { Lower right row number    }
                      DAttr    : BYTE); { Display attribute         }

CONST
 IntrCall    = 16;   { ROM Video BIOS call        }
 ServiceCall =  6;   { Scroll window up service   }

VAR
 SDDOSRec : Registers;

BEGIN
 SErr := ScrnNoErr;
 WITH SDDOSRec DO
  BEGIN
   AH := ServiceCall;
   AL := NumLines;
   CH := ULy - 1;
   CL := ULx - 1;
   DH := LRy - 1;
   DL := LRx - 1;
   BH := DAttr
  END; { WITH }
 INTR(IntrCall, SDDOSRec)
END;  { ScrollUp }


{====================}
{   INITIALIZATION   }
{====================}

BEGIN { UNIT ScrnStuf }
 IF SNumRow = 0 THEN SNumRow := 24;
 SErr := ScrnNoErr
END.  { UNIT ScrnStuf }

