{ SCRNSTUF.PAS }

UNIT ScrnStuf;

{
Description:  PC-specific video routines

Author:       Don Taylor
Date:         6/14/87
Last revised: 09/30/1988  14:21:06
Application:  IBM PC and compatibles
}


{--------------------}
     INTERFACE
{--------------------}

USES
 Crt,          { Built-in video routines                                     }
 Dos;          { Built-in DOS interface routines                             }

TYPE  { Public Data Types }
 AdapterType = (MonoAdapter, ColorAdapter);
 CursorType  = (Single, Double, Half, Block, None);

FUNCTION  ScrnAdapter   : AdapterType;         { Type of video adapter       }
FUNCTION  ScrnSeg       : WORD;                { Segment of video memory     }
FUNCTION  ScrnMaxColumn : BYTE;                { Last column on screen       }
FUNCTION  ScrnMaxRow    : BYTE;                { Last row on screen          }

PROCEDURE GetCursor(VAR Cursor : CursorType);  { Read current cursor type    }
PROCEDURE SetCursor(Cursor : CursorType);      { Select type of cursor       }
PROCEDURE SaveCursorPosn(VAR x,y : BYTE);      { Save cursor's position      }
PROCEDURE RestoreCursorPosn(x,y : BYTE);       { Restore cursor's position   }

PROCEDURE GetDisplayStatus( VAR x    : BYTE;        { Cursor x position      }
                            VAR y    : BYTE;        { Cursor y position      }
                            VAR c    : CursorType;  { Cursor shape           }
                            VAR a    : BYTE;        { Text attribute         }
                            VAR wmin : WORD;        { Window minimum         }
                            VAR wmax : WORD);       { Window maximum         }

PROCEDURE SetDisplayStatus( x        : BYTE;        { Cursor x position      }
                            y        : BYTE;        { Cursor y position      }
                            c        : CursorType;  { Cursor shape           }
                            a        : BYTE;        { Text attribute         }
                            wmin     : WORD;        { Window minimum         }
                            wmax     : WORD);       { Window maximum         }

PROCEDURE AbsGOTOXY(x, y : BYTE);   { GOTOXY absolute (ignore window limit)  }

PROCEDURE BlinkOn;                  { Set blinking mode                      }
PROCEDURE BlinkOff;                 { Set non-blinking mode                  }

PROCEDURE ScrollDown( NumLines : BYTE;  { Number of lines to scroll          }
                      ULx      : BYTE;  { Upper left column number           }
                      ULy      : BYTE;  { Upper left row number              }
                      LRx      : BYTE;  { Lower right column number          }
                      LRy      : BYTE;  { Lower right row number             }
                      DAttr    : BYTE); { Display attribute                  }

PROCEDURE ScrollUp  ( NumLines : BYTE;  { Number of lines to scroll          }
                      ULx      : BYTE;  { Upper left column number           }
                      ULy      : BYTE;  { Upper left row number              }
                      LRx      : BYTE;  { Lower right column number          }
                      LRy      : BYTE;  { Lower right row number             }
                      DAttr    : BYTE); { Display attribute                  }


{--------------------}
   IMPLEMENTATION
{--------------------}

VAR  { Private Variables }
 SCursShape  : WORD;        { Holds cursor shape during screen saves         }
 SScrnSeg    : WORD;        { Start of video adapter memory                  }
 SAdapter    : AdapterType; { Type of video adapter                          }
 SCursor     : CursorType;  { Type of cursor                                 }

 SNumCol : INTEGER ABSOLUTE $0000:$044A;  { Max number of display columns    }
 SNumRow : BYTE    ABSOLUTE $0000:$0484;  { Max number of display lines      }


{--------------------}

FUNCTION ScrnMaxColumn : BYTE;

BEGIN
 ScrnMaxColumn := SNumCol
END; { ScrnMaxColumn }


{--------------------}

FUNCTION ScrnMaxRow : BYTE;

BEGIN
 ScrnMaxRow := SNumRow + 1
END; { ScrnMaxColumn }


{--------------------}

PROCEDURE GetAdapterType;

CONST
 IntrCall    = 16;
 ServiceCall = 15;

VAR
 Result     : INTEGER;
 AdapterRec : Registers;

BEGIN
 WITH AdapterRec DO
  BEGIN
   AH := ServiceCall;
   INTR(IntrCall, AdapterRec);
   Result := AL;
   CASE Result OF
    0..6   : SAdapter := ColorAdapter;
    7      : SAdapter := MonoAdapter;
   ELSE
    SAdapter := MonoAdapter
   END { CASE }
  END { WITH }
END;  { GetAdapterType }


{--------------------}

FUNCTION ScrnAdapter : AdapterType;

BEGIN
 ScrnAdapter := SAdapter
END; { ScrnAdapter }


{--------------------}

FUNCTION ScrnSeg : WORD;

BEGIN
 ScrnSeg := SScrnSeg
END; { ScrnSeg }


{--------------------}

PROCEDURE InitScreenAddress;

CONST
 MonoSeg  = $B000;
 ColorSeg = $B800;

BEGIN
 CASE SAdapter OF
  MonoAdapter  : SScrnSeg := MonoSeg;
  ColorAdapter : SScrnSeg := ColorSeg
 END; { CASE }
END;  { InitScreenAddress }


{--------------------}

PROCEDURE GetCursor(VAR Cursor : CursorType);

CONST
 IntrCall    = 16;
 ServiceCall =  3;

VAR
 FirstLine : INTEGER;
 LastLine  : INTEGER;
 CursorRec : Registers;

BEGIN
 WITH CursorRec DO
  BEGIN
   AH := ServiceCall;
   INTR(IntrCall, CursorRec);
   FirstLine := CH;
   LastLine  := CL
  END; {WITH}

 CASE FirstLine OF
  0  : Cursor := Block;
  4  : Cursor := Half;
  6  : Cursor := Double;
  7  : IF LastLine = 7
        THEN Cursor := Single
        ELSE Cursor := Half;
  12 : Cursor := Double;
  13 : Cursor := Single;
  32 : Cursor := None
 END { CASE }
END; { GetCursor }


{--------------------}

PROCEDURE SetCursor(Cursor : CursorType);

CONST
 IntrCall    = 16;
 ServiceCall =  1;

VAR
 FirstLine : INTEGER;
 LastLine  : INTEGER;
 CursorRec : Registers;

BEGIN
 CASE Cursor OF
  Single : IF SAdapter = ColorAdapter
            THEN BEGIN
                  FirstLine := 7;
                  LastLine  := 7
                 END
            ELSE BEGIN
                  FirstLine := 13;
                  LastLine  := 13
                 END;
  Double : IF SAdapter = ColorAdapter
            THEN BEGIN
                  FirstLine := 6;
                  LastLine  := 7
                 END
            ELSE BEGIN
                  FirstLine := 12;
                  LastLine  := 13
                 END;
  Half   : IF SAdapter = ColorAdapter
              THEN BEGIN
                  FirstLine := 4;
                  LastLine  := 7
                 END
            ELSE BEGIN
                  FirstLine :=  7;
                  LastLine  := 13
                 END;
  Block  : IF SAdapter = ColorAdapter
              THEN BEGIN
                  FirstLine := 0;
                  LastLine  := 7
                 END
            ELSE BEGIN
                  FirstLine :=  0;
                  LastLine  := 13
                 END;
  None   : BEGIN
            FirstLine := 32;
            LastLine  := 32
           END;
 END; { CASE }

 WITH CursorRec DO
  BEGIN
   AH := ServiceCall;
   CH := FirstLine;
   CL := LastLine;
   INTR(IntrCall, CursorRec)
  END { WITH }
END;  { SetCursor }


{--------------------}

PROCEDURE AbsGOTOXY(x, y : BYTE);

CONST
 IntrCall    = 16;   { ROM Video BIOS call         }
 ServiceCall =  2;   { Set cursor position service }

VAR
 CurDOSRec : Registers;

BEGIN
 WITH CurDOSRec DO
  BEGIN
   AH := ServiceCall;
   BH := 0;      { Base video page          }
   DH := y - 1;  { 0,0 is upper left corner }
   DL := x - 1
  END; { WITH }
 INTR(IntrCall, CurDOSRec)
END; { AbsGOTOXY }


{--------------------}

PROCEDURE SaveCursorPosn(VAR x,y : BYTE);

BEGIN
 x := WhereX;
 y := WhereY
END; { SaveCursorPosn }


{--------------------}

PROCEDURE RestoreCursorPosn(x,y : BYTE);

BEGIN
 GOTOXY(x,y)
END; { RestoreCursorPosn }


{--------------------}

PROCEDURE GetDisplayStatus( VAR x    : BYTE;        { Cursor x position  }
                            VAR y    : BYTE;        { Cursor y position  }
                            VAR c    : CursorType;  { Cursor shape       }
                            VAR a    : BYTE;        { Text attribute     }
                            VAR wmin : WORD;        { Window minimum     }
                            VAR wmax : WORD);       { Window maximum     }

BEGIN
 SaveCursorPosn(x,y);
 GetCursor(c);
 a    := TextAttr;
 wmin := WindMin;
 wmax := WindMax
END; { SaveDisplayStatus }


{--------------------}

PROCEDURE SetDisplayStatus( x    : BYTE;            { Cursor x position  }
                            y    : BYTE;            { Cursor y position  }
                            c    : CursorType;      { Cursor shape       }
                            a    : BYTE;            { Text attribute     }
                            wmin : WORD;            { Window minimum     }
                            wmax : WORD);           { Window maximum     }

BEGIN
 Window(LO(wmin) + 1, HI(wmin) + 1, LO(wmax) + 1, HI(wmax) + 1);
 RestoreCursorPosn(x,y);
 SetCursor(c);
 TextAttr := a
END; { RestoreDisplayStatus }


{--------------------}

PROCEDURE BlinkOn;

BEGIN
 TextAttr := TextAttr OR $80
END; { BlinkOn }


{--------------------}

PROCEDURE BlinkOff;

BEGIN
 TextAttr := TextAttr AND $7F
END; { BlinkOff }


{--------------------}

PROCEDURE ScrollDown( NumLines : BYTE;  { Number of lines to scroll }
                      ULx      : BYTE;  { Upper left column number  }
                      ULy      : BYTE;  { Upper left row number     }
                      LRx      : BYTE;  { Lower right column number }
                      LRy      : BYTE;  { Lower right row number    }
                      DAttr    : BYTE); { Display attribute         }

CONST
 IntrCall    = 16;   { ROM Video BIOS call        }
 ServiceCall =  7;   { Scroll window down service }

VAR
 SDDOSRec : Registers;

BEGIN
 WITH SDDOSRec DO
  BEGIN
   AH := ServiceCall;
   AL := NumLines;
   CH := ULy - 1;
   CL := ULx - 1;
   DH := LRy - 1;
   DL := LRx - 1;
   BH := DAttr
  END; { WITH }
 INTR(IntrCall, SDDOSRec)
END;  { ScrollDown }


{--------------------}

PROCEDURE ScrollUp  ( NumLines : BYTE;  { Number of lines to scroll }
                      ULx      : BYTE;  { Upper left column number  }
                      ULy      : BYTE;  { Upper left row number     }
                      LRx      : BYTE;  { Lower right column number }
                      LRy      : BYTE;  { Lower right row number    }
                      DAttr    : BYTE); { Display attribute         }

CONST
 IntrCall    = 16;   { ROM Video BIOS call        }
 ServiceCall =  6;   { Scroll window up service   }

VAR
 SDDOSRec : Registers;

BEGIN
 WITH SDDOSRec DO
  BEGIN
   AH := ServiceCall;
   AL := NumLines;
   CH := ULy - 1;
   CL := ULx - 1;
   DH := LRy - 1;
   DL := LRx - 1;
   BH := DAttr
  END; { WITH }
 INTR(IntrCall, SDDOSRec)
END;  { ScrollUp }


{====================}
{   INITIALIZATION   }
{====================}

BEGIN { UNIT ScrnStuf }
 IF SNumRow = 0 THEN SNumRow := 24;
 GetAdapterType;
 InitScreenAddress
END.  { UNIT ScrnStuf }

