/*
        File:           STMT.CPP
        
        Revision:       2.0 Beta
        
        Date:           14-Mar-94

        Author:         Dale Hunscher
        
        Description:
        
        This file is the implementation of the SQL statement class. It
        provides a functional interface to SQL statements that do
        not return result sets (e.g., INSERT, UPDATE, and DELETE
        statements). It also forms the base class for cursors,
        which handle SQL statements and other operations that 
        do return result sets.
        
        Engineers:
        DAH             Dale A. Hunscher
        
        Revision History
        ================
        Date            Who             Did What
        -------------------------------------------------------------
    28-May-94   DAH     Replaced definition of the
                        RegisterError() function.
                        
        /////////////////////////////////////////////////////////////
        ///////////////////// NOTICE ////////////////////////////////
        /////////////////////////////////////////////////////////////

        Copyright (c) 1993-1995 by INTERSOLV, Inc. All rights reserved.

        Information in this document is subject to change without
        notice and does not represent a commitment on the part of
        INTERSOLV, Inc. This software is provided under
        a license agreement or non-disclosure agreement. The software
        may be used and/or copied only in accordance with the terms
        of the governing agreement. It is against the law to copy
        the software on any medium except as specifically allowed
        in the governing agreement. No part of this software may be
        reproduced or transmitted in any form or by any means,
        electronic or mechanical, including photocopying, recording,
        or information storage and retrieval systems, for any purpose
        other than the licensee's personal use, without the express
        written permission of INTERSOLV, Inc.
        
        /////////////////////////////////////////////////////////////
*/

#include <sql.hpp>
#if defined(__WATCOMC__)
    #include <mem.h> // for memset
#else
    #include <memory.h> // for memset
#endif

/***************************************************

    odbcSTMT
    
    Constructor.
    
    Arguments:

                podbcCONNECT    pC
                
                        Connection object pointer.
                
                LPUCSTR                 lpszStmt
                
                        SQL statement string. Default value
                        is NULL.
                        
                BOOL                    bPrepare
                
                        If TRUE (non-zero), SQLPrepare should
                        be invoked to prepare the SQL statement. 
                        Default value is FALSE.
                        
                BOOL                    bExecute
                
                        If TRUE (non-zero), SQLExecute should
                        be invoked (if bPrepare was non-zero)
                        or SQLExecDirect should be invoked
                        (if bprepare was zero) to execute the SQL
                        statement.  Default value is FALSE.
                        
                cpsPARMBIND     psParmBindings
                
                        If non-NULL, address of array of structures
                        defining the binding of parameters in the
                        SQL statement to members of the structure
                        containing parameter values at execution
                        time. Default value is NULL.
                        
                UWORD                   uParmCount
                
                        Count of array elements in psParmBindings.
                        Default value is 0.
                        
                PTR                             pvParmStruct
                
                        Address of structure containing parameter
                        values. Default value is NULL.
    
***************************************************/

odbcSTMT::odbcSTMT(
                podbcCONNECT            pC,
                LPUCSTR                 lpszSentStmt,
                BOOL                    bPrepare,
                BOOL                    bExecute,
                cpsPARMBIND             psParmBindings,
                UWORD                   uParmCount,
                PTR                     pvParmStruct
                ) : odbcBASE()
        {
        pConn = pC;
        // make sure we retrieve status info during construction
        AutoRetrieve(odbcREPSUCCESSWITHINFO);
        AutoReport(odbcNOREPORT);
        // initialize state variables
        pParmBindings = psParmBindings;
        ParmCount = uParmCount;
        pParms = pvParmStruct;
        lpszStmt = lpszSentStmt;
        iParm = 0;
        iParmRow = 0;
        iParamOptRowCount = 0;
        bParmsBound 
        = bError 
        = bPrepared 
        = bExecuted 
                = FALSE;

        // call driver to allocate a statement
        SetRC(SQLAllocStmt(
                        pConn->GetHdbc(),
                        &hstmt));
                        
        // if we succeeded...
        if (sqlsuccess())
                {
                if (lpszStmt)           // ...and there is a statement...
                        {
                        
                        if (bPrepare)   // if we are supposed to prepare
                                {
                                // prepare will bind parameters
                                Prepare();
                                } // bPrepare
                        else
                                SetParams();    // bind parameters explicitly
                                            
                        // we may or may not have prepared; now we see
                        // if we should execute. if so, call DoStmt().
                        if (!bError && bExecute)
                                {
                                // we may need data if we did an ExecDirect.
                                bError = !DoStmt();
                                } // bExecute
                                
                        } // lpszStmt
                else 
                        SetParams(); // bind parameters explicitly
                } // sqlsuccess

        // If any error occurred during construction,
        // the info has been retrieved.
        // Now inherit error handling from connection.
        ErrHandler          = pC->ErrHandler;
        bGetErrorInfo       = pC->bGetErrorInfo;
        bReportErrorInfo    = pC->bReportErrorInfo;
        hwnd                = pC->hwnd;
        flags               = pC->flags;


        pConn->RegisterStmt(this);
        };

odbcSTMT::odbcSTMT(
                        podbcCONNECT    pC,
                        LPCSTR                  lpszSentStmt,
                        BOOL                    bPrepare,
                        BOOL                    bExecute,
                        cpsPARMBIND     psParmBindings,
                        UWORD                   uParmCount,
                        PTR                             pvParmStruct
                        ) : odbcBASE()
        {
        pConn = pC;
        // make sure we retrieve status info during construction
        AutoRetrieve(odbcREPSUCCESSWITHINFO);
        AutoReport(odbcNOREPORT);
        // initialize state variables
        pParmBindings = psParmBindings;
        ParmCount = uParmCount;
        pParms = pvParmStruct;
        lpszStmt = (LPUCSTR)lpszSentStmt;
        iParm = 0;
        iParmRow = 0;
        iParamOptRowCount = 0; 
        bParmsBound
        = bError
        = bPrepared
        = bExecuted
                = FALSE;
        // call driver to allocate a statement
        SetRC(SQLAllocStmt(
                        pConn->GetHdbc(),
                        &hstmt));
                        
        // if we succeeded...
        if (sqlsuccess())
                {
                if (lpszStmt)           // ...and there is a statement...
                        {
                        
                        if (bPrepare)   // if we are supposed to prepare
                                {
                                // prepare will bind parameters
                                Prepare();
                                } // bPrepare
                        else
                                SetParams();    // bind parameters explicitly
                                            
                        // we may or may not have prepared; now we see
                        // if we should execute. if so, call DoStmt().
                        if (!bError && bExecute)
                                {
                                // we may need data if we did an ExecDirect.
                                bError = !DoStmt();
                                } // bExecute
                                
                        } // lpszStmt
                else 
                        SetParams(); // bind parameters explicitly
                } // sqlsuccess
                
        // If any error occurred during construction,
        // the info has been retrieved.
        // Now inherit error handling from connection.
        ErrHandler          = pC->ErrHandler;
        bGetErrorInfo       = pC->bGetErrorInfo;
        bReportErrorInfo    = pC->bReportErrorInfo;
        hwnd                = pC->hwnd;
        flags               = pC->flags;

        pConn->RegisterStmt(this);
        };

/***************************************************

    ~odbcSTMT

    Destructor.
***************************************************/

odbcSTMT::~odbcSTMT()
        {
        if (hstmt)
        {
                SQLFreeStmt(
                        hstmt,
                        SQL_DROP
                        );
        hstmt = NULL;
        }
    if ( pConn && !IsBadReadPtr( pConn, sizeof(*pConn)))
            pConn->UnregisterStmt(this);
        };

/***************************************************

    Cancel
    
    Invoke SQLCancel.
***************************************************/

RETCODE odbcSTMT::Cancel(void)
                {
                SetRC(SQLCancel(hstmt));
                return lastRC();
                };

/***************************************************

    Close
    
    Call SQLFreeStmt with SQL_CLOSE argument.
***************************************************/

RETCODE odbcSTMT::Close(void)
                {
                SetRC(SQLFreeStmt(
                hstmt,
                SQL_CLOSE
                ));

                return lastRC();
                };

/***************************************************

    ResetParams
    
    Remove old parameter bindings by calling SQLFreeStmt
    with SQL_RESET_PARAMS flag.
***************************************************/

RETCODE odbcSTMT::ResetParams(void)
                {
                SetRC(SQLFreeStmt(
                hstmt,
                SQL_RESET_PARAMS
                ));

                return lastRC();
                };

/***************************************************

    ExecDirect
    
    Invoke SQLExecDirect.
***************************************************/

RETCODE odbcSTMT::ExecDirect(
                LPUCSTR szSqlStr
                )
                {
                SetRC(SQLExecDirect(
                                hstmt,
                                (LPUSTR)szSqlStr,
                                SQL_NTS));
                                
                if (sqlsuccess())
                        bExecuted = TRUE;               

                return lastRC();
                };

/***************************************************

    Execute
    
    Invoke SQLExecute.
***************************************************/

RETCODE odbcSTMT::Execute(void)
                {
                SetRC(SQLExecute(hstmt));

                if (sqlsuccess())
                        bExecuted = TRUE;               

                return lastRC();
                };

/***************************************************

    RowCount
    
    Get count of rows affected by the statement. Overloaded 
    to two versions: one returns a result code, and the count
    is returned in the signed double-word pointed to by pcrow;
    the other version returns the count directly.
***************************************************/

RETCODE odbcSTMT::RowCount(
                SDWORD      *pcrow
                )
                {
                SetRC(SQLRowCount(
                                hstmt,
                                pcrow
                                ));

                return lastRC();
                };


SDWORD odbcSTMT::RowCount(void)
                {
                SDWORD sdword;
                SetRC(SQLRowCount(
                                hstmt,
                                &sdword
                                ));

                return sdword;
                };

/***************************************************

    Prepare
    
    Invoke SQLPrepare; bind parameters if parameter
    bindings need to be done.
***************************************************/

RETCODE odbcSTMT::Prepare(
                LPUCSTR szSqlStr
                )
                {
                SetRC(SQLPrepare(
                                hstmt,
                                (LPUSTR)szSqlStr,
                                SQL_NTS));

                return lastRC();
                };

RETCODE odbcSTMT::Prepare(void)
                {
                Prepare(lpszStmt ? lpszStmt : (LPUCSTR)"");
                if (!sqlsuccess())
                        bError = TRUE;
                else
                        {           
                        // we have prepared. Bind parameters now,
                        // if there are any to bind directly.
                        bPrepared = TRUE;
                        if (bParmsBound == TRUE)
                                ResetParams();
                        SetParams();
                        } // sqlsuccess()
                        
                return lastRC();
                }
                
/***************************************************

    SetParams
    
    Invoke SetParam member function passing array of
    parameter bindings.
***************************************************/

BOOL    odbcSTMT::SetParams(void)
                {
                if (
                        pParmBindings   // there are more than
                        && ParmCount    // zero parameter bindings
                   )
                        {
                        // bind multiple parameters automatically
                        SetParam(
                                pParmBindings,
                                ParmCount,
                                pParms
                                );
                        bError = !sqlsuccess();
                        } // pParmBindings && ParmCount
                else
                        return FALSE;

                return sqlsuccess();
                }

/***************************************************

    SetParam

    Invoke SQLSetParam to bind a parameter.
***************************************************/

RETCODE odbcSTMT::SetParam(
                UWORD       ipar,
                SWORD       fCType,
                SWORD       fSqlType,
                UDWORD      cbColDef,
                SWORD       ibScale,
                PTR         rgbValue,
                SDWORD      *pcbValue)
                {
                SetRC(SQLSetParam(
                                hstmt,
                                ipar,
                                fCType,
                                fSqlType,
                                cbColDef,
                                ibScale,
                                rgbValue,
                                pcbValue));

                if (sqlsuccess())
                        bParmsBound = TRUE;

                return lastRC();
                };

/***************************************************

    SetParam
    
    Bind multiple parameters as represented by the
    array of sPARMBIND structures pointed to by
    psParmBindings. The count of structures is
    passed as the uCount argument.If desired, these 
    are bound to the structure pointed to by pvBuf. 
    (You can also use the constants SQL_DATA_AT_EXEC 
    and SQL_NULL_DATA to indicate run-time parameter
    binding or null data values; see sqlstruc.hpp
    for instructions).
    
    One disadvantage is that this function binds
    only a single value to each parameter.
***************************************************/

RETCODE odbcSTMT::SetParam(
			cpsPARMBIND psParmBindings,
			UWORD 		uCount,
			PTR 		pvBuf
		)
		{
		LPSTR rgbBuf = (LPSTR)pvBuf;
		LPSTR pBuf;
		
		for (iParm = 0;
			iParm < uCount;
			iParm++
			)
			{
			// if the iOffset member contains odbcUSE_RGBVALUE, 
			// send the rgbValue member value rather than the computed
			// offset into the structure pointed to by pvBuf 
			if (psParmBindings[iParm].iOffset != odbcUSE_RGBVALUE)
				pBuf = (LPSTR)(PTR)(rgbBuf+psParmBindings[iParm].iOffset);
			else
				pBuf = (LPSTR)psParmBindings[iParm].rgbValue;
			
			// bind the column as directed.	
#if (ODBCVER >= 0x0200)
            // see if we can use new ODBC API function SQLBindParameter via odbcSTMT
            // member function BindParameter().
            if ( pConn->GetFunctions(
                    SQL_API_SQLBINDPARAMETER ) != FALSE )
                {
                // bind the column as directed.
                BindParameter(
                        psParmBindings[iParm].iParm,
                        psParmBindings[iParm].fParamType == 0 ?
                            SQL_PARAM_INPUT :
                            psParmBindings[iParm].fParamType,
                        psParmBindings[iParm].fCType,
                        psParmBindings[iParm].fSqlType,
                        PrecisionForSqlType(
                            psParmBindings[iParm].fSqlType,
                            psParmBindings[iParm].cbColDef
                            ),
                        psParmBindings[iParm].ibScale,
                        pBuf,
                        psParmBindings[iParm].cbColDef,
                        (SDWORD *)(&psParmBindings[iParm].cbValue)
                        );
                }
            else
                {
                // bind the column as directed.
                SetParam(
                        psParmBindings[iParm].iParm,
                        psParmBindings[iParm].fCType,
                        psParmBindings[iParm].fSqlType,
                        PrecisionForSqlType(
                            psParmBindings[iParm].fSqlType,
                            psParmBindings[iParm].cbColDef
                            ),
                        psParmBindings[iParm].ibScale,
                        pBuf,
                        (SDWORD *)(&psParmBindings[iParm].cbValue)
                        );
                }
#else
            // bind the column as directed.
            SetParam(
                    psParmBindings[iParm].iParm,
                    psParmBindings[iParm].fCType,
                    psParmBindings[iParm].fSqlType,
                    PrecisionForSqlType(
                            psParmBindings[iParm].fSqlType,
                            psParmBindings[iParm].cbColDef
                            ),
                    psParmBindings[iParm].ibScale,
                    pBuf,
                    (SDWORD *)(&psParmBindings[iParm].cbValue)
                    );
#endif

			if (lastRC() != SQL_SUCCESS)
				break;

			}

		return lastRC();
		}
		
/***************************************************

    RegisterError

        Get more information on the most recent error code
        from an ODBC operation. Results can be retrieved using
        member functions in the parent odbcBASE class.

        This function calls the base class member function Error()
        with arguments appropriate for this object type.
***************************************************/

RETCODE odbcSTMT::RegisterError(void)
                {
        return Error(
              pConn->GetHenv(),
              (pConn->GetHdbc() != NULL) ?
                 pConn->GetHdbc() :
                 SQL_NULL_HDBC,
                          (hstmt != NULL &&
                pConn->GetHdbc() != NULL) ?
                 hstmt :
                 SQL_NULL_HSTMT
             );
                };

/***************************************************

    GetStmtOption
    
    GetStmtOption returns the value of a given
    option for this statement.
    
***************************************************/

UDWORD odbcSTMT::GetStmtOption(UWORD fOption)
        {
        UDWORD Ret = 0;
        
        SetRC(SQLGetStmtOption(
                        hstmt,
                        fOption,
                        &Ret));
                        
    return Ret;
        }
        

/***************************************************

    SetStmtOption
    
    SetStmtOption sets the value of a given
    option for this statement.
***************************************************/

RETCODE odbcSTMT::SetStmtOption(UWORD fOption, UDWORD ulValue)
        {
        SetRC(SQLSetStmtOption(hstmt, fOption, ulValue));
                
        return lastRC();
        }
                

/***************************************************

    ParamData
    
    Invoke SQLParamData.
***************************************************/

RETCODE odbcSTMT::ParamData(PTR        *prgbValue)
        {
        SetRC(
                SQLParamData(
                        hstmt,
                        prgbValue
                        ));
                        
        return lastRC();
        }

/***************************************************

    PutData
    
    Invoke SQLPutData.
***************************************************/

RETCODE odbcSTMT::PutData(
        PTR rgbValue,
    SDWORD     cbValue)
        {
        SetRC(
                SQLPutData(
                        hstmt,
                        rgbValue,
                        cbValue
                        ));
                        
        return lastRC();
        }

/***************************************************

    DescribeParam
    
    Invoke SQLDescribeParam. Overloaded to two forms.  The first
    simply passes through the call to SQLDescribeParam.  The second 
    overloaded form returns result in an sPARMBIND struct.
    
    All members of the struct are first set to zeros.  The
    fSqlType, cbColDef, and ibScale members are set by the
    SQLDescribeParam call.  The value for nullability is put 
    in the cbValue member: a non-zero value will indicate that
    the parameter is nullable.  The default C data type is
    put in the fCType member.
    
***************************************************/

RETCODE odbcSTMT::DescribeParam(
                UWORD      ipar,
                SWORD      *pfSqlType,
                UDWORD     *pcbColDef,
                SWORD      *pibScale,
                SWORD      *pfNullable
                )
    {
        SetRC(
                SQLDescribeParam(
                        hstmt,
                ipar,
                pfSqlType,
                pcbColDef,
                pibScale,
                pfNullable)
                );
                
        return lastRC();
        }
                
RETCODE odbcSTMT::DescribeParam(
                UWORD      ipar,
                psPARMBIND pParmBind
                )
    {
    SWORD fNullable;
    
    memset(pParmBind, 0, sizeof(sPARMBIND));
    
        SetRC(
                SQLDescribeParam(
                        hstmt,
                ipar,
                &pParmBind->fSqlType,
                &pParmBind->cbColDef,
                &pParmBind->ibScale,
                &fNullable)
                );
                
    if (sqlsuccess()) 
        {
        pParmBind->fCType = CTypeFromSqlType(pParmBind->fSqlType);
        pParmBind->cbValue = (SDWORD)fNullable;
        }
                        
        return lastRC();
        }
                
/***************************************************

    CTypeFromSqlType
    
    Given SQL data type, return corresponding default 
    C data type.
    
***************************************************/
SWORD   odbcSTMT::CTypeFromSqlType(
                                SWORD fSqlType
                                )
        {
        // determine C type for SQL type
        switch(fSqlType)
                {
        case SQL_CHAR:
        case SQL_VARCHAR:
        case SQL_LONGVARCHAR:
        case SQL_DECIMAL:
        case SQL_NUMERIC:
        case SQL_BIGINT:
                return SQL_C_CHAR;

                        
        case SQL_BIT:
                return SQL_C_BIT;

        case SQL_TINYINT:
                return SQL_C_TINYINT;


        case SQL_SMALLINT:
                return SQL_C_SHORT;


        case SQL_INTEGER:
                return SQL_C_LONG;


        case SQL_REAL:
                return SQL_C_FLOAT;

        case SQL_FLOAT:
        case SQL_DOUBLE:
                return SQL_C_DOUBLE;


        case SQL_BINARY:
        case SQL_VARBINARY:
        case SQL_LONGVARBINARY:
                return SQL_C_BINARY;


        case SQL_DATE:
                return SQL_C_DATE;

        case SQL_TIME:
                return SQL_C_TIME;


        case SQL_TIMESTAMP:
                return SQL_C_TIMESTAMP;

                } // end switch
                
    return 0;                           
}
                       
/***************************************************

    NumParams
    
    Invoke SQLNumParams.
    
    Overloaded to two forms: the first is a pass-through
    version, the second takes no parameters and returns
    the value.  A zero return may indicate an error; use
    the sqlsuccess() member function to check.
***************************************************/

RETCODE odbcSTMT::NumParams(
                SWORD      *pcpar
                )
    {
        SetRC(
                SQLNumParams(
                        hstmt,
                pcpar)
                );
                
        return lastRC();
        }
                

SWORD odbcSTMT::NumParams(void)
    {
    SWORD par = 0;
    
        SetRC(
                SQLNumParams(
                        hstmt,
                (SWORD     *)&par)
                );
                
        return par;
        }
                
/***************************************************

    ParamOptions
    
    Invoke SQLParamOptions. Overloaded to two versions.
    The first is a pass-through call, the second uses the
    address of the internal data member iParmRow as the
    pirow argument.  If an error occurs processing an
    insert or update, call GetParmRow() to retrieve iParmRow,
    the parameter data set index where the error occurred.
***************************************************/

RETCODE odbcSTMT::ParamOptions(
    UDWORD     crow,
    UDWORD     *pirow)
    {
    SetRC(
        SQLParamOptions(
                hstmt,
                iParamOptRowCount = crow,
                pirow));
                
    return lastRC();
    }

RETCODE odbcSTMT::ParamOptions(UDWORD crow)
    {
    SetRC(
        SQLParamOptions(
                hstmt,
                iParamOptRowCount = crow,
                &iParmRow));
                
    return lastRC();
    }


/***************************************************

    DoStmt
    
    Execute a statement using SQLExecute or SQLExecDirect
    depending on the value of the bPrepared flag
    (set when SQLPrepare is successfully executed).
***************************************************/

BOOL odbcSTMT::DoStmt(void)
        {
        if (bPrepared)
                Execute();
        else
                ExecDirect(lpszStmt);

        return sqlsuccess();
        }

// new in v2.0

RETCODE odbcSTMT::BindParameter
              (
                  UWORD           ipar,
                  SWORD           fParamType,
              SWORD       fCType,
                  SWORD           fSqlType,
              UDWORD      cbColDef,
              SWORD       ibScale,
              PTR         rgbValue,
                  SDWORD          cbValueMax,
              SDWORD     *pcbValue
              )
    {
        SetRC( SQLBindParameter
                        (
                        hstmt,
                            ipar,
                            fParamType,
                        fCType,
                            fSqlType,
                        cbColDef,
                        ibScale,
                        rgbValue,
                            cbValueMax,
                        pcbValue
                        ));
        return lastRC() ;
    }

// end new in v2.0
        
