/*
        File:           STMT.HPP
        
        Revision:       2.0 Release
        
        Date:           12-26-93
        
        Author:         Dale Hunscher
        
        Description:
        
        This file is the interface to the SQL statement class. It
        provides a functional interface to SQL statements that do
        not return result sets (e.g., INSERT, UPDATE, and DELETE
        statements). It also forms the base class for cursors,
        which handle SQL statements and other operations that 
        do return result sets.
        
        Engineers:
        DAH             Dale A. Hunscher
        
        Revision History
        ================
        Date            Who     Did What
        -------------------------------------------------------------
        28-May-94       DAH     Added ODBC v2.0 support.
        /////////////////////////////////////////////////////////////
        ///////////////////// NOTICE ////////////////////////////////
        /////////////////////////////////////////////////////////////
                                                                             
        Copyright (c) 1993-1995 by INTERSOLV, Inc. All rights reserved.
        
        Information in this document is subject to change without
        notice and does not represent a commitment on the part of
        INTERSOLV, Inc. This software is provided under
        a license agreement or non-disclosure agreement. The software
        may be used and/or copied only in accordance with the terms
        of the governing agreement. It is against the law to copy
        the software on any medium except as specifically allowed
        in the governing agreement. No part of this software may be 
        reproduced or transmitted in any form or by any means, 
        electronic or mechanical, including photocopying, recording,
        or information storage and retrieval systems, for any purpose
        other than the licensee's personal use, without the express
        written permission of INTERSOLV, Inc.
        
        /////////////////////////////////////////////////////////////
*/

#if !defined(_STMT_HPP)
#define _STMT_HPP

///////////////////////////////////////////////////////////
//////////////////////// statements
///////////////////////////////////////////////////////////

class odbcEXPORTED odbcSTMT : public odbcBASE {

protected:
/***************************************************

    pConn
    
    Pointer to owning connection object.
***************************************************/

    podbcCONNECT        pConn;
    
/***************************************************

    hstmt
    
    ODBC statement handle.
***************************************************/

        HSTMT           hstmt;
/***************************************************

    lpszStmt
    
    SQL statement string.
***************************************************/

        LPUCSTR                 lpszStmt;
/***************************************************

    pParmBindings
    
    Array of parameter bindings represented by
    sPARMBIND structs.
***************************************************/

        cpsPARMBIND     pParmBindings;
/***************************************************

    ParmCount
    
    count of sPARMBIND structs in array.
***************************************************/

        UWORD                   ParmCount;
/***************************************************

    pParms
    
    Pointer to parameter data structure.
***************************************************/

        PTR                     pParms;
/***************************************************

    bError
    
    Non-zero if an error occurred (useful if error
    occurs during construction).
***************************************************/

        BOOL                    bError;
/***************************************************

    bPrepared
    
    Non-zero if we have prepared the statement.
***************************************************/

        BOOL                    bPrepared;
/***************************************************

    bParmsBound
    
    Non-zero if we have bound the parameters represented
    by the array of sPARMBIND structures.
***************************************************/

        BOOL                    bParmsBound;
/***************************************************

    bExecuted
    
    Non-zero if we have executed the SQL statement
    via a call to Execute or ExecDirect member functions.
***************************************************/

        BOOL                    bExecuted;
        
/***************************************************

    iParm
    
    Index of last sPARMBIND structure processed.
***************************************************/

        UWORD                   iParm;

/***************************************************

    iParmRow

    Index of last parameter set processed, when multiple
    parameter sets are indicated by calling ParamOptions.
***************************************************/

        UDWORD                  iParmRow;

// new in v2.0
/***************************************************

    iParamOptRowCount

    Number of parameter rows passed in call to
    ParamOptions.
***************************************************/

        UDWORD                  iParamOptRowCount;
// end new in v2.0

public:
/***************************************************

    odbcSTMT
    
    Constructor.
    
    Arguments:

                podbcCONNECT    pC
                
                        Connection object pointer.
                
                LPUCSTR                 lpszStmt
                
                        SQL statement string. Default value
                        is NULL.
                        
                BOOL                    bPrepare
                
                        If TRUE (non-zero), SQLPrepare should
                        be invoked to prepare the SQL statement. 
                        Default value is FALSE.
                        
                BOOL                    bExecute
                
                        If TRUE (non-zero), SQLExecute should
                        be invoked (if bPrepare was non-zero)
                        or SQLExecDirect should be invoked
                        (if bprepare was zero) to execute the SQL
                        statement.  Default value is FALSE.
                        
                cpsPARMBIND     psParmBindings
                
                        If non-NULL, address of array of structures
                        defining the binding of parameters in the
                        SQL statement to members of the structure
                        containing parameter values at execution
                        time. Default value is NULL.
                        
                UWORD                   uParmCount
                
                        Count of array elements in psParmBindings.
                        Default value is 0.
                        
                PTR                             pvParmStruct
                
                        Address of structure containing parameter
                        values. Default value is NULL.
    
***************************************************/

        odbcSTMT(
                        podbcCONNECT    pC,
                        LPUCSTR                 lpszStmt                = NULL,
                        BOOL                    bPrepare                = FALSE,
                        BOOL                    bExecute                = FALSE,
                        cpsPARMBIND     psParmBindings  = NULL,
                        UWORD                   uParmCount              = 0,
                        PTR                             pvParmStruct    = NULL
                        );

        odbcSTMT(
                        podbcCONNECT    pC,
                        LPCSTR                  lpszStmt,
                        BOOL                    bPrepare                = FALSE,
                        BOOL                    bExecute                = FALSE,
                        cpsPARMBIND     psParmBindings  = NULL,
                        UWORD                   uParmCount              = 0,
                        PTR                             pvParmStruct    = NULL
                        );

/***************************************************

    ~odbcSTMT
    
    Destructor.
***************************************************/

        virtual ~odbcSTMT();

/***************************************************

    SetParmBindings
    
    Set value of protected data members that define
    bindings between members of parameter structure
    and SQL statement parameters, as defined by the
    array of sPARMBIND structs..
***************************************************/

        virtual void            SetParmBindings
                        (
                        cpsPARMBIND psParmBindings      = NULL,
                        UWORD           uParmCount              = 0,
                        PTR                     pvParmStruct    = NULL
                        )
                {
                pParmBindings = psParmBindings;
                ParmCount = uParmCount;
                pParms = pvParmStruct;
                bExecuted = bParmsBound = FALSE;
                };
        
/***************************************************

    LastParmProcessed
    
    Zero-based index of the last parameter processed
    by SetParams() (useful if an error occurs).
***************************************************/

        virtual UWORD           LastParmProcessed(void)
                {
                return iParm;
                };

/***************************************************

    GetHstmt
    
    Get ODBC HSTMT associated with this statement.
***************************************************/

        virtual HSTMT GetHstmt(void) { return hstmt;};
                                
/***************************************************

    GetParmBindings
    
    Get parameter bindings array address, count,
    and struct address.  Note that the first and
    third arguments are addresses of pointers (pointers
    to pointers).
***************************************************/

        virtual void            GetParmBindings
                        (
                        cpsPARMBIND         *   ppsParmBindings,
                        UWORD     *                     puParmCount,
                        PTR     *                       ppvParmStruct
                        )
                {
                *ppsParmBindings = pParmBindings;
                *puParmCount = ParmCount;
                *ppvParmStruct = pParms;
                }
                
/***************************************************

    SetStmt
    
    Sets the SQL command string associated with 
    the statement. You may also need to call
    ResetParams() if parameter bindings have
    taken place.
***************************************************/

        virtual LPUCSTR                 SetStmt(LPUCSTR Stmt)
                {
                LPUCSTR Temp = lpszStmt;
                lpszStmt = Stmt;
                return Temp;
                };
                
        virtual LPCSTR          SetStmt(LPSTR Stmt)
                {
                LPCSTR Temp = (LPCSTR)lpszStmt;
                lpszStmt = (LPUCSTR)Stmt;
                return Temp;
                };
                
/***************************************************

    GetStmt
    
    Get current SQL command string associated with
    the statement.
***************************************************/

        virtual LPUCSTR         GetStmt(void)
                {
                return lpszStmt;
                };


/***************************************************

    GetParmRow
    
    Get index of last parameter set processed, when multiple
    parameter sets are indicated by calling ParamOptions.
***************************************************/

        virtual UDWORD  GetParmRow(void)
                {
                return iParmRow;
                };
                                
/***************************************************

    ErrorFlag
    
    Gets current value of internal error flag; useful
    if determining if an error occurred during construc-
    tion when return codes were not available.
***************************************************/

        virtual BOOL    ErrorFlag(void)
                {
                return bError;
                };
                
/***************************************************

    Prepared
    
    TRUE if SQLPrepare was called successfully
    already on this statement object.
***************************************************/

        virtual BOOL    Prepared(void)
                {
                return bPrepared;
                };
                
/***************************************************

    ParmsBound
    
    Returns TRUE if parameter bindings have been
    done on this statement.
***************************************************/

        virtual BOOL    ParmsBound(void)
                {
                return bParmsBound;
                };
                
/***************************************************

    Executed
    
    Returns TRUE if a successful call to SQLExecute
    or SQLExecDirect has occurred on this object.
***************************************************/

        virtual BOOL    Executed(void)
                {
                return bExecuted;
                };
                
        ////////////////////////////////////////////////////////
        ////////////////////// core functions //////////////////
        ////////////////////////////////////////////////////////

/***************************************************

    Cancel
    
    Invoke SQLCancel.
***************************************************/

        virtual RETCODE Cancel(void);

/***************************************************

    Close
    
    Call SQLFreeStmt with SQL_CLOSE argument.
***************************************************/

        virtual RETCODE Close(void);

/***************************************************

    ResetParams
    
    Remove old parameter bindings by calling SQLFreeStmt
    with SQL_RESET_PARAMS flag.
***************************************************/

        virtual RETCODE ResetParams(void);

/***************************************************

    RowCount
    
    Get count of rows affected by the statement. Overloaded 
    to two versions: one returns a result code, and the count
    is returned in the signed double-word pointed to by pcrow;
    the other version returns the count directly.
***************************************************/

        virtual RETCODE RowCount(
                SDWORD  *pcrow
                );

        virtual SDWORD RowCount(void);

/***************************************************

    Prepare
    
    Invoke SQLPrepare; bind parameters if parameter
    bindings need to be done. Overloaded version without
    argument prepares statement stored internally.
***************************************************/

        virtual RETCODE Prepare(LPUCSTR szSqlStr);
        
        virtual RETCODE Prepare(LPSTR szSqlStr)
                        {
                        return Prepare((LPUCSTR)szSqlStr);
                        };
        
        virtual RETCODE Prepare(void);
        
/***************************************************

    ExecDirect
    
    Invoke SQLExecDirect.
***************************************************/

        virtual RETCODE ExecDirect(LPUCSTR szSqlStr);

        virtual RETCODE ExecDirect(LPSTR szSqlStr)
                {
                return ExecDirect((LPUCSTR)szSqlStr);
                };

/***************************************************

    Execute
    
    Invoke SQLExecute.
***************************************************/

        virtual RETCODE Execute(void);

/***************************************************

    SetParams
    
    Invoke SetParam member function passing internal array of
    parameter bindings.
***************************************************/

    virtual BOOL        SetParams(void);
    
/***************************************************

    SetParam
    
    Invoke SQLSetParam to bind a parameter.
***************************************************/

        virtual RETCODE SetParam(
                UWORD       ipar,
                SWORD       fCType,
                SWORD       fSqlType,
                UDWORD      cbColDef,
                SWORD       ibScale,
                PTR         rgbValue,
                SDWORD      *pcbValue);

/***************************************************

    SetParam
    
    Bind multiple parameters as represented by the
    array of sPARMBIND structures pointed to by
    psParmBindings. The count of structures is
    passed as the uCount argument.If desired, these 
    are bound to the structure pointed to by pvBuf. 
    (You can also use the constants SQL_DATA_AT_EXEC 
    and SQL_NULL_DATA to indicate run-time parameter
    binding or null data values; see sqlstruc.hpp
    for instructions).
    
    One disadvantage is that this function binds
    only a single value to each parameter.
***************************************************/

        virtual RETCODE SetParam(
                cpsPARMBIND psParmBindings,
                UWORD           uCount,
                PTR                     pvBuf
                );
                
/***************************************************

    RegisterError
    
        Get more information on the most recent error code
        from an ODBC operation. Results can be retrieved using
        member functions in the parent odbcBASE class.
        
        This function calls the base class member function Error()
        with arguments appropriate for this object type.
***************************************************/

        virtual RETCODE RegisterError(void);

        ////////////////////////////////////////////////////////
        ////////////////////// level 1 functions ///////////////
        ////////////////////////////////////////////////////////
        
/***************************************************

    GetStmtOption
    
    GetStmtOption returns the value of a given
    option for this statement.
    
***************************************************/

        virtual UDWORD GetStmtOption(UWORD fOption);

/***************************************************

    SetStmtOption
    
    SetStmtOption sets the value of a given
    option for this statement.
***************************************************/

        virtual RETCODE SetStmtOption(UWORD fOption, UDWORD ulValue);

/***************************************************

    ParamData
    
    Invoke SQLParamData.
***************************************************/

        virtual RETCODE ParamData(PTR        *prgbValue);

/***************************************************

    PutData
    
    Invoke SQLPutData.
***************************************************/

        virtual RETCODE PutData(
                        PTR prgbValue,
                SDWORD     cbValue);
    
/***************************************************

    DescribeParam
    
    Invoke SQLDescribeParam.
***************************************************/

        virtual RETCODE DescribeParam(
                UWORD      ipar,
                SWORD      *pfSqlType,
                UDWORD     *pcbColDef,
                SWORD      *pibScale,
                SWORD      *pfNullable
                );
               
        virtual RETCODE DescribeParam(
                UWORD      ipar,
                psPARMBIND pParmBind
                );
                
/***************************************************

    CTypeFromSqlType
    
    Given SQL data type, return corresponding default 
    C data type.
    
***************************************************/
        virtual SWORD   CTypeFromSqlType(
                                SWORD fSqlType
                                );
                                
/***************************************************

    NumParams
    
    Invoke SQLNumParams.
***************************************************/

        virtual RETCODE NumParams(
                SWORD      *pcpar
                );

        virtual SWORD NumParams(void);

/***************************************************

    ParamOptions
    
    Invoke SQLParamOptions.
***************************************************/

        virtual RETCODE ParamOptions(
                        UDWORD     crow,
                        UDWORD     *pirow);
    
        virtual RETCODE ParamOptions(UDWORD crow);
    
    ////////////////////////////////////////////////
    ///////////// statement executors //////////////
    ////////////////////////////////////////////////
    
/***************************************************

    DoStmt
    
    Execute a statement using SQLExecute or SQLExecDirect
    depending on the value of the bPrepared flag
    (set when SQLPrepare is successfully executed).
***************************************************/

        virtual BOOL DoStmt(void);
                
/***************************************************

    pParmStruct
    
    Returns address of parameter structure.
***************************************************/

    virtual PTR pParmStruct(void)
        {
        return pParms;
        };
    
// new in v2.0

#if (ODBCVER >= 0x0200)
/***************************************************

    BindParameter

    Bind storage for parameter data.
***************************************************/

    virtual RETCODE BindParameter
              (
              UWORD     ipar,
              SWORD     fParamType,
              SWORD     fCType,
              SWORD     fSqlType,
              UDWORD    cbColDef,
              SWORD     ibScale,
              PTR       rgbValue,
              SDWORD    cbValueMax,
              SDWORD *  pcbValue
              ) ;

#endif // #if (ODBCVER >= 0x0200)

// end new in v2.0

};

#endif // if !defined(_STMT_HPP)
