/*
    File:           QUERITER.HPP
    
    Revision:       2.0 Release
    
    Date:           14-Mar-94

    Author:         Dale Hunscher
    
    Description:
    
    This file describes the interface to the various standard
    cursor classes supporting ODBC library information functions 
    that return result sets.
    
    Each of these is effectively just a cursor; all the specialized
    types do is bind the columns in the result set to an
    appropriate data structure.
    
    These are useful as examples of how to use an array of sCOLBIND 
    structures to bind columns in a result set to a specialized
    data structure.  While this would be an unwieldy approach to an 
    ad hoc query, it is an effective way to produce reusable queries.
    
    Following the pattern here, you can produce specialized cursor
    classes that "know" how to perform a query and produce output
    from the query in the form of a data structure.

    Engineers:
    DAH             Dale A. Hunscher
    
    Revision History
    ================
    Date            Who             Did What
    -------------------------------------------------------------
    03-Dec-94       DAH             Fixed odbcSYSINFOITERATOR::DoQuery
                                    so it tries _really hard_ to get
                                    bidirectional scrolling enabled.

    /////////////////////////////////////////////////////////////
    ///////////////////// NOTICE ////////////////////////////////
    /////////////////////////////////////////////////////////////
                                         
    Copyright (c) 1993-1995 by INTERSOLV, Inc. All rights reserved.
    
    Information in this document is subject to change without
    notice and does not represent a commitment on the part of
    INTERSOLV, Inc. This software is provided under
    a license agreement or non-disclosure agreement. The software
    may be used and/or copied only in accordance with the terms
    of the governing agreement. It is against the law to copy
    the software on any medium except as specifically allowed
    in the governing agreement. No part of this software may be 
    reproduced or transmitted in any form or by any means, 
    electronic or mechanical, including photocopying, recording,
    or information storage and retrieval systems, for any purpose
    other than the licensee's personal use, without the express
    written permission of INTERSOLV, Inc.
    
    /////////////////////////////////////////////////////////////
*/

#if !defined(_QUERITER_HPP)
#define _QUERITER_HPP

/**********************************************************
    odbcSYSINFOITERATOR

    Base iterator class.
**********************************************************/
    class odbcEXPORTED odbcSYSINFOITERATOR : public odbcCURSOR {

    protected:

    /**********************************************************
    bCanScroll

    TRUE if scrolling cursor behavior is enabled.
    **********************************************************/
    BOOL bCanScroll;

    public:

    /**********************************************************
    ~odbcSYSINFOITERATOR

    Destructor.
    **********************************************************/

    virtual ~odbcSYSINFOITERATOR() ;

    /**********************************************************
    odbcSYSINFOITERATOR

    Constructor for iterator class to iterate result set.
    **********************************************************/

    odbcSYSINFOITERATOR( podbcCONNECT   pC );

    /**********************************************************
    DoQuery

    Process query.
    **********************************************************/

    virtual BOOL  DoQuery(void);
    /***************************************************

    GetFirst

    Call Fetch() to get first row in the result set
    (or ExtFetchFirst() to get first rowset in result set).
    Returns non-zero value if it succeeds, zero if
    it fails.
    ***************************************************/

    virtual BOOL GetFirst(void);

    /***************************************************

    GetNext

    Call Fetch() to get next row in the result set
    (or ExtFetchNext() to get next rowset in result set).
    Returns non-zero value if it succeeds, zero if
    it fails.
    ***************************************************/

    virtual BOOL GetNext(void);

    /***************************************************

    GetLast

    Call Fetch() to get last rowset in the result set.
    Only valid if extended fetch operations are in use.
    Returns non-zero value if it succeeds, zero if
    it fails.
    ***************************************************/

    virtual BOOL GetLast(void);

    /***************************************************

    GetPrev

    Call ExtFetchPrev() to get prior rowset in the result set.
    Only valid if extended fetch operations are in use.
    Returns non-zero value if it succeeds, zero if
    it fails.
    ***************************************************/

    virtual BOOL GetPrev(void);


    }; // end odbcSYSINFOITERATOR class


/**********************************************************
    odbcStatisticsIterator

    Iterator class to process result set for SQLStatistics.
**********************************************************/

    class odbcEXPORTED odbcStatisticsIterator : public odbcSYSINFOITERATOR {

    protected:
    
/**********************************************************
    szQualifier

    Table qualifier to pass to SQLStatistics.
**********************************************************/

        LPUSTR                                  szQualifier;
        
/**********************************************************
    szOwner
    
    Owner name to pass to SQLStatistics.
**********************************************************/

        LPUSTR                                  szOwner;

/**********************************************************
    szName
    
    Table name to pass to SQLStatistics.
**********************************************************/

        LPUSTR                                  szName;

/**********************************************************
    fUnique
    
    Uniqueness argument to pass to SQLStatistics.
**********************************************************/

        UWORD                                   fUnique;

/**********************************************************
    fAccuracy
    
    Accuracy flag to pass to SQLStatistics.
**********************************************************/

        UWORD                                   fAccuracy;

/**********************************************************
    Set
    
    Structure for result set.
**********************************************************/

        sSQLSTATISTICSRESULTSET Set;

    public:
/**********************************************************
    odbcStatisticsIterator
    
    Constructor for iterator class to process result set 
    for SQLStatistics.
**********************************************************/

        odbcStatisticsIterator
            (
            podbcCONNECT                    pC,
        LPUSTR                                  szTableQualifier,
        LPUSTR                                  szTableOwner,
        LPUSTR                                  szTableName,
        UWORD                                   fUnique,
        UWORD                                   fAccuracy
            );

        odbcStatisticsIterator
            (
            podbcCONNECT                    pC,
        LPSTR                                   szTableQualifier,
        LPSTR                                   szTableOwner,
        LPSTR                                   szTableName,
        UWORD                                   fUnique,
        UWORD                                   fAccuracy
            );

/**********************************************************
    ~odbcStatisticsIterator

    Destructor.
**********************************************************/

        virtual ~odbcStatisticsIterator();

/**********************************************************
    DoQuery

    Process query.
**********************************************************/

        virtual BOOL  DoQuery(void);

/**********************************************************
    pStatisticsResultSet
    
    Returns pointer to row data structure for result
    set from SQLStatistics.
**********************************************************/

    virtual cpsSQLSTATISTICSRESULTSET pStatisticsResultSet(void)
        {
        return (cpsSQLSTATISTICSRESULTSET)
            pResultSet();
        };
    };

/**********************************************************
    odbcColumnIterator
    
    Iterator class to process result set for SQLColumns.
**********************************************************/

    class odbcEXPORTED odbcColumnIterator : public odbcSYSINFOITERATOR {
    protected:
/**********************************************************
    szQualifier
    
    Table qualifier to pass to SQLColumns.
**********************************************************/

        LPUSTR                                  szQualifier;
        
/**********************************************************
    szOwner
    
    Owner name to pass to SQLColumns.
**********************************************************/

        LPUSTR                                  szOwner;

/**********************************************************
    szName
    
    Table name to pass to SQLColumns.
**********************************************************/

        LPUSTR                                  szName;

/**********************************************************
    szColName
    
    Column name to pass to SQLColumns.
**********************************************************/

        LPUSTR szColName;        

/**********************************************************
    Set
    
    Structure for result set.
**********************************************************/

        sSQLCOLUMNSRESULTSET Set;
    public:
/**********************************************************
    odbcColumnIterator
    
    Constructor for iterator class to process result set 
    for SQLColumns.
**********************************************************/

        odbcColumnIterator(
            podbcCONNECT    pC,
        LPUSTR                  szTableQualifier,
        LPUSTR                  szTableOwner,
        LPUSTR                  szTableName,
        LPUSTR                  szColumnName
            );
            
        odbcColumnIterator(
            podbcCONNECT    pC,
        LPSTR                   szTableQualifier,
        LPSTR                   szTableOwner,
        LPSTR                   szTableName,
        LPSTR                   szColumnName
            );
            
/**********************************************************
    ~odbcColumnIterator
    
    Destructor.
**********************************************************/

        virtual ~odbcColumnIterator();
        
/**********************************************************
    DoQuery
    
    Process query.
**********************************************************/

        virtual BOOL  DoQuery(void);
                
/**********************************************************
    pColumnsResultSet
    
    Returns pointer to row data structure for result
    set from SQLColumns.
**********************************************************/

    virtual cpsSQLCOLUMNSRESULTSET pColumnsResultSet(void)
        {
        return (cpsSQLCOLUMNSRESULTSET)
        pResultSet();
        };
    }; // end odbcColumnIterator class

/**********************************************************
    odbcTableIterator
    
    Iterator class to process result set for SQLTables.
**********************************************************/

    class odbcEXPORTED odbcTableIterator : public odbcSYSINFOITERATOR {
    protected:
/**********************************************************
    szQualifier
    
    Table qualifier to pass to SQLTables.
**********************************************************/

        LPUSTR                                  szQualifier;
        
/**********************************************************
    szOwner

    Owner name to pass to SQLTables.
**********************************************************/

        LPUSTR                                  szOwner;

/**********************************************************
    szName
    
    Table name to pass to SQLTables.
**********************************************************/

        LPUSTR                                  szName;

/**********************************************************
    szType
    
    Table type to pass to SQLTables.
**********************************************************/

        LPUSTR                          szType;        

/**********************************************************
    Set
    
    Structure for result set.
**********************************************************/

        sSQLTABLESRESULTSET Set;
    public:
/**********************************************************
    odbcTableIterator
    
    Constructor for iterator class to process result set 
    for SQLTables.
**********************************************************/

        odbcTableIterator(
            podbcCONNECT    pC,
        LPUSTR                  szTableQualifier,
        LPUSTR                  szTableOwner,
        LPUSTR                  szTableName,
        LPUSTR                  szTableType
            );
            
        odbcTableIterator(
            podbcCONNECT    pC,
        LPSTR                   szTableQualifier,
        LPSTR                   szTableOwner,
        LPSTR                   szTableName,
        LPSTR                   szTableType
            );
            
/**********************************************************
    ~odbcTableIterator
    
    Destructor.
**********************************************************/

        virtual ~odbcTableIterator();
        
/**********************************************************
    DoQuery
    
    Process query.
**********************************************************/

        virtual BOOL  DoQuery(void);
                
/**********************************************************
    pTablesResultSet
    
    Returns pointer to row data structure for result
    set from SQLTables.
**********************************************************/

    virtual cpsSQLTABLESRESULTSET pTablesResultSet(void)
        {
        return (cpsSQLTABLESRESULTSET)
        pResultSet();
        };
    };

/**********************************************************
    odbcSpecialColumnIterator

    Iterator class to process result set for
    SQLSpecialColumns.
**********************************************************/

    class odbcEXPORTED odbcSpecialColumnIterator : public odbcSYSINFOITERATOR {
    protected:
/**********************************************************
    szQualifier
    
    Table qualifier to pass to SQLSpecialColumns.
**********************************************************/

        LPUSTR                                  szQualifier;
        
/**********************************************************
    szOwner
    
    Owner name to pass to SQLSpecialColumns.
**********************************************************/

        LPUSTR                                  szOwner;

/**********************************************************
    szName
    
    Table name to pass to SQLSpecialColumns.
**********************************************************/

        LPUSTR                                  szName;

/**********************************************************
    Type
    
    Type argument to pass to SQLSpecialColumns.
**********************************************************/

        UWORD                                   Type;

/**********************************************************
    Scope
    
    Scope argument to pass to SQLSpecialColumns.
**********************************************************/

        UWORD                                   Scope;

/**********************************************************
    Nullable
    
    Nullable argument to pass to SQLSpecialColumns.
**********************************************************/

        UWORD                                   Nullable;        

/**********************************************************
    Set
    
    Structure for result set.
**********************************************************/

        sSQLSPECIALCOLRESULTSET Set;
    public:
/**********************************************************
    odbcSpecialColumnIterator
    
    Constructor for iterator class to process result set 
    for SQLSpecialColumns.
**********************************************************/

        odbcSpecialColumnIterator(
            podbcCONNECT    pC,
        UWORD                   fColType,
        LPUSTR                  szTableQualifier,
        LPUSTR                  szTableOwner,
        LPUSTR                  szTableName,
        UWORD                   fScope,
        UWORD                   fNullable
            );
            
        odbcSpecialColumnIterator(
            podbcCONNECT    pC,
        UWORD                   fColType,
        LPSTR                   szTableQualifier,
        LPSTR                   szTableOwner,
        LPSTR                   szTableName,
        UWORD                   fScope,
        UWORD                   fNullable
            );
            
/**********************************************************
    ~odbcSpecialColumnIterator
    
    Destructor.
**********************************************************/

        virtual ~odbcSpecialColumnIterator();
        
/**********************************************************
    DoQuery

    Process query.
**********************************************************/

        virtual BOOL  DoQuery(void);
                
/**********************************************************
    pSpecialColsResultSet
    
    Returns pointer to row data structure for result
    set from SQLSpecialColumns.
**********************************************************/

    virtual cpsSQLSPECIALCOLRESULTSET pSpecialColsResultSet(void)
        {
        return (cpsSQLSPECIALCOLRESULTSET)
        pResultSet();
        };
    }; // end of class odbcSpecialColumnIterator

/**********************************************************
    odbcTypeInfoIterator

    Iterator class to process result set for 
    SQLTypeInfo.
**********************************************************/

    class odbcEXPORTED odbcTypeInfoIterator : public odbcSYSINFOITERATOR {
    protected:
    
/**********************************************************
    Set
    
    Structure for result set.
**********************************************************/

        sSQLTYPEINFORESULTSET Set;
    public:
/**********************************************************
    odbcTypeInfoIterator

    Constructor.
**********************************************************/

        odbcTypeInfoIterator(
            podbcCONNECT    pC
            );
            
/**********************************************************
    ~odbcTypeInfoIterator
    
    Destructor.
**********************************************************/

        virtual ~odbcTypeInfoIterator();

/**********************************************************
    DoQuery

    Process query.
**********************************************************/

        virtual BOOL  DoQuery(void);
                
/**********************************************************
    pTypeInfoResultSet
    
    Returns pointer to row data structure for result
    set from SQLTypeInfo.
**********************************************************/

    virtual cpsSQLTYPEINFORESULTSET pTypeInfoResultSet(void)
        {
        return (cpsSQLTYPEINFORESULTSET)
        pResultSet();
        };
    }; // end of class odbcTypeInfoIterator

// new in v2.0

/**********************************************************
    odbcColumnPrivilegesIterator
    
    Iterator class to process result set for SQLColumnPrivileges.
**********************************************************/

    class odbcEXPORTED odbcColumnPrivilegesIterator :
                 public odbcSYSINFOITERATOR {
    protected:
/**********************************************************
    szQualifier
    
    Table qualifier to pass to SQLColumnPrivileges.
**********************************************************/

        LPUSTR                                  szQualifier;
        
/**********************************************************
    szOwner
    
    Owner name to pass to SQLColumnPrivileges.
**********************************************************/

        LPUSTR                                  szOwner;

/**********************************************************
    szName
    
    Table name to pass to SQLColumnPrivileges.
**********************************************************/

        LPUSTR                                  szName;

/**********************************************************
    szColName

    Column name to pass to SQLColumnPrivileges.
**********************************************************/

        LPUSTR szColName;

/**********************************************************
    Set
    
    Structure for result set.
**********************************************************/

        sCOLUMNPRIVILEGESRESULTSET Set;
    public:
/**********************************************************
    odbcColumnPrivilegesIterator
    
    Constructor for iterator class to process result set 
    for SQLColumnPrivileges.
**********************************************************/

        odbcColumnPrivilegesIterator(
            podbcCONNECT    pC,
        LPUSTR                  szTableQualifier,
        LPUSTR                  szTableOwner,
        LPUSTR                  szTableName,
        LPUSTR                  szColumnName
            );
            
        odbcColumnPrivilegesIterator(
            podbcCONNECT    pC,
        LPSTR                   szTableQualifier,
        LPSTR                   szTableOwner,
        LPSTR                   szTableName,
        LPSTR                   szColumnName
            );

/**********************************************************
    ~odbcColumnPrivilegesIterator
    
    Destructor.
**********************************************************/

        virtual ~odbcColumnPrivilegesIterator();

/**********************************************************
    DoQuery

    Process query.
**********************************************************/

        virtual BOOL  DoQuery(void);
                
/**********************************************************
    pColumnPrivilegesResultSet

    Returns pointer to row data structure for result
    set from SQLColumnPrivileges.
**********************************************************/

    virtual cpsCOLUMNPRIVILEGESRESULTSET pColumnPrivilegesResultSet(void)
        {
        return (cpsCOLUMNPRIVILEGESRESULTSET)
        pResultSet();
        };
    }; // end odbcColumnPrivilegesIterator class

/**********************************************************
    odbcTablePrivilegesIterator
    
    Iterator class to process result set for SQLTablePrivileges.
**********************************************************/

    class odbcEXPORTED odbcTablePrivilegesIterator :
                 public odbcSYSINFOITERATOR {
    protected:
/**********************************************************
    szQualifier

    Table qualifier to pass to SQLTablePrivileges.
**********************************************************/

        LPUSTR                                  szQualifier;
        
/**********************************************************
    szOwner
    
    Owner name to pass to SQLTablePrivileges.
**********************************************************/

        LPUSTR                                  szOwner;

/**********************************************************
    szName
    
    Table name to pass to SQLTablePrivileges.
**********************************************************/

        LPUSTR                                  szName;

/**********************************************************
    Set
    
    Structure for result set.
**********************************************************/

        sTABLEPRIVILEGESRESULTSET Set;
    public:
/**********************************************************
    odbcTablePrivilegesIterator
    
    Constructor for iterator class to process result set 
    for SQLTablePrivileges.
**********************************************************/

        odbcTablePrivilegesIterator(
            podbcCONNECT    pC,
        LPUSTR                  szTableQualifier,
        LPUSTR                  szTableOwner,
        LPUSTR                  szTableName
            );
            
        odbcTablePrivilegesIterator(
            podbcCONNECT    pC,
        LPSTR                   szTableQualifier,
        LPSTR                   szTableOwner,
        LPSTR                   szTableName
            );

/**********************************************************
    ~odbcTablePrivilegesIterator
    
    Destructor.
**********************************************************/

        virtual ~odbcTablePrivilegesIterator();
        
/**********************************************************
    DoQuery

    Process query.
**********************************************************/

        virtual BOOL  DoQuery(void);
                
/**********************************************************
    pTablePrivilegesResultSet

    Returns pointer to row data structure for result
    set from SQLTablePrivileges.
**********************************************************/

    virtual cpsTABLEPRIVILEGESRESULTSET pTablePrivilegesResultSet(void)
        {
        return (cpsTABLEPRIVILEGESRESULTSET)
        pResultSet();
        };
    }; // end odbcTablePrivilegesIterator class

/**********************************************************
    odbcProceduresIterator

    Iterator class to process result set for SQLProcedures.
**********************************************************/

    class odbcEXPORTED odbcProceduresIterator :
                 public odbcSYSINFOITERATOR {
    protected:
/**********************************************************
    szQualifier
    
    Proc qualifier to pass to SQLProcedures.
**********************************************************/

        LPUSTR                                  szQualifier;
        
/**********************************************************
    szOwner
    
    Owner name to pass to SQLProcedures.
**********************************************************/

        LPUSTR                                  szOwner;

/**********************************************************
    szName

    Proc name to pass to SQLProcedures.
**********************************************************/

        LPUSTR                                  szName;

/**********************************************************
    Set
    
    Structure for result set.
**********************************************************/

        sPROCEDURESRESULTSET Set;
    public:
/**********************************************************
    odbcProceduresIterator
    
    Constructor for iterator class to process result set 
    for SQLProcedures.
**********************************************************/

        odbcProceduresIterator(
            podbcCONNECT    pC,
        LPUSTR                  szProcQualifier,
        LPUSTR                  szProcOwner,
        LPUSTR                  szProcName
            );
            
        odbcProceduresIterator(
            podbcCONNECT    pC,
        LPSTR                   szProcQualifier,
        LPSTR                   szProcOwner,
        LPSTR                   szProcName
            );

/**********************************************************
    ~odbcProceduresIterator
    
    Destructor.
**********************************************************/

        virtual ~odbcProceduresIterator();
        
/**********************************************************
    DoQuery

    Process query.
**********************************************************/

        virtual BOOL  DoQuery(void);
                
/**********************************************************
    pProceduresResultSet

    Returns pointer to row data structure for result
    set from SQLProcedures.
**********************************************************/

    virtual cpsPROCEDURESRESULTSET pProceduresResultSet(void)
        {
        return (cpsPROCEDURESRESULTSET)
        pResultSet();
        };
    }; // end odbcProceduresIterator class

/**********************************************************
    odbcProcedureColumnsIterator

    Iterator class to process result set for SQLProcedureColumns.
**********************************************************/

    class odbcEXPORTED odbcProcedureColumnsIterator :
                 public odbcSYSINFOITERATOR {
    protected:
/**********************************************************
    szQualifier
    
    Proc qualifier to pass to SQLProcedureColumns.
**********************************************************/

        LPUSTR                                  szQualifier;
        
/**********************************************************
    szOwner
    
    Owner name to pass to SQLProcedureColumns.
**********************************************************/

        LPUSTR                                  szOwner;

/**********************************************************
    szName

    Proc name to pass to SQLProcedureColumns.
**********************************************************/

        LPUSTR                                  szName;

/**********************************************************
    szColName

    Column name to pass to SQLColumnPrivileges.
**********************************************************/

        LPUSTR                  szColName;

/**********************************************************
    Set
    
    Structure for result set.
**********************************************************/

        sPROCEDURECOLUMNSRESULTSET Set;
    public:
/**********************************************************
    odbcProcedureColumnsIterator

    Constructor for iterator class to process result set 
    for SQLProcedureColumns.
**********************************************************/

        odbcProcedureColumnsIterator(
            podbcCONNECT    pC,
        LPUSTR                  szProcQualifier,
        LPUSTR                  szProcOwner,
        LPUSTR                  szProcName,
        LPUSTR                  szColumnName
            );
            
        odbcProcedureColumnsIterator(
            podbcCONNECT    pC,
        LPSTR           szProcQualifier,
        LPSTR           szProcOwner,
        LPSTR           szProcName,
        LPSTR           szColumnName
            );

/**********************************************************
    ~odbcProcedureColumnsIterator
    
    Destructor.
**********************************************************/

        virtual ~odbcProcedureColumnsIterator();
        
/**********************************************************
    DoQuery

    Process query.
**********************************************************/

        virtual BOOL  DoQuery(void);

/**********************************************************
    pProcedureColumnsResultSet

    Returns pointer to row data structure for result
    set from SQLProcedureColumns.
**********************************************************/

    virtual cpsPROCEDURECOLUMNSRESULTSET pProcedureColumnsResultSet(void)
        {
        return (cpsPROCEDURECOLUMNSRESULTSET)
        pResultSet();
        };
    }; // end odbcProcedureColumnsIterator class

/**********************************************************
    odbcPrimaryKeysIterator

    Iterator class to process result set for SQLPrimaryKeys.
**********************************************************/

    class odbcEXPORTED odbcPrimaryKeysIterator :
                 public odbcSYSINFOITERATOR {
    protected:
/**********************************************************
    szQualifier
    
    Table qualifier to pass to SQLPrimaryKeys.
**********************************************************/

        LPUSTR                                  szQualifier;
        
/**********************************************************
    szOwner
    
    Owner name to pass to SQLPrimaryKeys.
**********************************************************/

        LPUSTR                                  szOwner;

/**********************************************************
    szName

    Table name to pass to SQLPrimaryKeys.
**********************************************************/

        LPUSTR                                  szName;

/**********************************************************
    Set
    
    Structure for result set.
**********************************************************/

        sPRIMARYKEYSRESULTSET Set;
    public:
/**********************************************************
    odbcPrimaryKeysIterator

    Constructor for iterator class to process result set 
    for SQLPrimaryKeys.
**********************************************************/

        odbcPrimaryKeysIterator(
            podbcCONNECT    pC,
        LPUSTR                  szTableQualifier,
        LPUSTR                  szTableOwner,
        LPUSTR                  szTableName
            );
            
        odbcPrimaryKeysIterator(
            podbcCONNECT    pC,
        LPSTR           szTableQualifier,
        LPSTR           szTableOwner,
        LPSTR           szTableName
            );

/**********************************************************
    ~odbcPrimaryKeysIterator
    
    Destructor.
**********************************************************/

        virtual ~odbcPrimaryKeysIterator();
        
/**********************************************************
    DoQuery

    Process query.
**********************************************************/

        virtual BOOL  DoQuery(void);

/**********************************************************
    pPrimaryKeysResultSet

    Returns pointer to row data structure for result
    set from SQLPrimaryKeys.
**********************************************************/

    virtual cpsPRIMARYKEYSRESULTSET pPrimaryKeysResultSet(void)
        {
        return (cpsPRIMARYKEYSRESULTSET)
        pResultSet();
        };
    }; // end odbcPrimaryKeysIterator class

/**********************************************************
    odbcForeignKeysIterator

    Iterator class to process result set for SQLForeignKeys.
**********************************************************/

    class odbcEXPORTED odbcForeignKeysIterator :
                 public odbcSYSINFOITERATOR {
    protected:
/**********************************************************
    szPkQualifier

    Primary key table qualifier to pass to SQLForeignKeys.
**********************************************************/

        LPUSTR                                  szPkQualifier;
        
/**********************************************************
    szPkOwner

    Primary key table owner name to pass to SQLForeignKeys.
**********************************************************/

        LPUSTR                                  szPkOwner;

/**********************************************************
    szPkName

    Primary key table name to pass to SQLForeignKeys.
**********************************************************/

        LPUSTR                                  szPkName;

/**********************************************************
    szFkQualifier

    Foreign key table qualifier to pass to SQLForeignKeys.
**********************************************************/

        LPUSTR                                  szFkQualifier;
        
/**********************************************************
    szFkOwner

    Foreign key table owner name to pass to SQLForeignKeys.
**********************************************************/

        LPUSTR                                  szFkOwner;

/**********************************************************
    szFkName

    Foreign key table name to pass to SQLForeignKeys.
**********************************************************/

        LPUSTR                                  szFkName;

/**********************************************************
    Set

    Structure for result set.
**********************************************************/

        sFOREIGNKEYSRESULTSET Set;
    public:
/**********************************************************
    odbcForeignKeysIterator

    Constructor for iterator class to process result set
    for SQLForeignKeys.
**********************************************************/

        odbcForeignKeysIterator(
            podbcCONNECT    pC,
        LPUSTR                  szPkTableQualifier,
        LPUSTR                  szPkTableOwner,
        LPUSTR                  szPkTableName,
        LPUSTR                  szFkTableQualifier,
        LPUSTR                  szFkTableOwner,
        LPUSTR                  szFkTableName
            );

        odbcForeignKeysIterator(
            podbcCONNECT    pC,
        LPSTR           szPkTableQualifier,
        LPSTR           szPkTableOwner,
        LPSTR           szPkTableName,
        LPSTR           szFkTableQualifier,
        LPSTR           szFkTableOwner,
        LPSTR           szFkTableName
            );

/**********************************************************
    ~odbcForeignKeysIterator
    
    Destructor.
**********************************************************/

        virtual ~odbcForeignKeysIterator();
        
/**********************************************************
    DoQuery

    Process query.
**********************************************************/

        virtual BOOL  DoQuery(void);

/**********************************************************
    pForeignKeysResultSet

    Returns pointer to row data structure for result
    set from SQLForeignKeys.
**********************************************************/

    virtual cpsFOREIGNKEYSRESULTSET pForeignKeysResultSet(void)
        {
        return (cpsFOREIGNKEYSRESULTSET)
        pResultSet();
        };
    }; // end odbcForeignKeysIterator class


// end new in v2.0

#endif
