#  Toc.pm

## Copyright (c) 1996, Lotus Development Corporation.
## All Rights Reserved.


require 5.000;

package Toc;
BEGIN { &main::Require("Toc.nls");		import Toc_nls; }

BEGIN { &main::Require("Scfg.pm"); 		import Scfg; }
BEGIN { &main::Require("MiscUtil.pl");	import MiscUtil; }



sub new {
	local($class) = @_;

	my $self = {};
	bless $self;

	%$self = (	

		'tocfile'		=> "",
		'toctextfile'	=> "",
		'disttype'		=> "",
		'distrev'		=> "",
		'error'			=> "",

		'optsets'		=> {},
		'filesets'		=> {},
		'optsets_text'	=> {},
		'filesets_text'	=> {},
	);

	return $self;
}


sub Init {
	local( $self, $tocfile, $toctextfile ) = @_;

	local( $tag, @value );
	local( $rev );


	local($toc_scfg)		= new Scfg ();
	local($toctext_scfg)	= new Scfg ();

	$self->{'tocfile'}		= $tocfile;
	$self->{'toctextfile'}	= $toctextfile;
	$self->{'error'}		= "";

	local( @optset_toc_reqlist )	= qw( platforms );
	local( @optset_toc_optlist )	= qw( required optional custom );
	local( @fileset_toc_reqlist )	= qw( platforms );
	local( @fileset_toc_optlist ) 	= qw( include );

	local( @optset_text_reqlist )	= qw(	desc 
											install_banner 
											installhelp_banner 
											platforms 
										);
	local( @optset_text_optlist )	= qw( );
	local( @fileset_text_reqlist )	= qw( desc );
	local( @fileset_text_optlist )	= qw( help );


	# load raw tocfile
	#
	if ( ! $toc_scfg->InitFromFile( $tocfile ) ) {
		$self->{'error'} = $toc_scfg->GetError();
		return 0;
	}

	# get required tocfile header (disttype & distrev)
	#
	$toc_scfg->GetNext( \$tag, \@value );
	if ( $tag ne "disttype" ) {
		$self->SetError("unexpected_key", $toc_scfg->{'linenum'}, $tag);
		return 0;
	}
	$self->{'disttype'} = $value[0];
	$toc_scfg->GetNext( \$tag, \@value );
	if ( $tag ne "distrev" ) {
		$self->SetError("unexpected_key", $toc_scfg->{'linenum'}, $tag);
		return 0;
	}
	$self->{'distrev'} = $value[0];


	# loop for parsing toc file
	#
	while (1) {

		$toc_scfg->GetNext( \$tag, \@value ) || last;

		if ( $tag eq "optset" ) {
			$self->ParseTocSection(
				$toc_scfg,
				$self->{'optsets'},
				$value[0],
				\@optset_toc_reqlist,
				\@optset_toc_optlist ) || last;
		}

		if ( $tag eq "fileset" ) {
            $self->ParseTocSection(
				$toc_scfg,
                $self->{'filesets'},
                $value[0], 
                \@fileset_toc_reqlist,
                \@fileset_toc_optlist ) || last;
		}
	}

	($self->{'error'} && ($self->{'error'} ne "EOD")) && return 0;
	$self->{'error'} = "";

	
	# now do toc text file

	# load raw toctextfile
    #
    if ( ! $toctext_scfg->InitFromFile( $toctextfile ) ) {
        $self->{'error'} = $toctext_scfg->GetError();
        return 0;
    }


	# loop for parsing toc text file
	#
	while (1) {

        $toctext_scfg->GetNext( \$tag, \@value ) || last;
 
        if ( $tag eq "optset" ) {
            $self->ParseTocSection(
				$toctext_scfg,
                $self->{'optsets_text'},
                $value[0],
                \@optset_text_reqlist,
                \@optset_text_optlist ) || last;
        }
 
        if ( $tag eq "fileset" ) {
            $self->ParseTocSection(
				$toctext_scfg,
                $self->{'filesets_text'},
                $value[0],
                \@fileset_text_reqlist,
                \@fileset_text_optlist ) || last;
        }
    }
 
    ($self->{'error'} && ($self->{'error'} ne "EOD")) && return 0; 
    $self->{'error'} = "";


	return 1;
}


sub ParseTocSection {
	local(	$self, 
			$scfg,
			$sectionlist_ref,
			$sectiontag,
			$reqlist_ref,
			$optlist_ref		) = @_;

	# parse entries keyed on "rev"; must get at least one

	local( $tag, @value );
	local( $rev ) = "";

	if ($ENV{'NI_DEBUG_PARSETOCSECTION'}) {
		print "ParseTocSection:\n";
		print "  sectionlist_ref : $sectionlist_ref\n";
		print "  sectiontag      : $sectiontag\n";
		print "  reqlist_ref     : @$reqlist_ref\n";
		print "  optlist_ref     : @$optlist_ref\n";
	}

	while (1) {
 
		$scfg->GetNext( \$tag, \@value ) || last;;
		($tag eq "rev") || last;
		$rev = $value[0];
 
		# create the new (empty) hash's for this sectiontag/rev
		# if necessary
		#
		if (ref($sectionlist_ref->{$sectiontag}) ne "HASH") {
			$sectionlist_ref->{$sectiontag} = {};
		}
		if (ref($sectionlist_ref->{$sectiontag}->{$rev}) ne "HASH") {
			$sectionlist_ref->{$sectiontag}->{$rev} = {};
		}

		if ( ! $scfg->ParseKeys(
			$reqlist_ref, $optlist_ref,
			$sectionlist_ref->{$sectiontag}->{$rev} ) ) {
	 
			$self->{'error'} = $scfg->GetError();
			return 0;
		}
	}

	# unknown scfg error
	#
	if ($scfg->GetError() && ($scfg->GetError() ne "EOD")) {
		$self->{'error'} = $scfg->GetError();
		return 0;
	}

	# EOD before getting rev
	#
	if (($scfg->GetError() eq "EOD") && (! $rev)) {
		$self->SetError("unexpected_eod");
		return 0;
	}

	# got unexpected tag before a rev
	#
	if ( ! $rev ) {
		$self->SetError("unexpected_key", $scfg->{'linenum'}, $key);
		return 0;
	}

	# else ok

	$scfg->RewindOne();
	return 1;
}


sub Dump {
	local( $self ) = @_;

	&DumpHash( 0, %$self );

}


sub SetError {
	local( $self, $errtype, @data ) = @_;

	if ( $errtype eq "unexpected_key" ) {

		# $data[0]: linenum
		# $data[1]: key

		$self->{'error'} =
			"$txt{'Parse_error_at_line'} " .
			$data[0] .
			"$txt{'.'}\n" .
			"$txt{'Unexpected_key'} $txt{'\"'}" .
			"$data[1]$txt{'\"'}$txt{'.'}\n";

	} elsif ( $errtype eq "unexpected_eod" ) {
		$self->{'error'} = "$txt{'unexpected_eod'}$txt{'.'}\n";
		
	} else {
		$self->{'error'} = "$txt{'Unknown_TOC_error'}$txt{'.'}\n";
	}
}


sub GetError {
	local( $self ) = @_;

	return $self->{'error'};
}


sub GetOptions {
	local( $self, $optset, $rev ) = @_;

	local($listref) = $self->{'optsets'}->{$optset}->{$rev};

	$listref = $listref->{'optional'}->{'orderlist'};

	if (ref($listref) eq "ARRAY") {
		return @{$listref};
	} else {
		return(());
	}
}


sub GetPlatforms {
	local( $self, $optset, $rev ) = @_;

	return 
		@{$self->{'optsets'}->{$optset}->{$rev}->{'platforms'}->{'orderlist'}};
}


sub IsFilesetPlatformCommon {
    local( $self, $fileset, $rev ) = @_;

	local($tmp) = 
		$self->{'filesets'}->{$fileset}->{$rev}->{'platforms'}->{'common'};

	return (ref($tmp) eq "HASH") ? 1 : 0;
}


sub GetOptsetText {
	local( $self, $optset, $optsetrev, $texttag ) = @_;

	return ${$self->{'optsets_text'}->{$optset}->{$optsetrev}->{$texttag}}[0];
}


sub GetOptsetPlatformText {
	local( $self, $optset, $optsetrev, $platform, $texttag ) = @_;

	local($tmp) = $self->{'optsets_text'}->{$optset}->{$optsetrev};
	return ${$tmp->{'platforms'}->{$platform}->{$texttag}}[0];
}


sub GetFilesetText {
	local( $self, $fileset, $rev, $texttag ) = @_;

	return ${$self->{'filesets_text'}->{$fileset}->{$rev}->{$texttag}}[0];
}


sub GetOptionalFilesetPrereqList {
	local( $self, $optset, $rev, $fileset ) = @_;

	local($tmp)		= $self->{'optsets'}->{$optset}->{$rev}->{'optional'};
	local($listref)	= $tmp->{$fileset}->{'prereq'};

	if (ref($listref) eq "ARRAY") {
		return @{$listref};
	} else {
		return (());
	}
}


sub GetOptionalFilesetPrereqOfList {
	local( $self, $optset, $rev, $fileset ) = @_;

	local($opt, @prereqlist, @prereqoflist);
	local(@optsetoptions) 	= $self->GetOptions( $optset, $rev );

	foreach $opt (@optsetoptions) {
		($opt eq $fileset) && next;
		@prereqlist = 
			$self->GetOptionalFilesetPrereqList( $optset, $rev, $opt );
		grep(/^$fileset$/, @prereqlist) && push(@prereqoflist, $opt);
	}

	return @prereqoflist;
}


sub GetReqdFsetList {
	local( $self, $optset, $rev ) = @_;

	local( $tmp );

    $tmp = $self->{'optsets'}->{$optset}->{$rev}->{'required'};

    if ((ref($tmp) eq "HASH") && (ref($tmp->{'orderlist'}) eq "ARRAY")) {
        return @{$tmp->{'orderlist'}};
    } else {
		return();
	}
}


sub GetOptPlatformArchivesList {
	local(	$self, $optset, $rev, 
			$optlist_ref, $pfmlist_ref, $outlist_ref ) = @_;
	

	local( $fileset, @tmppfmlist, $platform, $tmp, $tmphashref, $fi );
	local( @filesets ) = ();
	local( %filesets_done ) = ();

	if ($ENV{'NIBE_DEBUG'}) {
		print "GetOptPlatformArchivesList...\n";
		print "optset         :$optset\n";
		print "rev            :$rev\n";
		print "optlist        :@$optlist_ref\n";
		print "pfmlist        :@$pfmlist_ref\n";
		print "outlist        :@$outlist_ref\n";
		print "Continue...\n";
		$foo=<STDIN>;
	}

	undef @$outlist_ref;

	# init the list of filesets to return info with 
	# any required filsets (may be empty)
	#
	@filesets = ($self->GetReqdFsetList( $optset, $rev ));
	
	# add the specified optional filesets to the list
	#
	push( @filesets, @$optlist_ref );

	# loop through fileset list in such a way that it can be added to 
	# while doing so
	#
	for ($fi=0; $fi<=$#filesets; ++$fi) {
		$fileset = $filesets[$fi];

		$filesets_done{$fileset} && next;
		$filesets_done{$fileset} = 1;


		if ($self->IsFilesetPlatformCommon( $fileset, $rev )) {
			@tmppfmlist = ("common");
		} else {
			@tmppfmlist = @$pfmlist_ref;
		}

		foreach $platform (@tmppfmlist) {

			# allocate a new hash (and archive) for this fileset & platform
			#
			$tmphashref = {};
			$tmphashref->{'archives'} = ();

			# fill-in except for archives
			#
			$tmphashref->{'prodopt_tag'}	= $fileset;
			$tmphashref->{'prodopt_desc'}	= 
				$self->GetFilesetText( $fileset, $rev, "desc" );
			if ($platform ne "common") {
				$tmphashref->{'platform_tag'}	= $platform;
				$tmphashref->{'platform_desc'}	= 
					$self->GetOptsetPlatformText(
						$optset, $rev, $platform, "desc" );
			}

			# add the archive file to list
			#
			$tmp = $self->{'filesets'}->{$fileset}->{$rev}->{'platforms'};
			$tmp = $tmp->{$platform}->{'file'};
			@{$tmphashref->{'archives'}} = (${$tmp}[0]);

			# add included fileset(s) to the todo list, if defined
			#
			$tmp = $self->{'filesets'}->{$fileset}->{$rev}->{'include'};
			if (ref($tmp) eq "ARRAY") {
				push( @filesets, @{$tmp} );
			}

			# add this fileset hash to the list which is returned
			push( @$outlist_ref, $tmphashref );
		}
	}

	return 1;
}
						

sub GetTotalSize {
	local(	$self, $optset, $rev, 
			$optlist_ref, $pfmlist_ref ) = @_;
	
	#	In:	$optset			optset as in toc
	#		$rev			rev of specified optset as in toc optset record
	#		$optlist_ref	list of product options to include in size 
	#						  (from toc optset "optional" record)
	#		$pfmlist_ref	list of platforms to include in size
	#						  (from toc optset "platforms" record)
	#
	#	Returns:			total of size fields from toc of set of
	#						  set of filesets as described by input lists


	local( $fileset, @tmppfmlist, $platform, $tmp, $fi );
	local( @filesets ) = ();
	local( %filesets_done ) = ();
	local( $size ) = 0;

	#print "optset  : $optset\n";
	#print "rev     : $rev\n";
	#print "optlist : " . @{$optlist_ref} . "\n";
	#print "pfmlist : " . @{$pfmlist_ref} . "\n";

    #
    @filesets = ($self->GetReqdFsetList( $optset, $rev ));

	# add the specified optional filesets to the list
	#
	push( @filesets, @$optlist_ref );

	# loop through fileset list in such a way that it can be added to 
	# while doing so (include-ed filesets)
	#
	for ($fi=0; $fi<=$#filesets; ++$fi) {
		$fileset = $filesets[$fi];

		$filesets_done{$fileset} && next;
		$filesets_done{$fileset} = 1;


		if ($self->IsFilesetPlatformCommon( $fileset, $rev )) {
			@tmppfmlist = ("common");
		} else {
			@tmppfmlist = @$pfmlist_ref;
		}

		foreach $platform (@tmppfmlist) {

			# add size
			#
			$size += $self->GetFilesetSize( $fileset, $rev, $platform );

			# add included fileset(s) to the todo list, if defined
			#
			$tmp = $self->{'filesets'}->{$fileset}->{$rev}->{'include'};
			if (ref($tmp) eq "ARRAY") {
				push( @filesets, @{$tmp} );
			}
		}
	}

	return $size;
}


sub GetFilesetSize {
	local( $self, $fileset, $rev, $platform ) = @_;

	local( $tmp );

	$tmp = $self->{'filesets'}->{$fileset}->{$rev}->{'platforms'};
	return ${$tmp->{$platform}->{'size'}}[0];
}


sub GetClosestPlatform {
	local(	$self,
			$optset,
			$rev,
			$arch,
			$major,
			$minor,	
			$error_ref ) = @_;

	local($tmp);

	undef $$error_ref;

	if ( ! defined(
		$self->{'optsets'}->{$optset}->{$rev}->{'platforms'}->{$arch}) ) {
		$$error_ref = "$txt{'platform_unavailable'}\n";
		return undef;
	}
	
	$tmp = $self->{'optsets'}->{$optset}->{$rev}->{'platforms'}->{$arch};

	if ($major < ${$tmp->{'major'}}[0]) {
		$$error_ref = "$txt{'platform_unavailable'}\n";
		return undef;
	}

	if ($major > ${$tmp->{'major'}}[0]) {
		$$error_ref = "$txt{'warn_greater_major'}\n";
		return $arch;
	}

	if ($minor < ${$tmp->{'minor'}}[0]) {
		$$error_ref = "$txt{'platform_unavailable'}\n";
		return undef;
	}

	if ($minor > ${$tmp->{'minor'}}[0]) {
		$$error_ref = "$txt{'warn_greater_minor'}\n";
		return $arch;
	}

	# golly, a perfect match
	return $arch;
}


sub HasCustom {
	local( $self, $optset, $optsetrev ) = @_;

	local($tmp);

	$tmp = $self->{'optsets'}->{$optset}->{$optsetrev}->{'custom'};

	return ${$tmp}[0] ? 1 : 0;
}



1;
