# Scfg.pm

## Copyright (c) 1996, Lotus Development Corporation.
## All Rights Reserved.


require 5.000;

package Scfg;
BEGIN { &main::Require("Scfg.nls"); 	import Scfg_nls; }

BEGIN { &main::Require("MiscUtil.pl"); 	import MiscUtil; }

#
#
#
#
#




sub new {
	local($class) = @_;

	my $self = {};
	bless $self;

	%$self = (	'file'				=> "",
				'linenum'			=> 0,
				'partial_lastline'	=> "",
				'error'				=> "" );

	@{$self->{'rawdata'}} = ();

	return $self;
}


sub InitFromFile {
	local( $self, $file ) = @_;

	# Reads lines of file at pathname $file into elements
	# of @{$self->{'rawdata'}}, chopping off newlines.
	# Returns 1 on success, 0 on failure and sets $self->{'error'} msg.
	# Empty file is error.


	local( *FILE );

	$self->{'file'}				= $file;
	$self->{'linenum'}			= 0;
	$self->{'partial_lastline'}	= "";
	$self->{'error'} 			= "";

	undef( @{$self->{'rawdata'}} );
	@{$self->{'rawdata'}} = ();


	if ( ! open( FILE, "<$self->{'file'}" ) ) {
		$self->{'error'} = 
			"$txt{'Cant_open_file'} " .
			"$txt{'\"'}$self->{'file'}$txt{'\"'}$txt{':'} $!\n";
		return 0;
	}

	while (<FILE>) {
		chop;
		push( @{$self->{'rawdata'}}, $_ );
	}
	close( FILE );
	
	if ( scalar( @{$self->{'rawdata'}} ) ) {
		return 1;
	} else {
		$self->{'error'} = 
			"$txt{'File'} $self->{'file'} $txt{'is_empty'}$txt{'.'}\n";
		return 0;
	}
}


sub GetNext {
	local( $self, $tag_ref, $list_ref, $taghash_ref ) = @_;

	# args
	# in:	$self
	# out:	$$tag_ref
	#		@$list_ref
	#		%$taghash_ref

	# Parses the next item from already init'd rawdata.
	# The scalar, array, & hash variables for which 
	# $tag_ref, $list_ref, & $taghash_ref provide references
	# (respectively) must exist. All three are re-initialized.
	# 
	# If the next item is "<tag>=<list>" (TAGLIST), 
	# $$tag_ref & @$list_ref are set.
	# If the next item is "<tag> { \n ... }" (TAGHASH),
	# $$tag_ref & %$taghash_ref are set. Subhash's are allocated dynamically
	#
	# Success returns 1, and failure sets $self->{'error'} msg 
	# ("EOD" if end of data) and returns 0.

	local( $origline, $type );

	$self->{'error'} = "";
	$$tag_ref = "";
	@$list_ref = ();
	defined($taghash_ref) && (%$taghash_ref=());

	$type = $self->GetNextLine( $tag_ref, $list_ref, \$origline );
	if ( ($type eq "EOD") || ($type eq "ERROR") ) {
		return 0;
	}

	if ($type eq "TAGLIST") {
	
		return 1;
	
	} elsif ($type eq "TAGHASH_BEGIN") {

		$tmpref = $self->GetTagHash();
		if ($tmpref) {
			%$taghash_ref = %$tmpref;
			return 1;
		} else {
			# $self->{'error'} already set by GetTagHash()
			return 0;
		}

	} else {

		# here is an unmatched TAGHASH_END

		$self->{'error'} = "$txt{'Error_in'} ";
		$self->{'error'} .=
			($self->{'file'} eq '-') ? "$txt{'data'} " :
			"$txt{'file'} $self->{'file'} ";
		$self->{'error'} .= 
			"$txt{'at_line'} $self->{'linenum'}$txt{':'}\n";
		$self->{'error'} .= "$txt{'\"'}$origline$txt{'\"'}\n";

		return 0;
	}
}


sub GetTagHash {
	local($self) = @_;

	# args
	# in:	$self
	# out:	-
	#
	# returns:	1 on success
	#			0 on failure ($self->{'error'} set as from GetNextLine())

	# Presumes to be called when just-previous item parsed was 
	# a TAGHASH decl. Dynamically creates an anonymous hash
	# and parses into it until the TAGHASH_END corresponding to the 
	# TAGHASH_BEGIN which prompted its call. Recurses when subhash encountered.

	local( $type, $tmptag, @tmplist, $origline );

	local( $taghash_ref ) = {};

	while ( 
		($type = $self->GetNextLine( \$tmptag, \@tmplist, \$origline ))
		ne "TAGHASH_END" )
	{
		if ( ($type eq "EOD") || ($type eq "ERROR") ) {
			return 0;
		}

		if ($type eq "TAGLIST") {

			@{$taghash_ref->{$tmptag}} = @tmplist;

		} elsif ($type eq "TAGHASH_BEGIN") {

			$taghash_ref->{$tmptag} = $self->GetTagHash();

			$taghash_ref->{$tmptag} || return 0;
		}
	}

	return $taghash_ref;
}


sub GetNextLine {
	local( $self, $tag_ref, $list_ref, $origline_ref ) = @_;

	# args
	# in:		$self
	# out:		$$tag_ref
	#			@$list_ref
	#			$$origline_ref

	# returns:	type:
	#				EOD
	#				ERROR				(sets $self->{'error'})
	#				TAGLIST				( <tag>=<list> )
	#				TAGHASH_BEGIN		( <tag> {      )
	#				TAGHASH_END			( }            )

	# Evaluates the next line from already init'd rawdata.
	# Skips over blank & comment lines.
	# Output arg vars are initialized, and $$origline_ref receives the raw line.
	# If it is a taglist (TAGLIST : <tag>=<list>), 
	# <tag> & <list> are parsed into $$tag_ref & @$list_ref respectively.
	# If it is the begining of a taghash (TAGHASH_BEGIN : <tag> { ),
	# tag is parsed into $$tag_ref.

	local( $right );

	$self->{'error'} = "";
	$$tag_ref = "";
	@$list_ref = ();
	$$origline_ref = "";

	# go forward to the next non-empty, non-comment line, and clean it.
	# do a bit of trickery that trailing }'s may be treated as separate lines.
	#
	while ( 1 ) {
		if ($self->{'linenum'} > $#{$self->{'rawdata'}}) {
			$self->{'error'} = "EOD";
			return "EOD";
		}

		if ($self->{'partial_lastline'}) {
			$_ = $self->{'partial_lastline'};
			$self->{'partial_lastline'}  = "";
			$$origline_ref = ${$self->{'rawdata'}}[ $self->{'linenum'} ];
		} else {
			$_ = $$origline_ref = ${$self->{'rawdata'}}[ $self->{'linenum'} ];
		}

		s/^\s+//;           # strip leading white
		s/\s+$//;           # strip trailing white
		if ( /^#/ || /^\s*$/ ) {
			# ignore empty- and comment-lines
			++$self->{'linenum'};
			next;
		}

		if (/^[^\}]+\}/) {
			$self->{'partial_lastline'} = $_;
			$self->{'partial_lastline'} =~ s/^[^\}]+//;
			s/\}.*$//;
		} elsif (/^\}.+$/) {
			$self->{'partial_lastline'} = $_;
			$self->{'partial_lastline'} =~ s/^\}//;
			$_ = '}';
		} else {
			++$self->{'linenum'};
		}

		last;
	}

	# we now have a raw line; see what it is
	
	if ( /^[^=\{\}]+=[^=\{\}]*$/ ) {
		
		#
		# "<tag>=<list>"
		#

		( $$tag_ref, $right ) = split( '=' );

		$$tag_ref =~ s/\s+$//;  # strip trailing white from tag
 
		@$list_ref = split( /,/, $right );
		grep( s/^\s+//, @$list_ref );
		grep( s/\s+$//, @$list_ref );
		grep( s/\\n/\n/g, @$list_ref );


		return "TAGLIST";
	
	} elsif ( /^[^=\{\}]+\s*\{\s*$/ ) {

		#
		# "<tag> {"
		#

		$$tag_ref = $_;
		$$tag_ref =~ s/\s*\{\s*$//;
 
		return "TAGHASH_BEGIN";

	} elsif ( /^\}$/ ) {
		
		# 
		# "}"
		#

		return "TAGHASH_END";

	} else {

		#
		# error
		#

		$self->{'error'} = "$txt{'Error_in'} ";
		$self->{'error'} .=
			($self->{'file'} eq '-') ? "$txt{'data'} " :
			"$txt{'file'} $self->{'file'} ";
		$self->{'error'} .= "$txt{'at_line'} $self->{'linenum'}$txt{':'}\n";
		$self->{'error'} .= "$txt{'\"'}$$origline_ref$txt{'\"'}\n";

		return "ERROR";
	}
}
	

sub ParseKeys {
	local( $self, $reqlist_ref, $optlist_ref, $outhash_ref, $error_ref ) = @_;

	# args
	# in:	$self
	#		$reqlist_ref	ref to list of required tags
	#		$optlist_ref	ref to list of optional tags
	# out:	%$outhash_ref	hash of tag=>value pairs
	#		$$error_ref		(optional; $self->{'error'} also set)


	# Starting with the next line in the already loaded raw data
	# expect all of reqlist & optionally optlist tags to be present,
	# in any order.
	#
	# Parse the corresponding tag values into tag=>value pairs
	# in the output hash, with sub-structures (arrays & hash's)
	# allocated dynamically.
	#
	# A tag not in reqlist or optlist encountered before all of
	# reqlist is an error, as is one already encountered before
	# all of reqlist.
	#
	# After getting all required tags, optional tags are retrieved 
	# until a non-specified tag, syntax error, or EOD occurs, any of which 
	# are not considered to be an error here.
	#
	# Returns 1 on success, else 0.


	local( $tag, @list, %taghash );
	local( %req_toget, %opt_toget );

	$self->{'error'} = "";
	undef %$outhash_ref;
	defined( $error_ref ) && ($$error_ref = "");

	foreach (@$reqlist_ref) {
		$req_toget{ $_ } = 1;
	}
	foreach (@$optlist_ref) {
        $opt_toget{ $_ } = 1;
    }

	# while not done and no error
	#
	while ( 1 ) {

		# get the next item and deal with it

		if ( ! $self->GetNext( \$tag, \@list, \%taghash ) ) {

			if ( (! %req_toget) && ($self->{'error'} eq "EOD")  ) {

				# ok: end of data, but we already got all required

				$self->{'error'} = "";
				return 1;
			} else {

				# not ok: end of data before all required tags
				# or something else bad

				defined( $error_ref ) && ($$error_ref = $self->{'error'});
				return 0;
			}
		}

		if ( &IsMember( $tag, @$reqlist_ref ) ) {

			# this tag is required

			if ( ! $req_toget{ $tag }) {
				
				# this tag has already been obtained 
				# which is an error if not all req tags
				# have been obtained, otherwise we're done

				if (%req_toget) {

					last; # goto errorland
				} else {
				
					# done
                	# "put back" the line we just got
	                --$self->{'linenum'};
	                return 1;
				}
			}

			# take it

			if ( @list ) {
				$outhash_ref->{$tag} = [];
				@{$outhash_ref->{$tag}} = @list;
			} elsif (%taghash) {
				$outhash_ref->{$tag} = {};
				%{$outhash_ref->{$tag}} = %taghash;
			} else {
				$outhash_ref->{$tag} = "";
			}

			delete( $req_toget{ $tag } );
			
		} elsif ( &IsMember( $tag, @$optlist_ref ) ) {

			# this tag is optional

			if ( ! $opt_toget{ $tag } ) {
	
				# this tag has already been obtained
				# which is an error if not all req tags
				# have been obtained, otherwise we're done

				if (%req_toget) {

					last; # goto errorland
				} else {
			
					# done
					# "put back" the line we just got
					--$self->{'linenum'};
					return 1;
				}
			}

			# take it

            if ( @list ) {
                $outhash_ref->{$tag} = [];
                @{$outhash_ref->{$tag}} = @list;
            } elsif (%taghash) {
                $outhash_ref->{$tag} = {};
                %{$outhash_ref->{$tag}} = %taghash;
            } else {
                $outhash_ref->{$tag} = "";
            }
 
			delete( $opt_toget{ $tag } );

		} else {

			# this tag is unexpected

			if ( ! %req_toget ) {

				# if all required tags already obtained, not an error

				# "put back" the line we just got
				--$self->{'linenum'};
				return 1;

			} else {

				last; # goto errorland
			}
		}

		if ( (! %req_toget) && (! %opt_toget) ) {
		
			# we got all required and optional: done
			return 1;
		}
	}
	
	# here only on error

	$self->{'error'} =
		"$txt{'Parse_error_at_line'} " .
		$self->{'linenum'} .
		"$txt{'.'}\n" .
		"$txt{'Unexpected_tag'} " .
		"$txt{'\"'}${tag}$txt{'\"'}$txt{'.'}\n";
	defined( $error_ref ) && ($$error_ref = $self->{'error'});
	return 0;
}

sub GetError {
	local($self) = @_;
	return $self->{'error'};
}


sub GetLineno {
	local($self) = @_;

	# return current line number from raw file/data

	return $self->{'linenum'};
}

sub RewindOne {
	local($self) = @_;

	# backup one line in raw file/data

	$self->{'linenum'} && --$self->{'linenum'};
}



package main;
1;
