
{*******************************************************}
{                                                       }
{       Delphi Visual Component Library                 }
{                                                       }
{       Copyright (c) 1995 Borland International        }
{                                                       }
{*******************************************************}

unit IStreams;

interface

uses VirtIntf, Classes, SysUtils;

type
  { TIMemoryStream }

  TIMemoryStream = class(TIStream)
  private
    FMemoryStream: TMemoryStream;
    FOwnStream: Boolean;
    FReserved: Byte;
    FModifyTime: Longint;
  public
    constructor Create(AMemoryStream: TMemoryStream);
    destructor Destroy; override;
    function Read(var Buffer; Count: Longint): Longint; override;
    function Write(const Buffer; Count: Longint): Longint; override;
    function Seek(Offset: Longint; Origin: Word): Longint; override;
    function GetModifyTime: Longint; override;
    property OwnStream: Boolean read FOwnStream write FOwnStream;
    property MemoryStream: TMemoryStream read FMemoryStream;
  end;

  { TIFileStream }

  TIFileStream = class(TIStream)
  private
    FFileStream: TFileStream;
  public
    constructor Create(const FileName: string; Mode: Word);
    destructor Destroy; override;
    function Read(var Buffer; Count: Longint): Longint; override;
    function Write(const Buffer; Count: Longint): Longint; override;
    function Seek(Offset: Longint; Origin: Word): Longint; override;
    function GetModifyTime: Longint; override;
    property FileStream: TFileStream read FFileStream;
  end;

  { TVirtualStream }

  TVirtualStream = class(TStream)
  private
    FIStream: TIStream;
  public
    constructor Create(AIStream: TIStream);
    function Read(var Buffer; Count: Longint): Longint; override;
    function Write(const Buffer; Count: Longint): Longint; override;
    function Seek(Offset: Longint; Origin: Word): Longint; override;
    function GetModifyTime: Longint;
  end;

  TExceptionHandler = procedure;

const
  ExceptionHandler: TExceptionHandler = nil;

implementation

procedure HandleException;
begin
  if @ExceptionHandler <> nil then ExceptionHandler;
end;

{ TIMemoryStream }

constructor TIMemoryStream.Create(AMemoryStream: TMemoryStream);
begin
  inherited Create;
  if AMemoryStream = nil then
    FMemoryStream := TMemoryStream.Create
  else FMemoryStream := AMemoryStream;
  FOwnStream := AMemoryStream = nil;
end;

destructor TIMemoryStream.Destroy;
begin
  if FOwnStream then FMemoryStream.Free;
  inherited Destroy;
end;

function TIMemoryStream.Read(var Buffer; Count: Longint): Longint;
begin
  try
    Result := FMemoryStream.Read(Buffer, Count);
  except
    HandleException;
    Result := 0;
  end;
end;

function TIMemoryStream.Write(const Buffer; Count: Longint): Longint;
begin
  try
    Result := FMemoryStream.Write(Buffer, Count);
  except
    HandleException;
    Result := 0;
  end;
end;

function TIMemoryStream.Seek(Offset: Longint; Origin: Word): Longint;
begin
  try
    Result := FMemoryStream.Seek(Offset, Origin);
  except
    HandleException;
    Result := -1;
  end;
end;

function TIMemoryStream.GetModifyTime: Longint;
begin
  Result := 0;
end;

{ TIFileStream }

constructor TIFileStream.Create(const FileName: string; Mode: Word);
begin
  inherited Create;
  FFileStream := TFileStream.Create(FileName, Mode);
end;

destructor TIFileStream.Destroy;
begin
  FFileStream.Free;
  inherited Destroy;
end;

function TIFileStream.Read(var Buffer; Count: Longint): Longint;
begin
  try
    Result := FFileStream.Read(Buffer, Count);
  except
    HandleException;
    Result := 0;
  end;
end;

function TIFileStream.Write(const Buffer; Count: Longint): Longint;
begin
  try
    Result := FFileStream.Write(Buffer, Count);
  except
    HandleException;
    Result := 0;
  end;
end;

function TIFileStream.Seek(Offset: Longint; Origin: Word): Longint;
begin
  try
    Result := FFileStream.Seek(Offset, Origin);
  except
    HandleException;
    Result := -1;
  end;
end;

function TIFileStream.GetModifyTime: Longint;
begin
  try
    Result := FileGetDate(FFileStream.Handle);
  except
    HandleException;
    Result := 0;
  end;
end;

{ TVirtualStream }

constructor TVirtualStream.Create(AIStream: TIStream);
begin
  inherited Create;
  FIStream := AIStream;
end;

function TVirtualStream.Read(var Buffer; Count: Longint): Longint;
begin
  Result := FIStream.Read(Buffer, Count);
end;

function TVirtualStream.Write(const Buffer; Count: Longint): Longint;
begin
  Result := FIStream.Write(Buffer, Count);
end;

function TVirtualStream.Seek(Offset: Longint; Origin: Word): Longint;
begin
  Result := FIStream.Seek(Offset, Origin);
end;

function TVirtualStream.GetModifyTime: Longint;
begin
  Result := FIStream.GetModifyTime;
end;

end.

