
{*******************************************************}
{                                                       }
{       Delphi Visual Component Library                 }
{       Editor for Filter Property (e.g. TOpenDialog)   }
{                                                       }
{       Copyright (c) 1995 Borland International        }
{                                                       }
{*******************************************************}

unit FiltEdit;

interface

uses WinTypes, WinProcs, Messages, Classes, Graphics, Forms, Controls, Tabs,
  Buttons, DsgnIntf, Grids, StdCtrls, ExtCtrls;

type
  TFilterEditor = class(TForm)
    OKBtn: TBitBtn;
    CancelBtn: TBitBtn;
    HelpBtn: TBitBtn;
    Bevel1: TBevel;
    procedure FormCreate(Sender: TObject);
    procedure HelpBtnClick(Sender: TObject);
  private
    procedure SetFilter(Value: string);
    function GetFilter: string;
  end;

  TFilterProperty = class(TStringProperty)
  public
    procedure Edit; override;
    function GetAttributes: TPropertyAttributes; override;
  end;

implementation

uses SysUtils, LibConst, LibHelp;


{$R *.DFM}

type
  TFilterGrid = class(TStringGrid)
  private
    FLimit: Integer;
  public
    function TotalChars: Integer;
    function GetEditLimit: Integer; override;
  end;

const
  NameCol = 0;
  FiltCol = 1;
  FirstRow = 1;

var
  FilterGrid: TFilterGrid;

{ TFilterGrid }

function TFilterGrid.TotalChars: Integer;
var
  R: Integer;
begin
  Result := 0;
  for R := FirstRow to RowCount - 1 do
    Result := Result + (Length(Cells[NameCol,R]) + Length(Cells[FiltCol,R]));
end;

function TFilterGrid.GetEditLimit: Integer;
begin
  Result := (FLimit - TotalChars) + Length(Cells[Col,Row]);
  if Result = 0 then Result := -1; { sets cell to read only }
end;

{ TFilterEditor }

procedure TFilterEditor.FormCreate(Sender: TObject);
begin
  HelpContext := hcDFilterEditor;
  FilterGrid := TFilterGrid.Create(Self);
  FilterGrid.BoundsRect := Bevel1.BoundsRect;
  with FilterGrid do
  begin
    ColCount := 2;
    FixedCols := 0;
    Font.Name :='MS Sans Serif';
    Font.Size := 8;
    Font.Style := [fsBold];
    RowCount := 25;
    ScrollBars := ssVertical;
    Options := [goFixedVertLine, goHorzLine, goVertLine, goEditing, goTabs, 
      goAlwaysShowEditor];
    FLimit := 240;
    Parent := Self;
    TabOrder := 1;
    ColWidths[NameCol] := ClientWidth div 2;
    ColWidths[FiltCol] := (ClientWidth div 2) - 1;
    DefaultRowHeight := Canvas.TextHeight('A');
    Cells[NameCol,0] := LoadStr(SFilterName);
    Cells[FiltCol,0] := LoadStr(SFilter);
  end;
  ActiveControl := FilterGrid;
end;

function TFilterEditor.GetFilter: string;

  function EmptyRow(r: Integer): Boolean;
  begin
    Result := True;
    with FilterGrid do
      if (Cells[NameCol,r] <> '') or (Cells[FiltCol,r] <> '') then
        Result := False;
  end;

var
  r: Integer;
begin
  Result := '';
  with FilterGrid do
  begin
    for r:=FirstRow to RowCount-1 do
    begin
      if not EmptyRow(r) then
      begin
        Result := Result+Cells[NameCol,r];
        Result := Result+'|';
        Result := Result+Cells[FiltCol,r];
        Result := Result+'|';
      end;
    end;
  end;
  r:=Integer(Result[0]);
  while Result[r] = '|' do
  begin
    Dec(Result[0]);
    r := r-1;
  end;
end;

procedure TFilterEditor.SetFilter(Value: string);
var
  Index: Byte;
  r, c: Integer;
begin
  if Value <> '' then
  begin
    r := FirstRow;
    c := NameCol;
    Index := Pos('|', Value);
    with FilterGrid do
    begin
      while Index > 0 do
      begin
        Cells[c, r] := Copy(Value, 1, Index - 1);
        if c = FiltCol then
        begin
          c := NameCol;
          if r = RowCount - 1 then
            RowCount := RowCount + 1;
          r := r + 1;
        end
        else c := FiltCol;
        Delete(Value, 1, Index);
        Index := Pos('|', Value);
      end;
      Cells[c, r] := Copy(Value, 1, Length(Value));
    end;
  end;
end;

{ TFilterProperty }

procedure TFilterProperty.Edit;
var
  temp: string;
  FilterEditor: TFilterEditor;
begin
  FilterEditor := TFilterEditor.Create(Application);
  try
    FilterEditor.SetFilter(GetValue);
    FilterEditor.ShowModal;
    if FilterEditor.ModalResult = mrOK then
      SetValue(FilterEditor.GetFilter);
  finally
    FilterEditor.Free;
  end;
end;

function TFilterProperty.GetAttributes: TPropertyAttributes;
begin
  Result := [paDialog];
end;

procedure TFilterEditor.HelpBtnClick(Sender: TObject);
begin
  Application.HelpContext(HelpContext);
end;

end.

