{
  RC resource script to Delphi form converter.

  Copyright (c) 1995 by Borland International
}

unit Rcexpert;

interface

uses
  SysUtils, WinTypes, WinProcs, Messages, Classes, Graphics, Controls,
  Forms, Dialogs, StdCtrls, Buttons, ExtCtrls, FileCtrl,
  Proxies, ExptIntf, ToolIntf, VirtIntf, IStreams, RCStatus, wincrt, Menus;

type
  TRCExpertDlg = class(TForm)
    Notebook1: TNotebook;
    Help: TBitBtn;
    Prev: TBitBtn;
    Next: TBitBtn;
    Image1: TImage;
    DirPath: TLabel;
    FileName: TEdit;
    Files: TFileListBox;
    Filters: TFilterComboBox;
    Drives: TDriveComboBox;
    Dirs: TDirectoryListBox;
    DirLabel: TLabel;
    IncludePath: TEdit;
    ShowForms: TCheckBox;
    Convert: TBitBtn;
    procedure NextClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure Notebook1PageChanged(Sender: TObject);
    procedure PrevClick(Sender: TObject);
    procedure FileNameKeyDown(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure FileNameKeyPress(Sender: TObject; var Key: Char);
    procedure FileNameChange(Sender: TObject);
    procedure FormDestroy(Sender: TObject);
    procedure FilesDblClick(Sender: TObject);
    procedure FilesClick(Sender: TObject);
    procedure HelpClick(Sender: TObject);
  private
    { Private declarations }
  public
    { Public declarations }
    function ValidFilename: Boolean;
  end;

  TRCExpert = class(TIExpert)
  public  { Methods required by Delphi to define the expert }
    function GetName: string; override;
    function GetComment: string; override;
    function GetGlyph: HBITMAP; override;
    function GetStyle: TExpertStyle; override;
    function GetState: TExpertState; override;
    function GetIDString: string; override;
    procedure Execute; override;
  public  { Additional items used to implement this expert }
    ErrorLog: TMemoryStream;
    procedure LogError(Sender: TObject; const Msg: string);
    procedure TransferErrorLog;
  private
    FileNames: TStringList;
    ModuleFlags: TCreateModuleFlags;
    StatusDlg: TRCCompileStatus;
    procedure CompileNextFile(Sender: TObject);
  end;

  TRCExpertOnHelpMenu = class(TRCExpert)
  public  { Methods required by Delphi to define the expert }
    function GetName: String; override;
    function GetStyle: TExpertStyle; override;
    function GetMenuText: String; override;
    function GetIDString: String; override;
  end;

procedure Register;

implementation

uses RCImport, DFM2PAS;

{$R *.DFM}
{$R rcexpert.res}

{ TRCExpert }
function TRCExpert.GetName: string;
begin
  Result := 'Resource Expert';
end;

function TRCExpert.GetComment: string;
begin
  Result := 'Convert RC resource scripts into Delphi forms and menus';
end;

function TRCExpert.GetGlyph: HBITMAP;
begin
  Result := LoadBitmap(HInstance, 'RCExpertBMP');
end;

function TRCExpert.GetStyle: TExpertStyle;
begin
  Result := esForm;
end;

function TRCExpert.GetState: TExpertState;
begin
  Result := [esEnabled];
end;

function TRCExpert.GetIDString: string;
begin
  Result := 'Borland.ResourceExpert';
end;

procedure TRCExpert.Execute;
var
  X: Integer;
  RCExpertDlg: TRCExpertDlg;
begin
  try
    FileNames := nil;
    try
      RCExpertDlg := TRCExpertDlg.Create(Application);
      try
        if (RCExpertDlg.ShowModal <> idOk) then Exit;
        ModuleFlags := [cmAddToProject, cmShowSource, cmUnNamed, cmMarkModified];
        if RCExpertDlg.ShowForms.Checked then
          Include(ModuleFlags, cmShowForm);
        ErrorLog := nil;
        FileNames := TStringList.Create;
        if RCExpertDlg.Files.SelCount = 0 then
          FileNames.Add(RCExpertDlg.Filename.Text)
        else
        begin
          for X := 0 to RCExpertDlg.Files.Items.Count-1 do
            if RCExpertDlg.Files.Selected[X] then
              FileNames.Add(RCExpertDlg.Files.Items[X]);
        end;
        RCIncludePath := RCExpertDlg.IncludePath.Text;
      finally
        RCExpertDlg.Free;
      end;

      StatusDlg := TRCCompileStatus.Create(Application);
      try
        StatusDlg.Execute(CompileNextFile);
      finally
        StatusDlg.Free;
        TransferErrorLog;
      end;
    finally
      Filenames.Free;
    end;
  except
  on E: Exception do
    if Assigned(ToolServices) then
      ToolServices.RaiseException(E.Message)
    else
      Raise;
  end;
end;

procedure TRCExpert.CompileNextFile(Sender: TObject);

  procedure ConvertFile(const RCName: string);
  var
    UnitIdent, FormIdent: TIDentString;
    FileName: TFileName;

    procedure GetNames;
    begin
      if Assigned(ToolServices) then
      begin
        if not ToolServices.GetNewModuleName(UnitIdent, FileName) then
          raise Exception.Create('Can''t create new module');
      end
      else
      begin
        UnitIdent := 'unit1';
        Filename := 'unit1.pas';
      end;
      FormIdent := 'Form' + Copy(UnitIdent, 5, 255);
    end;

  var
    R: TRCConverter;
    List : TStringList; { principal resources in RC project, by resource id }
    XRefs: TStringList; { list of controls referring to icons, bmps, etc.   }
    PAS, DFM: TIMemoryStream;
    X, Index, Rounds: Integer;
    Frm: TForm;
    Obj: TObject;
    Success: Boolean;
  begin
    DFM := nil;
    PAS := nil;
    List:= nil;
    XRefs:=nil;
    R := TRCConverter.Create(RCName);
    R.OnStatus := StatusDlg.ShowStatus;
    StatusDlg.ShowStatus(nil, stProjectName, ExpandFileName(RCName), 0);
    R.OnError := LogError;
    try
      DFM := TIMemoryStream.Create(TMemoryStream.Create);
      DFM.OwnStream := True;
      PAS := TIMemoryStream.Create(TMemoryStream.Create);
      PAS.OwnStream := True;
      List := TStringList.Create;
      XRefs := TStringList.Create;
      R.ConvertResources(List, XRefs);
      if List = nil then Exit;
      for X := 0 to XRefs.Count - 1 do
      begin
        { Resolve cross-reference links }
        Index := List.IndexOf(XRefs[X]);
        if Index <> -1 then
        begin
          Obj := XRefs.Objects[x];
          if Obj is TBitbtn then
            TBitBtn(Obj).Glyph.Assign(List.Objects[Index] as TGraphic)
          else
          if Obj is TImage then
            TImage(Obj).Picture.Assign(List.Objects[Index] as TGraphic);
        end;
      end;
      StatusDlg.ShowStatus(nil, stActionLabel, 'Submitting: ', 0);
      for X := 0 to List.Count - 1 do
      begin
        if not (List.Objects[x] is TForm) then Continue;
        { fixup form, unit name }
        Frm := TForm(List.Objects[x]);
        Rounds := 0;
        Success := True;
        { Try to use the RC dialog identifier for the form name and
          form class name (already set) in the Delphi project.  If
          the CreateModule call fails, presumeably due to a name conflict,
          discard the RC names and get guaranteed unique (but less meaningful)
          names from the IDE. }
        GetNames; { Get new unit and form file names }
        repeat
          if Frm.Name = '' then
          begin
            TProxyForm(Frm).RenameClass('T'+FormIdent);
            Frm.Name := FormIdent;
          end;
          StatusDlg.ShowStatus(nil, stFileName, Frm.Name, 0);
          DFM.MemoryStream.Position := 0;
          DFM.MemoryStream.WriteComponentRes(Frm.Name, Frm);
          DFM.MemoryStream.Position := 0;
          PAS.MemoryStream.SetSize(0);
          FormSourceFromResource(UnitIdent, DFM.MemoryStream, PAS.MemoryStream);
          DFM.MemoryStream.Position := 0;
          PAS.MemoryStream.Position := 0;
          if Assigned(ToolServices) then
            Success := ToolServices.CreateModule(FileName, PAS, DFM, ModuleFlags);
          if not Success then
          begin
            LogError(Self, 'Error encountered adding form "'+Frm.Name+'" to current project,');
            LogError(Self, '  possibly due to a name conflict with another form in the project.');
            Frm.Name := '';  { force the form class to be renamed next loop }
            GetNames;
            LogError(Self,'Renaming form to "'+FormIdent+'".');
          end;
          Inc(Rounds);
        until Success or (Rounds > 2);
        if (Rounds > 2) then
          LogError(Self, 'Unable to add form "'+Frm.Name+'" to current project.');
        StatusDlg.BringToFront;
        Application.ProcessMessages;
      end;
    finally
      PAS.Free;
      DFM.Free;
      R.Free;
      XRefs.Free;  { XRefs does not own its contents }
      if List <> nil then    { but List does }
      begin
        for X := 0 to List.Count-1 do
          TObject(List.Objects[x]).Free;
        List.Free;
      end;
    end;
  end;

var
  X: Integer;
begin
  for X := 0 to Filenames.Count-1 do
    ConvertFile(Filenames[X]);
end;


procedure TRCExpert.LogError(Sender: TObject; const Msg: string);
const
  CRLF: string[2] = #13#10;
begin
  if ErrorLog = nil then
  begin
    ErrorLog := TMemoryStream.Create;
    LogError(nil, 'Error log for RC project: '+StatusDlg.RCFile.Caption);
  end;
  ErrorLog.Write(Msg[1], Length(Msg));
  ErrorLog.Write(CRLF[1], 2);
end;

procedure TRCExpert.TransferErrorLog;
var
  IStrm: TIMemoryStream;
begin
  if (ErrorLog = nil) then Exit;
  ErrorLog.Position := 0;
  IStrm := TIMemoryStream.Create(ErrorLog);
  try
    IStrm.OwnStream := True;
    if ToolServices = nil then Exit;
    ToolServices.CreateModule('ErrorLog.txt', IStrm, nil,
         [cmShowSource, cmUnNamed, cmMarkModified]);
  finally
    IStrm.Free;
  end;
end;

{ GetName and GetIDString MUST return unique strings. }
function TRCExpertOnHelpMenu.GetName: String;
begin
  Result := 'Resource Expert (help)';
end;

function TRCExpertOnHelpMenu.GetIDString: String;
begin
  Result := inherited GetIDString + '.1';
end;

function TRCExpertOnHelpMenu.GetStyle: TExpertStyle;
begin
  Result := esStandard;
end;

function TRCExpertOnHelpMenu.GetMenuText: String;
begin
  Result := '&Resource Expert...';
end;



procedure Register;
begin
  RegisterLibraryExpert(TRCExpert.Create);
  RegisterLibraryExpert(TRCExpertOnHelpMenu.Create);
end;

{ TRC2DFM form }

procedure TRCExpertDlg.FormCreate(Sender: TObject);
var
  Buf: array [0..255] of Char;
begin
  Image1.Picture.Bitmap.Handle := LoadBitmap(HInstance, 'RCExpertBMP');
  Notebook1.PageIndex := 0;
  GetPrivateProfileString('Resource Expert','RCIncludePath','',Buf,Sizeof(Buf), 'Delphi.ini');
  IncludePath.SetTextBuf(Buf);
end;

procedure TRCExpertDlg.FormDestroy(Sender: TObject);
var
  Buf: array [0..255] of Char;
begin
  IncludePath.GetTextBuf(Buf, SizeOf(Buf));
  WritePrivateProfileString('Resource Expert','RCIncludePath',Buf,'Delphi.ini');
end;

procedure TRCExpertDlg.NextClick(Sender: TObject);
begin
  if (Notebook1.PageIndex <> 1) or ValidFilename then
    Notebook1.PageIndex := Notebook1.PageIndex + 1
  else
    Files.ApplyFilePath(Filename.Text);
end;

procedure TRCExpertDlg.Notebook1PageChanged(Sender: TObject);
begin
  case Notebook1.PageIndex of
    0 : begin
          Prev.Enabled := False;
          Next.Enabled := True;
        end;
    1 : begin
          Prev.Enabled := True;
          Next.Enabled := ValidFilename;
          Filename.SetFocus;
        end;
    2 : begin
          Next.Visible := True;
          Convert.Visible := False;
        end;
    3 : begin
          Convert.Visible := True;
          Next.Visible := False;
        end;
  end;
end;

procedure TRCExpertDlg.PrevClick(Sender: TObject);
begin
  Notebook1.PageIndex := Notebook1.PageIndex - 1;
end;

function TRCExpertDlg.ValidFilename: Boolean;
var
  DirInfo: TSearchRec;
begin
  Result := False;
  if (Pos('*',Filename.Text) <> 0) or (Pos(';',Filename.Text) <> 0) then Exit;
  if FindFirst(Filename.Text, faAnyFile, DirInfo) = 0 then
  begin
    Result := (DirInfo.Attr and (faDirectory or faVolumeID)) = 0;
    FindClose(DirInfo);
  end;
end;

procedure TRCExpertDlg.FileNameKeyDown(Sender: TObject; var Key: Word;
  Shift: TShiftState);
begin
  if (Key = vk_Return) and (Shift = []) then
    Files.ApplyFilePath(Filename.Text);
  SendMessage(Files.Handle, lb_SetSel, 0, -1);
end;

procedure TRCExpertDlg.FileNameKeyPress(Sender: TObject; var Key: Char);
begin
  if Key = #13 then Key := #0;
end;

procedure TRCExpertDlg.FileNameChange(Sender: TObject);
begin
  Next.Enabled := ValidFilename;
end;

procedure TRCExpertDlg.FilesDblClick(Sender: TObject);
begin
  Next.Click;
end;

procedure TRCExpertDlg.FilesClick(Sender: TObject);
begin
  FileName.Text := Files.Items[Files.ItemIndex];
end;

procedure TRCExpertDlg.HelpClick(Sender: TObject);
begin
  winprocs.WinHelp(Handle, 'RCEXPERT.HLP', Help_Context, 10);
end;

end.
