/***********************************************************************
IE-EX3.C: Import/Export example program #3
by MSW
Copyright (c) 1992 by cc:Mail, Inc.  All rights reserved.

This source code, any derivatives of it, and the resulting object code
may be freely used and/or copied, so long as this copyright notice is
included and remains unmodified.  cc:Mail, Inc. does not make any
representation or warranty, express or implied with respect to this
source code.  cc:Mail, Inc. disclaims any liability whatsoever for any
use of such code.

************************************************************************

    This program is an example of how to write an Import/Export message
    format file so that the ITEMSIZE parameter of Import can be used.
    This is recommended when the contents of a text item may contain a
    keyword (e.g. "Message:").  In particular, this program shows how
    to write the sizes of the various parts of the message without
    calculating all the sizes beforehand.

    For this example, the message is written to the file CCMAIL.IMP,
    has a single recipient, and has a variable number of text items.
    The usage is:
        ie-ex3 author recip txtfil1 [txtfil2 [txtfil3 ...]]

            where author   is the message author (enclose in quotes if
                           it contains spaces)
                  recip    is the single message recipient (enclose in
                           quotes if it contains spaces)
                  txtfil1  is the file containing primary text item
                  txtfil2+ are the files containing secondary text items
                           (optional)

    IE-EX3 returns an ERRORLEVEL of 0 for success and 1 for any error.

    Note that no testing is done on the specified files to determine if
    they are in fact valid text items.  This could be added to a
    production program.

    Build with Microsoft C version 5.1 or later using any memory model.
***********************************************************************/


// Include file(s)
#include <fcntl.h>
#include <io.h>
#include <share.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys\types.h>
#include <sys\stat.h>

// Define constants
#define BUFFSIZE    1024        // Size of buffer for copying text items

// Define file statics
static char allocSpace[] = "          \r\n";
                                // Spaces to write section size over
static int hOut = -1;           // File handle for output file

// Declare prototypes
int copyFile(char *name);
void endSection(long *posP);
void startSection(long *posP);
void writeStr(char *str);


/***********************************************************************
IE-EX3 main routine (described above)
***********************************************************************/
int main(int argc, char *argv[])
{
    int i;
    long posMsg, posCnt, posItm;

    // Check argument count
    if (argc < 4) {
        fprintf(stderr,
            "Usage: ie-ex3 author recip txtfil1 [txtfil2 ...]\n");
        return  1;
    }

    // Open output file (binary mode)
    if ((hOut = sopen("CCMAIL.IMP",
            O_WRONLY | O_BINARY | O_CREAT | O_TRUNC, SH_DENYWR,
            S_IREAD | S_IWRITE)) == -1) {
        fprintf(stderr, "ie-ex3: Error opening output file\n");
        return  1;
    }

    // Do the work
    writeStr("Message:\r\n");
    startSection(&posMsg);
    writeStr("From: ");
    writeStr(argv[1]);
    writeStr("\r\nTo: ");
    writeStr(argv[2]);
    writeStr("\r\nSubject: IE-EX3 program\r\nContents:\r\n");
    startSection(&posCnt);
    for (i=3; i<argc; i++) {
        writeStr("\r\nText item: From file ");
        writeStr(argv[i]);
        writeStr("\r\n");
        startSection(&posItm);
        if (!copyFile(argv[i])) {
            break;
        }
        endSection(&posItm);
    }
    endSection(&posCnt);
    endSection(&posMsg);

    // Close output file (ignore errors) and exit with no error
    close(hOut);
    return  0;
} // main


/***********************************************************************
Routine    : copyFile
Description: Copy the specified file to the output file.  Both files are
           : accessed in binary mode (it is assumed that the output file
           : is already opened in binary mode).
Inputs     : name       input file specification (null-terminated)
Uses       : hOut
Returns    : Zero (false) if the input file cannot be opened
***********************************************************************/
int copyFile(char *name)
{
    char buff[BUFFSIZE];
    int hIn, readCnt;

    // Open input file (binary mode)
    if ((hIn = sopen(name, O_RDONLY | O_BINARY, SH_DENYNO)) == -1) {
        fprintf(stderr, "ie-ex3: Error opening %s\n", name);
        return  0;
    }

    // Copy input file to output file (ignore read problems)
    while ((readCnt = read(hIn, buff, sizeof(buff))) > 0) {
        if (write(hOut, buff, readCnt) != readCnt) {
            fprintf(stderr, "ie-ex3: Error writing to output file\n");
            close(hIn);
            close(hOut);
            exit(1);
        }
    }

    // Close input file (ignore errors) and return success
    close(hIn);
    return  1;
} // copyFile


/***********************************************************************
Routine    : endSection
Description: Go back and write the size of the section started when
           : startSection was called.  This function and startSection
           : must be called in matching pairs with the same argument.
Inputs     : posP       same pointer passed to startSection
Uses       : hOut
***********************************************************************/
void endSection(long *posP)
{
    char tmp[sizeof(allocSpace)];
    long curPos;

    // Remember current position
    if ((curPos = lseek(hOut, 0L, SEEK_CUR)) == -1L) {
        fprintf(stderr, "ie-ex3: Error determining file position\n");
        close(hOut);
        exit(1);
    }

    // Return to file position to write section size
    if (lseek(hOut, *posP, SEEK_SET) == -1L) {
        fprintf(stderr, "ie-ex3: Error seeking output file\n");
        close(hOut);
        exit(1);
    }

    // Write the section size
    sprintf(tmp, "%lu",
        curPos - *posP - strlen(allocSpace));
    writeStr(tmp);

    // Restore position
    if (lseek(hOut, curPos, SEEK_SET) == -1L) {
        fprintf(stderr, "ie-ex3: Error seeking output file\n");
        close(hOut);
        exit(1);
    }
} // endSection


/***********************************************************************
Routine    : startSection
Description: Remember where the start of a section is and allocate space
           : to write the section size later.  This function and
           : endSection must be called in matching pairs with the same
           : argument.
Inputs     : posP       pointer to a variable in which to save the
           :            starting position of the section
Uses       : hOut
***********************************************************************/
void startSection(long *posP)
{
    // Remember current file offset
    if ((*posP = lseek(hOut, 0L, SEEK_CUR)) == -1L) {
        fprintf(stderr, "ie-ex3: Error determining file position\n");
        close(hOut);
        exit(1);
    }

    // Allocate space in output file to write size later
    writeStr(allocSpace);
} // startSection


/***********************************************************************
Routine    : writeStr
Description: Write the specified string to the output file, checking for
           : errors.
Inputs     : str        string to write (null-terminated)
Uses       : hOut
***********************************************************************/
void writeStr(char *str)
{
    unsigned len;

    len = strlen(str);
    if (write(hOut, str, len) != len) {
        fprintf(stderr, "ie-ex3: Error writing to output file\n");
        close(hOut);
        exit(1);
    }
} // writeStr

// End of IE-EX3.C

