#include <string.h>
#include <scl1.h>
#include <scl1keys.h>
#include <scl1clor.h>

/**************************************************************************
	Shows the use of the Fields2 and FieldCheck functions and how to create
	an user defined field check function.

	The data entry field has a Select type field, one LineEditor and two
	mouse buttons.

	The field check function will:

		1) Check for F1 key to display context sensitive help.
		2) Check if user has selected YES or NO. The LineEditor
		   field color is grayed in the case of NO.
		3) Keep the cursor off the LineEditor field if the user has
		   selected NO and the LineEditor field is grayed out.
		4) Check before exit if cancel is not selected and YES has been
		   selected if the user has fill out the LineEditor field.  */

int OurFieldCheck(FData2 *fd2);   /* our CheckF prototype */

int Color1=WHITE_BLUE;            /* Colors are initialized */
int Color2=BLACK_CYAN;            /* for a color monitor */
int Color3=BROWN_BLACK+HIGHLIGHT;

/* Our first field will be a Select type */

SData1 sd1[]={           /* This is Select's first structure */
	 11,37,"YES",
	 11,47,"NO",
	 0};

/* Select's secondary structure: */

SData2 sd2={23,11,16,"Create backup files:",0,0,0,0};

/* Our second field will be a LineEditor type
   This structure defines the LineEditor configuration */

LEData led={23,13,16,"Backup Directory:",112,13,34,30,30,
	 CC_PATH+CC_CAPITALIZE, 0,0,0,0,0,0,2,1,0,0,0,0,0,0,0};

/* The next two fields will be of the MouseButton type. */

MBData mb1={23,112,15,28,15,33,15,28," OK ",0,0,0,0};

MBData mb2={23,112,15,42,15,51,15,42," CANCEL ",0,0,0,0};

FData1 fd1[]={   /* This is the FData1 structure: */
	 SELECT,sd1,&sd2,OurFieldCheck,
	 LINE_EDITOR,&led,0,OurFieldCheck,
	 MOUSE_BUTTON,&mb1,0,OurFieldCheck,
	 MOUSE_BUTTON,&mb2,0,OurFieldCheck,
	 0};

char F1Text[]=" Press F1 for Help ";

main()
{
FData2 fd2; /* Fields2 secondary structure */

if(Video()==MONO)         /* Verify the type of display and change */
	{                     /* colors if necessary */
	Color1=WHITE_BLACK;
	Color2=BLACK_WHITE;
	Color3=BLACK_WHITE+HIGHLIGHT;
	}

/* Clear the screen and draw a box */

Cls(Color1,10,13,16,66);
Box(Color1,0,10,13,16,66);

WriteScreen(Color1,16,Center(F1Text),F1Text);

/* Call Fields2 with a F_INIT message */

Fields2(F_INIT,fd1,&fd2);

/* allocate buffer for LineEditor field */

Fields2(F_ALLOC,fd1,&fd2);

/* Tell Fields2 to modify colors to new values */

Fields2(F_COLORS,fd1,&fd2,Color1,Color2);

/* Draw fields */

Fields2(F_DRAW,fd1,&fd2);

InitMouse(IM_SHOW);

/* Activate, now our CheckF will be in control of program flow */

Fields2(F_ACTIVE,fd1,&fd2);

/* Check the FData2 structure to see if the user presses ESCAPE
   or selects the CANCEL mouse button for exiting */

if(fd2.ActiveField==3 || fd2.EventInfo==ESC)

	exit(-1);   /* CANCEL selected */
else
	exit(0);    /* OK selected */
}

/* This is the custom field check function */

int OurFieldCheck(FData2 *fd2)
{
int Mess;

/* Let SCL1 FieldCheck function do most of the work */

Mess=FieldCheck(fd2);

/* 1) We will watch the F1 help key. It will be reported by Fields2 with an
   ILLEGAL_KEY message in the FData2 EventInfo structure element.*/

if(fd2->Message == LE_ILLEGAL_KEY && fd2->EventInfo==F1)
	{

	/* display help, fd2->ActiveField = active field number */

	Help(fd2->ActiveField);

	/* return POSITION_STAY message so that the cursor is not moved
	   to another field */

	Mess=F_POSITION_STAY;
	}

/* 2) Check if the Select field is active and if a S_NEW_POSITION message,
	  was send by Select. Thsi message is received each time the user
	  changes his selection. We'll check current position to validate
	  or invalidate the LineEditor field */

else if(fd2->ActiveField==0 && fd2->Message==S_NEW_POSITION)
	{
	if(sd2.Position==0)			/* YES selected */
		{
		led.PColor=Color1;
		led.FColor=Color2;		/* normal colors */
		}
	else

	/* NO selected, use gray color */

		{
		led.PColor=BLACK_BLUE + HIGHLIGHT;
		led.FColor=BLACK_BLACK + HIGHLIGHT;
		}

	LineEditor(LE_DRAW,&led);	/* redraw LineEditor */
	}

/* 3) The FIELDSTART message is send each time a new field receives control
   We'll check this message and if the LineEditor field is active we'll
   check its color. If it is set to gray color it means that the user has
   selected not to create backup files. In this case we'll move to the
   next or previous field. By checking the fd2->EventInfo we'll know if
   the user had previously pressed TAB (for the next field) or SHIFT+TAB */

else if(fd2->ActiveField==1 && fd2->Message==F_FIELDSTART)
	{
	if(led.FColor==BLACK_BLACK + HIGHLIGHT)
		{
		if(fd2->EventInfo==TAB)
			fd2->ActiveField=2;      /* next field */
		else
			fd2->ActiveField=0;		/* previous field */
		Mess=F_SET_POSITION;
		}
	}

/* 4) When the user wants to exit we'll check if a string has been entered
   in the LineEditor field if Select is set to YES and the OK mouse
   button was selected. We know we are about to exit fields when FieldCheck
   returns and F_EXIT message. We'll also check if the key that was
   used to exit was ESCAPE or the CANCEL button was selected, in these
   cases the user has selected to cancel the operation */

else if(Mess==F_EXIT && fd2->EventInfo != ESC && fd2->ActiveField != 3)
	{

	/* if Select is in YES position and no directory has been entered */

	if(sd2.Position==0 && strlen(led.Buffer)==0)
		{

		/* display error, change message to POSITION_STAY and move the
		cursor to the LineEditor field */

		MessageOn(Color3,"Directory must be entered!");
		Mess=F_POSITION_STAY;
		fd2->ActiveField=1;
		WaitKeyMouse();
		MessageOff();
		}
	}

 /* Mess holds the message Fields2 will receive, a POSITION_STAY message
	or the message returned by FieldCheck */

return(Mess);
}

char AllHelp[]=" TAB-next  SHIFT TAB-previous ";

Help(int field)
{

/* save cursor status and turn it off */

PushCursor();
CursorOff();

/* field tells us the field number that is active, we will
   use it to display context sensitive help */

switch(field)
	{
	case 0: /* first field */

		MessageOn(Color3,"Use arrow keys to select if you\nwant to createbackup files.");
		break;

	case 1: /* second field */

		MessageOn(Color3,"Type Backup files directory.");
		break;

	case 2: /* third field */

		MessageOn(Color3,"Press ENTER to validate\nyour selection.");
		break;

	case 3: /* fourth field */

		MessageOn(Color3,"Press ENTER or ESC to\ncancel your selection.");
		break;
	}

WriteScreen(Color3,12,Center(AllHelp),AllHelp);

 /* wait for mouse or keystroke, remove message, restore cursor */

WaitKeyMouse();
MessageOff();
PopCursor();
}
