/********************************************************************/ 
/*  Licensed Materials - Property of IBM                            */ 
/*                                                                  */ 
/*                                                                  */ 
/* Copyright (C) International Business Machines Corp., 1994.       */ 
/* Copyright (C) Apple Computer, Inc., 1994                         */ 
/*                                                                  */ 
/*  US Government Users Restricted Rights -                         */ 
/*  Use, duplication, or disclosure restricted                      */ 
/*  by GSA ADP Schedule Contract with IBM Corp.                     */ 
/*                                                                  */ 
/********************************************************************/ 
/*
   File:    XMPPart.h

   Contains:   Class definition for XMPPart

   Written by: Joshua Susser

   Protocols:  DragAndDrop, Embedding, Frames, Imaging, Linking,
            Memory Management, Part Activation, Storage, UI Events, Undo

*/

#ifndef _PART_
#define _PART_

#ifndef _PLFMDEF_
#include "PlfmDef.h"
#endif

#ifndef _PSTOBJ_
#include "PstObj.h"
#endif

#ifndef _XMPTYPES_
#include "XMPTypes.h"
#endif

#ifndef _DRAFT_
#include "Draft.h"   // For XMPDraftKey
#endif


//==============================================================================
// Theory of Operation
//==============================================================================

/*
   This file defines class XMPPart.  Part behavior is implemented in Part
   Handlers.  A given Handler may handle many Parts, but there is one XMPPart
   instance for each individual Part.
*/

//==============================================================================
// Constants
//==============================================================================

const XMPID kNewGroup = 0;

//==============================================================================
// Classes defined in this interface
//==============================================================================

class XMPPart;
class XMPEmbeddedFramesIterator;

//==============================================================================
// Classes used by this interface
//==============================================================================

class XMPCanvas;
class XMPFacet;
class XMPFrame;
class XMPLink;
class XMPPersistentObject;
class XMPShape;
class XMPStorageUnit;
class XMPStorageUnitView;
class XMPTransform;
class XMPDragItemIterator;
class XMPSession;
class XMPTypeSet;

struct XMPPoint;

//==============================================================================
// XMPPart
//==============================================================================

#define kXMPPartID "appl:part$class,1.0.0"

class XMPPart : public XMPPersistentObject
{
   friend class XMPDraft;
   friend class XMPEmbeddedFramesIterator;

public:

   //-------------------------------------------------------------------------
   // From DragAndDrop protocol
   //-------------------------------------------------------------------------

   XMPVMethod void FulfillPromise(XMPStorageUnitView *promiseSUView)
      = 0;
   
      // fulfills a promise which has been put out.
   
   XMPVMethod void DropCompleted(XMPPart* destPart, XMPDropResult dropResult)
      = 0;
      
      // NotifyDropComplete is a peculiar method. If DragAndDrop::StartDrag 
      // is synchronous, the return code to DragAndDrop::StartDrag is the 
      // drop result. Part::NotifyDropComplete does not need to be called. 
      // However, if it is required that DragAndDrop::StartDrag returns
      // immediately, there needs to be a way to notify the result of 
      // the drop.

   XMPVMethod MRESULT DragEnter(XMPDragItemIterator* dragInfo,   //dwf
                       XMPFacet* facet,
                       XMPPoint where)
      = 0;

      // the mouse has entered a frame of this part during a drag

   XMPVMethod MRESULT DragWithin(XMPDragItemIterator* dragInfo,   //dwf
                        XMPFacet* facet,
                        XMPPoint where)
      = 0;

      // the mouse has moved in a frame of this part during a drag

   XMPVMethod void DragLeave(XMPFacet* facet,
                       XMPPoint where)
      = 0;

      // the mouse has left a frame of this part during a drag

   XMPVMethod XMPDropResult Drop(XMPDragItemIterator* dropInfo,
                          XMPFacet* facet,
                          XMPPoint where)
      = 0;

      // dragging finished, something was dropped in a frame of this part

   //-------------------------------------------------------------------------
   // From Embedding protocol
   //-------------------------------------------------------------------------

   XMPVMethod void ContainingPartPropertiesChanged(XMPFrame* frame,
                                       XMPStorageUnit* propertyUnit)
      = 0;

      // Used by my ContainingPart to inform me of changes to content
      // properties of its proxy for me.

   // ********* for containing parts *********

   XMPVMethod XMPStorageUnit* GetContainingPartProperties(XMPFrame* frame)
      = 0;

      // Used by an embedded Part to ask me of changes to content
      // properties of my proxy for it.

   XMPVMethod void RevealFrame(XMPFrame* embeddedFrame)
      = 0;
      
      // scroll my contents to reveal an embedded frame.
      // ask my containing part to help if I'm not revealed yet.

   XMPVMethod void EmbeddedFrameSpec(XMPFrame* embeddedFrame, XMPObjectSpec* spec)
      = 0;
      
      // construct an object specifier describing an embedded frame.
      // ask my containing part to help by constructing a specifier
      // for my display frame

   XMPVMethod XMPEmbeddedFramesIterator* CreateEmbeddedFramesIterator()
      = 0;

      // create and return an iterator for my embedded frames list

   //-------------------------------------------------------------------------
   // From Frames protocol
   //-------------------------------------------------------------------------

   XMPVMethod void AddDisplayFrame(XMPFrame* frame)
      = 0;

      // add a new frame viewing the part

   XMPVMethod void AttachSourceFrame(XMPFrame* frame, XMPFrame* sourceFrame)
      = 0;

      // If sourceFrame is not NULL, this part should replicate the embedded
      // frames of sourceFrame in the new frame

   XMPVMethod void RemoveDisplayFrame(XMPFrame* frame)
      = 0;

      // remove a frame viewing the part

   XMPVMethod void CloseDisplayFrame(XMPFrame* frame)
      = 0;

      // close embedded frames in specified display frame

   XMPVMethod void FrameShapeChanged(XMPFrame* frame)
      = 0;

      // inform the part that a view of it has been reshaped

   XMPVMethod void ViewTypeChanged(XMPFrame* frame)
      = 0;

      // change the type of view in one of my frames
      // may be used to collapse a "frame view" into an "icon view"

   XMPVMethod void PresentationChanged(XMPFrame* frame)
      = 0;

      // change the kind of presentation in one of my frames
      // used to distinguish between content/palette/dialog/etc.
      // part may correct frame's presentation if not supported

   XMPVMethod void WritePartInfo(XMPPtr partInfo,
                          XMPStorageUnitView* storageUnitView)
      = 0;

      // write out the partInfo data onto a pre-focused StorageUnitView

   XMPVMethod XMPPtr ReadPartInfo(XMPFrame* frame, XMPStorageUnitView* storageUnitView)
      = 0;

      // read the partInfo data from a pre-focused StorageUnitView

   XMPVMethod XMPID Open(XMPFrame* frame)
      = 0;

      // open a window and add a display frame for me into it
      // if frame == nil, open in a new frame
      // return ID of the window created

   //---------------------------------
   // for containing parts

   XMPVMethod XMPFrame* CreateEmbeddedFrame(XMPFrame* containingFrame,
                                  XMPShape* frameShape,
                                  XMPTransform* externalTransform,
                                  XMPPart* embedPart,
                                  XMPTypeToken viewType,
                                  XMPTypeToken presentation,
                                  XMPID frameGroupID,
                                  XMPBoolean isOverlaid)
      = 0;

      // create a new frame and embed a part in it.
      
   XMPVMethod void RemoveEmbeddedFrame(XMPFrame* embeddedFrame)
      = 0;

      // remove a frame viewing an embedded part

   XMPVMethod XMPShape* RequestFrameShape(XMPFrame* embeddedFrame, XMPShape* frameShape)
      = 0;

      // an embedded part asks to change the shape of its frame

   XMPVMethod void UsedShapeChanged(XMPFrame* embeddedFrame)
      = 0;

      // inform the part that a view of it has been reshaped

   XMPVMethod XMPShape* AdjustBorderShape(XMPFrame* embeddedFrame, XMPShape* shape)
      = 0;

      // adjust the display of the active frame border of an
      // embedded frame.  shape param is requsted shape. return
      // value is that shape obscured by other contents.

   //-------------------------------------------------------------------------
   // From Facet protocol
   //-------------------------------------------------------------------------

   XMPVMethod void FacetAdded(XMPFacet* facet)
      = 0;

      // inform the part that a facet has been added to one of its frames

   XMPVMethod void FacetRemoved(XMPFacet* facet)
      = 0;

      // inform the part that a facet has been removed from one of its frames

   XMPVMethod void ClipShapeChanged(XMPFacet* facet)
      = 0;

      // inform the part that a view of it has been re-clipped

   XMPVMethod void ExternalTransformChanged(XMPFacet* facet)
      = 0;

      // inform the part that a view of it has been re-positioned

   //-------------------------------------------------------------------------
   // From Imaging protocol
   //-------------------------------------------------------------------------

   XMPVMethod void Draw(XMPFacet* facet, XMPShape* invalidShape)
      = 0;

      // image the part in the specified facet of one of its frames
      // only pixels in the invalidShape need to be painted

   XMPVMethod void CanvasUpdated(XMPCanvas* canvas)
      = 0 ;

      // the canvas has been drawn on asynchronously. this part is
      // the owner of the canvas, so it must copy the bits up to
      // the parent canvas.

   XMPVMethod void HighlightChanged(XMPFrame* frame)
      = 0 ;

      // notification that the highlight state of a display frame
      // of this part has been changed.  part can get new value
      // with frame->GetHightlight().

   XMPVMethod XMPULong GetPrintResolution(XMPFrame* frame)
      = 0 ;

      // answer the resolution in dots per inch the part requires
      // to image the specified frame on a printer canvas

   //-------------------------------------------------------------------------
   // From Linking protocol
   //-------------------------------------------------------------------------

   XMPVMethod XMPLink* CreateLink(XMPPtr data, XMPULong size)
         = 0;

      // create a new link object, put in the initial data
      // and return it to the caller.
      // This call to be implemented for the real or proxy source part
      // created by the destination part. 

   XMPVMethod void LinkUpdated(XMPLink* updatedLink, XMPChangeID id)
         = 0;

      // the specified link has been updated, please read and use
      // the incoming data.
      // The link object makes this call.

   XMPVMethod void RevealLink(XMPLinkSource* linkSource)
         = 0;

   XMPVMethod void   EmbeddedFrameChanged(XMPFrame* frame, XMPChangeID change)
         = 0;

   XMPVMethod void   LinkStatusChanged(XMPFrame* frame)
         = 0;

   //-------------------------------------------------------------------------
   // From Part Activation protocol
   //-------------------------------------------------------------------------

   XMPVMethod XMPBoolean BeginRelinquishFocus(XMPTypeToken focus,
                                    XMPFrame* ownerFrame,
                                    XMPFrame* proposedFrame)
         = 0;

      // Default is to return TRUE
      
   XMPVMethod void CommitRelinquishFocus(XMPTypeToken focus, 
                                XMPFrame* ownerFrame,
                                XMPFrame* proposedFrame)
         = 0;
                     
      // Must succeed
      // If part of proposedFrame is self then flicker can be avoided by doing nothing
      
   XMPVMethod void AbortRelinquishFocus(XMPTypeToken focus,
                               XMPFrame* ownerFrame,
                               XMPFrame* proposedFrame)
         = 0;
                          
      // Default is a noop
      // Part handlers must implement this if they do more in the Prepare method
      // than simply return TRUE
      // For example, a part requests a focus set owned by several different parts.
      // One part is downloading some data. It is willing to stop downloading, but if other
      // parts refuse to relinquish their focus set, the downloading part would like to resume
      // downloading.

   XMPVMethod void FocusAcquired(XMPTypeToken focus,
                          XMPFrame* ownerFrame)
         = 0;
   
      // A notification to the part that one of its frames has acquired
      // the specified focus set
      // Called when a document is opened, for example

   XMPVMethod void FocusLost(XMPTypeToken focus,
                       XMPFrame* ownerFrame)
         = 0;
                           
      // A notification to the part that a focus has been forcibly removed from it.
      // This might be done because a communication connection has been broken, for example  
            

   //-------------------------------------------------------------------------
   // From Storage protocol
   //-------------------------------------------------------------------------

   XMPVMethod void CloneInto(XMPDraftKey key, XMPStorageUnit* storageUnit, XMPStorageUnit* initiatingFrame)
      = 0;

      // copy this part to another storage unit
      
   //-------------------------------------------------------------------------
   // From Binding protocol
   //-------------------------------------------------------------------------

   XMPVMethod void ExternalizeKinds(XMPTypeSet* kindset)
      = 0;

      // Externalize all of the kinds in kindset that this part know how to

   XMPVMethod void ChangeKind(XMPType kind)
      = 0;

      // Set the preferred kind of the part to 'kind' and perform whatever 
      // storage operations necessary
      
   //-------------------------------------------------------------------------
   // From UI Events protocol
   //-------------------------------------------------------------------------

   XMPVMethod XMPBoolean HandleEvent(XMPEventData event,
                             XMPFrame* frame,
                             XMPFacet* facet)
      = 0;

      // some event happened in one of my frames.  answer whether i handled it.
            
   XMPVMethod void MouseEnter(XMPFacet* facet, XMPPoint where)
      = 0;
   
      // Called when the mouse is moved over a facet (mouse button up)

   XMPVMethod void MouseWithin(XMPFacet* facet, XMPPoint where)
      = 0;
   
      // Called when the mouse is moved within a facet (mouse button up)
         
   XMPVMethod void MouseLeave(XMPFacet* facet)
      = 0;
   
      // Called when the mouse is moved out of a facet (mouse button up)

   XMPVMethod void AdjustMenus(XMPFrame* frame)
      = 0;
      
      // Called by the shell (on Macintosh), just before calling MenuSelect
      // or MenuKey, so that the part can enable/disable items

   //-------------------------------------------------------------------------
   // From Undo protocol
   //-------------------------------------------------------------------------

   XMPVMethod void UndoAction(XMPActionData actionState)
      = 0;

      // Tells the part to Undo the action described by actionState
      
   XMPVMethod void RedoAction(XMPActionData actionState)
      = 0;

      // Tells the part to Redo the action described by actionState
   
   XMPVMethod void DisposeActionState(XMPActionData actionState,
                              XMPDoneState doneState)
      = 0;

      // Tells the part to dispose of the action data.
      // itsDoneState is kDone or kReDone if the action was on the document's,
      // Undo stack and kUndone if it was on the Redo stack
   
   XMPVMethod void WriteActionState(XMPPtr actionState,
                            XMPStorageUnitView* storageUnitView)
      = 0;

      // write out the actionState data onto a pre-focused StorageUnitView

   XMPVMethod XMPPtr ReadActionState(XMPStorageUnitView* storageUnitView)
      = 0;

      // read the actionState data from a pre-focused StorageUnitView

protected:

   //-------------------------------------------------------------------------
   // constructors/destructors
   //-------------------------------------------------------------------------

   XMPPart();

      // constructor - called only by XMPDraft

   virtual ~XMPPart();

      // destructor - called only by XMPDraft
      
   XMPVMethod void InitPart(XMPStorageUnit* storageUnit)
      = 0;
      
      // first time initializer - called only by XMPDraft
      
   XMPVMethod void InitPartFromStorage(XMPStorageUnit* storageUnit)
      = 0;

      // persistent store initializer - called only by XMPDraft
};


//==============================================================================
// XMPEmbeddedFramesIterator
//==============================================================================

class XMPEmbeddedFramesIterator
{

public:

   //-------------------------------------------------------------------------
   // from Embedding protocol:
   //-------------------------------------------------------------------------

   XMPVMethod XMPFrame* First() = 0;
   XMPVMethod XMPFrame* Next() = 0;
   XMPVMethod XMPBoolean   IsNotComplete() = 0;
};

#endif // _PART_
