/********************************************************************/ 
/*  Licensed Materials - Property of IBM                            */ 
/*                                                                  */ 
/*                                                                  */ 
/* Copyright (C) International Business Machines Corp., 1994.       */ 
/* Copyright (C) Apple Computer, Inc., 1994                         */ 
/*                                                                  */ 
/*  US Government Users Restricted Rights -                         */ 
/*  Use, duplication, or disclosure restricted                      */ 
/*  by GSA ADP Schedule Contract with IBM Corp.                     */ 
/*                                                                  */ 
/********************************************************************/ 
/*
   File:    LinkSrcM.h

   Contains:   The XMPMacLinkSource class definition

   Written by: Nick Pilch & Craig Carper

*/

#ifndef _LINKSRCM_
#define _LINKSRCM_

#ifndef _PLFMDEF_
#include "PlfmDef.h"
#endif

#ifndef _PSTOBJ_
#include "PstObj.h"
#endif

#ifndef _PLFMTYPE_
#include "PlfmType.h"
#endif

#ifndef __ALIASES__
#include <Aliases.h>
#endif

#ifndef __EDITIONS__
#include <Editions.h>
#endif

//==============================================================================
// Theory of Operation
//==============================================================================

/*
   This class is used to represent the source side of OpenDoc links. XMPDrafts
   create and own these links. The source XMPPart will ask the draft to create
   an XMPLinkSource and an XMPLink pair, and return the XMPLink object to the
   the destination part.  The destination part uses the XMPLink object to
   extract the contents of the link.
*/

//==============================================================================
// Classes defined in this interface
//==============================================================================

class XMPLinkSource;

//==============================================================================
// Classes used by this interface
//==============================================================================

class XMPStorageUnit;
class XMPPart;
class XMPPartList;
class XMPLink;

//==============================================================================
// XMPMacLinkSource
//==============================================================================

#define kXMPMacLinkSourceID "appl:xmpmaclinksource$class,1.0.0"

class XMPMacLinkSource : public XMPAbsLinkSource
{
  public:

   XMPVMethod XMPBoolean Lock(XMPULong wait, XMPLinkKey* key);

      // Acquire exclusive access to the content storage unit of the link
      // for the current thread. The value kXMPTrue is returned if the lock is
      // granted; the key argument is set to a valid link key.  Access is
      // granted if the current thread already holds the lock.  Nested calls
      // to Lock must be balanced by an equal number of calls to Unlock to 
      // relinquish the lock.

   XMPVMethod void Unlock(XMPLinkKey key);
      
      // Relinquish exclusive access to the content storage unit of the link.
      // The argument should be a non-zero key returned by Lock.

   XMPVMethod XMPStorageUnit* GetContentStorageUnit(XMPLinkKey key);

      // used by the source part to get the storage unit containing the 
      //  contents of the link.  The key parameter should be obtained by 
      //  aquiring the lock for this thread. 
      //  formerly named: GetStorageUnit

   XMPVMethod void ContentChanged(XMPChangeID id);

      // used by the source to make sure all destinations are notified
      //  of the updated link contents
      //  formerly named: Updated

   XMPVMethod void Clear(XMPChangeID id, XMPLinkKey key);

      // Empties the contents of the link. The key parameter
      // should be obtained by aquiring the lock for this thread.
      // The next call to GetContentStorageUnit returns an empty storage unit.
   
   XMPVMethod XMPChangeID GetChangeID();

      // Returns the change id of the link content.  Can be used by subscribers
      // to determine when new data is available.  The returned value should be
      // tested for equality only, as there is no implicit ordering of change ids.

   XMPVMethod void SetAutoExport(XMPBoolean automatic);

      // If automatic is kXMPTrue, export updates to other documents when the 
      // link's draft is saved.  If automatic is kXMPFalse, updates are exported
      // only when the Export method is called. This setting defaults to kXMPTrue.

   XMPVMethod XMPBoolean IsAutoExport();

      // Returns kXMPTrue if updates are automatically exported to other documents
      // when the link's draft is saved.

   XMPVMethod XMPLink* GetLink();
      
      // Return the link object associated with this link source.

   //---------------------------------
   // For XMPLink objects only

   XMPVMethod void ShowSourceContent();

      // Causes the source part of the link to become active, makes the source content
      // visible, and displays the link border.  Throws an exception if the source
      // part of the link cannot be located, or if the link has been broken at the
      // source.
      
   XMPVMethod XMPBoolean KeyValid(XMPLinkKey key);

      // Returns kXMPTrue if the argument key is valid.
      
   //---------------------------------
   // For XMPLinkManager

   XMPVMethod void UseLinkEdition(
                     EditionContainerSpec editionContainer,
                     XMPFileSpec* documentSpec,
                     XMPULong sectionID,
                     XMPLinkKey key);

      // subscribe to the argument edition container using the unique
      // section ID.

   XMPVMethod void ReadLinkEdition(XMPLinkKey key);
   
      // link should update itself by reading its edition file

   //-------------------------------------------------------------------------
   // From Memory protocol
   //-------------------------------------------------------------------------
   
   XMPVMethod XMPSize Purge(XMPSize size);

      // Gets rid of any unnecessary memory by purging memory blocks
      // or flushing structures out to disk
               
   //-------------------------------------------------------------------------
   // From Persistent Object protocol
   //-------------------------------------------------------------------------

   XMPVMethod void Externalize();

      // used by the Draft to get the link to externalize the necessary
      // parts of its persistent store; probably just calls fSU->Externalize();
   
   XMPVMethod void ReleaseAll();

      // used by the Draft to get the link to release all references to
      //  persistent obects.

   //-------------------------------------------------------------------------
   // From Initialization protocol
   //-------------------------------------------------------------------------

   XMPNVMethod void InitLinkSource(
                     XMPStorageUnit* storageUnit,
                     XMPPart* sourcePart,
                     XMPLink* link);

   XMPNVMethod void InitLinkSourceFromStorage(XMPStorageUnit* storageUnit);

      // used by the Draft to get the link to internalize the necessary
      // parts of its persistent store; probably just calls fSU->Internalize();

   //-------------------------------------------------------------------------
   // constructors/destructors
   //-------------------------------------------------------------------------

   XMPMacLinkSource();

   virtual  ~XMPMacLinkSource();

  private:

   SectionHandle  fSection;      // the section record
   XMPStorageUnit*   fContentSU;    // the storage unit containing the link's content
   XMPChangeID    fChangeID;     // change id of current link content
   XMPBoolean     fAutoExport;   // true if link should be exported on save
   XMPLink*    fLink;         // associated link object
   XMPPart*    fSourcePart;

   XMPMethod void ThrowIfInvalidKey(XMPLinkKey key);
   XMPMethod XMPStorageUnit* CloneFromMemoryContainer(XMPHandle containerHandle);
};

#endif // _LINKSRCM_
