/********************************************************************/ 
/*  Licensed Materials - Property of IBM                            */ 
/*                                                                  */ 
/*                                                                  */ 
/* Copyright (C) International Business Machines Corp., 1994.       */ 
/* Copyright (C) Apple Computer, Inc., 1994                         */ 
/*                                                                  */ 
/*  US Government Users Restricted Rights -                         */ 
/*  Use, duplication, or disclosure restricted                      */ 
/*  by GSA ADP Schedule Contract with IBM Corp.                     */ 
/*                                                                  */ 
/********************************************************************/ 
/*
   File:    LinkM.h

   Contains:   The XMPMacLink class definition

   Written by: Nick Pilch & Craig Carper

*/

#ifndef _LINKM_
#define _LINKM_

#ifndef _PLFMDEF_
#include "PlfmDef.h"
#endif

#ifndef _PSTOBJ_
#include "PstObj.h"
#endif

#ifndef _PLFMTYPE_
#include "PlfmType.h"
#endif

#ifndef __ALIASES__
#include <Aliases.h>
#endif

#ifndef __EDITIONS__
#include <Editions.h>
#endif

//==============================================================================
// Theory of Operation
//==============================================================================

/*
   This class is used to represent the destination side of OpenDoc links.
   XMPDrafts create and own these links. The source XMPPart will ask the draft
   to create an XMPLinkSource and an XMPLink pair, and return the XMPLink
   object to the the destination part.  The destination part uses the XMPLink
   object to extract the contents of the link.
*/

//==============================================================================
// Classes defined in this interface
//==============================================================================

class XMPLink;

//==============================================================================
// Classes used by this interface
//==============================================================================

class XMPStorageUnit;
class XMPPart;
class XMPPartList;
class XMPLinkSource;

//==============================================================================
// XMPLink
//==============================================================================

#define kXMPMacLinkID "appl:xmpmaclink$class,1.0.0"

class XMPMacLink : public XMPAbsLink
{
  public:

   XMPVMethod XMPBoolean Lock(XMPULong wait, XMPLinkKey* key);

      // Acquire exclusive access to the content storage unit of the link
      // for the current thread. The value kXMPTrue is returned if the lock is
      // granted; the key argument is set to a valid link key.  Access is
      // granted if the current thread already holds the lock.  Nested calls
      // to Lock must be balanced by an equal number of calls to Unlock to 
      // relinquish the lock.

   XMPVMethod void Unlock(XMPLinkKey key);
      
      // Relinquish exclusive access to the content storage unit of the link.
      // The argument should be a non-zero key returned by Lock.

   XMPVMethod XMPStorageUnit* GetContentStorageUnit(XMPLinkKey key);

      // used by destination parts to get the storage unit
      //  containing the contents of the link.  The key parameter
      //  should be obtained by aquiring the lock for this thread. 
      //  formerly named: GetStorageUnit

   XMPVMethod void RegisterDependent(XMPPart* clientPart, XMPChangeID id);

      // used by the destination to inform the link of its interest in the
      //  contents of the link. The parts LinkUpdated() method is called
      //  immediately if the links change id differs from the argument.
      //  The constant kChangeUnknown may be supplied if the part does not
      //  keep track of link change ids.
      //  formerly named: Notify
      
   XMPVMethod void UnregisterDependent(XMPPart* clientPart);

      // unregister dependent part previously registered with RegisterDependent.
      
   XMPVMethod XMPChangeID GetChangeID();

      // Returns the change id of the link content.  Can be used by destinations
      //  to determine when new data is available.  The returned value should be
      //  tested for equality only, as there is no implicit ordering of change ids.

   XMPVMethod XMPError GetStatus();

      // Returns the status of the link.  A non-zero value indicates a non-fatal
      // problem which the part may ignore or bring to the users attention.
      // The most common problem is a broken cross-document destination.
      // This situation may be temporary if the source document re-establishes
      // the cross-document link.

   XMPVMethod void ShowSourceContent();

      // Causes the source part of the link to become active, makes the source content
      // visible, and displays the link border.  Throws an exception if the source
      // part of the link cannot be located, or if the link has been broken at the
      // source.
      
   //---------------------------------
   // For XMPLinkSource

   XMPVMethod void UpdateDependents(XMPChangeID id);
      
      // Notify dependent parts of a change to link contents.

   //---------------------------------
   // For XMPLinkManager

   XMPVMethod AliasHandle CreateLinkEdition(
                        EditionContainerSpec editionContainer,
                        XMPFileSpec* documentSpec,
                        XMPULong sectionID,
                        XMPLinkKey key);

      // create an edition file using the argument container
      // and unique section ID.

   XMPVMethod AliasHandle GetLinkEdition(XMPLinkKey key);

      // returns an alias to the edition file maintained by this link
   
   XMPVMethod void UpdateLinkEdition(XMPLinkKey key);

      // Export the links contents to destinations in other documents.  
      // An exception is thrown if the link does not maintain a cross-document link
      // or if the source is in another document.

   //-------------------------------------------------------------------------
   // From Memory protocol
   //-------------------------------------------------------------------------
   
   XMPVMethod XMPSize Purge(XMPSize size);

      // Gets rid of any unnecessary memory by purging memory blocks
      // or flushing structures out to disk
               
   //-------------------------------------------------------------------------
   // From Persistent Object protocol
   //-------------------------------------------------------------------------

   XMPVMethod void Externalize();

      // used by the Draft to get the link to externalize the necessary
      // parts of its persistent store; probably just calls fSU->Externalize();
   
   XMPVMethod void ReleaseAll();

      // used by the Draft to get the link to release all references to
      //  persistent obects.

   //-------------------------------------------------------------------------
   // From Initialization protocol
   //-------------------------------------------------------------------------

   XMPNVMethod void InitLink(XMPStorageUnit* storageUnit, XMPLinkSource* linkSource);

   XMPNVMethod void InitLinkFromStorage(XMPStorageUnit* storageUnit);

      // used by the Draft to get the link to internalize the necessary
      // parts of its persistent store; probably just calls fSU->Internalize();
      

   //-------------------------------------------------------------------------
   // constructors/destructors
   //-------------------------------------------------------------------------

   XMPMacLink();

   virtual  ~XMPMacLink();

  private:

   XMPPartList*   fPartList;     // the list of dependent parts
   SectionHandle  fSection;      // the section record for exported link
   XMPLinkSource* fLinkSource;   // associated link source object

   XMPMethod void ThrowIfInvalidKey(XMPLinkKey key);
   XMPMethod XMPHandle GetTextValue(XMPLinkKey key);
   XMPMethod XMPHandle CloneToMemoryContainer(XMPLinkKey key);
   XMPMethod void WriteLinkEdition(FSSpecPtr documentSpec, XMPLinkKey key);
};

#endif // _LINKM_
