/********************************************************************/ 
/*  Licensed Materials - Property of IBM                            */ 
/*                                                                  */ 
/*                                                                  */ 
/* Copyright (C) International Business Machines Corp., 1994.       */ 
/* Copyright (C) Apple Computer, Inc., 1994                         */ 
/*                                                                  */ 
/*  US Government Users Restricted Rights -                         */ 
/*  Use, duplication, or disclosure restricted                      */ 
/*  by GSA ADP Schedule Contract with IBM Corp.                     */ 
/*                                                                  */ 
/********************************************************************/ 
/*
   File:    Except.h

   Contains:   Exception handling macros

   Written by: Richard Rodseth, Steve Smith, Jens Alfke

*/


/*
   THEORY OF OPERATION
   
   To come. See LibraryManager.h
   
   This file simply uses the Apple Shared Library Manager exceptions.

   Example Usage:
   
      XMPFoo* obj = kXMPNULL;
      XMPVolatile(obj);       // if it's modified in TRY, and used in CATCH
                        // it must be kept out of registers
      
      TRY
      
         obj = new XMPFoo;
         FailNULL(obj, kXMPErrOutOfMemory);  // Until global new throws exception
         obj->Initialize();   // Initialization that can fail
      
      CATCH(kXMPErrOutOfMemory)
      
         DeleteObject(&obj);     // i.e. if (obj) { delete obj; obj == NULL }
      
      CATCH_ALL
      
         // Do some stuff
         RERAISE; // or: Fail(error);
               
      ENDTRY
   
   Limitations: 
      One Exception type (a long integer value)
      Catch block must rethrow errors it doesn't handle
      No automatic calling of destructors
      Use of VOLATILE (See below)
      Maybe some others
   
*/

//
//--------------------------------------<<  >>--------------------------------------

#ifndef _EXCEPT_
#define _EXCEPT_

#ifndef _XMPTYPES_
#include "XMPTypes.h"
#endif

#ifndef _ERRORDEF_
#include "ErrorDef.h"
#endif
// ErrorDef.h is included here as a convenience
// if you are including Except.h in order to throw exceptions, 
// you probably need the exception codes as well.
// -Tantek elik (93.12.03)

#define APPLEINCL_LibraryManager
#ifndef APPLEINCLUDED
   #include <Apple.h>
#endif


#define XMPVolatile(x) Volatile(x)

#define exceptionMask   0x0000FFFF
#define methodMask      0x00FF0000
#define classMask    0xFF000000

#define kExceptionList  100
#define kClassList      101

void WriteToFile( char* expType = kXMPNULL, XMPError error = 0, char* msg = kXMPNULL);
void THROW_IF_ERROR(XMPError error, char* msg = kXMPNULL);
void THROW(XMPError error, char* msg = kXMPNULL);
void THROW_IF_NULL(void* value, char* msg = kXMPNULL);

// Logging.
// LOG has the same syntax as printf but produces a SysBreakStr, which will
// write the result to the log window in VoodooMonkey. (It will also cause
// a breakpoint, unfortunately.) You should put a "\n" at the end of the
// format string if you want to end the line there.
//
// To enable logging in a source file, you must redefine the symbol LOGGING
// as "1" in that file, otherwise LOG statements will not be compiled. Make
// sure to #undef the symbol before you re-#define it, as some compilers
// (e.g. Symantec) won't redefine an already-defined symbol.
//
// LOG statements do not generate any code if logging is off.

#define LOGGING 0    // Redefine as 1 in source file to enable logging

#define LOG    if(!XMPDebug || !LOGGING) ; else _Log

// Warnings.
// WARN has the same syntax as printf but produces a SysBreakStr.
// Warnings are disabled (and generate no code) when XMPDebug is off.

#define WARN   if(!XMPDebug) ; else _Warn

// Assertion macros.
// These all cause a SysBreakStr if the condition evaluates to false or 0.
// Leading "W" is a Warning: it doesn't raise an exception.
// Trailing "M" means special Message displayed instead of condition.

#define ASSERT( COND, ERR )   \
   if(!XMPDebug || (COND)) ; else _AssertionFailed( #COND, __FILE__, ERR )
#define ASSERTM( COND, ERR, MSG )   \
   if(!XMPDebug || (COND)) ; else _AssertionFailed( #COND, __FILE__, ERR, MSG )
#define WASSERT( COND ) \
   if(!XMPDebug || (COND)) ; else _AssertionFailed( #COND, __FILE__, 0)
#define WASSERTM( COND, MSG ) \
   if(!XMPDebug || (COND)) ; else _AssertionFailed( #COND, __FILE__, 0, MSG)

// Functions that underly these macros:
void _Log            ( char *fmt, ... );
void _Warn           ( char *fmt, ... );
void _AssertionFailed   ( char *cond,  char *fileName,
                     XMPError, char *msg = kXMPNULL );

#endif // _EXCEPT_
