/********************************************************************/ 
/*  Licensed Materials - Property of IBM                            */ 
/*                                                                  */ 
/*                                                                  */ 
/* Copyright (C) International Business Machines Corp., 1994.       */ 
/* Copyright (C) Apple Computer, Inc., 1994                         */ 
/*                                                                  */ 
/*  US Government Users Restricted Rights -                         */ 
/*  Use, duplication, or disclosure restricted                      */ 
/*  by GSA ADP Schedule Contract with IBM Corp.                     */ 
/*                                                                  */ 
/********************************************************************/ 
/*
  File:    Draft.h

  Contains:  definition for XMPDraft

  Written by:  Vincent Lo, Tantek I. Celik

*/
#ifndef _DRAFT_
#define _DRAFT_

#ifndef _XMPTYPES_
#include "XMPTypes.h"
#endif

#ifndef _PSTOBJ_
#include "PstObj.h"
#endif

#ifndef _STORAGEU_
#include "StorageU.h"
#endif

//==============================================================================
// Constants
//==============================================================================

enum {
  kDPNone       = 0,
  kDPTransient    = 1,
  kDPReadOnly      = 2,
  kDPSharedWrite    = 3,
  kDPExclusiveWrite  = 4
};

typedef XMPULong XMPDraftPermissions;

typedef const enum {
  kInvisibleBlocks  = 1,
  kAllBlocks      = 2,
  kVisibleBlocks    = 3
}  XMPPurgePriority;

typedef enum {
  kXMPCloneCopy  = 0,
  kXMPCloneCut,
  kXMPClonePaste,
  kXMPCloneDuplicate,
  kXMPCloneAll        // For private use only
} XMPCloneKind;

//==============================================================================
// Scalar Types
//==============================================================================
typedef XMPULong  XMPVersionID;
typedef  XMPID    XMPDraftID;
typedef  XMPID    XMPLinkID;
typedef  XMPISOStr  XMPDraftName;

//==============================================================================
// Classes defined in this interface
//==============================================================================
class XMPDraft;

//==============================================================================
// Classes used by this interface
//==============================================================================
class  XMPCanvas;
class  XMPDocument;
class  XMPFrame;
class  XMPLink;
class  XMPLinkSource;
class  XMPLinkSpec;
class  XMPPart;
class  XMPPartTable;
class  XMPShape;
class  XMPEmbeddedContainer;
class  XMPAEHashTable;
class  IDList;
class  XMPStorageUnit;
class  VersionList;
class  XMPDragDropStorageUnit;

//==============================================================================
// XMPDraft
//==============================================================================

#define kXMPDraftID "appl:xmpdraft$class,1.0.0"

class XMPDraft : public XMPRefCntObject
{
  friend class XMPLinkIterator;  
  friend class XMPLinkSourceIterator;  

public:

  XMPMethod  XMPDocument*  GetDocument();
  
  XMPMethod  XMPDraftID    GetID();

  XMPMethod  XMPDraftName  GetName();

  XMPMethod  void      SetName(XMPDraftName name);

  XMPMethod  XMPStorageUnit*  GetDraftProperties();
  
  XMPMethod  void IncrementRefCount();
    
  XMPMethod  void Release();

  XMPMethod  XMPDraftPermissions  GetPermissions();

  XMPMethod  XMPStorageUnit*  CreateStorageUnit();
  XMPMethod  XMPStorageUnit*  GetStorageUnit(XMPStorageUnitID id);
  XMPMethod  void      RemoveStorageUnit(XMPStorageUnit* storageUnit);
  
  XMPMethod  XMPDraftKey    BeginClone(XMPCloneKind kind);
  
  XMPMethod  void      EndClone(XMPDraftKey key);
  
  XMPMethod  void      AbortClone(XMPDraftKey key);

  XMPMethod  XMPBoolean    ChangedFromPrev();  

  XMPMethod  void      SetChangedFromPrev();  

  XMPMethod  void      RemoveFromDocument();

  XMPMethod  XMPDraft*    RemoveChanges();

  XMPMethod  XMPDraft*    Externalize();
  
  XMPMethod   XMPDraft*    SaveToAPrevious(XMPDraft* to);

  XMPMethod XMPFrame*  CreateFrame(
              XMPFrame*    containingFrame,
              XMPShape*     frameShape,
              XMPPart*     part,
              XMPTypeToken  viewType,
              XMPTypeToken  presentation,
              XMPULong     frameGroup,
              XMPBoolean    isRoot,
              XMPBoolean     isOverlaid);
       
  XMPMethod XMPFrame*  GetFrame(XMPStorageUnitID id);
  XMPMethod void    RemoveFrame(XMPFrame* frame);

  XMPMethod XMPPart*  CreatePart(XMPType partType,XMPEditor optionalEditor);
  XMPMethod XMPPart*  GetPart(XMPStorageUnitID id);
  XMPMethod void    RemovePart(XMPPart* part);

  XMPMethod XMPLinkSpec*    CreateLinkSpec (XMPPart* part, XMPPtr data, XMPULong size);
  XMPMethod XMPLinkSource*  CreateLinkSource(XMPPart* part);
  XMPMethod XMPLinkSource*   GetLinkSource(XMPStorageUnitID id);
  XMPMethod XMPLink*      GetLink(XMPStorageUnitID id,XMPLinkSpec* linkSpec);
  XMPMethod void        RemoveLink(XMPLink* link);
  XMPMethod void        RemoveLinkSource(XMPLinkSource* link);

private:

  XMPMethod XMPSize Purge(XMPSize size);

  XMPMethod XMPStorageUnit* CreateSU(XMPStorageUnitID id, const XMPType suType);

  XMPMethod void  CreateVersion();
  XMPMethod void  OpenVersion();
  XMPMethod void  CloseVersion();
  XMPMethod void  DestroyVersion();
  XMPMethod void  FlushVersion();

  XMPMethod XMPPersistentObject* GetPersistentObject(XMPStorageUnitID id);
  XMPMethod void  ReleasePersistentObject(XMPPersistentObject* object);
  XMPMethod void  RemovePersistentObject(XMPPersistentObject* object);
  XMPMethod void  CreateCollections();
  XMPMethod void  DeleteCollections();
  XMPMethod void  ExternalizeCollections();
  XMPMethod void  CloseCollections();
  XMPMethod void  OpenCollections();
  XMPMethod XMPBoolean  AreEmptyCollections();
  
  XMPMethod void  FailIfNotExclusiveWrite();
  
  XMPMethod XMPLink*  CreateLink(XMPLinkSource* linkSource);

  XMPMethod void  PrepareLinksForClone();
  XMPMethod void  AnnotateLinksForNextClone();

  XMPDocument*      fDocument;
  XMPDraftID        fID;
  XMPVersionID      fVersionID;
  XMPVersionID      fPrevVersionID;
    
  XMPDraftPermissions    fPermissions;
  XMPEmbeddedContainer*  fEmbeddedContainer;
  XMPBoolean        fChangedFromPrev;
  XMPBoolean        fIsNewDraft;
  XMPBoolean        fExternalized;

  XMPAEHashTable*      fPersistentObjects;
  XMPAEHashTable*      fReleasedPersistentObjects;
  XMPAEHashTable*      fStorageUnits;
  XMPAEHashTable*      fReleasedStorageUnits;
  
  IDList*          fIDList;
  XMPStorageUnit*      fDraftProperties;
  
  XMPAEHashTable*      fClonedSUIDs;
  XMPDraft*        fDestDraft;
  XMPDraftKey        fCurrentKey;
  XMPULong        fLockCount;
  XMPCloneKind      fCloneKind;

  XMPHeap          fHeap;
  
public:  // private by convention

  XMPDraft();

  XMPVMethod ~XMPDraft();

  XMPMethod void InitDraft(XMPDocument* document, XMPDraftID id, XMPDraftPermissions perms);

  XMPMethod void  ReleaseFrame(XMPFrame* frame);
  XMPMethod void  ReleasePart(XMPPart* part);
  XMPMethod void  ReleaseLink(XMPLink* link);
  XMPMethod XMPDraft*    ReleaseStorageUnit(XMPStorageUnitID id);
  
  XMPMethod void   SetChangedFromPrevFlag(XMPBoolean changed);

  XMPMethod IDList*    GetIDList();

  XMPMethod void Reinitialize(XMPDraftPermissions perms);
  XMPMethod void Open();
  XMPMethod void Close();
  XMPMethod void Abort();
  XMPMethod void Flush();
  XMPMethod XMPBoolean  IsNewDraft();
  XMPMethod XMPEmbeddedContainer*  GetEmbeddedContainer();
  XMPMethod CMContainer  GetCMContainer();
  XMPMethod XMPBoolean  NeedExternalizing();
  XMPMethod XMPBoolean  IsChangedFromPrev(VersionList* versionList);
  XMPMethod XMPAEHashTable* GetClonedSUIDs(XMPDraft* destDraft);

  XMPMethod XMPStorageUnit* CreateDragDropStorageUnit();
  XMPMethod XMPStorageUnit* GetDragDropStorageUnit(XMPStorageUnitID id);
  
  XMPMethod  XMPBoolean  IsValidDraftKey(XMPDraftKey key);
  XMPMethod  XMPHeap  GetHeap();
  
  XMPMethod  XMPLinkID        GetPersistentLinkID(XMPLink* link);
  XMPMethod  XMPStorageUnitID    GetLinkID(XMPLinkID linkID);
  XMPMethod  XMPLinkIterator*    GetXMPLinkIterator();
  XMPMethod  XMPLinkSourceIterator*  GetXMPLinkSourceIterator();
};

#endif  // _DRAFT_
