/********************************************************************/ 
/*  Licensed Materials - Property of IBM                            */ 
/*                                                                  */ 
/*                                                                  */ 
/* Copyright (C) International Business Machines Corp., 1994.       */ 
/* Copyright (C) Apple Computer, Inc., 1994                         */ 
/*                                                                  */ 
/*  US Government Users Restricted Rights -                         */ 
/*  Use, duplication, or disclosure restricted                      */ 
/*  by GSA ADP Schedule Contract with IBM Corp.                     */ 
/*                                                                  */ 
/********************************************************************/ 
/*
  File:    AltPoly.h

  Contains:  OpenDoc polygon: optional C++ savvy classes

  Written by:  Jens Alfke

  Theory of Operation:
  
    This is an alternate definition of XMPPolygon and XMPContour. The data format is
    identical, but the structs defined here have a lot of useful methods including
    constructors, accessors and conversion operators.
    
    To use these instead of the regular structs defined in Polygon.h, just include
    this header file _before_ Polygon.h. An easy way to do this is to include it
    first.
  
    QuickDraw GX users take note:
    XMPContour is identical in data format to a gxPolygon.
    XMPPolygon is identical in data format to a gxPolygons. (See <graphics types.h>)
    
    You cannot directly instantiate XMPPolygons (by declaring an XMPPolygon variable,
    or via "new XMPPolygon") since they are variable-size objects. The compiler will
    complain if you try. Instead, use the static "New" methods.
*/


#ifndef _ALTPOLY_
#define _ALTPOLY_

#ifdef _POLYGON_
  #error "Must include AltPoly.h *before* Polygon.h!"
#else
  #define _POLYGON_    /* Make sure Polygon.h does NOT get included later! */
#endif

#ifndef _OBJECT_
#include "XMPObj.h"
#endif

#include <stddef.h>          // for size_t

#if PLATFORM_MACINTOSH
   #define APPLEINCL_QuickDraw
   #ifndef APPLEINCLUDED
     #include <Apple.h>
   #endif
//JYS: Commented out these headers
/*
  #ifndef __FIXMATH__
  #include <FixMath.h>        // Must include before GX headers...
  #endif
  #ifndef graphicsTypesIncludes
  #include "graphics types.h"      // for gxShape type
  #endif
*/
#endif


//==============================================================================
// Classes used in this interface
//==============================================================================

struct XMPRect;
class XMPStorageUnit;
class XMPTransform;

//==============================================================================
// XMPContour
//==============================================================================

#define kXMPContourID "appl:xmpcontour$class,1.0.0"

struct XMPContour {
  public:
  
  XMPSLong  nVertices;
  XMPPoint  vertex[1];    // Array size is actually nVertices
  
  XMPContour* NextContour( )        const  {return (XMPContour*)&vertex[nVertices];}
  XMPBoolean  IsRectangular( )      const;
  XMPBoolean  AsRectangle( XMPRect* )    const;
#if PLATFORM_MACINTOSH
  PolyHandle  AsQDPolygon( )        const;
#endif
  
  XMPBoolean  operator== ( const XMPContour& )      const;
  XMPBoolean  operator!= ( const XMPContour &c )      const  {return !(*this==c);}
    // Equality tests not fully robust yet; may generate false negatives.

  private:
  
  XMPContour( ) { }      // Don't allow foreigners to create contours themselves
  friend struct XMPPolygon;
};

//==============================================================================
// XMPPolygon
//==============================================================================

#define kXMPPolygonID "appl:XMPPolygon$class,1.0.0"

struct XMPPolygon {
  public:
  
  // DATA MEMBERS:
  
  XMPSLong  nContours;
  XMPContour  firstContour;        // Rest of contours follow after first
                // firstContour field does not exist if nContours==0!
  
  // CREATION:
  
  static XMPPolygon*  New( XMPSLong nVertices );
  static XMPPolygon*  New( XMPSLong nVertices, const XMPPoint *vertices );
  static XMPPolygon*  New( XMPSLong nContours, const XMPSLong *contourVertices );
  static XMPPolygon*  New( const XMPRect& );
  static XMPPolygon*  NewFromPolygon( const XMPPolygon* );
  static XMPPolygon*  NewFromSU( XMPStorageUnit* );
#if PLATFORM_MACINTOSH
  static XMPPolygon*  NewFromGXShape( gxShape );    // Accepts rect, polygon(s), path(s)
#endif
  
#ifdef __DEBUG_ALLOC__
  void    operator delete( void *obj, const char *file, size_t len )  {::delete (obj);}    //Keeps compiler happy
#else
  void    operator delete( void *obj )  {::delete obj;}    //Keeps compiler happy
#endif

  // ACCESSORS:  (Alternatives to directly accessing the data members.)
  
  XMPSLong  GetNContours( )                const  {return nContours;}
  XMPContour*  FirstContour( )                    {return &firstContour;}
  const XMPContour*  FirstContour( )            const  {return &firstContour;}
  XMPSLong  GetNVertices( XMPSLong contour )      const;
  XMPPoint  GetVertex( XMPSLong contour, XMPSLong vertex ) const;
  XMPPoint  SetVertex( XMPSLong contour, XMPSLong vertex, const XMPPoint& );
  
  // GEOMETRY:
  
  void    ComputeBoundingBox( XMPRect* )        const;
  XMPBoolean  IsRectangular( )              const;
  void    Transform( XMPTransform* );
  
  XMPBoolean  operator== ( const XMPPolygon& )      const;
  XMPBoolean  operator!= ( const XMPPolygon &poly )    const  {return !(*this==poly);}
    // Equality tests not fully robust yet; may generate false negatives.
  
  // TESTS:
  
  XMPSLong  Contains( XMPPoint, XMPBoolean *onEdge )  const;
  XMPBoolean  IsEmpty( )                  const;
  
  // CONVERSIONS:
  
  XMPBoolean  AsRectangle( XMPRect* )            const;  // Returns false if impossible
#if PLATFORM_MACINTOSH
  RgnHandle  AsQDRegion( )                const;
#endif
  
  // MEMORY:
  
  XMPULong  GetDataSize( )                const;
  XMPPolygon*  Copy( )                    const;
  
  private:
  
#ifdef __DEBUG_ALLOC__
  void*    operator new( size_t, const char *, size_t, long );
#else
  void*    operator new( size_t, long );
#endif
  XMPPolygon( ) { }    // Don't allow foreigners to create polygons themselves
};

#endif //_ALTPOLY_