/********************************************************************/ 
/*  Licensed Materials - Property of IBM                            */ 
/*                                                                  */ 
/*                                                                  */ 
/* Copyright (C) International Business Machines Corp., 1994.       */ 
/* Copyright (C) Apple Computer, Inc., 1994                         */ 
/*                                                                  */ 
/*  US Government Users Restricted Rights -                         */ 
/*  Use, duplication, or disclosure restricted                      */ 
/*  by GSA ADP Schedule Contract with IBM Corp.                     */ 
/*                                                                  */ 
/********************************************************************/ 
/*
  File:    AltPoint.h

  Contains:  C++ savvy points and rects (alternate XMPPoint, XMPRect)

  Written by:  Jens Alfke

  Notes:
  
  These are alternate definitions of the XMPPoint and XMPRect structs.
  These definitions have the same size and data format and can be used
  interchangeably with the basic definitions; but they're much more C++
  savvy, with constructors, operators, conversions, and utility methods.
  
  To use these classes instead of the defaults, just include "AltPoint.h"
  as the first thing in your source file. It has to be included first so
  it can override the default struct definitions in PlfmType.h.
  
  This API and implementation are **NOT** an official part of the OpenDoc
  API, just handy utilities for C++ programmers.
*/


#ifndef _ALTPOINT_
#define _ALTPOINT_

#ifndef PLATFORM_OS2
#define PLATFORM_OS2 1
#endif


// Make sure that built-in structs do not get defined by PlfmType.h.
#if !defined(_POINT_) && !defined(_RECT_)
  #define _POINT_
  #define _RECT_
#else
  #error "Must include AltPoint.h *before* XMPTypes.h or PlfmType.h!"
#endif


#ifndef _PLFMTYPE_
#include "PlfmType.h"    // For XMPCoordinate
#endif

#include "os2def.h"

#ifndef __IMAGSTUB
#include "imagstub.h"
#endif

typedef XMPFixed XMPCoordinate;

//==============================================================================
// XMPPoint
//==============================================================================

struct XMPPoint {
  public:
  
  // CONTENTS:
  
  XMPCoordinate x, y;
  
  // CONSTRUCTORS:
  
  XMPPoint( ) { }
  
  XMPPoint( XMPCoordinate xx, XMPCoordinate yy )
          {x=xx; y=yy;}
  
  XMPPoint( const XMPPoint& );        // Copy constructor
  
  // ASSIGNMENT:
  
  XMPPoint& operator= ( const XMPPoint& );  // Copy from another pt
  
  // MODIFICATION:
  
  inline void  Clear( )
          {x=y=0;}
  inline void  Set( XMPCoordinate xx, XMPCoordinate yy )
          {x=xx; y=yy;}
  void  Offset( XMPCoordinate x, XMPCoordinate y );
  void  operator+=( const XMPPoint& );
  void  operator-=( const XMPPoint& );
  
  // ACCESSORS:

  XMPSShort  IntX( )    const;    // Returns X-coord as (16bit) integer
  XMPSShort  IntY( )    const;    // Returns Y-coord as (16bit) integer
  
  // COMPARISON:
  
  Boolean  operator==( const XMPPoint& )  const;
  Boolean  operator!=( const XMPPoint& )  const;
//-------------------------YJ's modification----------------------------
  
  // MAC TOOLBOX CONVENIENCES:
/*  
#ifdef PLATFORM_MACINTOSH
  XMPPoint( Point );              // Construct from QD point
  XMPPoint& operator= ( const Point& );    // Copy from a QD Point
  Point  AsQDPoint( )          const;  // Convert to integer (QD) point
  void  operator+=( const    Point& );      // Add/subtract QD point
  void  operator-=( const    Point& );
#endif
*/

#ifdef PLATFORM_OS2
   // Point is typedef'd as POINTL in PM
   XMPPoint(Point );                     // Construct from POINTL
   XMPPoint& operator= ( const Point&);  // Copy construcror
   Point   AsPOINTL()   const;           // Convert to OS/2 POINTL
   void    operator +=( const  Point&);
   void    operator -=( const  Point&);
#endif

//---------------------------YJ's modification------------------------------
};


//==============================================================================
// XMPRect
//==============================================================================

struct XMPRect {
  public:
  
  // CONTENTS:
  
  XMPCoordinate left, top, right, bottom;
    
  // CONSTRUCTORS:
  
  XMPRect( )    { }
  XMPRect( XMPCoordinate l, XMPCoordinate t,
          XMPCoordinate r, XMPCoordinate b )
      {left=l; top=t; right=r; bottom=b; }
  XMPRect( const XMPPoint&, const XMPPoint& );  // Any 2 oppos121ite pts
  XMPRect( const XMPPoint &topLeft, XMPCoordinate width, XMPCoordinate height );

//#ifdef PLATFORM_OS2
//#if PLATFORM_MACINTOSH
  XMPRect( const Rect& );
//#endif
  
  // ASSIGNMENT:
  
//#if PLATFORM_MACINTOSH
//  #ifdef PLATFORM_OS2 
  XMPRect& operator= ( const Rect& );
//#endif
  
  // MODIFICATION:
  
  void  Clear( );
  void  Set( XMPCoordinate l, XMPCoordinate b, XMPCoordinate r, XMPCoordinate t );
  void  Set( const XMPPoint&, XMPCoordinate width, XMPCoordinate height );
  void  SetInt( short l, short b, short r, short t );
  void  Offset( XMPCoordinate x, XMPCoordinate y );
  void  Offset( const XMPPoint& );
  void  Inset( XMPCoordinate x, XMPCoordinate y );
  
  void  operator&= ( const XMPRect& );  // Intersect with rectangle
  void  operator|= ( const XMPRect& );  // Union with rectangle
  void  operator|= ( const XMPPoint& );    // Expand to fit point
    
  // ACCESSORS
  
  const XMPPoint& TopLeft( )                    const
                {return *(XMPPoint*)&left;}
  XMPPoint&    TopLeft( )
                {return *(XMPPoint*)&left;}
  const XMPPoint& BotRight( )                    const
                {return *(XMPPoint*)&right;}
  XMPPoint&    BotRight( )
                {return *(XMPPoint*)&right;}
  XMPCoordinate  Width( )                    const
                {return right-left;}
  XMPCoordinate  Height( )                    const
                {return  bottom > top ? bottom-top : top - bottom;}
//#if PLATFORM_MACINTOSH
  void AsOS2Rect(Rect &)       const;
//  void      AsQDRect( Rect& )                const;

  
  // TESTING

  Boolean  operator==( const XMPRect& )            const;
  Boolean  operator!=( const XMPRect &r )            const
                {return !(*this==r);}
  
  Boolean IsEmpty( )                      const;
  Boolean Contains( const XMPPoint& )              const;
  Boolean Contains( const XMPRect& )              const;
  Boolean Intersects( const XMPRect& )            const;
};


#endif //_ALTPOINT_
