/********************************************************************/ 
/*  Licensed Materials - Property of IBM                            */ 
/*                                                                  */ 
/*                                                                  */ 
/* Copyright (C) International Business Machines Corp., 1994.       */ 
/* Copyright (C) Apple Computer, Inc., 1994                         */ 
/*                                                                  */ 
/*  US Government Users Restricted Rights -                         */ 
/*  Use, duplication, or disclosure restricted                      */ 
/*  by GSA ADP Schedule Contract with IBM Corp.                     */ 
/*                                                                  */ 
/********************************************************************/ 
/*
  File:    AEHshTbl.h

  Contains:  Interface to AppleEvent hash table code and XMPAEHashTable
        class.

  Written by:  Nick Pilch

*/

#ifndef _AEHSHTBL_
#define _AEHSHTBL_

#ifndef _PLFMDEF_
#include "PlfmDef.h"
#endif

#ifndef _PLFMTYPE_
#include "PlfmType.h"
#endif


//==============================================================================
// Theory of Operation
//==============================================================================

/*
  C Interface to some of the routines in HashTabl.A and class wrapper.

  The class simply provides an object-oriented wrapper to the AE hash table
  functions found in HashTable.A. It removes access to the MemHooks parameter. (MemHooks is set to NULL and SysHeap is set to false.)
  It hides the Table parameter since that is a member variable of the class.
*/

//==============================================================================
// Scalar Types
//==============================================================================

typedef void *KeyPtr;
typedef void *HEntryPtr;


//==============================================================================
// Error Codes
//==============================================================================

const short kAEErrAlreadyExists  = -1722;
const short kAEErrNotFound    = -1723;
const short kAEErrEndOfTable  = -1724;

//----------
// XMP types
//----------

typedef KeyPtr XMPKeyPtr;
typedef HEntryPtr XMPEntryPtr;

class    HCollection;
typedef  HCollection *HashTable;
class    HCollectionCursor;
typedef  HCollectionCursor *HashTableCursor;

//==============================================================================
// Classes defined in this interface
//==============================================================================

class XMPAEHashTable;
class XMPAEHashTableIterator;

//==============================================================================
// XMPAEHashTable
//==============================================================================

#define kXMPAEHashTableID "appl:xmpaehashtable$class,1.0.0"

class XMPAEHashTable
{

  public:

  XMPAEHashTable();
  XMPVMethod void Initialize(XMPULong numEntries, XMPUShort keySize,
                XMPUShort valueSize, XMPBoolean inSysHeap);
    // kXMPErrOutOfMemory is thrown if the table cannot be created.

  virtual ~XMPAEHashTable();

  XMPVMethod void      ReplaceEntry(XMPKeyPtr key, XMPEntryPtr value);
    // Replace and/or add an entry. Pass a POINTER to the key as well as a
    //  a POINTER to the value to be added. kXMPErrOutOfMemory is thrown if
    //  the entry cannot be added.

  XMPVMethod void      RemoveEntry(XMPKeyPtr key);
    // Pass a POINTER to the key.

  XMPVMethod XMPBoolean  GetValue(XMPKeyPtr key, XMPEntryPtr value);
    // Pass a POINTER to the key as well as a POINTER to the value to be
    //  retrieved. kXMPTrue is returned if the key was found, kXMPFalse
    //  otherwise.
    
  XMPVMethod XMPBoolean  Exists(XMPKeyPtr key);
    // Check to see if a key exists. This function is not much faster than
    //  GetValue and should probably only be used when you do not intend to
    //  fetch the value immediately. kXMPTrue is returned if the key exists,
    //  kXMPFalse, otherwise.
  
  private:

  HashTable  fAEHashTable;
  int        fKeySize, fEntrySize;
  
  friend XMPAEHashTableIterator;
  HashTable  GetAEHashTable();
};

//==============================================================================
// XMPAEHashTableIterator
//
//  This iterator is only meant to be used in the the context of a for loop,
//  e.g.:
//
//  XMPAEHashTableIterator iter;
//  for (iter.First(key, value); iter.IsNotComplete(); iter.Next(key, value))
//  {
//    ...
//  }
//
//==============================================================================

#define kXMPAEHashTableIteratorID "appl:xmpaehashtableiterator$class,1.0.0"

class XMPAEHashTableIterator
{
  public:
  XMPAEHashTableIterator(XMPAEHashTable* table);
  ~XMPAEHashTableIterator();
  void    First(XMPKeyPtr key, XMPEntryPtr value);
  void    Next(XMPKeyPtr key, XMPEntryPtr value);
  XMPBoolean  IsNotComplete();
  private:
  HashTable       fTable;
  HashTableCursor cursor;
};
#endif // _AEHSHTBL_