                       F i l e    I n f o r m a t i o n

* DESCRIPTION
Manage portions of EMS as a heap by TurboPower Software.

* ASSOCIATED FILES
EMSHEAP.PAS
EMSHEAP.ASM
EMSHEAP.OBJ
EMSHEAP.TPU
GETEMSP.PAS
TESTHEAP.EXE
TESTHEAP.PAS

* KEYWORDS
PROGRAM PASCAL SOURCE V4.0 HEAP

==========================================================================
}
{EMSHEAP.PAS Copyright (C) 1988, by TurboPower Software}

{$R-,V-,S-,F-,B-,I-}
{.$DEFINE Debug}

unit EmsHeap;
  {-Manage portions of EMS as a heap}

{************************************************************************
  Specifications:
    If EMS is not installed, calls normal heap routines instead.
    Smallest structure allocated is 1 byte.
    Largest contiguous structure 16K bytes.
    Heap manager assumes it owns EMS, mapping pages at will and expecting
      pages to remain mapped from call to call.
    Although allocation returns a pointer, this pointer must be specially
      dereferenced before use: EmsP(MyEmsPointer)^.
    Space for the free list is allocated at base of first available page.
      Each free list entry uses 6 bytes. Maximum free list entries: 2730.
    Does not take advantage of EMS 4.0 or EEMS for increased page window
      size.
    Call EmsResult after each call to check status. Error codes defined
      below.
 ************************************************************************}

interface

const
  EmsSuccess = 0;                 {Returned by EmsResult when success}
  EmsAllocError = 1;              {Error allocating physical EMS space}
  EmsMapError = 2;                {Error mapping EMS page}
  EmsOutOfMemory = 3;             {Insufficient memory}
  EmsInvalidGet = 4;              {Request for block larger than PageSize}
  EmsOutOfFreeList = 5;           {Free list overflow}
  EmsInvalidFree = 6;             {Request to free impossible block}
  EmsErrorCode = $FFFF;           {Returned by low level functions if error}

  HaltOnError : Boolean = False;  {True to halt when EMS error occurs}

type
  EmsPtr =
    record
      PageOfs : Word;             {Offset within logical EMS page}
      PageNum : Word;             {Logical EMS page where located}
    end;

var
  EmsAvailable : Boolean;         {True if EMS available}
  EmsAllocated : Boolean;         {True when EMS heap allocated}
  FrameSeg : Word;                {Page frame segment}
  FreeCnt : Word;                 {Count of free list entries}
  EmsPAddr : Pointer;             {Address of pointer conversion routine}

procedure InitEmsHeap(TotalBytes, FreeListBytes : LongInt);
  {-Allocate EMS space for use by manager}

procedure ExitEmsHeap;
  {-Deallocate EMS space -- automatically called when program shuts down}

procedure GetEms(var P; ReqSize : Word);
  {-Allocate space from EMS heap, returning special EMS pointer}

procedure FreeEms(var P; ReqSize : Word);
  {-Free allocated EMS heap space}

function EmsMemAvail : LongInt;
  {-Return total EMS heap space available}

function EmsMaxAvail : LongInt;
  {-Return largest EMS heap block available}

function EmsP(P : Pointer) : Pointer;
  {-Return physical pointer for an EMS pointer.
   This version for speed when it's not assured that EMS is allocated}
  inline
  ($80/$3E/>EmsAllocated/$00/     {CMP BYTE PTR [>EmsAllocated],0}
   $74/$06/                       {JZ  XferPtr}
   $FF/$1E/>EmsPAddr/             {CALL DWORD PTR [>EmsPAddr]}
   $EB/$02/                       {JMP SHORT EmsPDone}
   {XferPtr:}
   $58/                           {POP AX}
   $5A);                          {POP DX}
  {EmsPDone:}

function GetEmsP(P : Pointer) : Pointer;
  {-Return physical pointer for an EMS pointer.
    Don't call unless EMS has been allocated!}

function EmsResult : Word;
  {-Return most recent error/status result}

  {$IFDEF Debug}
procedure DumpFreeList;
  {-Write out the free list. for debug}
  {$ENDIF}

  {--------------------- low-level EMS routines -----------------------}

function EmsInstalled : Boolean;
  {-Returns true if EMM is installed}

function EmsPagesAvail : Word;
  {-Returns the number of available pages, or EmsErrorCode in case of error}

function EmsPageFramePtr : Pointer;
  {-Returns the page frame base address as a pointer to the page frame}

function AllocateEmsPages(NumPages : Word) : Word;
  {-Allocates pages and returns a handle, or EmsErrorCode}

function MapEmsPage(Handle, LogicalPage : Word; PhysicalPage : Byte) : Boolean;
  {-Maps the page window, returning true if successful}

function DeallocateEmsHandle(Handle : Word) : Boolean;
  {-Deallocates the indicated handle and the memory associated with it}

  {=======================================================================}

implementation

const
  PageSize = 16384;               {Bytes in an EMS page}
  MaxPP = 3;                      {Highest physical page in window, 0..MaxPP}
  MaxFreeList = 2730;             {PageSize div SizeOf(FreeRec)}

type
  FreeRec =
    record
      PageNum : Word;
      PageOfs : Word;
      Size : Word;
    end;
  FreeRecPtr = ^FreeRec;
  FreeList = array[1..MaxFreeList] of FreeRec;
  FreeListPtr = ^FreeList;

  MapRec =
    record
      LogicalPage : Word;
      Touched : Word;
    end;
  MapArray = array[0..MaxPP] of MapRec;

var
  Status : Word;                  {Status, like a Turbo runtime error}
  Handle : Word;                  {EMS handle when allocated}
  PagesFree : Word;               {Number of EMS pages still untouched}
  CurPage : Word;                 {Current logical page for allocation}
  CurOfst : Word;                 {Offset within current logical page}
  MaxFree : Word;                 {Highest allowed free list entry}
  TimeCount : Word;               {Quasi-time for LRU replacement}
  FreePtr : Pointer;              {Free list pointer, should be 0:0}
  FP : FreeListPtr;               {Physical (mapped) free list pointer}
  FTop : FreeRecPtr;              {Pointer to top free list record}
  PageMap : MapArray;             {Maps logical page to physical}
  SaveExit : Pointer;             {Previous exit handler}

  {$L EMSHEAP.OBJ}
  function EmsPagesAvail : Word; external;
  function EmsPageFramePtr : Pointer; external;
  function AllocateEmsPages(NumPages : Word) : Word; external;
  function MapEmsPage(Handle, LogicalPage : Word; PhysicalPage : Byte) : Boolean; external;
  function DeallocateEmsHandle(Handle : Word) : Boolean; external;
  procedure MergeFreeList; external;
  function GetEmsP(P : Pointer) : Pointer; external;

  function EmsInstalled : Boolean;
    {-Returns true if EMM is installed}
  var
    F : file;
    SaveMode : Byte;
  begin
    SaveMode := FileMode;
    FileMode := 0;
    Assign(F, 'EMMXXXX0');
    Reset(F);
    if IoResult = 0 then begin
      EmsInstalled := True;
      Close(F);
    end else
      EmsInstalled := False;
    FileMode := SaveMode;
  end;

  procedure Error(ErrStatus : Word);
    {-Hook for error handling, called whenever error is detected}
  begin
    Status := ErrStatus;
    if HaltOnError then begin
      WriteLn('EMS heap error ', Status);
      Halt(1);
    end;
  end;

  procedure InitEmsHeap(TotalBytes, FreeListBytes : LongInt);
    {-Allocate EMS space for use by manager}
  const
    InitOfst = 2;                 {Starting offset of free list, NOT ZERO}
  var
    PagesR : Word;
    PagesA : Word;
    PP : Word;
  begin
    if EmsAvailable then
      if not EmsAllocated then begin
        {Allocate the overall heap block from EMS}
        PagesR := (TotalBytes+PageSize-1) div PageSize;
        PagesA := EmsPagesAvail;
        if PagesA < PagesR then
          PagesR := PagesA;
        Handle := AllocateEmsPages(PagesR);
        if Handle = EmsErrorCode then begin
          Error(EmsAllocError);
          Exit;
        end;
        EmsAllocated := True;

        {Initialize the heap manager}
        PagesFree := PagesR;
        CurPage := 0;
        CurOfst := InitOfst;
        FreeCnt := 0;
        TimeCount := 0;
        for PP := 0 to MaxPP do
          with PageMap[PP] do begin
            LogicalPage := $FFFF;
            Touched := 0;
          end;

        {Allocate the free list}
        if FreeListBytes > PageSize-InitOfst then
          FreeListBytes := PageSize-InitOfst;
        FreePtr := Ptr(0, InitOfst);
        Inc(CurOfst, FreeListBytes);
        MaxFree := FreeListBytes div SizeOf(FreeRec);
      end;
  end;

  procedure ExitEmsHeap;
    {-Deallocate EMS space -- automatically called when program shuts down}
  var
    Junk : Boolean;
  begin
    if EmsAllocated then begin
      Junk := DeallocateEmsHandle(Handle);
      PagesFree := 0;
      MaxFree := 0;
      EmsAllocated := False;
    end;
  end;

  procedure AddFreeList(PNum, POfs, Siz : Word);
    {-Add block to free list}
  begin
    {Exit in case of a do-nothing request}
    if Siz = 0 then
      Exit;

    {Assure there's space for one more item on free list}
    if FreeCnt >= MaxFree then begin
      Error(EmsOutOfFreeList);
      Exit;
    end;

    {Add item to the free list}
    FP := GetEmsP(FreePtr);
    Inc(FreeCnt);
    FTop := @FP^[FreeCnt];
    with FTop^ do begin
      Size := Siz;
      PageNum := PNum;
      PageOfs := POfs;
    end;

    {$IFDEF Debug}
    WriteLn(^M^J'Page  Offset  Size');
    WriteLn(PNum:4, '  ', POfs:5, '  ', Siz:5, ' freed');
    {$ENDIF}

    {Merge free blocks where possible}
    MergeFreeList;

    {$IFDEF Debug}
    DumpFreeList;
    {$ENDIF}
  end;

  procedure GetEms(var P; ReqSize : Word);
    {-Allocate space from EMS heap, returning special EMS pointer}
  var
    Pt : Pointer absolute P;
    F : Word;
  begin
    if EmsAllocated then begin

      {Validate the request}
      if (ReqSize = 0) or (ReqSize > PageSize) then begin
        Pt := nil;
        Error(EmsInvalidGet);
        Exit;
      end;

      if FreeCnt > 0 then begin
        {Scan the free list for a suitable block}
        FP := GetEmsP(FreePtr);
        for F := 1 to FreeCnt do
          with FP^[F] do
            if Size >= ReqSize then begin
              {Return the first free block that's large enough}
              Pt := Ptr(PageNum, PageOfs);
              if Size > ReqSize then begin
                {Adjust the remaining size of the free block}
                Inc(PageOfs, ReqSize);
                Dec(Size, ReqSize);
              end else begin
                {Remove the free block}
                FP^[F] := FP^[FreeCnt];
                Dec(FreeCnt);
              end;
              Exit;
            end;
      end;

      {Check for space at the top of the current page}
      if PageSize-CurOfst >= ReqSize then begin
        Pt := Ptr(CurPage, CurOfst);
        Inc(CurOfst, ReqSize);
        Exit;
      end;

      {See if another page is available}
      if PagesFree <= 1 then begin
        Pt := nil;
        Error(EmsOutOfMemory);
        Exit;
      end;

      {Add what's left of the current page to the free list}
      AddFreeList(CurPage, CurOfst, PageSize-CurOfst);
      if Status <> EmsSuccess then
        Exit;

      {Return pointer at start of next page}
      Inc(CurPage);
      Dec(PagesFree);
      Pt := Ptr(CurPage, 0);
      CurOfst := ReqSize;

    end else
      GetMem(Pt, ReqSize);
  end;

  procedure FreeEms(var P; ReqSize : Word);
    {-Free allocated EMS heap space}
  var
    EPt : EmsPtr absolute P;
    Pt : Pointer absolute P;
  begin
    if EmsAllocated then begin
      with EPt do begin
        {Validate free request}
        if ((PageNum > CurPage) or
            (PageOfs > PageSize) or
            (ReqSize > PageSize))
        then begin
          Error(EmsInvalidFree);
          Exit;
        end;
        {Add item to free list}
        AddFreeList(PageNum, PageOfs, ReqSize);
        if Status <> EmsSuccess then
          Exit;
      end;
      Pt := nil;
    end else
      FreeMem(Pt, ReqSize);
  end;

  function EmsMemAvail : LongInt;
    {-Return total EMS heap space available}
  var
    Tot : LongInt;
    F : Word;
  begin
    if EmsAllocated then begin
      {Start with space in all pages not completely used}
      Tot := LongInt(PagesFree)*PageSize;
      {Reduce by space used in current page}
      Dec(Tot, CurOfst);
      {Add space in the free list}
      FP := GetEmsP(FreePtr);
      for F := 1 to FreeCnt do
        Inc(Tot, FP^[F].Size);
      EmsMemAvail := Tot;
    end else
      EmsMemAvail := MemAvail;
  end;

  function EmsMaxAvail : LongInt;
    {-Return largest EMS heap block available}
  var
    Max : Word;
    F : Word;
  begin
    if EmsAllocated then begin
      if PagesFree > 1 then
        {One or more complete pages are still available}
        EmsMaxAvail := PageSize
      else begin
        {Start with space remaining in current page}
        Max := PageSize-CurOfst;
        {Scan the free list for larger chunks}
        FP := GetEmsP(FreePtr);
        for F := 1 to FreeCnt do
          with FP^[F] do
            if Size > Max then
              Max := Size;
        EmsMaxAvail := Max;
      end;
    end else
      EmsMaxAvail := MaxAvail;
  end;

  function EmsResult : Word;
    {-Return most recent error/status result}
  begin
    EmsResult := Status;
    Status := EmsSuccess;
  end;

  {$IFDEF Debug}
  procedure DumpFreeList;
    {-Write out the free list}
  var
    F : Word;
  begin
    if EmsAllocated then begin
      FP := GetEmsP(FreePtr);
      for F := 1 to FreeCnt do
        with FP^[F] do
          WriteLn(PageNum:4, '  ', PageOfs:5, '  ', Size:5);
    end;
  end;
  {$ENDIF}

  {$F+}
  procedure AutoExit;
    {-Automatically called when program ends}
  begin
    ExitProc := SaveExit;
    ExitEmsHeap;
  end;
  {$F-}

begin
  Status := EmsSuccess;
  EmsAllocated := False;
  FreeCnt := 0;
  EmsAvailable := EmsInstalled;
  if EmsAvailable then begin
    FreePtr := EmsPageFramePtr;
    FrameSeg := Seg(FreePtr^);
  end;
  SaveExit := ExitProc;
  ExitProc := @AutoExit;
  EmsPAddr := @GetEmsP;
end.

