                       F i l e    I n f o r m a t i o n

* DESCRIPTION
Program containing two useful tools for statistical programs: a function
to evaluate the area under the gaussian ( or normal ) curve and a
procedure to reverse the first, that is given an area under the gaussian
curve it will return the corresponding Z. Author Marc Lalibert. Ver T1.0.

* ASSOCIATED FILES
GAUSS.PAS
GAUSS.DAT
GAUSS.TPM
GAUSS.TXT
README.WP5

* KEYWORDS
TURBO PASCAL V4.0 MATH CURVE ALGORITHM GAUSSIAN

==========================================================================
}
PROGRAM GaussianCurveByTabulatedValues;

{ This program demonstrate the use of two simple statistical tools.

The first one simply calculate for a given Z the area under the gaussian
curve between minus infinity and Z. This function use tabulated values and
interpolation. It is more accurate than the finest tables I've been able to
found. It is be accurate to more than five decimal places and might be accurate
to 10 or 12 decimal places.

By the way the tabulated values the program use have been generated by the
Numerical Toolbox using the Romberg Algorithm ( ROMBERG.PAS ).

The second one is a bit more tricky. It approximate for a given area under
the gaussian curve the corresponding Z. It use the iterative root-finding
Secant Algorithm.

This program have been written by Marc Lalibert, Qubec.
Last modified on 88/07/23.

I have been inspired by a program called NORMAL.PAS writen by Bob Fruit. }

{$IFOPT N+}
  TYPE
    FLOAT = DOUBLE;
{$ELSE}
  TYPE
    FLOAT = REAL;
{$ENDIF}

CONST
  ZMin : FLOAT = -8.0;                 { The minimum tabulated value of Z. }
  MinZ = 0;                            { The rank of the first tabulated }
                                         { value, ZMin. }
  ZMax : FLOAT = 0.0;                  { The maximum tabulated value of Z. }
  MaxZ = 80;                           { The rank of the last tabulated }
                                         { value, ZMax. }
  DeltaZ : FLOAT = 0.1;                { The spacing between each Z. }
  Zero : FLOAT = 1E-15;                { Zero, nearly enough. }
  DataFileName= 'GAUSS.DAT';           { Name of the file containing the data. }

VAR
  TabulatedValues : ARRAY[  MinZ..MaxZ ] OF FLOAT;
                                       { This array contains the values of the }
                                         { gaussian distribution. }
  SQRT2PI : FLOAT;                     { Constant := SQRT( 2*PI ) }
  SQRDZ2 : FLOAT;                      { Constant := SQR( DeltaZ ) / 2 }
  Z : FLOAT;                           { The value of Z. }
  A : FLOAT;                           { The area. }
  Ch : CHAR;                           { Dummy. }
  Error : WORD;                        { The value of the error in FindZ, if any. }
  I : WORD;                            { Dummy }

PROCEDURE Initialization;
{ Initialize the program. }

VAR
  I : WORD;                            { Dummy. }
  Data : TEXT;                         { The file containing the tabulated }
                                         { values. }

BEGIN
  ASSIGN( Data,DataFileName );
  RESET( Data );
  FOR I := MinZ TO MaxZ DO
    READLN( Data,TabulatedValues[ I ] );
  CLOSE( Data );
  SQRT2PI := SQRT( 2*PI );
  SQRDZ2 := DeltaZ*DeltaZ/2;
END; { PROCEDURE Initialization }


FUNCTION Gauss1stDerivative( Z : FLOAT ) : FLOAT;
{ Calculate the first derivative of the Gaussian function at Z. }

BEGIN
  Gauss1stDerivative := - Z*EXP( -Z*Z/2 )/SQRT2PI
END; { PROCEDURE Gauss1stDerivative }

FUNCTION AreaUnderGauss( Z : FLOAT ) : FLOAT;
{ Calculate the area under the gaussian curve between minus infinity and Z.
  Use tabulated values and a clamped cubic spline to make an interpolation
  between them. }

VAR
  I : WORD;                            { The subscript of TabulatedValues cor- }
                                         { responding to the tabulated value }
                                         { immediatelity lower than Z. }
  ZPositive : BOOLEAN;                 { TRUE if Z is positive. }
  S : FLOAT;                           { A normalized distance with a value }
                                         { between 0 and 1. It is equal to }
                                         { ( Z - ( ZMin + I*DeltaZ ) )/DeltaZ. }
  T : FLOAT;                           { 1 - S. }
  Tempo : FLOAT;                       { Dummy variable. }

BEGIN
  IF Z >= 0 THEN
  BEGIN
    IF ABS( Z ) <= Zero THEN
    BEGIN
      AreaUnderGauss := 0.5;
      EXIT
    END
    ELSE BEGIN
      Z := -Z;
      ZPositive := TRUE
    END
  END
  ELSE ZPositive := FALSE;
  IF Z <= ZMin THEN
  BEGIN
    IF ZPositive THEN AreaUnderGauss := 1
    ELSE AreaUnderGauss := 0;
    EXIT
  END;
  I := TRUNC( ( Z - ZMin )/DeltaZ );
  S := ( Z - ( ZMin + I * DeltaZ ) ) / DeltaZ;
  T := 1 - S;

  { Interpolation. }

  Tempo := T*TabulatedValues[ I ] -
           SQRDZ2*T*T*S*Gauss1stDerivative( ZMin + I*DeltaZ );
  INC( I );
  Tempo := Tempo + S*TabulatedValues[ I ] -
           SQRDZ2*S*S*T*Gauss1stDerivative( Zmin + I*DeltaZ );

  IF ZPositive THEN AreaUnderGauss := 1 - Tempo
  ELSE AreaUnderGauss := Tempo
END; { FUNCTION AreaUnderGauss }


PROCEDURE FindZ(
  A : FLOAT;                           { The area under the normal curve. }
  VAR Z : FLOAT;                       { The value of Z to find. }
  VAR Count : WORD;                    { The number of cycles required. }
  VAR Error : WORD );                  { 0 if no error found;
                                         1 if area <= 0;
                                         2 if area >= 1;
                                         3 if Count > CountMax }

{ Given the area under the normal curve, find the value of the
  corresponding Z. }

CONST
  CountMax = 100;                      { The maximum number of }
                                         { iterations allowed. }

VAR
  Guess1,Guess2,Guess3 : FLOAT;        { Successive guess of Z. }
  A1,A2 : FLOAT;                       { The areas corresponding to Guess1 }
                                         { and Guess2. }
  DeltaA : FLOAT;                      { The difference between A2 and A. }
                                         { Used to speed-up things. }
BEGIN
  Z := 0;
  Count := 0;
  Error := 0;
  IF ( A <= Zero ) OR ( A >= ( 1 - Zero ) ) THEN
  BEGIN
    IF A <= Zero THEN Error := 1 ELSE Error := 2;
    EXIT
  END;
  IF ABS( 0.5 - A ) <= Zero THEN Z := 0
  ELSE BEGIN
    Guess1 := 0.0;
    A1 := 0.5;
    IF A > 0.5 THEN
    BEGIN
      Guess2 := 2;
      A2 := AreaUnderGauss( Guess2 );
      DeltaA := A2 - A;
      IF ( A - 0.5 ) < ABS( DeltaA ) THEN
      BEGIN
        Guess1 := 2;
        Guess2 := 0.0;
        A1 := A2;
        A2 := 0.5;
        DeltaA := 0.5 - A;
      END;
      WHILE ( ABS( DeltaA ) >= Zero ) AND ( Count <= CountMax ) DO
      BEGIN
        Guess3 := Guess2 - ( Guess2 - Guess1 )*DeltaA/( A2 - A1 );
        A1 := A2;
        A2 := AreaUnderGauss( Guess3 );
        DeltaA := A2 - A;
        Guess1 := Guess2;
        Guess2 := Guess3;
        INC( Count )
      END { WHILE }
    END
    ELSE BEGIN
      Guess2 := -2;
      A2 := AreaUnderGauss( Guess2 );
      DeltaA := A2 - A;
      IF ABS( 0.5 - A ) < ABS( DeltaA ) THEN
      BEGIN
        Guess1 := -2;
        Guess2 := 0.0;
        A1 := A2;
        A2 := 0.5;
        DeltaA := 0.5 - A
      END;
      WHILE ( ABS( DeltaA/A ) >= Zero ) AND ( Count <= CountMax ) DO
      BEGIN
        Guess3 := Guess2 - ( Guess2 - Guess1 )*DeltaA/( A2 - A1 );
        A1 := A2;
        A2 := AreaUnderGauss( Guess3 );
        DeltaA := A2 - A;
        Guess1 := Guess2;
        Guess2 := Guess3;
        INC( Count )
      END { WHILE }
    END; { IF }
    Z := Guess2;
    IF Count > CountMax THEN Error := 3
  END
END; { PROCEDURE FindZ }

BEGIN
  Initialization;
  WRITELN( 'This is a demonstration program for two statistical tools.' );
  WRITELN;
  WRITELN( 'The first one, AreaUnderGauss, evaluate the area between minus infinity' );
  WRITELN( 'and a given value of Z. ( Note: Z := ( Average - X )/ Standard deviation ).' );
  WRITELN;
  WRITELN( 'The second is the reverse of the first. It calculate the Z corresponding' );
  WRITELN( 'to the area under the gaussian curve.' );
  WRITELN;
  REPEAT
    WRITE( 'Enter a value for Z : ' );
    READLN( Z );
    A := AreaUnderGauss( Z );
    WRITELN( 'The area is ',A );
    WRITELN( 'Now calculating back Z... ' );
    FindZ( A,Z,I,Error );
    CASE Error OF
      0 : WRITELN( 'Z = ',Z,' and ',I,' iterations have been required.' );
      1 : WRITELN( 'ERROR : The Area is less than or equal to 0.' );
      2 : WRITELN( 'ERROR : The Area is greater than or equal to 1.' );
      3 : WRITELN( 'ERROR : After ',I,' iterations the program stopped. Last guess is ',Z )
    END;
    WRITELN;
    WRITE( 'Do you want to try again ? ( Y/N ) ' );
    READLN( Ch );
    WRITELN;
  UNTIL NOT( Ch IN [ 'y','Y' ] );
END.

