{
                       F i l e    I n f o r m a t i o n 

* DESCRIPTION
Unit and demomstration for Virtual arrays, auto-select RAM-only or RAM
disk large arrays. Make, Dispose, Load, Save up to 32 MB arrays. Requires
TPRARRAY.TPU and TPVARRAY.TPU from Turbo Professional 4.0 by TurboPower
Software. Author: R. Jones.

* ASSOCIATED FILES
VIRTUAL.PAS
TESTVIRT.PAS


* KEYWORDS
TURBO PASCAL V4.0 ARRAY RAM DISK
==========================================================================
}
unit virtual;

{
  The virtual array unit "VIRTUAL.TPU".  This unit and the demo program
  "testvirt.pas" both require Turbo Professional 4.0 by TurboPower Software.
  (408-438-8608).

  This unit was developed to assist me in handling very large arrays of
  real numbers in my ophthalmologic research.  Please feel free to use
  and modify the routines as needed.  As presently configured, the unit
  will automatically determine if sufficient RAM exists to set up large
  arrays on the heap.  If insufficient RAM is available for the complete
  array, a virtual array is created (RAM/disk based), but only if sufficient
  disk storage space is available.  The storage routine even searches
  other disk drives (A-J) for space, returning to the default drive/path
  after storing the array and updating the file name.

  With respect to file names, a routine is used which automatically
  creates sequentially number files name "TmpFxxxx.@%@", where the
  xxxx's are digits from "0000" to "9999".  These file names are
  re-used if the included DisposeArray routine is used to dispose
  of previously allocated arrays.  See below for more details.
  You can override the automatic allocation of file names by supplying
  a file name when calling MakeRealArray.

  Also of note:  by changing the definition of the VirtualRec record
  from type "real" to "string", along with a couple of changes
  indicated in comment sections in the MakeRealArray function, one
  could create arrays for other data types very simply, since the
  other routines past only the record structure.


  Author:  Robert L. Jones
           4521 Campus Drive, #111
           Irvine, CA  92715

  CompuServe User ID 71251,2566

  Released to the public domain 9/5/88.
}


interface

USES tpRarray, tpVarray, dos;

TYPE
  VirtualRec = record
                 A     : tparray;   { RAM/disk based real array }
                 FName : string;    { File name associated with array }
                 UH    : boolean;   { Use Heap space }
               END;

{ Definitions }
function DisposeArray (VAR VR : VirtualRec; EraseFile : boolean) : boolean;
function LoadArray (VAR FName : string; VAR VR : VirtualRec) : boolean;
function MakeRealArray (VAR VR : VirtualRec; row,col:word;
                        VAR FName : string ) : boolean;
function SaveArray (VAR VR : VirtualRec; OverWrite : boolean) : boolean;


implementation

CONST
  TempFileNumber : string[4] = '0000';   { unit global initializing string }
  MaxFileSize = 8000000;                 { 8 MB maximum, may inc to 32 MB }
VAR
  FreeMem : longint;
  value : real;
  i : integer;


function DisposeArray (VAR VR : VirtualRec; EraseFile : boolean) : boolean;
{
  If using heap (UH) is true, simply remove record from heap.  If not,
  then cyclically dispose of file names down to "0000", decrementing the
  const variable TempFileNumber.  The array is erased from the disk if
  EraseFile is TRUE (this boolean has no affect upon RAM-only arrays).
  TempFileNumber is not decremented if the final letter in FName <> '@',
  since a user supplied file name must be in effect.  If no error is
  reported during the disposal, the function returns TRUE.
}
VAR
  i, code : integer;
  Err : word;
BEGIN
  DisposeArray := TRUE;
  IF not (VR.UH) THEN
    BEGIN
      IF VR.FName[length(VR.FName)] = '@' THEN
        {
          FName was probably assigned by AssignFileName, therefore decrement
          the global TempFileNumber string.
        }
        BEGIN
          TempFileNumber :=
              COPY(VR.FName,length(VR.FName)-7,length(VR.FName)-3);
          VAL(TempFileNumber,i,code);
          dec(i);
          IF (i < 0) THEN i := 0;
          STR(i,TempFileNumber);
          WHILE (length(TempFileNumber) < 4) DO
             TempFileNumber := '0'+TempFileNumber;
        END;
      tpVarray.DisposeA (VR.A, EraseFile);    { erases file }
      Err := tpVarray.ErrorA;
      IF Err <> 0 THEN DisposeArray := FALSE;
   END
 ELSE
   BEGIN
     tpRarray.DisposeA (VR.A);    { removes file from heap }
     Err := tpRarray.ErrorA;
     IF Err <> 0 THEN DisposeArray := FALSE;
   END;
END;


function LoadArray (VAR FName : string; VAR VR : VirtualRec) : boolean;
{
  Function searches disk for file "FName", returning FALSE if not
  found, or an error occurs during loading.  RAM-only or RAM/disk arrays
  are automatically selected based on available memory and file size.
  Array data are returned in variable VR.  Successful allocation returns TRUE.
}
VAR
  Err : word;
  FreeMem : longint;
  S : searchrec;
BEGIN
  LoadArray := FALSE;
  FreeMem := longint (MemAvail shr 1);   { 1/2 of all heap space }
  FindFirst(FName, $3F, S);
  IF (DosError <> 0) OR (FreeMem < 1024) THEN exit;
  LoadArray := TRUE;
  VR.FName := FName;
  IF ((1.1 * s.size) < FreeMem) THEN
    BEGIN
      VR.UH := TRUE;
      tpRarray.LoadA (VR.A, FName);
      Err := tpRarray.ErrorA;
      IF Err <> 0 THEN LoadArray := FALSE;
    END
  ELSE
    BEGIN
      VR.UH := FALSE;
      tpVarray.LoadA (VR.A, FName, FreeMem);
      Err := tpVarray.ErrorA;
      IF Err <> 0 THEN LoadArray := FALSE;
    END;
END;


function SaveArray (VAR VR : VirtualRec; OverWrite : boolean) : boolean;
{
  Function first searches disk for file "VR.FName".  If OverWrite is FALSE
  and file is found, or an error occurs during saving, the function
  returns FALSE.  If OverWrite is TRUE the function will overwrite any
  file of the same name.  RAM-only or RAM/disk arrays are automatically
  selected based on the VR.UH flag.  (VR.FName was provided by you when
  MakeRealArray was first called.)
}
VAR
  Err : word;
  FreeMem : longint;
  S : searchrec;
BEGIN
  SaveArray := FALSE;
  FindFirst(VR.FName, $3F, S);
  IF (DosError = 0) AND not(OverWrite) THEN exit;  { prevents erasing data }
  SaveArray := TRUE;
  IF (VR.UH) THEN
    BEGIN
      tpRarray.StoreA (VR.A, VR.FName);
      Err := tpRarray.ErrorA;
      IF Err <> 0 THEN SaveArray := FALSE;
    END
  ELSE
    BEGIN
      tpVarray.StoreA (VR.A);
      Err := tpVarray.ErrorA;
      IF Err <> 0 THEN SaveArray := FALSE;
    END;
END;


procedure AssignFileName (VAR FName : string);
{
  Cyclically assign file names using a coding scheme to generate names
  unlikely to be found on a disk ("*.@%@"), where the "*" is TmpFxxxx.
  "xxxx" can vary from "0001" to "9999"; and will else default to "0000".

  9999 files is rather optimistic since DOS will not allow that many files
  to be open at any one time.  To ensure a reasonable number, such as 10-15,
  check your config.sys file to see if it contains a statement like
  "FILES = 25" (25 or 30 is OK).
}
VAR
  i, code : integer;
BEGIN
  VAL(TempFileNumber,i,code);
  INC(i);
  IF (i > 9999) THEN i := 0;
  STR(i,TempFileNumber);
  WHILE (length(TempFileNumber) < 4) DO TempFileNumber:='0'+TempFileNumber;
  FName := 'TmpF' + TempFileNumber + '.@%@';
END;


function ExamineDisks (VAR MemReq : longint; VAR currdrive : string) : string;
{
  Search current drive, then drives A-J (without repeating current)
  for sufficient disk space on which to store data array.  Sufficient
  here is 110% of the actual memory required by the virtual array
  elements, to allow for overhead and header storage information
  (the 1.1 value is thus somewhat arbitrary).  The OK boolean precludes
  crashes from open drives, etc.
}
CONST
  i : byte = 1;
VAR
  OK : boolean;
  tempdrive : string;
BEGIN
  ExamineDisks := '';
  {$i-} GetDir(0,currdrive); OK:=(IOResult=0); {$i+}    { determine current }
  IF (DiskFree(0) > (1.1 * MemReq)) THEN
     BEGIN
       ExamineDisks := currdrive;
       exit;
     END;
  REPEAT         { search all potential drives A-J }
    {$i-} GetDir(i,tempdrive); OK:=(IOResult=0); {$i+}
    IF (UpCase(currdrive[1]) <> UpCase(tempdrive[1])) THEN
      IF (DiskFree(i) > (1.1 * MemReq)) THEN
        BEGIN              { exit upon finding drive with sufficient memory }
          ExamineDisks := tempdrive;
          exit;
        END;
    INC(i);
  UNTIL (i = 10);   { 10  -->  drive = J }
END;


function MakeRealArray (VAR VR : VirtualRec; row,col:word;
                        VAR FName : string ) : boolean;
  {
    MakeRealArray first attempts to allocate enough space for the array on
    the heap using the TProf unit "tpRarray.tpu".  If insufficient space is
    available, the virtual array unit (tpVarray.tpu) is called.  If neither
    arrays can be created, MakeRealArray returns a FALSE boolean result.  A
    second boolean flag within the VirtualRec is used to indicate whether
    RAM-only or a RAM/disk virtual array is being used.  This flag is
    crucial in order to know which TProf routines such as ClearA or DisposeA
    to use.  That is, to initial an TProf array when both tpRarray.tpu and
    tpVarray.tpu are included, one needs to call the unit indentifier.  For
    example:   "tpRarray.ClearA(VR.A, value, tpRarray.fastinit);".  In a
    program one can use an IF/ELSE statement to distinguish between the two
    states:  "IF (VR.UH) THEN ... ELSE ...;".  Examples can be found in this
    function code below as well as in the demo program.

    While a complete description of T.Prof arrays is in their manual, a brief
    review of the array structure can help.  All arrays are zero-based.
    That is for a 10 element array, the array begins at 0 and continues
    through 9 (ie, 0..9).  Therefore, access is 1 less than the column
    or row number which you want to locate.

    Heap space is allocated a call to MemAvail.  1/2 of this value is
    assigned to "FreeMem".  (Depending on your application, you may prefer
    to change all MemAvail calls in VIRTUAL.TPU to MaxAvail [MemAvail is all
    free RAM; MemAvail is largest block of free RAM].)  The space required
    by the array is determined by the number of rows & columns and the
    "SizeOf" the element type.  At least 1024 bytes must be available on
    the heap to impliment a RAM/disk virtual array.  All values are
    initialized to the value of "value" which is presently set to 0.

    By changing the definition of "value" from type real to type string,
    for example, one could use the same function for large string records.
    To use this function in this way requires you to change the type of
    variable used in the SizeOf function.  (After performing these changes
    you probably would want to copy the function and re-name it MakeStrArray.)

    If the size of the requested array is too large for RAM, MakeRealArray
    calls ExamineDisks to find a diskdrive containing enough space to
    store the array.  (See above for details.)  If none is found MakeRealArray
    returns FALSE.  32 MB is absolute maximum for any single virtual array;
    default maximum is a constant, set at 8 MB.

    FName will be the default name of the virtual array file, unless
    FName = '', in which case the internal routine AssignFileName will be
    called to provide automatically assigned names.  This feature is useful
    for temporary data arrays for intermediate results.

    The call "Err := tpXarry.ErrorA;" prevents the T.Prof. routine from
    acting upon an error condition similar to calling IOResult in Turbo Pascal.
    These calls are used by VIRTUAL.TPU to avoid any program disruption.
  }

CONST
  value : real = 0.0;    { may change to "value : string = '';" }
VAR
  FreeMem,MemReq : longint;
  Err : word;
  x,y : byte;
  OK  : boolean;
  currdrive,tempdrive : string;
BEGIN
  MakeRealArray := FALSE;
  FreeMem := longint (MemAvail shr 1);   { 1/2 of all heap space }
  MemReq  := longint (SizeOf(real)) * row * col;

  IF ((1.1 * MemReq) < FreeMem) THEN
    { only use heap if sufficient RAM is present for the array }
    BEGIN
       VR.UH := TRUE;    { set VirtualRec flag to indicate RAM-only usage }
       VR.FName := FName;
       MakeRealArray := TRUE;

       { can change from real to string to allocate different array types }
       tpRarray.MakeA (VR.A, row, col, SizeOf(real));
       Err := tpRarray.ErrorA;
       IF Err <> 0 THEN MakeRealArray := FALSE;

       tpRarray.ClearA (VR.A, value, tpRarray.fastinit);
       Err := tpRarray.ErrorA;
       IF Err <> 0 THEN MakeRealArray := FALSE;

       exit;
    END  { of RAM-only array }
  ELSE
    BEGIN
      { exit IF the memory requirements are greater than the 8 MB max size }
      IF ((1.1 * MemReq) > MaxFileSize) THEN exit;

      { empirically derived limits which seem to work well for virtual arrays }
      IF ((FreeMem > 0.004 * MemReq) AND (FreeMem > 1024)) THEN
        BEGIN
          IF (FName = '') THEN AssignFileName(VR.FName)
          ELSE VR.FName := FName;
          VR.UH := FALSE;  { set VirtualRec flag to indicate RAM/disk usage }

          { locate a drive with sufficient storage requirements }
          tempdrive := ExamineDisks(MemReq, currdrive);

          { re-set original drive }
          {$i-} GetDir(0,currdrive); OK:=(IOResult=0); {$i+}
          IF (tempdrive = '') THEN exit;  { insufficient storage on A-J }

          { update file name to storage drive/path }
          IF tempdrive[length(tempdrive)] <> '\' THEN tempdrive := tempdrive + '\';
          VR.FName := tempdrive + VR.FName;

          MakeRealArray := TRUE;

          { can change from real to string to allocate different array types }
          tpVarray.MakeA (VR.A, row, col, SizeOf(real), VR.FName, FreeMem);
          Err := tpVarray.ErrorA;
          IF Err <> 0 THEN MakeRealArray := FALSE;

          tpVarray.ClearA (VR.A, value, tpVarray.fastinit);
          Err := tpVarray.ErrorA;
          IF Err <> 0 THEN MakeRealArray := FALSE;

          { re-set original drive }
          {$i-} ChDir(currdrive); OK:=(IOResult=0); {$i+}
        END;
    END; { of ELSE use virtual array:  RAM/disk based array }
END;  { of function MakeRealArray }

BEGIN
  {
    Turn-off Turbo Professional large array error check reporting.
    Errors, however, are still monitored, but functions simply return
    FALSE rather than TRUE if any should occur.  This method avoids
    messing-up screens, etc.
  }
  HaltOnError := FALSE;
  WriteError := FALSE;
  RangeCheck := FALSE;
  KeepDiskCurrent := TRUE;   { see T.Prof. manual for details }
END.
