UNIT Math;

INTERFACE

USES IOStuff, Crt, Dos;

FUNCTION Parse(S : AnyStr;VAR Error : Boolean):Real;

{ Function Parse evaluates the mathmatical expression }
{ contained in string S and returns the value of the  }
{ expression.  The string may contain values, all the }
{ normal operators, parentheses and function references. }
{ This parser is similar to Namir Shammas's parser }
{ presented in The Turbo Pascal Toolbook.  This version}
{ removes the logic for variables, modifies the flow and}
{ corrects several bugs }

IMPLEMENTATION

CONST
     MaxStack = 100;   { Max size of the expression stack }
TYPE
     OType = (OPER,FUNC,PUSH,EOLN);

     RPNRecord = Record
                  OpType   : OType;
                  Index    : Integer;
                  Priority : Integer;
                  Value    : Real;
                End;

     RPNArray = Array[1..MaxStack] of RPNRecord;
 {===============================================================}
 PROCEDURE PrepareS(VAR S: AnyStr);
 { Remove blanks and convert to uppercase }

 VAR
    I    : Integer;
    OutS : AnyStr;
 BEGIN
    OutS := '';
    For I := 1 to Length(S) do
       If S[I] <> ' ' then OutS := OutS + UpCase(S[I]);
    S := OutS;
 END;

 {===============================================================}
 PROCEDURE CheckParen(VAR S: AnyStr; VAR Error : Boolean);
 { Make sure expression has matching left and right parentheses }

 VAR
    C     : Char;
    NOpen : Integer;
    NClos : Integer;
    I,L   : Integer;
    OK    : Boolean;
 BEGIN
    NOpen := 0;
    NClos := 0;
    OK := True;
    I := 1;
    L := Length(S);
    While (I <= L) and OK do Begin
      C := S[I];
      If C = '(' then Begin
        Inc(NOpen);
        OK := NClos <= NOpen;
      End;
      If C = ')' then Begin
        Inc(NClos);
        OK := NClos <= NOpen;
      End;
      Inc(I);
    End;
    If NOpen <> NClos then Error := True;
 END;

 {===============================================================}
 PROCEDURE ScanForMinus(VAR S: AnyStr);
 { Scan for any plus signs or minus signs that indicate positive }
 { or negative numbers and are not the operations plus or minus. }
 { Replace minus with ! and plus with @ }

 VAR
    I,L  : Integer;
 BEGIN
    If S[1] = '-' then S[1] := '!';
    If S[1] = '+' then S[1] := '@';
    I := 2;
    L := Length(S);
    While I <= L do begin
      If S[I] in ['(','+','-','*','/'] then begin
        Inc(I);
        If S[I] = '-' then S[I] := '!';
        If S[I] = '+' then S[I] := '@';
      End;
      Inc(I);
    End;
 END;

 {===============================================================}
 PROCEDURE ScanFunctions(VAR S: AnyStr; VAR Error : Boolean);
 { Scan the expression string for legal functions.  Replace the }
 { function name with a function number }
 {   Function     Name  Num }
 {  Absolute Value ABS = 1  }
 {  Square Root    SQR = 2  }
 {  Logarithm      LOG = 3  }
 {  Exponential    EXP = 4  }
 {  Sine           SIN = 5  }
 {  Cosine         COS = 6  }
 {  Tangent        TAN = 7  }
 {  ArcTangent     ATN = 8  }

 CONST
    FunctName : Array[1..8] of String[4]=
    ('ABS(','SQR(','LOG(','EXP(','SIN(','COS(','TAN(','ATN(');
 VAR
    I,J,Ptr  : Integer;
    FunctNum : String[3];
    Ch       : Char;
 BEGIN
   For I := 1 to 8 do Begin
     Ptr := Pos(FunctName[I],S);
     While Ptr > 0 do Begin
       S[Ptr+3] := '{';
       Str(I:3,FunctNum);
       For J := 1 to 3 do S[Ptr+J-1] := FunctNum[J];
       Ptr := Pos(FunctName[I],S);
     End;
   End;
                 { Now check for bogus functions }
   I := 2;
   J := Length(S);

   While (I <= J) and (Error = False )do Begin
     Ch := S[I-1];
     If (S[I] = '(') and
     (not (Ch in ['+','-','*','/','^','!','@','(',')']))
     then Error := True
     else Inc(I);
   End;

   S := S + '|';
 END;


 {===============================================================}
 PROCEDURE MakeRPNExpr(VAR S   : AnyStr;
                       VAR RPN : RPNArray;
                       VAR Error : Boolean);
 { Converts groomed expression string S into Reverse Polish Notation }
 { Array RPN.  RPN then may be evaluated to find the value of the }
 { expression }

 CONST
     ShiftOffset = 10;
 VAR
    TailPtr, ExprPtr, LastPtr, Offset, StackHeight : Integer;
    I, J, L : Integer;
    ScanCh, C1, C2 : Char;
    R : RPNRecord;

 {===============================================================}
 PROCEDURE PushStack;
 VAR
   ExprVar   : AnyStr;
   I, Dummy  : Integer;
   RealDummy : Real;
   ErrorLoc  : Integer;
 BEGIN
   ExprVar := '';
        { Pull a substring off the string into ExprVar }
   For I := LastPtr to (ExprPtr-1) do ExprVar := ExprVar + S[I];

        { Add a zero in front of numbers starting with decimal. }
        { this circumvents a Turbo Pascal bug in the VAL routine }
        { which will not handle numbers beginning with '.'}
   If ExprVar[1] = '.' then Insert('0',ExprVar,1);

   Inc(StackHeight);
   RPN[StackHeight].Priority := 0;

        { Convert ExprVar into RealDummy -- a real variable }
        { and push RealDummy onto the RPN stack }
   Val(ExprVar,RealDummy,ErrorLoc);
   If ErrorLoc > 0 then Error := True;
   RPN[StackHeight].OpType := PUSH;
   RPN[StackHeight].Index := 0;
   RPN[StackHeight].Value := RealDummy;
 END;

 {===============================================================}
 PROCEDURE DropOpStack(PriorityLevel : Integer);
 { Drop stack and pull pending operations from tail to front }
 BEGIN
   While (RPN[TailPtr].Priority >= PriorityLevel)
       and (TailPtr <= MaxStack) do Begin

      Inc(StackHeight);
      RPN[StackHeight] := RPN[TailPtr];
      Inc(TailPtr);
   End;
 END;

 {===============================================================}
 PROCEDURE SavePending;
 { Save pending operations in the tail of the RPN array }
 BEGIN
   Dec(TailPtr);
   RPN[TailPtr] := R;
 END;

 {===============================================================}
 PROCEDURE CheckStatus;
 { Pull operations off pending stack depending on priorities }
 BEGIN
   If LastPtr < ExprPtr then PushStack;
   LastPtr := ExprPtr + 1;
   If TailPtr = (MaxStack + 1) then SavePending
   Else If R.Priority > RPN[TailPtr].Priority
      then SavePending
      Else Begin
        DropOpStack(R.Priority);
        SavePending;
      End;
 END;

 {===============================================================}
 FUNCTION GetFunctNum : Integer;
 { Extract the coded function number from the expression string }
 VAR
   Nu : String[3];
   I,Dummy1, Dummy2 : Integer;

 BEGIN
   Nu := '';
   For I := 1 to 3 do
     If S[ExprPtr-3+I] in ['0'..'9'] then Nu := Nu + S[ExprPtr-3+I];
   Val(Nu,Dummy1,Dummy2);
   GetFunctNum := Dummy1;
 END;

 {===============================================================}
 { Start of procedure MakeRPNExpr }
 BEGIN
   Error := False;
   For I := 1 to MaxStack do RPN[I].OpType := EOLN;
                        { First groom the expression string }
   PrepareS(S);
   CheckParen(S,Error);
   ScanForMinus(S);
   ScanFunctions(S,Error);
   ExprPtr := 1;
   LastPtr := 1;
   L := Length(S);
   Offset := 0;
   StackHeight := 0;
   TailPtr := MaxStack + 1;
   While (ExprPtr <= L) and (Error = False) do Begin
     ScanCh := S[ExprPtr];

                            { Scan for operation characters }
     If ScanCh in ['-','+','*','/','!','@','^','(',')','{','|']
     then Begin             { Set priority and push operation on stack }
       Case ScanCh Of
      '-' : Begin
             R.OpType := OPER;
             R.Index := 1;
             R.Priority := 1 + Offset;
             R.Value := 0.0;
             CheckStatus;
            End;
      '+' : Begin
             R.OpType := OPER;
             R.Index := 2;
             R.Priority := 1 + Offset;
             R.Value := 0.0;
             CheckStatus;
            End;
      '*' : Begin
             R.OpType := OPER;
             R.Index := 3;
             R.Priority := 2 + Offset;
             R.Value := 0.0;
             CheckStatus;
            End;
      '/' : Begin
             R.OpType := OPER;
             R.Index := 4;
             R.Priority := 2 + Offset;
             R.Value := 0.0;
             CheckStatus;
            End;
      '!' : Begin
             R.OpType := OPER;
             R.Index := 6;
             R.Priority := 3 + Offset;
             R.Value := 0.0;
             CheckStatus;
            End;
      '@' : Begin
             R.OpType := OPER;
             R.Index := 7;
             R.Priority := 3 + Offset;
             R.Value := 0.0;
             CheckStatus;
            End;
      '^' : Begin
             R.OpType := OPER;
             R.Index := 5;
             R.Priority := 4 + Offset;
             R.Value := 0.0;
             CheckStatus;
            End;
      '(' : Begin
             Offset := Offset + ShiftOffset;
             LastPtr := ExprPtr + 1;
            End;
      ')' : Begin
             If LastPtr < ExprPtr then PushStack;
             DropOpStack(Offset);
             Offset := Offset - ShiftOffset;
             LastPtr := ExprPtr + 1;
            End;
      '{' : Begin
             R.OpType := FUNC;
             R.Index := GetFunctNum;
             R.Priority := 5 + Offset;
             R.Value := 0.0;
             LastPtr := ExprPtr + 1;
             Offset := Offset + ShiftOffset;
             CheckStatus;
            End;
      '|' : Begin
             If LastPtr < ExprPtr then PushStack;
             DropOpStack(0);
            End;
      End; {case}
    End; {if}
    Inc(ExprPtr);
  End; {While}
 END;

{===============================================================}
FUNCTION Parse(S : AnyStr;VAR Error : Boolean):Real;
VAR
   RPN         : RPNArray;
   Stack       : Array [1..MaxStack] of Real;
   I, StackPtr : Integer;

BEGIN

    { First convert the expression string to Reverse Polish Notation }
  MakeRPNExpr(S,RPN,Error);

    { Then collapse the RPN stack to evaluate the expression }
  If Error then Exit
  Else Begin
    For I := 1 to MaxStack do Stack[I] := 0.0;
    I := 1;
    StackPtr := 0;
    While (RPN[I].OpType <> EOLN) do begin
  Case RPN[I].OpType of
        OPER : Begin
            Case RPN[I].Index of
                                    { handle everyday operation here }
              1 : Stack[StackPtr-1] := Stack[StackPtr-1] - Stack[StackPtr];
              2 : Stack[StackPtr-1] := Stack[StackPtr-1] + Stack[StackPtr];
              3 : Stack[StackPtr-1] := Stack[StackPtr-1] * Stack[StackPtr];
              4 : Stack[StackPtr-1] := Stack[StackPtr-1] / Stack[StackPtr];
              5 : Stack[StackPtr-1] := exp(Stack[StackPtr] * LN(Stack[StackPtr-1]));
              6 : Stack[StackPtr]   := -Stack[StackPtr];   {Minus}
              7 : Stack[StackPtr]   :=  Stack[StackPtr];   {Plus}
            End; {case}
            If RPN[I].Index < 6 then Dec(StackPtr);
          End;
        FUNC : Begin
            Case RPN[I].Index of
                                   { handle functions here }
              1 : Stack[StackPtr] := Abs(Stack[StackPtr]);
              2 : Stack[StackPtr] := Sqrt(Stack[StackPtr]);
              3 : Stack[StackPtr] := Ln(Stack[StackPtr]);
              4 : Stack[StackPtr] := Exp(Stack[StackPtr]);
              5 : Stack[StackPtr] := Sin(Stack[StackPtr]);
              6 : Stack[StackPtr] := Cos(Stack[StackPtr]);
              7 : Stack[StackPtr] := Sin(Stack[StackPtr])
                                     /Cos(Stack[StackPtr]);
              8 : Stack[StackPtr] := ArcTan(Stack[StackPtr]);
            End;
          End;

       PUSH   : Begin
                      Inc(StackPtr);
                      Stack[StackPtr] := RPN[I].Value;
                    End;
        EOLN       : Begin End;
      End;
      Inc(I);
    End;
    Parse := Stack[1];
  End;
END;


END.