/* Copyright (C) 1993, 1994 Aladdin Enterprises.  All rights reserved.
  
  This file is part of Aladdin Ghostscript.
  
  Aladdin Ghostscript is distributed with NO WARRANTY OF ANY KIND.  No author
  or distributor accepts any responsibility for the consequences of using it,
  or for whether it serves any particular purpose or works at all, unless he
  or she says so in writing.  Refer to the Aladdin Ghostscript Free Public
  License (the "License") for full details.
  
  Every copy of Aladdin Ghostscript must include a copy of the License,
  normally in a plain ASCII text file named PUBLIC.  The License grants you
  the right to copy, modify and redistribute Aladdin Ghostscript, but only
  under certain conditions described in the License.  Among other things, the
  License requires that the copyright notice and this notice be preserved on
  all copies.
*/

/* zfilter.c */
/* Filter creation */
#include "memory_.h"
#include "ghost.h"
#include "errors.h"
#include "oper.h"
#include "gsstruct.h"
#include "ialloc.h"
#include "stream.h"
#include "strimpl.h"
#include "sfilter.h"
#include "srlx.h"
#include "ifilter.h"
#include "files.h"		/* for filter_open, file_d'_buffer_size */

/* Define null handlers for the BCP out-of-band signals. */
private int
no_bcp_signal_interrupt(stream_state *st)
{	return 0;
}
private int
no_bcp_request_status(stream_state *st)
{	return 0;
}

/* <source> .filter_ASCIIHexEncode <file> */
int
zAXE(os_ptr op)
{	return filter_write(op, 0, &s_AXE_template, NULL, false);
}

/* <target> .filter_ASCIIHexDecode <file> */
int
zAXD(os_ptr op)
{	return filter_read(op, 0, &s_AXD_template, NULL, false);
}

/* <source> .filter_BCPEncode <file> */
int
zBCPE(os_ptr op)
{	return filter_write(op, 0, &s_BCPE_template, NULL, false);
}

/* <target> .filter_BCPDecode <file> */
int
zBCPD(os_ptr op)
{	stream_BCPD_state state;
	state.signal_interrupt = no_bcp_signal_interrupt;
	state.request_status = no_bcp_request_status;
	return filter_read(op, 0, &s_BCPD_template, (stream_state *)&state, false);
}

/* <target> .filter_NullEncode <file> */
int
zNullE(os_ptr op)
{	return filter_write(op, 0, &s_NullE_template, NULL, false);
}

/* <source> <bool> .filter_PFBDecode <file> */
int
zPFBD(os_ptr op)
{	stream_PFBD_state state;
	check_type(*op, t_boolean);
	state.binary_to_hex = op->value.boolval;
	return filter_read(op, 1, &s_PFBD_template, (stream_state *)&state, false);
}

/* ------ RunLength filters ------ */

/* <target> <record_size> .filter_RunLengthEncode <file> */
int
zRLE(register os_ptr op)
{	stream_RLE_state state;
	check_int_leu(*op, max_uint);
	state.record_size = op->value.intval;
	return filter_write(op, 1, &s_RLE_template, (stream_state *)&state, false);
}

/* <source> .filter_RunLengthDecode <file> */
int
zRLD(os_ptr op)
{	return filter_read(op, 0, &s_RLD_template, NULL, false);
}

/* <source> <EODcount> <EODstring> .filter_SubFileDecode <file> */
int
zSFD(os_ptr op)
{	stream_SFD_state state;
	check_type(op[-1], t_integer);
	check_read_type(*op, t_string);
	if ( op[-1].value.intval < 0 )
		return_error(e_rangecheck);
	state.count = op[-1].value.intval;
	state.eod.data = op->value.const_bytes;
	state.eod.size = r_size(op);
	return filter_read(op, 2, &s_SFD_template, (stream_state *)&state, r_is_local(op));
}

/* <source> .filter_TBCPEncode <file> */
int
zTBCPE(os_ptr op)
{	return filter_write(op, 0, &s_TBCPE_template, NULL, false);
}

/* <target> .filter_TBCPDecode <file> */
int
zTBCPD(os_ptr op)
{	stream_BCPD_state state;
	state.signal_interrupt = no_bcp_signal_interrupt;
	state.request_status = no_bcp_request_status;
	return filter_read(op, 0, &s_TBCPD_template, (stream_state *)&state, false);
}

/* ------ Utilities ------ */

/* Set up an input filter. */
const stream_procs s_new_read_procs =
{	s_std_noavailable, s_std_noseek, s_std_read_reset,
	s_std_read_flush, s_filter_close
};
int
filter_read(os_ptr op, int npop, const stream_template *template,
  stream_state *st, bool local)
{	uint min_size = template->min_out_size;
	bool save_local = ialloc_is_local(idmemory);
	register os_ptr sop = op - npop;
	stream *s;
	stream *sstrm;
	int is_temp;
	int code;
	ialloc_set_local(idmemory, local || r_is_local(sop));
	/* Check to make sure that the underlying data */
	/* can function as a source for reading. */
	switch ( r_type(sop) )
	{
	case t_string:
		check_read(*sop);
		sstrm = s_alloc(imemory, "filter_read(string stream)");
		if ( sstrm == 0 )
		{	code = gs_note_error(e_VMerror);
			goto out;
		}
		sread_string(sstrm, sop->value.bytes, r_size(sop));
		is_temp = 1;
		break;
	case t_file:
		check_read_known_file(sstrm, sop,
				      return_error(e_invalidfileaccess));
		code = file_ensure_buf(sstrm,
				       template->min_in_size +
				       sstrm->state->template->min_out_size);
		if ( code < 0 )
		  goto out;
		is_temp = 0;
		break;
	default:
		code = sread_proc(sop, &sstrm);
		if ( code < 0 )
		  goto out;
		is_temp = 1;
		break;
	   }
	if ( min_size < 128 )
	  min_size = file_default_buffer_size;
	code = filter_open("r", min_size, (ref *)sop,
			   &s_new_read_procs, template, st, &s);
	if ( code < 0 )
		goto out;
	s->strm = sstrm;
	s->strm_is_temp = is_temp;
	pop(npop);
out:	ialloc_set_local(idmemory, save_local);
	return code;
}

/* Set up an output filter. */
const stream_procs s_new_write_procs =
{	s_std_noavailable, s_std_noseek, s_std_write_reset,
	s_std_write_flush, s_filter_close
};
int
filter_write(os_ptr op, int npop, const stream_template *template,
  stream_state *st, bool local)
{	uint min_size = template->min_in_size;
	bool save_local = ialloc_is_local(idmemory);
	register os_ptr sop = op - npop;
	stream *s;
	stream *sstrm;
	int is_temp;
	int code;
	ialloc_set_local(idmemory, local || r_is_local(sop));
	/* Check to make sure that the underlying data */
	/* can function as a sink for writing. */
	switch ( r_type(sop) )
	{
	case t_string:
		check_write(*sop);
		sstrm = s_alloc(imemory, "filter_write(string)");
		if ( sstrm == 0 )
		{	code = gs_note_error(e_VMerror);
			goto out;
		}
		swrite_string(sstrm, sop->value.bytes, r_size(sop));
		is_temp = 1;
		break;
	case t_file:
		check_write_known_file(sstrm, sop);
		code = file_ensure_buf(sstrm,
				       template->min_out_size +
				       sstrm->state->template->min_in_size);
		if ( code < 0 )
		  goto out;
		is_temp = 0;
		break;
	default:
		code = swrite_proc(sop, &sstrm);
		if ( code < 0 )
		  goto out;
		is_temp = 1;
		break;
	}
	if ( min_size < 128 )
	  min_size = file_default_buffer_size;
	code = filter_open("w", min_size, (ref *)sop,
			   &s_new_write_procs, template, st, &s);
	if ( code < 0 )
		goto out;
	s->strm = sstrm;
	s->strm_is_temp = is_temp;
	pop(npop);
out:	ialloc_set_local(idmemory, save_local);
	return code;
}

/* ------ Initialization procedure ------ */

op_def zfilter_op_defs[] = {
	{"1.filter_ASCIIHexEncode", zAXE},
	{"1.filter_ASCIIHexDecode", zAXD},
	{"1.filter_BCPEncode", zBCPE},
	{"1.filter_BCPDecode", zBCPD},
	{"1.filter_NullEncode", zNullE},
	{"2.filter_PFBDecode", zPFBD},
	{"2.filter_RunLengthEncode", zRLE},
	{"1.filter_RunLengthDecode", zRLD},
	{"1.filter_SubFileDecode", zSFD},
	{"1.filter_TBCPEncode", zTBCPE},
	{"1.filter_TBCPDecode", zTBCPD},
	op_def_end(0)
};
