/* Copyright (C) 1992, 1993 Aladdin Enterprises.  All rights reserved.
  
  This file is part of Aladdin Ghostscript.
  
  Aladdin Ghostscript is distributed with NO WARRANTY OF ANY KIND.  No author
  or distributor accepts any responsibility for the consequences of using it,
  or for whether it serves any particular purpose or works at all, unless he
  or she says so in writing.  Refer to the Aladdin Ghostscript Free Public
  License (the "License") for full details.
  
  Every copy of Aladdin Ghostscript must include a copy of the License,
  normally in a plain ASCII text file named PUBLIC.  The License grants you
  the right to copy, modify and redistribute Aladdin Ghostscript, but only
  under certain conditions described in the License.  Among other things, the
  License requires that the copyright notice and this notice be preserved on
  all copies.
*/

/* gxdcconv.c */
/* Conversion between device color spaces for Ghostscript */
#include "gx.h"
#include "gxdcconv.h"			/* interface */
#include "gxdcolor.h"			/* for gxcmap.h */
#include "gxdevice.h"			/* ditto */
#include "gxcmap.h"
#include "gxfarith.h"
#include "gxlum.h"
#include "gzstate.h"

/* Note: the color model conversion algorithms are taken from */
/* Rogers, Procedural Elements for Computer Graphics, pp. 401-403. */

/* ------ Conversion between HSB and RGB ------ */

/* Convert RGB to HSB. */
void
color_rgb_to_hsb(floatp r, floatp g, floatp b, float hsb[3])
{	frac red = float2frac(r), green = float2frac(g), blue = float2frac(b);
#define rhue hsb[0]
#define rsat hsb[1]
#define rbri hsb[2]
	if ( red == green && green == blue )
	   {	rhue = 0;	/* arbitrary */
		rsat = 0;
		rbri = r;	/* pick any one */
	   }
	else
	   {	/* Convert rgb to hsb */
		frac V, Temp;
		long diff, H;
		V = (red > green ? red : green);
		if ( blue > V ) V = blue;
		Temp = (red > green ? green : red);
		if ( blue < Temp ) Temp = blue;
		diff = V - Temp;
		if ( V == red )
		    H = (green - blue) * frac_1_long / diff;
		else if ( V == green )
		    H = (blue - red) * frac_1_long / diff + 2 * frac_1_long;
		else /* V == blue */
		    H = (red - green) * frac_1_long / diff + 4 * frac_1_long;
		if ( H < 0 ) H += 6 * frac_1_long;
		rhue = H / (frac_1 * 6.0);
		rsat = diff / (float)V;
		rbri = frac2float(V);
	   }
#undef rhue
#undef rsat
#undef rbri
}

/* Convert HSB to RGB. */
void
color_hsb_to_rgb(floatp hue, floatp saturation, floatp brightness, float rgb[3])
{	if ( saturation == 0 )
	   {	rgb[0] = rgb[1] = rgb[2] = brightness;
	   }
	else
	   {	/* Convert hsb to rgb. */
		/* We rely on the fact that the product of two */
		/* fracs fits into an unsigned long. */
		floatp h6 = hue * 6;
		ulong V = float2frac(brightness);	/* force arithmetic to long */
		frac S = float2frac(saturation);
		int I = (int)h6;
		ulong F = float2frac(h6 - I);		/* ditto */
		/* M = V*(1-S), N = V*(1-S*F), K = V*(1-S*(1-F)) = M-N+V */
		frac M = V * (frac_1_long - S) / frac_1_long;
		frac N = V * (frac_1_long - S * F / frac_1_long) / frac_1_long;
		frac K = M - N + V;
		frac R, G, B;
		switch ( I )
		   {
		default: R = V; G = K; B = M; break;
		case 1: R = N; G = V; B = M; break;
		case 2: R = M; G = V; B = K; break;
		case 3: R = M; G = N; B = V; break;
		case 4: R = K; G = M; B = V; break;
		case 5: R = V; G = M; B = N; break;
		   }
		rgb[0] = frac2float(R);
		rgb[1] = frac2float(G);
		rgb[2] = frac2float(B);
#ifdef DEBUG
if ( gs_debug_c('c') )
{		dprintf7("[c]hsb(%g,%g,%g)->VSFI(%ld,%d,%ld,%d)->\n",
			 hue, saturation, brightness, V, S, F, I);
		dprintf6("   RGB(%d,%d,%d)->rgb(%g,%g,%g)\n",
			 R, G, B, rgb[0], rgb[1], rgb[2]);
}
#endif
	   }
}

/* ------ Color space conversion ------ */

/* Only 4 of the 6 conversions are implemented here; */
/* the other 2 (Gray to RGB/CMYK) are trivial. */
/* The CMYK to RGB algorithms specified by Adobe are, e.g., */
/*	R = 1.0 - min(1.0, C + K)	*/
/* but we get much better results with */
/*	R = (1.0 - C) * (1.0 - K)	*/

/* Convert RGB to Gray. */
frac
color_rgb_to_gray(frac r, frac g, frac b, const gs_state *pgs)
{	return (r * (unsigned long)lum_red_weight +
		g * (unsigned long)lum_green_weight +
		b * (unsigned long)lum_blue_weight +
		(lum_all_weights / 2))
	    / lum_all_weights;
}

/* Convert RGB to CMYK. */
/* Note that this involves black generation and undercolor removal. */
void
color_rgb_to_cmyk(frac r, frac g, frac b, const gs_state *pgs,
  frac cmyk[4])
{	frac c = frac_1 - r, m = frac_1 - g, y = frac_1 - b;
	frac k = (c < m ? min(c, y) : min(m, y));
	/* The default UCR and BG functions are pretty arbitrary.... */
	frac bg =
		(pgs->black_generation == NULL ? frac_0 :
		 gx_map_color_frac(pgs, k, black_generation));
	signed_frac ucr =
		(pgs->undercolor_removal == NULL ? frac_0 :
		 gx_map_color_frac(pgs, k, undercolor_removal));
	/* Adobe specifies, e.g., */
	/*	C = max(0.0, min(1.0, 1 - R - UCR)) */
	/* but in order to match our improved CMYK->RGB mapping, we use */
	/*	C = max(0.0, min(1.0, 1 - R / (1 - UCR)) */
	if ( ucr == frac_1 )
		cmyk[0] = cmyk[1] = cmyk[2] = 0;
	else
	{	float denom = frac2float(frac_1 - ucr);	/* unscaled */
		float v;
		v = (float)frac_1 - r / denom;	/* unscaled */
		cmyk[0] =
		  (is_fneg(v) ? frac_0 : v >= (float)frac_1 ? frac_1 : (frac)v);
		v = (float)frac_1 - g / denom;	/* unscaled */
		cmyk[1] =
		  (is_fneg(v) ? frac_0 : v >= (float)frac_1 ? frac_1 : (frac)v);
		v = (float)frac_1 - b / denom;	/* unscaled */
		cmyk[2] =
		  (is_fneg(v) ? frac_0 : v >= (float)frac_1 ? frac_1 : (frac)v);
	}
	cmyk[3] = bg;
	if_debug7('c', "[c]RGB 0x%x,0x%x,0x%x -> CMYK 0x%x,0x%x,0x%x,0x%x\n",
		  r, g, b, cmyk[0], cmyk[1], cmyk[2], cmyk[3]);
}

/* Convert CMYK to Gray. */
frac
color_cmyk_to_gray(frac c, frac m, frac y, frac k, const gs_state *pgs)
{	frac not_gray = color_rgb_to_gray(c, m, y, pgs);
	return (not_gray > frac_1 - k ?		/* gray + k > 1.0 */
		frac_0 : frac_1 - (not_gray + k));
}

/* Convert CMYK to RGB. */
void
color_cmyk_to_rgb(frac c, frac m, frac y, frac k, const gs_state *pgs,
  frac rgb[3])
{	switch ( k )
	{
	case frac_0:
		rgb[0] = frac_1 - c;
		rgb[1] = frac_1 - m;
		rgb[2] = frac_1 - y;
		break;
	case frac_1:
		rgb[0] = rgb[1] = rgb[2] = frac_0;
		break;
	default:
	{	ulong not_k = frac_1 - k;
		/* Compute not_k * (frac_1 - v) / frac_1 efficiently. */
		ulong prod;
#define deduct_black(v)\
  (prod = (frac_1 - (v)) * not_k, frac_1_quo(prod))
		rgb[0] = deduct_black(c);
		rgb[1] = deduct_black(m);
		rgb[2] = deduct_black(y);
	}
	}
	if_debug7('c', "[c]CMYK 0x%x,0x%x,0x%x,0x%x -> RGB 0x%x,0x%x,0x%x\n",
		  c, m, y, k, rgb[0], rgb[1], rgb[2]);
}
