//-------------------------------------------------------------------
//  FILE:               CYSOURCE.C
//
//  DESCRIPTION:        Main source module for the application
//
//  AUTHOR:             Carlos Yu ( 8/11/94 )
//                      Copyright (c) 1994, CYSoft Incorporated
//-------------------------------------------------------------------

#define STRICT
#include <windows.h>
#include <windowsx.h>

#include <stdlib.h>
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <time.h>

#include "errors.h"  
#include "comment.h" 
#include "resource.h"

//  Help File

#include "cysource.inc"

#define                 MAX_LINES                50

//--------------
//  Structures
//--------------

typedef struct APP_SETTINGS
{                                      
    char                Company [ 31 ],
                        Programmer [ 31 ];

    UINT                Msg,
                        ListKey;
        
    BOOL                HotKey,
                        CtrlKey,
                        Borland,
                        ShiftKey,
                        Microsoft,
                        MouseClick,                                                
                        Powerbuilder;
};

typedef struct CMT_TYPE
{
    BOOL                File,
                        Class,
                        Member,
                        Function;
};

static struct APP_SETTINGS _AppSet;
static struct CMT_TYPE     _CmtType;

//-------------
//  Constants
//-------------

static HWND             _hMainDlg = 0;

static char             _szAppName [] = "CYSource";

#ifdef _BORLAND40

extern const HINSTANCE _cdecl _hInstance;

#else

HINSTANCE               _hInstance;

#endif

//-----------------------
//  Function Prototypes
//-----------------------

char*                   GetCommentYear ( void );
char*                   GetCommentDate ( void );

void                    InsertText ( HWND hWnd );

void                    ScrollToTop ( HWND hWnd );

void                    CenterWindow ( HWND hWnd, BOOL IsParent );

void                    SendBuffer ( HWND hWnd, char* cmtBuffer );

void                    PBComment ( HWND hWnd );

void                    FileComment ( HWND hWnd );

void                    FuncComment ( HWND hWnd );

void                    InitListKey ( HWND hWnd );

void                    ClassComment ( HWND hWnd );

void                    MemberComment ( HWND hWnd );

void                    BcwFileComment ( HWND hWnd, HWND hookWnd );

void                    PositionCursor ( HWND hWnd, int iUp, int iRight );

BOOL                    CheckError ( APP_ERROR appError );

APP_ERROR               LoadSettings ( void );

APP_ERROR               SaveSettings ( void );

APP_ERROR               GetDlgSettings ( HWND hWnd );

APP_ERROR               SetDlgSettings ( HWND hDlg );

APP_ERROR               CreateMainWnd ( HWND *pHwnd );

APP_ERROR               RegisterMainWnd ( HINSTANCE hInstance );

LRESULT CALLBACK        CYMainProc ( HWND hWnd, UINT uMsg, WPARAM
                            wParam, LPARAM lParam );

BOOL CALLBACK           MainDlgProc ( HWND hWnd, UINT uMsg, WPARAM
                            wParam, LPARAM lParam );

BOOL CALLBACK           CommentDlgProc ( HWND hWnd, UINT uMsg, WPARAM
                            wParam, LPARAM lParam );

//-------------------------------------------------------------------
//  FUNCTION:           WinMain
//
//  DESCRIPTION:        Entry point of the application
//
//  ARGUMENTS:          See windows help
//
//  RETURNS:            See windows help
//-------------------------------------------------------------------

#ifdef _BORLAND40

#pragma argsused

#endif

int PASCAL              WinMain ( HINSTANCE hInstance, HINSTANCE
                            hPrevInstance, LPSTR lpszCmdLine,
                            int nCmdShow )
{
    MSG                 msg;
    HWND                hWnd;
    HINSTANCE           hBwcc,
                        hComment;

    //-----------------------------------------------
    //  Allow only one instance for the application
    //-----------------------------------------------

    if ( 0 == hPrevInstance )
    { 
        //----------------------------
        //  Save the Instance Handle  
        //----------------------------
        
#ifndef _BORLAND40

            _hInstance = hInstance;
            
#endif
                    
        //----------------------
        //  Load All DLL's
        //----------------------

        hBwcc = LoadLibrary ( "BWCC.DLL" );
        hComment = LoadLibrary ( "COMMENT.DLL" );

        if ( HINSTANCE_ERROR > hBwcc )
        {
            CheckError ( CY_BWCCDLL );

            return ( 0 );
        }

        if ( HINSTANCE_ERROR > hComment )
        {
            CheckError ( CY_HOOKDLL );

            return ( 0 );
        }

        //---------------------------------------
        //  Register and create the main window
        //---------------------------------------

        if ( CheckError ( RegisterMainWnd ( _hInstance ) ) )
        {
            //--------------------------------------
            //  The window could not be registered
            //--------------------------------------
            //--------------------------
            //  Unload all DLL's
            //--------------------------

            if ( HINSTANCE_ERROR < hBwcc )
                FreeLibrary ( hBwcc );

            if ( HINSTANCE_ERROR < hComment )
                FreeLibrary ( hComment );

            return ( 0 );
        }

        if ( CheckError ( CreateMainWnd ( &hWnd ) ) )
        {
            //-----------------------------------
            //  The window could not be created
            //-----------------------------------
            //--------------------------
            //  Unload all DLL's
            //--------------------------

            if ( HINSTANCE_ERROR < hBwcc )
                FreeLibrary ( hBwcc );

            if ( HINSTANCE_ERROR < hComment )
                FreeLibrary ( hComment );

            return ( 0 );
        }
        else
            ShowWindow ( hWnd, SW_MINIMIZE );

        //-----------------------------------------
        //  Initialize the comment type structure
        //-----------------------------------------

        _CmtType.File = TRUE;
        _CmtType.Class = FALSE;
        _CmtType.Member = FALSE;
        _CmtType.Function = FALSE;

        //-------------------------------
        //  Activate the hook procedure
        //-------------------------------

        SetHook ( TRUE, _hMainDlg );
    }
    else
    {
        //-------------------------------------
        //  The application is already active
        //-------------------------------------

        MessageBox ( 0, "The application is already running",
            "CYSoft Commenter", MB_OK | MB_SYSTEMMODAL );

        return ( 0 );
    }

    //----------------
    //  Message Loop
    //----------------

    while ( GetMessage ( &msg, NULL, 0, 0 ) )
    {
        //-----------------------------
        //  Check for dialog messages
        //-----------------------------

        if ( !IsDialogMessage ( _hMainDlg, &msg ) )
        {
            //--------------------------------------------
            //  If it is not a dialog message then go on
            //--------------------------------------------

            TranslateMessage ( &msg );
            DispatchMessage  ( &msg );
        }
    }

    //--------------------------
    //  Unload all DLL's
    //--------------------------

    if ( HINSTANCE_ERROR < hBwcc )
        FreeLibrary ( hBwcc );

    if ( HINSTANCE_ERROR < hComment )
        FreeLibrary ( hComment );

    //-------------------------------
    //  Release the hook procedure
    //-------------------------------

    SetHook ( FALSE, hWnd );

    return ( 0 );
}


//-------------------------------------------------------------------
//  FUNCTION:           CheckError
//
//  DESCRIPTION:        Checks for errors
//
//  ARGUMENTS:          APP_ERROR - enumerated return code
//
//  RETURNS:            TRUE if an error occurred,  FALSE otherwise
//-------------------------------------------------------------------

BOOL                    CheckError ( APP_ERROR appError )
{
    char                errMsg [ 255 ];

    if ( CY_SUCCESS != appError )
    {
        //--------------------------
        //  Load the error message
        //--------------------------

        if ( 0 == LoadString ( _hInstance, appError, errMsg, MAXSTRING ) )
            strcpy ( errMsg, "An unknown error occurred" );

        //--------------------------
        //  Show the error message
        //--------------------------

        MessageBox ( 0, errMsg, "ERROR", MB_OK | MB_SYSTEMMODAL );

        return ( TRUE );
    }

    return ( FALSE );
}

//-------------------------------------------------------------------
//  FUNCTION:           RegisterMainWnd
//
//  DESCRIPTION:        Register the application's main window
//
//  ARGUMENTS:          hInstance - the app's instance handle
//
//  RETURNS:            CY_SUCCESS on success
//-------------------------------------------------------------------

APP_ERROR               RegisterMainWnd ( HINSTANCE hInstance )
{
    WNDCLASS            wc;

    wc.style          = 0;
    wc.lpfnWndProc    = CYMainProc;
    wc.cbClsExtra     = 0;
    wc.cbWndExtra     = 0;
    wc.hInstance      = hInstance;
    wc.hIcon          = LoadIcon ( hInstance, _szAppName );
    wc.hCursor        = LoadCursor ( NULL, IDC_ARROW );
    wc.hbrBackground  = NULL;
    wc.lpszMenuName   = NULL;
    wc.lpszClassName  = _szAppName;

    if ( 0 == RegisterClass ( &wc ) )
        return ( CY_REGISTER );

    return ( CY_SUCCESS );
}

//-------------------------------------------------------------------
//  FUNCTION:           CreateMainWnd
//
//  DESCRIPTION:        Create the application's main window
//
//  ARGUMENTS:          pHwnd - pointer to the window handle
//
//  RETURNS:            CY_SUCCESS on success
//-------------------------------------------------------------------

APP_ERROR               CreateMainWnd ( HWND *pHwnd )
{
    //---------------------
    //  Create the window
    //---------------------

    *pHwnd = CreateWindow ( _szAppName, "Source Commenter",
                 WS_OVERLAPPED | WS_VISIBLE | WS_CLIPCHILDREN |
                 WS_CAPTION | WS_SYSMENU | WS_MINIMIZEBOX,
                 CW_USEDEFAULT, CW_USEDEFAULT, CW_USEDEFAULT,
                 CW_USEDEFAULT, NULL, NULL, _hInstance, 0 );

    if ( 0 == *pHwnd )
    {
        //----------------------------------------
        //  The main window could not be created
        //----------------------------------------

        return ( CY_MAINCREATE );
    }

    return ( CY_SUCCESS );
}


//-------------------------------------------------------------------
//  FUNCTION:           CYMainProc
//
//  DESCRIPTION:        Main window procedure
//
//  ARGUMENTS:          See windows help
//
//  RETURNS:            See windows help
//-------------------------------------------------------------------

LRESULT CALLBACK        CYMainProc ( HWND hWnd, UINT uMsg, WPARAM
                            wParam, LPARAM lParam )
{
    LRESULT             lResult = 0;
    BOOL                bCallDefault = FALSE;

    //--------------------------------
    //  Process the message received
    //--------------------------------

    switch ( uMsg )
    {
        case WM_CREATE:

            //--------------------------
            //  Create the main dialog
            //--------------------------

            _hMainDlg = CreateDialog ( _hInstance, _szAppName, hWnd,
                            ( DLGPROC ) MainDlgProc );

            if ( 0 == _hMainDlg )
            {
                //-----------------------------------
                //  The dialog could not be created
                //-----------------------------------

                CheckError ( CY_MAINDLGCREATE );

                lResult = -1;
            }
            else
            {
                //------------------------------------------
                //  Adjust the window to the dialog's size
                //------------------------------------------

                RECT    rWin, rDlg;

                GetWindowRect ( hWnd, &rWin );
                GetWindowRect ( _hMainDlg, &rDlg );

                MoveWindow ( hWnd, rWin.left, rWin.right,
                    rDlg.right - rDlg.left,
                    rDlg.bottom - rDlg.top + GetSystemMetrics ( SM_CYCAPTION ),
                    FALSE );

                lResult = 0;
            }

            break;


        case WM_DESTROY:

            DestroyWindow ( _hMainDlg );
            PostQuitMessage ( 0 );

            break;

        case WM_SIZE:

            //---------------------------
            //  The window size changed
            //---------------------------

            if ( SIZE_MINIMIZED != wParam )
                CenterWindow ( hWnd, FALSE );

            break;

        default:

            bCallDefault = TRUE;

            break;

    }

    //-------------------------------------------------
    //  See if the default procedure should be called
    //-------------------------------------------------

    if ( bCallDefault )
        lResult = DefWindowProc ( hWnd, uMsg, wParam, lParam );

    return ( lResult );
}



//-------------------------------------------------------------------
//  FUNCTION:           LoadSettings
//
//  DESCRIPTION:        Load the default settings for the dialog
//                      values
//
//  ARGUMENTS:          NONE
//
//  RETURNS:            CY_SUCCESS on success
//-------------------------------------------------------------------

APP_ERROR               LoadSettings ()
{
    char                tmpBuffer [ 255 ],
                        iniFilename [ 255 ];

    //-----------------------
    //  Create the filename
    //-----------------------

    sprintf ( iniFilename, "%s.INI", _szAppName );

    //------------------------
    //  Get the INI settings
    //------------------------       
    //------------------------
    //  Activation Procedure  
    //------------------------
    
    GetPrivateProfileString ( "SETTINGS", "Hot Key", "FALSE",
        tmpBuffer, 30, iniFilename );

    if ( 0 == strcmp ( tmpBuffer, "TRUE" ) )
        _AppSet.HotKey = TRUE;
    else                        
        _AppSet.HotKey = FALSE;        
        
    GetPrivateProfileString ( "SETTINGS", "Mouse Click", "TRUE",
        tmpBuffer, 30, iniFilename );

    if ( 0 == strcmp ( tmpBuffer, "TRUE" ) )
        _AppSet.MouseClick = TRUE;
    else                        
        _AppSet.MouseClick = FALSE;        
        
    //---------------------
    //  Hot key character  
    //---------------------
        
    GetPrivateProfileString ( "SETTINGS", "Key List Index", "8",
        tmpBuffer, 30, iniFilename );
        
    _AppSet.ListKey = atoi ( tmpBuffer );
                
    //----------------
    //  Message Type
    //----------------

    GetPrivateProfileString ( "SETTINGS", "Activate", "WM_RBUTTONDOWN",
        tmpBuffer, 30, iniFilename );

    if ( 0 == strcmp ( tmpBuffer, "WM_MBUTTONDOWN" ) )
        _AppSet.Msg = WM_MBUTTONDOWN;
    else if ( _AppSet.HotKey )
        _AppSet.Msg = WM_KEYDOWN;
    else 
        _AppSet.Msg = WM_RBUTTONDOWN;       
        
    SetActiveMsg ( _AppSet.Msg );
    
    //--------
    //  Keys  
    //--------

    GetPrivateProfileString ( "SETTINGS", "ShiftKey", "TRUE",
        tmpBuffer, 30, iniFilename );

    if ( 0 == strcmp ( tmpBuffer, "TRUE" ) )
        _AppSet.ShiftKey = TRUE;
    else                        
        _AppSet.ShiftKey = FALSE;    

    SetShiftKey ( _AppSet.ShiftKey );
    
    GetPrivateProfileString ( "SETTINGS", "CtrlKey", "TRUE",
        tmpBuffer, 30, iniFilename );

    if ( 0 == strcmp ( tmpBuffer, "TRUE" ) )
        _AppSet.CtrlKey = TRUE;
    else                        
        _AppSet.CtrlKey = FALSE;    

    SetCtrlKey ( _AppSet.CtrlKey );    

    //--------------
    //  Programmer
    //--------------

    GetPrivateProfileString ( "SETTINGS", "Programmer", "Carlos Yu",
        _AppSet.Programmer, 30, iniFilename );

    //-----------
    //  Company
    //-----------

    GetPrivateProfileString ( "SETTINGS", "Company", "CYSoft Incorporated",
        _AppSet.Company, 30, iniFilename );

    //-----------
    //  Borland
    //-----------

    GetPrivateProfileString ( "SETTINGS", "Borland", "TRUE",
        tmpBuffer, MAXSTRING, iniFilename );

    if ( 0 == strcmp ( tmpBuffer, "TRUE" ) )
        _AppSet.Borland = TRUE;
    else
        _AppSet.Borland = FALSE;

    //-------------
    //  Microsoft
    //-------------

    GetPrivateProfileString ( "SETTINGS", "Microsoft", "TRUE",
        tmpBuffer, MAXSTRING, iniFilename );

    if ( 0 == strcmp ( tmpBuffer, "TRUE" ) )
        _AppSet.Microsoft = TRUE;
    else
        _AppSet.Microsoft = FALSE;

    //----------------
    //  Powerbuilder
    //----------------

    GetPrivateProfileString ( "SETTINGS", "Powerbuilder", "TRUE",
        tmpBuffer, MAXSTRING, iniFilename );

    if ( 0 == strcmp ( tmpBuffer, "TRUE" ) )
        _AppSet.Powerbuilder = TRUE;
    else
        _AppSet.Powerbuilder = FALSE;

    return ( CY_SUCCESS );
}


//-------------------------------------------------------------------
//  FUNCTION:           SaveSettings
//
//  DESCRIPTION:        Save the default settings for the dialog
//                      values
//
//  ARGUMENTS:          NONE
//
//  RETURNS:            CY_SUCCESS on success
//-------------------------------------------------------------------

APP_ERROR               SaveSettings ()
{
    char                tmpBuffer [ 255 ],
                        iniFilename [ 255 ];

    //-----------------------
    //  Create the filename
    //-----------------------

    sprintf ( iniFilename, "%s.INI", _szAppName );

    //------------------------
    //  Get the INI settings
    //------------------------
    //------------------------
    //  Activation Procedure  
    //------------------------
    
    if ( TRUE == _AppSet.HotKey )
        strcpy ( tmpBuffer, "TRUE" );
    else                             
        strcpy ( tmpBuffer, "FALSE" );    
                                             
    WritePrivateProfileString ( "SETTINGS", "Hot Key",
        tmpBuffer, iniFilename );             
        
    if ( TRUE == _AppSet.MouseClick )
        strcpy ( tmpBuffer, "TRUE" );
    else                             
        strcpy ( tmpBuffer, "FALSE" );    
                                             
    WritePrivateProfileString ( "SETTINGS", "Mouse Click",
        tmpBuffer, iniFilename );                     
        
    //---------------------
    //  Hot key character  
    //---------------------
    
    itoa ( _AppSet.ListKey, tmpBuffer, 10 );
        
    WritePrivateProfileString ( "SETTINGS", "Key List Index",
        tmpBuffer, iniFilename );
    
    //----------------
    //  Message Type
    //----------------

    if ( _AppSet.Msg == WM_MBUTTONDOWN )
        strcpy ( tmpBuffer, "WM_MBUTTONDOWN" );
    else if ( _AppSet.Msg == WM_KEYDOWN )
        strcpy ( tmpBuffer, "WM_KEYDOWN" );
    else 
        strcpy ( tmpBuffer, "WM_RBUTTONDOWN" );
        
    WritePrivateProfileString ( "SETTINGS", "Activate",
           tmpBuffer, iniFilename );        
           
    //--------
    //  Keys  
    //--------
                                     
    if ( TRUE == _AppSet.ShiftKey )
        strcpy ( tmpBuffer, "TRUE" );
    else                             
        strcpy ( tmpBuffer, "FALSE" );    
                                             
    WritePrivateProfileString ( "SETTINGS", "ShiftKey",
        tmpBuffer, iniFilename );           
        
    if ( TRUE == _AppSet.CtrlKey )
        strcpy ( tmpBuffer, "TRUE" );
    else                             
        strcpy ( tmpBuffer, "FALSE" );    
                                             
    WritePrivateProfileString ( "SETTINGS", "CtrlKey",
        tmpBuffer, iniFilename );                   

    //--------------
    //  Programmer
    //--------------

    WritePrivateProfileString ( "SETTINGS", "Programmer",
        _AppSet.Programmer, iniFilename );

    //-----------
    //  Company
    //-----------

    WritePrivateProfileString ( "SETTINGS", "Company",
        _AppSet.Company, iniFilename );

    //-----------
    //  Borland
    //-----------

    if ( TRUE == _AppSet.Borland )
        strcpy ( tmpBuffer, "TRUE" );
    else
        strcpy ( tmpBuffer, "FALSE" );

    WritePrivateProfileString ( "SETTINGS", "Borland",
        tmpBuffer, iniFilename );

    //-------------
    //  Microsoft
    //-------------

    if ( TRUE == _AppSet.Microsoft )
        strcpy ( tmpBuffer, "TRUE" );
    else
        strcpy ( tmpBuffer, "FALSE" );

    WritePrivateProfileString ( "SETTINGS", "Microsoft",
        tmpBuffer, iniFilename );

    //----------------
    //  Powerbuilder
    //----------------

    if ( TRUE == _AppSet.Powerbuilder )
        strcpy ( tmpBuffer, "TRUE" );
    else
        strcpy ( tmpBuffer, "FALSE" );

    WritePrivateProfileString ( "SETTINGS", "Powerbuilder",
        tmpBuffer, iniFilename );

    return ( CY_SUCCESS );
}


//-------------------------------------------------------------------
//  FUNCTION:           GetDlgSettings
//
//  DESCRIPTION:        Move the current values in the dialog into
//                      the _AppSet structure
//
//  ARGUMENTS:          NONE
//
//  RETURNS:            CY_SUCCESS on success
//-------------------------------------------------------------------

APP_ERROR               GetDlgSettings ( HWND hWnd )
{
    long                tmpInt;

    char                tmpBuffer [ 255 ];

    //-----------------------------------
    //  Check the copyright information
    //-----------------------------------

    SendMessage ( GetDlgItem ( hWnd, 601 ), WM_GETTEXT,
        ( WPARAM ) MAXSTRING, ( LPARAM ) tmpBuffer );

    if ( 0 != strcmp ( tmpBuffer, "SOURCE COMMENTER 2.0" ) )
        MessageBox ( 0, "The copyright information has been changed",
            "ERROR", MB_SYSTEMMODAL );

    SendMessage ( GetDlgItem ( hWnd, 603 ), WM_GETTEXT,
        ( WPARAM ) MAXSTRING, ( LPARAM ) tmpBuffer );

    if ( 0 != strcmp ( tmpBuffer, "Copyright (c) 1994, CYSoft Incorporated" ) )
        MessageBox ( 0, "The copyright information has been changed",
            "ERROR", MB_SYSTEMMODAL );

    //---------------------------
    //  Get the checkbox values
    //---------------------------
    //-----------
    //  Borland
    //-----------

    tmpInt = SendMessage ( GetDlgItem ( hWnd, IDC_BORLAND ),
                 BM_GETCHECK, 0, 0L );

    if ( 1 == tmpInt )
        _AppSet.Borland = TRUE;
    else
        _AppSet.Borland = FALSE;

    //-------------
    //  Microsoft
    //-------------

    tmpInt = SendMessage ( GetDlgItem ( hWnd, IDC_MICROSOFT ),
                 BM_GETCHECK, 0, 0L );

    if ( 1 == tmpInt )
        _AppSet.Microsoft = TRUE;
    else
        _AppSet.Microsoft = FALSE;


    //----------------
    //  Powerbuilder
    //----------------

    tmpInt = SendMessage ( GetDlgItem ( hWnd, IDC_POWERBUILDER ),
                 BM_GETCHECK, 0, 0L );

    if ( 1 == tmpInt )
        _AppSet.Powerbuilder = TRUE;
    else
        _AppSet.Powerbuilder = FALSE;

    //-------------------------
    //  Get the string values
    //-------------------------
    //--------------
    //  Programmer
    //--------------

    SendMessage ( GetDlgItem ( hWnd, IDC_PROGRAMMER ), WM_GETTEXT,
        ( WPARAM ) MAXSTRING, ( LPARAM ) tmpBuffer );

    if ( strlen ( tmpBuffer ) > 29 )
        return ( CY_PROGLEN );

    strcpy ( _AppSet.Programmer, tmpBuffer );

    //-----------
    //  Company
    //-----------

    SendMessage ( GetDlgItem ( hWnd, IDC_COMPANY ), WM_GETTEXT,
        ( WPARAM ) MAXSTRING, ( LPARAM ) tmpBuffer );

    if ( strlen ( tmpBuffer ) > 29 )
        return ( CY_COMPANYLEN );

    strcpy ( _AppSet.Company, tmpBuffer );

    //---------------------------
    //  Get radio button values
    //---------------------------
    //-----------
    //  Message
    //-----------

    tmpInt = SendMessage ( GetDlgItem ( hWnd, IDC_RIGHTCLICK ),
                 BM_GETCHECK, 0, 0L );

    if ( 1 == tmpInt )
        _AppSet.Msg = WM_RBUTTONDOWN;

    tmpInt = SendMessage ( GetDlgItem ( hWnd, IDC_MIDDLECLICK ),
                 BM_GETCHECK, 0, 0L );

    if ( 1 == tmpInt )
        _AppSet.Msg = WM_MBUTTONDOWN;
        
    //--------
    //  Keys  
    //--------
        
    tmpInt = SendMessage ( GetDlgItem ( hWnd, IDC_SHIFTKEY ),
                 BM_GETCHECK, 0, 0L );                 

    if ( 1 == tmpInt )
        _AppSet.ShiftKey = TRUE;        
    else                                 
        _AppSet.ShiftKey = FALSE;            
        
    tmpInt = SendMessage ( GetDlgItem ( hWnd, IDC_CTRLKEY ),
                 BM_GETCHECK, 0, 0L );                 

    if ( 1 == tmpInt )
        _AppSet.CtrlKey = TRUE;        
    else                                 
        _AppSet.CtrlKey = FALSE;                    
        
    //---------------------------------------
    //  Get the Mouse Click/Hot Key Setting 
    //---------------------------------------
    
    tmpInt = SendMessage ( GetDlgItem ( hWnd, IDC_MOUSE ),
                 BM_GETCHECK, 0, 0L );

    if ( 1 == tmpInt )
        _AppSet.MouseClick = TRUE;
    else
        _AppSet.MouseClick = FALSE;    
        
    tmpInt = SendMessage ( GetDlgItem ( hWnd, IDC_HOTKEY ),
                 BM_GETCHECK, 0, 0L );

    if ( 1 == tmpInt )
        _AppSet.HotKey = TRUE;
    else
        _AppSet.HotKey = FALSE;              
        
    //---------------------------
    //  Get the listbox setting        
    //---------------------------
    
    _AppSet.ListKey = ( UINT ) SendMessage ( GetDlgItem ( hWnd, IDC_LISTCHAR ),
                          LB_GETCURSEL, 0, 0L );

    return ( CY_SUCCESS );
}


//-------------------------------------------------------------------
//  FUNCTION:           SetDlgSettings
//
//  DESCRIPTION:        Move the current values in the _AppSet
//                      structure into the dialog
//
//  ARGUMENTS:          hDlg - the dialog's window handle
//
//  RETURNS:            CY_SUCCESS on success
//-------------------------------------------------------------------

APP_ERROR               SetDlgSettings ( HWND hDlg )
{                                              
    //---------------------------
    //  Set the checkbox values
    //---------------------------
    //-----------
    //  Borland
    //-----------

    SendMessage ( GetDlgItem ( hDlg, IDC_BORLAND ),
        BM_SETCHECK, _AppSet.Borland, 0L );

    //-------------
    //  Microsoft
    //-------------

    SendMessage ( GetDlgItem ( hDlg, IDC_MICROSOFT ),
        BM_SETCHECK, _AppSet.Microsoft, 0L );

    //----------------
    //  Powerbuilder
    //----------------

    SendMessage ( GetDlgItem ( hDlg, IDC_POWERBUILDER ),
        BM_SETCHECK, _AppSet.Powerbuilder, 0L );

    //-------------------------
    //  Set the string values
    //-------------------------
    //--------------
    //  Programmer
    //--------------

    SendMessage ( GetDlgItem ( hDlg, IDC_PROGRAMMER ), WM_SETTEXT,
        0, ( LPARAM ) _AppSet.Programmer );

    //-----------
    //  Company
    //-----------

    SendMessage ( GetDlgItem ( hDlg, IDC_COMPANY ), WM_SETTEXT,
        0, ( LPARAM ) _AppSet.Company );

    //-----------
    //  Message
    //-----------      
    
    SendMessage ( GetDlgItem ( hDlg, IDC_RIGHTCLICK ),
        BM_SETCHECK, 0, 0L );                 
        
    SendMessage ( GetDlgItem ( hDlg, IDC_MIDDLECLICK ),
        BM_SETCHECK, 0, 0L );                           
    
    if ( WM_MBUTTONDOWN == _AppSet.Msg )
        SendMessage ( GetDlgItem ( hDlg, IDC_MIDDLECLICK ),
            BM_SETCHECK, 1, 0L );
    else
        SendMessage ( GetDlgItem ( hDlg, IDC_RIGHTCLICK ),
            BM_SETCHECK, 1, 0L );

    //--------
    //  Keys    
    //--------    

    SendMessage ( GetDlgItem ( hDlg, IDC_SHIFTKEY ),
        BM_SETCHECK, _AppSet.ShiftKey, 0L );    
        
    SendMessage ( GetDlgItem ( hDlg, IDC_CTRLKEY ),
        BM_SETCHECK, _AppSet.CtrlKey, 0L );                              
        
    //---------------------------
    //  Set the listbox setting        
    //---------------------------
    
    SendMessage ( GetDlgItem ( hDlg, IDC_LISTCHAR ),
                 LB_SETCURSEL, _AppSet.ListKey, 0L );         
        
    //---------------------------------------
    //  Set the Mouse Click/Hot Key Setting 
    //---------------------------------------
    
    SendMessage ( GetDlgItem ( hDlg, IDC_MOUSE ),
                 BM_SETCHECK, _AppSet.MouseClick, 0L );
        
    SendMessage ( GetDlgItem ( hDlg, IDC_HOTKEY ),
                 BM_SETCHECK, _AppSet.HotKey, 0L );
                 
    if ( _AppSet.MouseClick )
        SendMessage ( hDlg, WM_COMMAND, IDC_MOUSE, 
            MAKELPARAM ( GetDlgItem ( hDlg, IDC_MOUSE ), BN_CLICKED ) );
    else                                                
        SendMessage ( hDlg, WM_COMMAND, IDC_HOTKEY, 
            MAKELPARAM ( GetDlgItem ( hDlg, IDC_HOTKEY ), BN_CLICKED ) );       

    return ( CY_SUCCESS );
}


//-------------------------------------------------------------------
//  FUNCTION:           CommentDlgProc
//
//  DESCRIPTION:        Comment type dialog window procedure
//
//  ARGUMENTS:          See windows help
//
//  RETURNS:            See windows help
//-------------------------------------------------------------------

#ifdef _BORLAND40

#pragma argsused

#endif

BOOL CALLBACK           CommentDlgProc ( HWND hWnd, UINT uMsg, WPARAM
                            wParam, LPARAM lParam )
{
    long                isCheck;

    BOOL                bResult = TRUE;

    //--------------------------------
    //  Process the message received
    //--------------------------------

    switch ( uMsg )
    {
        case WM_COMMAND:

            switch ( LOWORD ( wParam ) )
            {
                case IDCANCEL:

                    EndDialog ( hWnd, 0 );

                    break;
                    
                case IDOK:

                    //----------------------------
                    //  Save the dialog settings
                    //----------------------------

                    isCheck = SendMessage ( GetDlgItem
                        ( hWnd, IDC_CMTFILE ), BM_GETCHECK, 0, 0L );

                    if ( 1 == isCheck )
                        _CmtType.File = TRUE;
                    else
                        _CmtType.File = FALSE;

                    isCheck = SendMessage ( GetDlgItem
                        ( hWnd, IDC_CMTFUNCTION ), BM_GETCHECK, 0, 0L );

                    if ( 1 == isCheck )
                        _CmtType.Function = TRUE;
                    else
                        _CmtType.Function = FALSE;

                    isCheck = SendMessage ( GetDlgItem
                        ( hWnd, IDC_CMTCLASS ), BM_GETCHECK, 0, 0L );

                    if ( 1 == isCheck )
                        _CmtType.Class = TRUE;
                    else
                        _CmtType.Class = FALSE;

                    isCheck = SendMessage ( GetDlgItem
                        ( hWnd, IDC_CMTMEMBER ), BM_GETCHECK, 0, 0L );

                    if ( 1 == isCheck )
                        _CmtType.Member = TRUE;
                    else
                        _CmtType.Member = FALSE;

                    //--------------------
                    //  Close the dialog
                    //--------------------

                    EndDialog ( hWnd, 1 );

                    break;
            }

            break;

        case WM_INITDIALOG:

            //---------------------
            //  Center the window
            //---------------------

            CenterWindow ( hWnd, FALSE );

            //---------------------------
            //  Initialize the controls
            //---------------------------

            SendMessage ( GetDlgItem ( hWnd, IDC_CMTFILE ),
                BM_SETCHECK, _CmtType.File, 0L );

            SendMessage ( GetDlgItem ( hWnd, IDC_CMTFUNCTION ),
                BM_SETCHECK, _CmtType.Function, 0L );

            SendMessage ( GetDlgItem ( hWnd, IDC_CMTCLASS ),
                BM_SETCHECK, _CmtType.Class, 0L );

            SendMessage ( GetDlgItem ( hWnd, IDC_CMTMEMBER ),
                BM_SETCHECK, _CmtType.Member, 0L );


            break;

        default:

           bResult = FALSE;

           break;

    }

    return ( bResult );
}


//-------------------------------------------------------------------
//  FUNCTION:           PBComment
//
//  DESCRIPTION:        Prepare the Powerbuilder comment buffer
//
//  ARGUMENTS:          hWnd - the handle of the Powerbuilder
//                             script window
//
//  RETURNS:            NONE
//-------------------------------------------------------------------

void far                PBComment ( HWND hWnd )
{                                               
    HWND                hArguments;
    
    DWORD               lineIdx;
    
    BOOL                isEvent, isObject;

    char                strCaption [ 255 ];

    int                 posCopy, posStart, loop, argCount;
    
    char                strObject [ 255 ], strSend [ 255 ],
                        strEventFunc [ 255 ], strWinText [ 255 ],
                        strArgText [ 255 ];  
                        
    //----------------------------
    //  Get the caption to parse
    //----------------------------

    GetWindowText ( GetParent ( GetParent ( GetParent ( hWnd ) ) ),
        strCaption, 254 );

    //---------------------
    //  Parse the caption
    //---------------------
    //-------------------------------------------
    //  Determine if it is a script or function
    //-------------------------------------------

    if ( 'S' == strCaption [ 0 ] )
    {
        //------------------
        //  It is a script
        //------------------

        posCopy = 0;

        isEvent = TRUE;

        //----------------------
        //  Get the event name
        //----------------------

        for ( loop = 9; loop < ( signed ) strlen ( strCaption ); loop++ )
        {
            if ( ( ' ' == strCaption [ loop - 1 ] )    &&
                 ( 'f' == strCaption [ loop ]     )    &&
                 ( 'o' == strCaption [ loop + 1 ] )    &&
                 ( 'r' == strCaption [ loop + 2 ] )    &&
                 ( ' ' == strCaption [ loop + 3 ] ) )
            {
                posStart = loop + 4;

                loop = strlen ( strCaption ) + 6;
            }
            else
            {
                strEventFunc [ posCopy ] = strCaption [ loop ];
                posCopy++;
            }
        }

        strEventFunc [ posCopy ] = '\0';

        //-----------------------
        //  Get the object name
        //-----------------------

        strcpy ( strObject, strCaption + posStart );
    }
    else if ( 'F' == strCaption [ 0 ] )
    {
        //--------------------
        //  It is a function
        //--------------------

        posCopy = 0;

        isEvent = FALSE;

        isObject = FALSE;

        //-------------------------
        //  Get the function name
        //-------------------------

        for ( loop = 11; loop < ( signed ) strlen ( strCaption ); loop++ )
        {
            if ( ( ' ' == strCaption [ loop - 1 ] )    &&
                 ( 'f' == strCaption [ loop ]     )    &&
                 ( 'o' == strCaption [ loop + 1 ] )    &&
                 ( 'r' == strCaption [ loop + 2 ] )    &&
                 ( ' ' == strCaption [ loop + 3 ] ) )
            {
                isObject = TRUE;

                posStart = loop + 4;

                loop = strlen ( strCaption ) + 5;

                strEventFunc [ posCopy ] = '\0';
            }
            else
            {
                strEventFunc [ posCopy ] = strCaption [ loop ];
                posCopy++;
                strEventFunc [ posCopy ] = '\0';
            }
        }

        strEventFunc [ posCopy ] = '\0';

        //-----------------------
        //  Get the object name
        //-----------------------

        if ( isObject )
            strcpy ( strObject, strCaption + posStart );
        else
            strObject [ 0 ] = '\0';
    }
    else
    {
        //----------------------------------
        //  Clicked in the DOS file editor
        //----------------------------------

        MessageBeep ( 0 );
        return;
    }

    //-----------------------------------------
    //  See if the date needs to be extracted
    //-----------------------------------------

    GetDlgSettings ( _hMainDlg );

    //-------------------------------
    //  Go to the top of the window
    //-------------------------------

    ScrollToTop ( hWnd );
    
    //--------------------------
    //  Set the argument count  
    //--------------------------
    
    argCount = 1;

    //-----------------------------
    //  Create the comment buffer
    //-----------------------------

    strcpy ( strSend, "\n//" );
    SendBuffer ( hWnd, strSend );

    LoadString ( _hInstance, STR_DASH, strSend, 100 );
    SendBuffer ( hWnd, strSend );

    strcpy ( strSend, " \n" );
    SendBuffer ( hWnd, strSend );

    if ( ( isEvent ) ||
         ( isObject ) )
    {
        //-------------------------
        //  Insert an object name
        //-------------------------

        sprintf ( strSend, "//  OBJECT:             %s\n//\n", strObject );
        SendBuffer ( hWnd, strSend );
    }

    if ( isEvent )
    {
        //-------------------------------
        //  Insert the event parameters
        //-------------------------------

        sprintf ( strSend, "//  EVENT:              %s\n//\n", strEventFunc );
        SendBuffer ( hWnd, strSend );

        strcpy ( strSend, "//  DESCRIPTION:        \n//\n" );
        SendBuffer ( hWnd, strSend );

        strcpy ( strSend, "//  INPUT PARAMS:       \n//\n" );
        SendBuffer ( hWnd, strSend );

        strcpy ( strSend, "//  OUTPUT PARAMS:      \n//\n" );
        SendBuffer ( hWnd, strSend );
    }
    else
    {
        //----------------------------------
        //  Insert the function parameters
        //----------------------------------


        sprintf ( strSend, "//  FUNCTION:           %s\n//\n", strEventFunc );
        SendBuffer ( hWnd, strSend );

        strcpy ( strSend, "//  DESCRIPTION:        \n//\n" );
        SendBuffer ( hWnd, strSend );
        
        //-------------------------------------
        //  Check the Combo Box for arguments  
        //-------------------------------------
        
        hArguments = GetWindow ( GetParent ( hWnd ), GW_HWNDFIRST ); 
        argCount = ( int ) SendMessage ( hArguments, CB_GETCOUNT, 0, 0L );
        --argCount;
        
        if ( 0 == argCount )
        {                            
            //-----------------------------------
            //  There are no function arguments  
            //-----------------------------------
                        
            strcpy ( strSend, "//  ARGUMENTS:          NONE" );
            SendBuffer ( hWnd, strSend );                      
            
            argCount = 1;
        }
        else
        {                        
            for ( loop = 0; loop < argCount; loop++ )
            {   
                //------------------------------------------
                //  Insert constants for the comment block  
                //------------------------------------------
                                  
                if ( 0 == loop )
                {                                
                    strcpy ( strSend, "//  ARGUMENTS:          " );
                    SendBuffer ( hWnd, strSend ); 
                }
                else
                {
                    strcpy ( strSend, "\n//                      " );
                    SendBuffer ( hWnd, strSend ); 
                }            
                
                //---------------------------------
                //  Get the current list box text  
                //---------------------------------                                         
                
                SendMessage ( hArguments, CB_GETLBTEXT, loop,
                    ( LPARAM ) strWinText );
                    
                //----------------------------------
                //  Extract only the variable name  
                //----------------------------------
                
                posCopy = 0;
                
                for ( posStart = 0; posStart < ( signed ) strlen ( strWinText ); 
                    posStart++ )
                {
                    //----------------------------
                    //  Look for the first space  
                    //----------------------------
                    
                    if ( ' ' == strWinText [ posStart ] )
                    {
                        posCopy = ++posStart;
                        posStart = strlen ( strWinText ) + 1;
                    }
                }
                
                //--------------------------
                //  Copy the variable name  
                //--------------------------
                
                strcpy ( strArgText, strWinText + posCopy );
                    
                sprintf ( strSend, "%s - ",
                    strArgText );
                    
                SendBuffer ( hWnd, strSend );                          
            }
        }    

        strcpy ( strSend, "\n//\n//  RETURNS:            \n//\n" );
        SendBuffer ( hWnd, strSend );
    }

    //-------------------------------------------
    //  Insert Author and Copyright Information
    //-------------------------------------------

    sprintf ( strSend, "//  AUTHOR:             %s ( %s )\n//\n",
        _AppSet.Programmer, GetCommentDate () );
    SendBuffer ( hWnd, strSend );


    sprintf ( strSend, "//                      Copyright (c), %s %s\n",
    GetCommentYear (), _AppSet.Company );
    SendBuffer ( hWnd, strSend );

    strcpy ( strSend, "//" );
    SendBuffer ( hWnd, strSend );

    LoadString ( _hInstance, STR_DASH, strSend, 100 );
    SendBuffer ( hWnd, strSend );

    strcpy ( strSend, " \n\n" );
    SendBuffer ( hWnd, strSend );
    
    //-----------------------------------------------
    //  Position the cursor in the description area
    //-----------------------------------------------
    
    if ( !_AppSet.CtrlKey )
        PositionCursor ( hWnd, ( 11 + --argCount ), 24 );
    else
    {
        int             iLineStart, iLineEnd, iCurLine;
        
        iLineStart = ( signed ) SendMessage ( hWnd, EM_LINEFROMCHAR, -1, 0L );
        
        iLineEnd = iLineStart - ( 11 + -- argCount );  
        
        iCurLine = 0;
        
        while ( iCurLine != iLineEnd )
        {
            SendMessage ( hWnd, WM_KEYDOWN, VK_LEFT, 0 );
            SendMessage ( hWnd, WM_KEYUP,   VK_LEFT, 0 );
            
            iCurLine = ( signed ) SendMessage ( hWnd, EM_LINEFROMCHAR, -1, 0L );
        }
    }
    
    //-------------------------
    //  Shake off the cobwebs  
    //-------------------------
    
    if ( _AppSet.ShiftKey )
    {
            lineIdx = SendMessage ( hWnd, EM_GETSEL, 0, 0L ); 
    
            SendMessage ( hWnd, EM_SETSEL, 0, MAKELPARAM (LOWORD ( lineIdx ), 
            LOWORD ( lineIdx ) ) );    
    }
}





//-------------------------------------------------------------------
//  FUNCTION:           SendBuffer
//
//  DESCRIPTION:        Send the characters to the specified
//                      window
//
//  ARGUMENTS:          hWnd - the handle of the Powerbuilder
//                             script window
//                      cmtBuffer - the buffer to send
//
//  RETURNS:            NONE
//-------------------------------------------------------------------

void                    SendBuffer ( HWND hWnd, char* cmtBuffer )
{
    int                 loop;    
    
    //-------------------------------------------
    //  Type the comment into the source window
    //-------------------------------------------

    for ( loop = 0; loop < ( signed ) strlen ( cmtBuffer ); loop++ )
    {
        if ( '\n' == cmtBuffer [ loop ] )
        {
            //--------------------------------
            //  The proper way to send a
            //  carriage return and linefeed
            //--------------------------------

            SendMessage ( hWnd, WM_KEYDOWN, VK_RETURN, ( LPARAM ) 1835009L );

            SendMessage ( hWnd, WM_CHAR, VK_RETURN, ( LPARAM ) 1835009L );

            SendMessage ( hWnd, WM_KEYUP, VK_RETURN, ( LPARAM ) 3223060481L );
        }
        else
            SendMessage ( hWnd, WM_CHAR, cmtBuffer [ loop ], 0 );
    }
}


//-------------------------------------------------------------------
//  FUNCTION:           ClassComment
//
//  DESCRIPTION:        Prepare a Class Comment for the window
//                      handle passed
//
//  ARGUMENTS:          hWnd - the handle of the Powerbuilder
//                             script window
//
//  RETURNS:            NONE
//-------------------------------------------------------------------

void                    ClassComment ( HWND hWnd )
{
    char                strSend [ 255 ];

    //-----------------------------------------
    //  See if the date needs to be extracted
    //-----------------------------------------

    GetDlgSettings ( _hMainDlg );

    //-----------------------------
    //  Create the comment buffer
    //-----------------------------

    strcpy ( strSend, "\n//" );
    SendBuffer ( hWnd, strSend );

    LoadString ( _hInstance, STR_DASH, strSend, 100 );
    SendBuffer ( hWnd, strSend );

    strcpy ( strSend, " \n" );
    SendBuffer ( hWnd, strSend );

    SendBuffer ( hWnd, "//  CLASS:              \n//\n" );

    SendBuffer ( hWnd, "//  DESCRIPTION:        \n//\n" );

    //-------------------------------------------
    //  Insert Author and Copyright Information
    //-------------------------------------------

    sprintf ( strSend, "//  AUTHOR:             %s ( %s )\n",
        _AppSet.Programmer, GetCommentDate () );
    SendBuffer ( hWnd, strSend );

    strcpy ( strSend, "//" );
    SendBuffer ( hWnd, strSend );

    LoadString ( _hInstance, STR_DASH, strSend, 100 );
    SendBuffer ( hWnd, strSend );

    strcpy ( strSend, " \n\n" );
    SendBuffer ( hWnd, strSend );

    //-----------------------------------------------
    //  Position the cursor in the description area
    //-----------------------------------------------

    PositionCursor ( hWnd, 7, 24 );
}


//-------------------------------------------------------------------
//  FUNCTION:           FuncComment
//
//  DESCRIPTION:        Prepare the function comment for the window
//                      handle passed
//
//  ARGUMENTS:          hWnd - the handle of the Powerbuilder
//                             script window
//
//  RETURNS:            NONE
//-------------------------------------------------------------------

void                    FuncComment ( HWND hWnd )
{
    char                strSend [ 255 ];

    //-----------------------------------------
    //  See if the date needs to be extracted
    //-----------------------------------------

    GetDlgSettings ( _hMainDlg );

    //-----------------------------
    //  Create the comment buffer
    //-----------------------------

    strcpy ( strSend, "\n//" );
    SendBuffer ( hWnd, strSend );

    LoadString ( _hInstance, STR_DASH, strSend, 100 );
    SendBuffer ( hWnd, strSend );

    strcpy ( strSend, " \n" );
    SendBuffer ( hWnd, strSend );

    SendBuffer ( hWnd, "//  FUNCTION:           \n//\n" );

    SendBuffer ( hWnd, "//  DESCRIPTION:        \n//\n" );

    SendBuffer ( hWnd, "//  ARGUMENTS:          NONE\n//\n" );

    SendBuffer ( hWnd, "//  RETURNS:            NONE\n//\n" );

    //-------------------------------------------
    //  Insert Author and Copyright Information
    //-------------------------------------------

    sprintf ( strSend, "//  AUTHOR:             %s ( %s )\n",
        _AppSet.Programmer, GetCommentDate () );
    SendBuffer ( hWnd, strSend );

    strcpy ( strSend, "//" );
    SendBuffer ( hWnd, strSend );

    LoadString ( _hInstance, STR_DASH, strSend, 100 );
    SendBuffer ( hWnd, strSend );

    strcpy ( strSend, " \n\n" );
    SendBuffer ( hWnd, strSend );

    //-----------------------------------------------
    //  Position the cursor in the description area
    //-----------------------------------------------

    PositionCursor ( hWnd, 11, 24 );
}


//-------------------------------------------------------------------
//  FUNCTION:           MemberComment
//
//  DESCRIPTION:        Prepare the member comment for the window
//                      handle passed
//
//  ARGUMENTS:          hWnd - the handle of the Powerbuilder
//                             script window
//
//  RETURNS:            NONE
//-------------------------------------------------------------------

void                    MemberComment ( HWND hWnd )
{
    char                strSend [ 255 ];

    //-----------------------------------------
    //  See if the date needs to be extracted
    //-----------------------------------------

    GetDlgSettings ( _hMainDlg );

    //-----------------------------
    //  Create the comment buffer
    //-----------------------------

    strcpy ( strSend, "\n//" );
    SendBuffer ( hWnd, strSend );

    LoadString ( _hInstance, STR_DASH, strSend, 100 );
    SendBuffer ( hWnd, strSend );

    strcpy ( strSend, " \n" );
    SendBuffer ( hWnd, strSend );

    SendBuffer ( hWnd, "//  CLASS:              \n//\n" );

    SendBuffer ( hWnd, "//  MEMBER:             \n//\n" );

    SendBuffer ( hWnd, "//  DESCRIPTION:        \n//\n" );

    SendBuffer ( hWnd, "//  ARGUMENTS:          NONE\n//\n" );

    SendBuffer ( hWnd, "//  RETURNS:            NONE\n//\n" );

    //-------------------------------------------
    //  Insert Author and Copyright Information
    //-------------------------------------------

    sprintf ( strSend, "//  AUTHOR:             %s ( %s )\n",
        _AppSet.Programmer, GetCommentDate () );
    SendBuffer ( hWnd, strSend );

    strcpy ( strSend, "//" );
    SendBuffer ( hWnd, strSend );

    LoadString ( _hInstance, STR_DASH, strSend, 100 );
    SendBuffer ( hWnd, strSend );

    strcpy ( strSend, " \n\n" );
    SendBuffer ( hWnd, strSend );

    //-----------------------------------------------
    //  Position the cursor in the description area
    //-----------------------------------------------

    PositionCursor ( hWnd, 13, 24 );
}


//-------------------------------------------------------------------
//  FUNCTION:           PositionCursor
//
//  DESCRIPTION:        Positions the cursor in the first item to
//                      fill for the comment block
//
//  ARGUMENTS:          hWnd - the handle of the window
//
//  RETURNS:            NONE
//-------------------------------------------------------------------

void                    PositionCursor ( HWND hWnd, int iUp, int iRight )
{
    int                 loop;

    //-----------------------
    //  Position the cursor
    //-----------------------
    
    for ( loop = 0; loop < iUp; loop++ )
    {
        SendMessage ( hWnd, WM_KEYDOWN, VK_UP, 0 );
        SendMessage ( hWnd, WM_KEYUP, VK_UP, 0 );
    }


    for ( loop = 0; loop < iRight; loop++ )
    {
        SendMessage ( hWnd, WM_KEYDOWN, VK_RIGHT, 0 );
        SendMessage ( hWnd, WM_KEYUP, VK_RIGHT, 0 );
    }
}


//-------------------------------------------------------------------
//  FUNCTION:           FileComment
//
//  DESCRIPTION:        Create a file comment block
//
//  ARGUMENTS:          hWnd - the handle of the window
//
//  RETURNS:            NONE
//-------------------------------------------------------------------

void                    FileComment ( HWND hWnd )
{
    int                 posLen, posStart, loop;
    char                strCaption [ 255 ], strSend [ 255 ], strTemp [ 255 ];

    //----------------------------------------
    //  Get the caption of the source window
    //----------------------------------------

    GetWindowText ( GetParent ( hWnd ), strTemp, 254 );

    //---------------------------------
    //  Get the length of the caption
    //---------------------------------

    posLen = strlen ( strTemp );

    //---------------------
    //  Parse the caption
    //---------------------

    for ( loop = posLen; loop > 0; loop-- )
    {
        //------------------------------------------
        //  Test for the beginning of the filename
        //------------------------------------------

        if ( ( '\\'  == strTemp [ loop ] )    ||
             ( ' '   == strTemp [ loop ] ) )
        {
            posStart = ++loop;
            loop = 0;
        }
    }

    //---------------------------
    //  Copy the parsed caption
    //---------------------------

    strcpy ( strCaption, strTemp + posStart );

    //-----------------------
    //  Remove the asterisk
    //-----------------------

    for ( loop = 0; loop < ( signed ) strlen ( strCaption ); loop++ )
        if ( '*' == strCaption [ loop ] )
            strCaption [ loop ] = ' ';

    //-----------------------------------------
    //  See if the date needs to be extracted
    //-----------------------------------------

    GetDlgSettings ( _hMainDlg );

    //-------------------------------
    //  Go to the top of the window
    //-------------------------------

    ScrollToTop ( hWnd );

    //-----------------------------
    //  Create the comment buffer
    //-----------------------------

    strcpy ( strSend, "\n//" );
    SendBuffer ( hWnd, strSend );

    LoadString ( _hInstance, STR_DASH, strSend, 100 );
    SendBuffer ( hWnd, strSend );

    strcpy ( strSend, " \n" );
    SendBuffer ( hWnd, strSend );

    sprintf ( strSend, "//  FILE:               %s\n//\n", strCaption );
    SendBuffer ( hWnd, strSend );

    SendBuffer ( hWnd, "//  DESCRIPTION:        \n//\n" );

    //-------------------------------------------
    //  Insert Author and Copyright Information
    //-------------------------------------------

    sprintf ( strSend, "//  AUTHOR:             %s ( %s )\n//\n",
        _AppSet.Programmer, GetCommentDate () );
    SendBuffer ( hWnd, strSend );

    sprintf ( strSend, "//                      Copyright (c), %s %s\n",
    GetCommentYear (), _AppSet.Company );
    SendBuffer ( hWnd, strSend );

    strcpy ( strSend, "//" );
    SendBuffer ( hWnd, strSend );

    LoadString ( _hInstance, STR_DASH, strSend, 100 );
    SendBuffer ( hWnd, strSend );

    strcpy ( strSend, " \n\n" );
    SendBuffer ( hWnd, strSend );

    //-----------------------------------------------
    //  Position the cursor in the description area
    //-----------------------------------------------

    PositionCursor ( hWnd, 7, 24 );
}


//-------------------------------------------------------------------
//  FUNCTION:           BcwFileComment
//
//  DESCRIPTION:        Create a file comment block
//
//  ARGUMENTS:          hWnd - the handle of the window
//
//  RETURNS:            NONE
//-------------------------------------------------------------------

void                    BcwFileComment ( HWND hWnd, HWND hookWnd )
{
    int                 posLen, posStart, loop;
    char                strCaption [ 255 ], strSend [ 255 ], strTemp [ 255 ];

    //----------------------------------------
    //  Get the caption of the source window
    //----------------------------------------

    GetWindowText ( GetParent ( hookWnd ), strTemp, 254 );

    //---------------------------------
    //  Get the length of the caption
    //---------------------------------

    posLen = strlen ( strTemp );

    if ( 0 == posLen )
    {
        MessageBox ( 0, "Unable to get the caption", "ERROR",
            MB_OK | MB_SYSTEMMODAL );

        return;
    }

    //---------------------
    //  Parse the caption
    //---------------------

    for ( loop = posLen; loop > 0; loop-- )
    {
        //------------------------------------------
        //  Test for the beginning of the filename
        //------------------------------------------

        if ( ( '\\'  == strTemp [ loop ] )    ||
             ( ':'   == strTemp [ loop ] ) )
        {
            posStart = ++loop;
            loop = 0;
        }
    }

    //---------------------------
    //  Copy the parsed caption
    //---------------------------

    strcpy ( strCaption, strTemp + posStart );

    //--------------------------------------------------
    //  Remove the extra char and convert to uppercase
    //--------------------------------------------------

    for ( loop = 0; loop < ( signed ) strlen ( strCaption ); loop++ )
    {
        strCaption [ loop ] = toupper ( strCaption [ loop ] );
        
        if ( ']' == strCaption [ loop ] )
            strCaption [ loop ] = ' ';
    }

    //-----------------------------------------
    //  See if the date needs to be extracted
    //-----------------------------------------

    GetDlgSettings ( _hMainDlg );

    //-----------------------------
    //  Create the comment buffer
    //-----------------------------

    strcpy ( strSend, "\n//" );
    SendBuffer ( hWnd, strSend );

    LoadString ( _hInstance, STR_DASH, strSend, 100 );
    SendBuffer ( hWnd, strSend );

    strcpy ( strSend, " \n" );
    SendBuffer ( hWnd, strSend );

    sprintf ( strSend, "//  FILE:               %s\n//\n", strCaption );
    SendBuffer ( hWnd, strSend );

    SendBuffer ( hWnd, "//  DESCRIPTION:        \n//\n" );

    //-------------------------------------------
    //  Insert Author and Copyright Information
    //-------------------------------------------

    sprintf ( strSend, "//  AUTHOR:             %s ( %s )\n//\n",
        _AppSet.Programmer, GetCommentDate () );
    SendBuffer ( hWnd, strSend );

    sprintf ( strSend, "//                      Copyright (c), %s %s\n",
    GetCommentYear (), _AppSet.Company );
    SendBuffer ( hWnd, strSend );

    strcpy ( strSend, "//" );
    SendBuffer ( hWnd, strSend );

    LoadString ( _hInstance, STR_DASH, strSend, 100 );
    SendBuffer ( hWnd, strSend );

    strcpy ( strSend, " \n\n" );
    SendBuffer ( hWnd, strSend );
}


//-------------------------------------------------------------------
//  FUNCTION:           CenterWindow
//
//  DESCRIPTION:        Centers the window in either the screen
//                      or its parent window
//
//  ARGUMENTS:          hWnd - handle of window to be centered
//                      IsParent - Center the window within its
//                                 parent
//
//  RETURNS:            NONE
//-------------------------------------------------------------------

void                    CenterWindow ( HWND hWnd, BOOL IsParent )
{
    RECT                rChild,
                        rParent;

    int                 pWidth,
                        pHeight,
                        cWidth,
                        cHeight;


    if ( TRUE == IsParent )
    {
        //------------------------------------
        //  Use the parent window dimensions
        //------------------------------------

        GetWindowRect ( GetParent ( hWnd ), &rParent );

        pWidth  = ( rParent.right - rParent.left ) / 2;
        pHeight = ( rParent.bottom - rParent.top ) / 2;
    }
    else
    {
        //-----------------------------
        //  Use the screen dimensions
        //-----------------------------

        pWidth  = GetSystemMetrics ( SM_CXFULLSCREEN ) / 2;
        pHeight = GetSystemMetrics ( SM_CYFULLSCREEN ) / 2;
    }

    //-----------------------------------
    //  Get the child window dimensions
    //-----------------------------------

    GetWindowRect ( hWnd, &rChild );

    cWidth  = ( rChild.right - rChild.left ) / 2;
    cHeight = ( rChild.bottom - rChild.top ) / 2;

    //--------------------------
    //  Center the main window
    //--------------------------

    SetWindowPos ( hWnd, NULL, pWidth - cWidth, pHeight - cHeight,
        100, 100, SWP_NOSIZE );
}


//-------------------------------------------------------------------
//  FUNCTION:           GetCommentDate
//
//  DESCRIPTION:        Gets the system date and time
//
//  ARGUMENTS:          NONE
//
//  RETURNS:            The system date/time in char format
//-------------------------------------------------------------------

char*                   GetCommentDate ()
{
    time_t              t;
    struct tm           *tBlock;
    static char         strDate [ 24 ];

    t = time ( NULL );
    tBlock = localtime ( &t );

    sprintf ( strDate, "%02d/%02d/%d - %02d:%02d:%02d", ++tBlock->tm_mon,
        tBlock->tm_mday, tBlock->tm_year, tBlock->tm_hour, tBlock->tm_min,
        tBlock->tm_sec );

    //---------------------
    //  Return the string
    //---------------------

    return ( strDate );
}


//-------------------------------------------------------------------
//  FUNCTION:           ScrollToTop
//
//  DESCRIPTION:        Positions the cursor to the top of the
//                      window
//
//  ARGUMENTS:          hWnd - the handle of the window
//
//  RETURNS:            NONE
//-------------------------------------------------------------------

void                    ScrollToTop ( HWND hWnd )
{
    //-------------------------------
    //  Go to the top of the window
    //-------------------------------

    SendMessage ( hWnd, WM_VSCROLL, ( WPARAM ) MAKELONG ( SB_THUMBPOSITION,
        0 ), 0L );

    //---------------------------------------------
    //  Send a mouse click to position the cursor
    //---------------------------------------------

    SendMessage ( hWnd, WM_LBUTTONDOWN, 0, MAKELPARAM ( 0, 0 ) );
    SendMessage ( hWnd, WM_LBUTTONUP,   0, MAKELPARAM ( 0, 0 ) );
}


//-------------------------------------------------------------------
//  FUNCTION:           InsertText
//
//  DESCRIPTION:        Key sequence to insert text from the
//                      the clipboard
//
//  ARGUMENTS:          hWnd - the window to receive the key sequence
//
//  RETURNS:            NONE
//-------------------------------------------------------------------

void                    InsertText ( HWND hWnd )
{
    //-------------------------
    //  Send the key sequence
    //-------------------------

    SendMessage ( hWnd, WM_KEYDOWN, VK_SHIFT, 1076494341L );

    SendMessage ( hWnd, WM_KEYUP, VK_INSERT, 3243376641L );

    SendMessage ( hWnd, WM_KEYUP, VK_SHIFT, 3223977985L );
}


//-------------------------------------------------------------------
//  FUNCTION:           GetCommentYear
//
//  DESCRIPTION:        Get the current year
//
//  ARGUMENTS:          NONE
//
//  RETURNS:            The year in char format
//-------------------------------------------------------------------

char*                   GetCommentYear ()
{
    time_t              t;
    struct tm           *tBlock;
    static char         strYear [ 6 ];

    t = time ( NULL );
    tBlock = localtime ( &t );

    tBlock->tm_year += 1900;

    itoa ( tBlock->tm_year, strYear, 10 );

    return ( strYear );
}


//-------------------------------------------------------------------
//  FUNCTION:           MainDlgProc
//
//  DESCRIPTION:        Main dialog window procedure
//
//  ARGUMENTS:          See windows help
//
//  RETURNS:            See windows help
//-------------------------------------------------------------------
                            
#ifdef _BORLAND40

#pragma argsused

#endif                                                  

BOOL CALLBACK           MainDlgProc ( HWND hWnd, UINT uMsg, WPARAM
                            wParam, LPARAM lParam )
{
    long                isCheck;         
    char                tmpBuffer [ 9 ];
    BOOL                bResult = TRUE;

    //--------------------------------
    //  Process the message received
    //--------------------------------

    switch ( uMsg )
    {
        case WM_COMMAND:

            switch ( LOWORD ( wParam ) )
            {             
                case IDC_RIGHTCLICK:

                    isCheck = SendMessage ( GetDlgItem ( hWnd,
                                  IDC_RIGHTCLICK ), BM_GETCHECK, 0, 0L );

                    if ( 1 == isCheck )
                    {
                        _AppSet.Msg = WM_RBUTTONDOWN;
                        SetActiveMsg ( WM_RBUTTONDOWN );
                    }   
                    
                    SendMessage ( GetDlgItem ( hWnd, IDC_HOTKEY ), 
                        BM_SETCHECK, 0, 0L );

                    SendMessage ( GetDlgItem ( hWnd, IDC_MOUSE ), 
                        BM_SETCHECK, 1, 0L );                    

                    break;

                case IDC_MIDDLECLICK:

                    isCheck = SendMessage ( GetDlgItem ( hWnd,
                                  IDC_MIDDLECLICK ), BM_GETCHECK, 0, 0L );

                    if ( 1 == isCheck )
                    {
                        _AppSet.Msg = WM_MBUTTONDOWN;
                        SetActiveMsg ( WM_MBUTTONDOWN );
                    }                        
                    
                    SendMessage ( GetDlgItem ( hWnd, IDC_HOTKEY ), 
                        BM_SETCHECK, 0, 0L );

                    SendMessage ( GetDlgItem ( hWnd, IDC_MOUSE ), 
                        BM_SETCHECK, 1, 0L );                    

                    break;
                    
                case IDC_CTRLKEY:

                    isCheck = SendMessage ( GetDlgItem ( hWnd,
                                  IDC_CTRLKEY ), BM_GETCHECK, 0, 0L );

                    if ( 1 == isCheck )
                        _AppSet.CtrlKey = TRUE;
                    else                        
                        _AppSet.CtrlKey = FALSE;
                        
                    SetCtrlKey ( _AppSet.CtrlKey );

                    break;                    
                    
                case IDC_SHIFTKEY:

                    isCheck = SendMessage ( GetDlgItem ( hWnd,
                                  IDC_SHIFTKEY ), BM_GETCHECK, 0, 0L );

                    if ( 1 == isCheck )
                        _AppSet.ShiftKey = TRUE;
                    else                        
                        _AppSet.ShiftKey = FALSE;
                        
                    SetShiftKey ( _AppSet.ShiftKey );

                    break;                                        
                    
                case IDC_HOTKEY:      
                case IDC_LISTCHAR:   
                
                    ShowWindow ( GetDlgItem ( hWnd, IDC_LISTCHAR ), SW_SHOW );
                
                    SendMessage ( GetDlgItem ( hWnd, IDC_HOTKEY ), 
                        BM_SETCHECK, 1, 0L );

                    SendMessage ( GetDlgItem ( hWnd, IDC_MOUSE ), 
                        BM_SETCHECK, 0, 0L );
                    
                    _AppSet.HotKey = TRUE;   
                    _AppSet.MouseClick = FALSE;                    
                    
                    SetActiveMsg ( WM_KEYDOWN );                           

                    ShowWindow ( GetDlgItem ( hWnd, IDC_RIGHTCLICK ), SW_HIDE );
                    ShowWindow ( GetDlgItem ( hWnd, IDC_MIDDLECLICK ), SW_HIDE );                                        
                    
                    _AppSet.ListKey = ( UINT ) SendMessage ( GetDlgItem ( hWnd, IDC_LISTCHAR ), 
                        LB_GETCURSEL, 0, 0L );
                    
                    SendMessage ( GetDlgItem ( hWnd, IDC_LISTCHAR ), LB_GETTEXT,
                        ( WPARAM ) _AppSet.ListKey, ( LPARAM ) tmpBuffer );
                        
                    SetVirtualKey ( tmpBuffer ); 
                    
                    break;                       
                    
                case IDC_MOUSE:

                    ShowWindow ( GetDlgItem ( hWnd, IDC_RIGHTCLICK ), SW_SHOW );
                    ShowWindow ( GetDlgItem ( hWnd, IDC_MIDDLECLICK ), SW_SHOW );                                                        
                
                    SendMessage ( GetDlgItem ( hWnd, IDC_HOTKEY ), 
                        BM_SETCHECK, 0, 0L );

                    SendMessage ( GetDlgItem ( hWnd, IDC_MOUSE ), 
                        BM_SETCHECK, 1, 0L );
                    
                    _AppSet.HotKey = FALSE;   
                    _AppSet.MouseClick = TRUE;                    
                    
                    isCheck = SendMessage ( GetDlgItem ( hWnd,
                                  IDC_MIDDLECLICK ), BM_GETCHECK, 0, 0L );

                    if ( 1 == isCheck )
                    {
                        _AppSet.Msg = WM_MBUTTONDOWN;
                        SetActiveMsg ( WM_MBUTTONDOWN );
                    }              
                    else           
                    {
                        _AppSet.Msg = WM_RBUTTONDOWN;
                        SetActiveMsg ( WM_RBUTTONDOWN );
                    }                                                     
                    
                    ShowWindow ( GetDlgItem ( hWnd, IDC_LISTCHAR ), SW_HIDE );                    
                    
                    break;                    

                case IDC_OKBTN:

                    ShowWindow ( GetParent ( hWnd ), SW_MINIMIZE );

                    break;

                case IDC_ABORTBTN:

                    //-------------------------
                    //  Save the INI settings
                    //-------------------------

                    GetDlgSettings ( hWnd );
                    SaveSettings ();

                    DestroyWindow ( GetParent ( hWnd ) );

                    break;
                    
                case IDC_HELPBTN:
                
                    WinHelp ( hWnd, "cysource.hlp", HELP_CONTENTS, 0L );
                        
                    break;

            }

            break;

        case WM_INITDIALOG:

            //-------------------------
            //  Align the main dialog
            //-------------------------

            SetWindowPos ( hWnd, 0, 0, 0, 100, 100, SWP_NOSIZE );
                                 
            //--------------------------
            //  Initialize the listbox  
            //--------------------------
                                 
            InitListKey ( hWnd );
            
            //------------------------
            //  Get the INI settings
            //------------------------

            LoadSettings ();
            SetDlgSettings ( hWnd );

            break;

        case WM_DESTROY:

            //-------------------------
            //  Save the INI settings
            //-------------------------

            GetDlgSettings ( hWnd );
            SaveSettings ();

            break;

        case WM_BORLAND:

            GetDlgSettings ( hWnd );  
            
            if ( TRUE == _AppSet.Borland )
            {
                //----------------------------------
                //  Create the comment type dialog
                //----------------------------------

                if ( 1 == DialogBox ( _hInstance, "CommentType", 0,
                    ( DLGPROC ) CommentDlgProc ) )
                {
                    //---------------------
                    //  Comment the class
                    //---------------------

                    if ( TRUE == _CmtType.Class )
                        ClassComment ( GetDlgItem ( hWnd, IDC_BORBUFFER ) );
                    else if ( TRUE == _CmtType.Function )
                        FuncComment ( GetDlgItem ( hWnd, IDC_BORBUFFER ) );
                    else if ( TRUE == _CmtType.Member )
                        MemberComment ( GetDlgItem ( hWnd, IDC_BORBUFFER ) );
                    else if ( TRUE == _CmtType.File )
                        BcwFileComment ( GetDlgItem ( hWnd, IDC_BORBUFFER ),
                            ( HWND ) lParam );
                    else
                        MessageBox ( 0, "Invalid comment type",
                            "Error", MB_OK | MB_SYSTEMMODAL );

                    //--------------------------------------------
                    //  Move the created comment block into
                    //  the clipboard then paste it into the IDE
                    //--------------------------------------------

                    SendDlgItemMessage ( hWnd, IDC_BORBUFFER, EM_SETSEL,
                        0, MAKELONG ( 0, -1 ) );

                    SendDlgItemMessage ( hWnd, IDC_BORBUFFER, WM_CUT, 0, 0 );

                    //----------------------------------------
                    //  Scroll to the proper insertion point
                    //----------------------------------------

                    if ( TRUE == _CmtType.File )
                    {
                        //-------------------------------
                        //  Go to the top of the window
                        //-------------------------------

                        ScrollToTop ( ( HWND ) lParam );
                    }


                    SendMessage ( ( HWND ) lParam, WM_PASTE, 0, 0 );
                }
            }

            break;

        case WM_MICROSOFT:
        
            GetDlgSettings ( hWnd );    
             
            if ( TRUE == _AppSet.Microsoft )
            {
                //----------------------------------
                //  Create the comment type dialog
                //----------------------------------

                if ( 1 == DialogBox ( _hInstance, "CommentType", 0,
                    ( DLGPROC ) CommentDlgProc ) )
                {
                    //---------------------
                    //  Comment the class
                    //---------------------

                    if ( TRUE == _CmtType.Class )
                        ClassComment ( ( HWND ) lParam );
                    else if ( TRUE == _CmtType.Function )
                        FuncComment ( ( HWND ) lParam );
                    else if ( TRUE == _CmtType.Member )
                        MemberComment ( ( HWND ) lParam );
                    else if ( TRUE == _CmtType.File )
                        FileComment ( ( HWND ) lParam );
                    else
                        MessageBox ( 0, "Invalid comment type",
                            "Error", MB_OK | MB_SYSTEMMODAL );
                }
            }

            break;

        case WM_POWERBUILDER:

            GetDlgSettings ( hWnd ); 
            
            if ( TRUE == _AppSet.Powerbuilder )
            {
                //---------------------------------
                //  Send the Powerbuilder comment
                //---------------------------------

                PBComment ( ( HWND ) lParam );
            }                    

            break;

        default:

           bResult = FALSE;

           break;

    }

    return ( bResult );
}                      


//-------------------------------------------------------------------- 
//  FUNCTION:           InitListKey
//
//  DESCRIPTION:        Initializes the list box
//
//  ARGUMENTS:          hWnd - dialog handle
//
//  RETURNS:            NONE
//
//  AUTHOR:             Carlos Yu ( 08/28/94 - 18:01:23 )
//-------------------------------------------------------------------- 

void                    InitListKey ( HWND hWnd )
{
    int                 loop;
    HWND                hList; 
    char                tmpBuffer [ 3 ];
    char                szKeys [] = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";
    
    //--------------------------
    //  Get the listbox handle  
    //--------------------------
    
    hList = GetDlgItem ( hWnd, IDC_LISTCHAR );    
    
    //---------------------------
    //  Initialize the list box
    //---------------------------    
    
    for ( loop = 1; loop < 25; loop++ )                   
    {
        sprintf ( tmpBuffer, "F%02d", loop );                        
        
        SendMessage ( hList, LB_ADDSTRING, 0, ( LPARAM ) tmpBuffer );
    }        
        
    for ( loop = 0; loop < ( signed ) strlen ( szKeys ); loop++ )
    {                                                  
        sprintf ( tmpBuffer, "%c", szKeys [ loop ] );   
            
        SendMessage ( hList, LB_ADDSTRING, 0, ( LPARAM ) tmpBuffer );
    }
}                                     