/* area:  Areacode Finder

  Program searches for telephone area codes/state information by
  area code, state name, or postal codes.

  Author:  Jason Mathews <mathews@nssdca.gsfc.nasa.gov>

  Copyright (C) 1992-94 by Jason Mathews.  Permission is granted to any
  individual or institution to use, copy or redistribute this software so long
  as it is not sold for profit, provided this copyright notice is retained.

  Usage: area ?			- help
	 area *			- list all area codes
	 area XXX [XXX...]	- find match
	    where XXX is an area code, state name, or postal code.

	- Two letter state/postal codes like TX for Texas, CA for California
	  can be used, otherwise type in the state name.
	- Enter two word state names like "New*Jersey".
	- Enter three digit area code to find state.
	- Enter "*" for a list of all Area Codes.

  Compilers: This program will compile and run under Turbo C 2.0 and UNIX gcc

    tcc -1 -Z -O -d area.c  (MS-DOS)
    gcc -O area.c -o area   (unix)

  History:

  V1.1 20-Apr-94  - Optimized postal code search.
		  - Added 210, 407, 508, 610, 708, 810, 903, 908,
		    909, 910, and 917 area codes.
		  - Updated 706 and 905 area codes.
		  - Added Canada, Bahamas (BA), Bermuda (BM) postal codes.

  V1.0 26-Nov-92  - Original version for MS-DOS/UNIX.

 *************************************************************************/

#include <stdio.h>
#include <string.h>
#include <ctype.h>

#define N_ZONES  69
#define N_CODES 168

typedef struct zone_s {
    char *code;
    char *stateName;
} zone_t;

zone_t areazones[N_ZONES] = {
"AL","Alabama",
"AK","Alaska",
"AZ","Arizona",
"AR","Arkansas",
"BC","British Columbia",
"CA","California",
"CN","Canada",
"CO","Colorado",
"CT","Connecticut",
"DE","Delaware",
"DC","Washington, District of Columbia",
"FL","Florida",
"GA","Georgia",
"HI","Hawaii",
"ID","Idaho",
"IL","Illinois",
"IN","Indiana",
"IA","Iowa",
"KS","Kansas",
"KY","Kentucky",
"LA","Louisiana",
"ME","Maine",
"MD","Maryland",
"MA","Massachusetts",
"MI","Michigan",
"MN","Minnesota",
"MS","Mississippi",
"MO","Missouri",
"MT","Montana",
"NE","Nebraska",
"NV","Nevada",
"NH","New Hampshire",
"NJ","New Jersey",
"NM","New Mexico",
"NY","New York",
"NC","North Carolina",
"ND","North Dakota",
"OH","Ohio",
"OK","Oklahoma",
"OR","Oregon",
"PA","Pennsylvania",
"PR","Puerto Rico",
"RI","Rhode Island",
"SC","South Carolina",
"SD","South Dakota",
"TN","Tennessee",
"TX","Texas",
"UT","Utah",
"VT","Vermont",
"VI","Virgin Islands",
"VA","Virginia",
"WA","Washington",
"WV","West Virginia",
"WI","Wisconsin",
"WY","Wyoming",
"  ","Wide area (tollfree)",
"  ","International Access",
"BA","Bahamas",
"BM","Bermuda",
"AB","Alberta",
"MB","Manitoba",
"NB","New Brunswick",
"NF","Newfoundland",
"NS","Nova Scotia",
"ON","Ontario",
"PE","Prince Edward Island",
"PO","Quebec",
"SK","Saskatchewan",
"YT","Yukon Territory"
};

typedef struct code_s {
    short zcode;	/* index into zonelist (0..n) */

    short areaCode;	/* local area code.
			 * if areacode is -1 then the list continues from
			 * the previous entry.
			 */

    char *list;		/* list of cities, counties, ... with same areacode.
			 * if list is NULL (0) then this area code covers
			 * the entire area.
			 */
} code_t;

code_t areacodes[N_CODES] = {
0,205,0,
1,907,0,
2,602,0,
3,501,0,
4,604,0, /* British Columbia */

5,209,"Fresno and Stockton",
5,213,"Los Angeles",
5,310,"Southern & Western CA",
5,408,"San Jose and Sunnyvale",
5,415,"Oakland and San Francisco",
5,510,"San Francisco",
5,619,"San Diego and the Imperial Valleya",
5,707,"Eureka, Napa, and Santa Rosa",
5,714,"Orange and Palm Springs",
5,805,"Bakersfield, Ventura, and Simi Valley",
5,818,"Suburban area near Los Angeles",
5,909,"Southwestern CA",
5,916,"Sacramento and South Tahoe",

/* CANADA */
6,604,"British Columbia",
6,403,"Alberta and Yukon",
6,306,"Saskatchewan",
6,204,"Manitoba",
6,416,"Toronto, Ont.",
6,519,"London, Ont.",
6,613,"Ottawa, Ont.",
6,705,"East Ontario",
6,807,"West Ontario",
6,905,"Southern Ontario",
6,418,"Northern Quebec",
6,514,"Montreal, Quebec",
6,819,"NW Quebec",
6,709,"Newfoundland",
6,506,"New Brunswick",
6,902,"Nova Scotia, Prince Edward Island",

7,303,"NW Colorado",
7,719,"SE Colorado",
8,203,0,
9,302,0,
10,202,0,
11,305,"Miami, Key West, Ft. Lauderdale",
11,407,"Eastern Florida",
11,813,"Ft. Myers, Winter Haven",
11,904,"Jacksonville",
12,404,"Atlanta, Rome",
12,706,"Augusta, Columbus",
12,912,"Savannah, Waycross",
13,808,0,
14,208,0,
15,217,"Springfield",
15,309,"Peoria",
15,312,"Chicago, Aurora, Waukegan",
15,618,"Alton, Mt. Vernon",
15,708,"Central Illinois",
15,815,"Rockford",
16,812,"Evansville",
16,219,"Gary, South Bend, Warsaw",
16,317,"Indianapolis, Kokomo",
17,712,"Council Bluffs",
17,515,"Des Moines",
17,319,"Dubuque",
18,316,"Wichita, Dodge City",
18,913,"Topeka, Lawrence, Manhattan, Salina",
19,502,"Louisville, Frankfort, Paducah, Shelbyville",
19,606,"Winchester",
20,504,"Baton Rouge, New Orleans",
20,318,"Lake Charles",
21,207,0,
22,301,"Bethesda, Greenbelt, Rockville, Silver Spring",
22,410,"Balitmore, Eastern Shore, Ellicott City",
23,508,"New Bedford, Worchester",
23,617,"Boston, Plymouth",
23,413,"Springfield",
24,313,"Detroit, Ann Arbor, Flint",
24,616,"Battle Creek, Grand Rapids, Kalamazoo",
24,517,"Lansing",
24,810,"Flint",
24,906,"Escanaba",
25,218,"Duluth",
25,612,"Minneapolis, St. Paul",
25,507,"Rochester",
26,601,0,
27,816,"Belton, Independence, Kansas City, Marshall, St. Joseph, Sedalia",
27,314,"St. Louis, Cape Girardeau, Columbia, Fulton, Hannibal,",
	27,-1,"Jefferson City, Mexico, Poplar Bluff, Rolla",
27,417,"Joplin, Springfield",
28,406,0,
29,402,"Omaha, Lincoln",
29,308,"North Platte, Scottsbluff",
30,702,0,
31,603,0,
32,609,"Atlantic City, Camden, Trenton",
32,201,"Newark, Hackensack, Patterson",
32,908,"Elizabeth, New Brunswick",
33,505,0,
34,518,"Albany, Schenectady",
34,607,"Binghamton",
34,716,"Buffalo, Niagara Falls, Rochester",
34,212,"New York City (Manhattan and Bronx)",
34,718,"New York City (Queens, Brooklyn and Staten Island)",
34,917,"New York City (All Locations)",
34,315,"Syracuse",
34,914,"White Plains",
35,704,"Charlotte, Salisbury",
35,910,"Greensboro, Winston-Salem",
35,919,"Raleigh",
36,701,0,
37,216,"Akron, Cleveland, Youngstown",
37,513,"Cincinnati, Dayton",
37,614,"Columbus",
37,419,"Toledo",
38,918,"Tulsa, Bartlesville, McAlester, Muskogee",
38,405,"Oklahoma City, Enid, Norman, Ponca City, Stillwater",
39,503,0,
40,215,"Philadelphia",
40,412,"Pittsburgh",
40,610,"Allentown",
40,717,"Harrisburg, Scranton",
40,814,"Erie",
41,809,"Anguilla, Antigua, Bahamas, Barbados, Bermuda, Cayman Islands,",
41,-1,"Dominican Republic, Jamaica, St. Lucia, Trinidad, Virgin Islands",
42,401,0,
43,803,0,
44,605,0,
45,615,"Nashville, Chattanooga",
45,901,"Memphis",
46,210,"San Antonio",
46,214,"Dallas, Ennis, Greenville, Jefferson, Longview, Sherman",
46,409,"Bay City, Beaumont, Bryan, College Station, Galveston, Huntsville",
46,512,"Austin, Brownsville, Corpus Christi, Del Rio, Eagle Pass,",
46,-1,"Laredo, McAllen, Victoria",
46,713,"Houston, Baytown, Pasadena",
46,806,"Amarillo, Dalhart, Lubbock",
46,817,"Fort Worth, Denton, Temple, Waco, Wichita Falls",
46,903,"Texarkana, Tyler",
46,915,"Abilene, Alpine, Big Spring, El Paso, Midland, Odessa",
47,801,0,
48,802,0,
49,809,0,
50,804,"Charlottesville, Newport News, Norfolk, Richmond",
50,703,"Arlington, Roanoke, Winchester",
51,206,"Seattle, Olympia, Vancouver",
51,509,"Spokane, Walla Walla, Yakima",
52,304,0,
53,414,"Milwaukee, Green Bay, Racine",
53,608,"La Crosse, Madison",
53,715,"Eau Claire, Wausau",
54,307,0,
55,800,0,

56,011,0, /* International */
57,809,0, /* Bahamas */
58,809,0, /* Bermuda */

/* CANADA */

59,403,0, /* Alberta */
60,204,0, /* Manitoba */
61,506,0, /* New Brunswick */
62,709,0, /* Newfoundland */
63,902,0, /* Nova Scotia */
64,416,"Toronto", /* Ontario */
64,519,"London",
64,613,"Ottawa",
64,705,"East Ontario",
64,807,"West Ontario",
64,905,"Southern Ontario",
65,902,0, /* Prince Edward */
66,418,"Quebec City",
66,514,"Montreal",
66,819,"Sherbrooke",
67,306,0, /* Saskatchewan */
68,403,0  /* Yukon Territory */
};

/* Printcode:	Print the entry for the specified areacode
 *
 *		Keep track of the previous zone and
 *		only print the state information when this changes.
 */
void PrintCode (int code)
{
	static int zone = -1;
	int newzone = areacodes[code].zcode;

	/*  check if entry is continued from the previous one
	 *  (identified by the -1 areacode) and already printed.
	 */
	if (areacodes[code].areaCode < 0) return;
	if (newzone != zone)
	{
		zone = newzone;
		printf("\n%s  %s area code(s):\n", areazones[zone].code, areazones[zone].stateName);
	}
	printf("    %03d %s\n", areacodes[code].areaCode,
	areacodes[code].list ? areacodes[code].list : "All locations.");

	/* Check if areacode list continues on next entry */
	while (++code < N_CODES && areacodes[code].areaCode == -1)
	   printf("        %s\n", areacodes[code].list);
}

/* FindAbbrev:	Find postal code in list of states matching the string s
 *
 * returns:	1 if match found.
 *              0 otherwise.
 */
int FindAbbrev (char *s)
{
	int  i;
	char ch1, ch2;	/* target characters in uppercase */
	ch1 = toupper(*s);
	ch2 = toupper(*(s+1));
	for (i=0; i < N_ZONES; i++)
	{
		if (ch1==areazones[i].code[0] && ch2==areazones[i].code[1])
		{
			int code = 0;
		    /*  Locate areacode in list and
		     *  print all entries for that state.
		     */
			while (areacodes[code].zcode < i) code++;
			while (areacodes[code].zcode==i)
			{
				PrintCode(code++);
			}
			return 1;
		}
	}
	return 0;
}

/* FindCode:	Find areacode in list
 *
 * returns:	1 if areacode is found.
 * 		0 otherwise.
 */
int FindCode (int code)
{
	int i;
	for (i=0; i < N_CODES; i++)
	{
		if (areacodes[i].areaCode == code)
		{
			PrintCode(i);
			return 1;
		}
	}
	return 0;
}

/* match:	match string s with wildcard pattern
 *
 * returns:	1 if the pattern matches the string
 * 		0 otherwise.
 */
int match (char *s, char *pattern)
{
    char symbol;
    char *old_pat = 0;	/* address of last wildcard pattern */
    char *old_target;   /* address of target string */

    while (*pattern && *s)
    {
	if (*pattern == '*')
	{
		old_pat = pattern++;
		old_target = s + 1;
		symbol = tolower(*pattern);
		while (symbol && *s && tolower(*s) != symbol) s++;

		/*  If the end of the pattern was reached
		 *  and the last character was a wildcard (*) then
		 *  the rest of the string is accepted
		 */
		if (!*pattern) return 1;
		/* else check s in regular case */
	}
	else symbol = tolower(*pattern);

	/* check if match failed */
	if (tolower(*s) != symbol)
	{
		/*  check if a wildcard was previously parsed, if so back up
		 *  to that pattern and try again.
		 */
		if (old_pat)
		{
			pattern = old_pat; /* reset wildcard pattern */
			s = old_target;
		}
		else return 0;
	}
	else /* increment strings */
	{
		s++;
		pattern++;
	}
    }

    /*  check if all the pattern has been matched,
     *  which is indicated by a NULL pattern string.
     */
     return ( !*pattern );
}


/* FindState:	Find first state in list of states matching the string s
 *
 * returns:	1 if state is found.
 *              0 otherwise.
 */
int FindState (char *s)
{
	int i;
	for (i=0; i < N_ZONES; i++)
	{
		if (match(areazones[i].stateName, s))
		{
			int code = 0;
			while (areacodes[code].zcode < i) code++;
			while (code < N_CODES && areacodes[code].zcode==i)
			{
				PrintCode(code++);
			}
			return 1;
		}
	}
    return 0;
} /* FindState */


int main (int argc, char **argv)
{
	int i;

	printf("   ** Area Code Finder **    (C) Jason Mathews 1992-94\n");
	if (argc==1 || *argv[1]=='?')
	{
		printf("\nProgram searches for telephone area codes,\n"
			"   as area xxx xxx xxx etc.\n"
			"   xxx is an Area Code or State name.\n"
			"   Two letter state/postal codes like TX for Texas, CA for California\n"
			"      can be used, otherwise type in the state name.\n"
			"   Enter two word state names like this: New*Jersey.\n"
			"   Enter area * for a list of all Area Codes.\n");
		return(0);
	}

	if (*argv[1] == '*' && !argv[1][1])
	{
		for (i=0; i < N_CODES;i++) PrintCode(i);
	}
	else for (i=1; i < argc; i++)
	{
		int len = strlen(argv[i]);
		int code;
		switch (len) {
		 case 2: /* find state zone: MD, NY, ...
			  * if there's no match then try the
			  * general search.
			  */
			if (!FindAbbrev(argv[i])) goto others;
			break;

		 case 3: /* find area code: 301, 718, ... */
			if (isdigit(*argv[i]))
			{
			  code = atoi(argv[i]);
			  if (code > 0 && FindCode(code))
				break;
			}
			/* otherwise try general search */

		 default: /* state name: new*york, marylard, ... */
 others:
			if (!FindState(argv[i]))
			  printf("\nNot found, %s.\n", argv[i]);
		}
	} /* for each argument */
    return(0);
}
