(*
 * Copyright (C) 1989 Eric Ng
 *
 * Egaint is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License, Version 1, as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * without any warranty whatsoever, without even the implied warranties
 * of merchantability or fitness for a particular purpose.  See the
 * enclosed GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with egaint; see the file COPYING.  If not, write to:
 *
 *       Free Software Foundation, Inc.
 *       675 Massachusetts Avenue
 *       Cambridge, Massachusetts 02139
 *
 *)

(*
 * The complete egaint 0.93.05 package can be obtained at either of
 * the following bulletin board systems:
 *
 *       Tom and Sue McDermet's The Odyssey
 *       A carrier of the SmartNet network
 *       Morris Plains, New Jersey
 *       (201) 984-6574
 *
 *       John Looker's Bandersnatch
 *       Phoenix Net #807/7
 *       Basking Ridge, New Jersey
 *       (201) 766-3801
 *
 * In addition, bug reports, modifications, and other assorted
 * queries can be directed, via Internet e-mail, to
 *
 *       erc@{mars,irss,inis}.njit.edu
 *
 * Please note that since I will be returning to college in the
 * fall, future versions of egaint may not be posted on the above
 * bulletin board systems.  The e-mail address, however, should
 * remain valid.
 *
 *)

(*
 * Egaint 0.93.05 was originally written in Turbo Pascal 4.0;
 * however, I have just received my upgrade copy of Turbo Pascal 5.5.
 * This new version of the compiler, so far, has not exhibited
 * any problems and compiled without change.  What this means,
 * through interpolation, is that it should (but is not guaranteed to)
 * compile with Turbo Pascal 5.0 (but I can't verify this).
 *
 *)

{$B-}
{$D-}
{$I-}
{$L-}
{$R-}
{$S-}
{$V-}

Program egaint;

 Uses
  Crt, Dos, Driver, Fonts, Graph;


 Const
  id              : String [6]  = 'egaint';
  version         : String [7]  = '0.93.05';
  copyright       : String [27] = 'Copyright (C) 1989 Eric Ng';
  copr            : String [22] = 'Copr (C) 1989 Eric Ng';

  nshapes         = 26;             { different shapes }
  shapesiz        = 4;              { max size of each shape (minus one) }
  xshapelevels    = 4;              { levels (classic, easy, medium, hard) }
  xshapeclassic   = 7;              { different classic shapes }
  xshapeeasy      = 13;             { different easy extended shapes }
  xshapemedium    = 19;
  xshapehard      = 26;             { different hard extended shapes }

  norients        = 3;              { different orientations }

  ncolors         = 7;              { different colors }
  nstyles         = 4;              { different styles }
  nstyletabs      = 4;              { different style tables }

  ngames          = 256;            { number of tournament games }

  rowmin          = 0;              { playing field coordinates in pixels }
  rowmax          = 337;
  colmin          = 250;
  colmax          = 392;

  pixelsperblock  = 14;             { pixels per block }
  blockrows       = 24;             { rows in blocks }
  xblockrows      = 25;             { rows in blocks (plus one) }
  blockcols       = 10;             { columns in blocks }

  initrow         = 0;              { initial row and column for mkshape }
  initcol         = 5;

  left            = -1;             { displacements for movement/rotation }
  right           = 1;

  maxlevel        = 10;             { maximum level }
  rowsperlevel    = 10;             { rows needed for level advancement }

  maxheight       = 12;             { maximum initial height }
  filladd         = 3;              { constants for fill }
  fillbase        = 3;

  dropdelay       = 20;             { constants for title drop }
  dropinc         = 5;

  clearlimit      = 5;

  bonusrowclear   = 3;              { bonus for clearing a row }
  bonusmultclear  = 2;              { bonus for clearing multiple rows }
  bonusnext       = 1;              { bonus for not using show next shape }
  bonusshadow     = 1;              { bonus for not using show shadow }
  bonushidden     = 3;              { bonus for using hidden blocks }

  info            = 0;              { information element in shape table }

  nhiscores       = 15;             { number of high scores }
  hiscorename     = 'egaint.rec';   { high score file name }
  configname      = 'egaint.rc';    { configuration file name }


 Type
  displaytype     = (color, mono, plasma);
  mesgcolors      = (normal, high);
  bufstr          = String [32];

  rinfotype       = Array [1..clearlimit] Of byte;

  hiscorerec      = Record
                     score      : longint;
                     level      : byte;
                     rowsclear  : word;
                     date       : String [8];
                     time       : String [8];
                     name       : bufstr;
                     version    : String [7]
                    End;


 Const
  shapetab        : Array [1..nshapes, 0..shapesiz, 1..2] Of shortint =
      { bar }       (((3, 2), ( 0, -1), ( 0,  1), ( 0,  2), ( 0,  0)),
      { tee }        ((3, 2), ( 0, -1), ( 1,  0), ( 0,  1), ( 0,  0)),
      { box }        ((3, 3), ( 1,  0), ( 0,  1), ( 1,  1), ( 0,  0)),
      { zig }        ((3, 3), ( 0, -1), ( 1,  0), ( 1,  1), ( 0,  0)),
      { zag }        ((3, 3), ( 1, -1), ( 1,  0), ( 0,  1), ( 0,  0)),
      { ell }        ((3, 3), ( 1, -1), ( 0, -1), ( 0,  1), ( 0,  0)),
      { lel }        ((3, 3), ( 0, -1), ( 0,  1), ( 1,  1), ( 0,  0)),
   { easy }          ((0, 0), ( 0,  0), ( 0,  0), ( 0,  0), ( 0,  0)),
                     ((1, 0), ( 0,  1), ( 0,  0), ( 0,  0), ( 0,  0)),
                     ((1, 1), ( 1,  1), ( 0,  0), ( 0,  0), ( 0,  0)),
                     ((2, 1), ( 1,  0), ( 0,  1), ( 0,  0), ( 0,  0)),
                     ((2, 1), ( 0, -1), ( 0,  1), ( 0,  0), ( 0,  0)),
      { 13 }         ((4, 3), ( 0, -2), ( 0, -1), ( 0,  1), ( 0,  2)),
   { medium }        ((2, 3), ( 1, -1), ( 1,  1), ( 0,  0), ( 0,  0)),
                     ((2, 4), ( 1, -1), ( 0,  1), ( 0,  0), ( 0,  0)),
                     ((2, 4), ( 0, -1), ( 1,  1), ( 0,  0), ( 0,  0)),
                     ((4, 4), ( 1, -1), ( 0, -1), ( 0,  1), ( 1,  1)),
                     ((4, 4), (-1, -1), (-1,  0), ( 1,  0), (-1,  1)),
      { 19 }         ((4, 5), ( 0, -1), (-1,  0), ( 1,  0), ( 0,  1)),
   { hard }          ((4, 5), ( 1, -1), ( 0, -1), (-1,  0), (-1,  1)),
                     ((4, 6), ( 1, -1), ( 0, -1), ( 0,  1), (-1,  1)),
                     ((4, 6), (-1, -1), ( 0, -1), ( 0,  1), ( 1,  1)),
                     ((4, 6), ( 2,  0), ( 1,  0), ( 0,  1), ( 0,  2)),
                     ((3, 7), (-1, -1), ( 1,  0), (-1,  1), ( 0,  0)),
                     ((3, 7), ( 1, -1), ( 2,  0), ( 1,  1), ( 0,  0)),
      { 26 }         ((4, 7), (-1, -1), ( 1, -1), (-1,  1), ( 1,  1)));

  shapecolortab   : Array [displaytype, 1..ncolors] Of byte =
   { color }        ((LightBlue, LightGreen, LightCyan, LightRed,
                      LightMagenta, Yellow, White),
   { mono }          (White, LightGray, White, LightGray, White, LightGray,
                      White),
   { plasma }        (LightGray, Red, Blue, LightGray, Red, Blue, LightGray));

  mesgcolortab    : Array [displaytype, mesgcolors] Of byte =
   { color }        ((LightGray, White),
   { mono }          (LightGray, White),
   { plasma }        (White, LightGray));

  filltab         : Array [1..nstyles] Of FillPatternType =
                    (($ff, $ff, $ff, $ff, $ff, $ff, $ff, $ff),
                     ($aa, $55, $aa, $55, $aa, $55, $aa, $55),
                     ($99, $cc, $66, $33, $99, $cc, $66, $33),
                     ($99, $33, $66, $cc, $99, $33, $66, $cc));

  timedelaytab    : Array [1..maxlevel] Of byte =
                    (10, 9, 8, 7, 6, 5, 4, 3, 2, 1);

  xshapetitles    : Array [1..xshapelevels] Of String [7] =
                    ('Classic',
                     'Easy',
                     'Medium',
                     'Hard');

  styleblocktitles: Array [1..nstyletabs] Of String[20] =
                    ('New',
                     'Classic',
                     'Pumped Full of Drugs',
                     'Really P.F.D.');

 Var
  shapecolors     : Array [1..ncolors] Of byte;
  field           : Array [0..xblockrows, 1..blockcols] Of boolean;
{ fieldshadows    : Array [1..blockcols] Of boolean; }
  hiscore         : Array [1..nhiscores] Of hiscorerec;
  styletab        : Array [1..ncolors, 1..nstyles] Of pointer;
  xstyletabs      : Array [1..nstyletabs, 1..ncolors, 1..nstyles] Of pointer;
  xshapetab       : Array [1..nshapes, 0..norients, 1..shapesiz, 1..2] Of
                    shortint;
  yshapetab       : Array [1..nshapes, 0..norients, 1..shapesiz, 1..2] Of
                    shortint;

  reg             : Registers;      { 8086 registers record }

  buf, buf2, buf3 : bufstr;
  colorhigh       : byte;
  colornormal     : byte;
  curtain         : Array [boolean] Of pointer;
  emptyrow        : pointer;
  fconfig         : Text;
  fhiscore        : File of hiscorerec;
  filler          : pointer;
  graphdriver     : integer;
  graphmode       : integer;
  savemode        : word;
{ shadows         : pointer; }

  bonus           : byte;
  rowsclear       : word;
  score           : longint;
  shapemap        : byte;

 Const
  endrun          : boolean     = False;
  page            : integer     = 0;
  xpage           : byte        = 1;

  display         : displaytype = color;
  height          : byte    = 0;
  level           : byte    = 5;
  shownext        : boolean = True;
  showshadow      : boolean = False;
  styleblocks     : byte    = 0;
  tournament      : boolean = False;
  tournamentgame  : byte    = 0;
  xshape          : byte    = 0;


 Function gettimer : longint;
  Inline($28/$e4/                   { sub ah,ah }
         $cd/$1a/                   { int 1ah   }
         $89/$d0/                   { mov ax,dx }
         $89/$ca);                  { mov dx,cx }


 Procedure dographics;
  Begin
   savemode := LastMode;
   DetectGraph(GraphDriver, GraphMode);
   Case GraphDriver Of
    EGA: Begin
          InitGraph(GraphDriver, GraphMode, '');
          SetGraphMode(EGAHi)
         End;
    VGA: Begin
          InitGraph(GraphDriver, GraphMode, '');
          SetGraphMode(VGAMed)
         End;
    Else
     Begin
      WriteLn('Sorry, but ', id,
              'requires either an EGA card with 256K or a VGA card.');
      Halt(0)
     End
   End
  End;


 Procedure dotext;
  Begin
   CloseGraph;
   TextMode(savemode)
  End;


 Procedure fillzero(Var s : bufstr);

  Var
   i              : integer;

  Begin
   For i := 1 To Length(s) Do
    If s[i] = #32 Then
     s[i] := '0'
  End;


 Procedure placewindow(x1, y1, x2, y2 : integer);
  Begin
   Rectangle(x1, y1, x2, y2);
   Bar(x2+1, y1+8, x2+3, y2);
   Bar(x1+8, y2+1, x2+3, y2+2)
  End;


 Procedure putshape(x, y : integer;
                    s    : byte;
                    p    : pointer);

  Var
   i              : integer;
   xs             : byte;

  Begin
   xs := shapetab[s, info, 1];
   PutImage(x, y, p^, XORPut);
   For i := 1 To xs Do
    PutImage(x+xshapetab[s, 0, i, 2], y+xshapetab[s, 0, i, 1], p^, XORPut)
  End;


 Procedure init;

  Var
   i, j, isiz     : integer;

  Procedure abortgraphics;
   Begin
    WriteLn(GraphErrorMsg(GraphResult));
    Halt(0)
   End; {-abortgraphics-}

  Begin {-init-}
   Randomize;

   Assign(fconfig, configname);
   Reset(fconfig);
   If IOResult = 0 Then
    While Not Eof(fconfig) Do
     Begin
      ReadLn(fconfig, buf3);
      If buf3[1] <> '#' Then
       Begin
        i := Pos('=', buf3);
        buf2 := Copy(buf3, 1, i-1);
        buf := Copy(buf3, i+1, Length(buf3)-i);
      { WriteLn(buf2);
        WriteLn(buf);
        ReadLn; }
        If buf2 = 'display' Then
         Case buf[1] Of
          'C', 'c': display := color;
          'M', 'm': display := mono;
          'P', 'p': display := plasma
         End;
        If buf2 = 'height' Then
         Begin
          Val(buf, i, j);
          If (j = 0) And (i In [0..2*maxheight]) Then
           height := i
         End;
        If buf2 = 'level' Then
         Begin
          Val(buf, i, j);
          If (j = 0) And (i In [1..maxlevel]) Then
           level := i
         End;
        If buf2 = 'shownext' Then
         Case buf[1] Of
          'Y', 'y': shownext := True;
          'N', 'n': shownext := False
         End;
        If buf2 = 'showshadow' Then
         Case buf[1] Of
          'Y', 'y': showshadow := False;
          'N', 'n': showshadow := False
         End;
        If buf2 = 'tournament' Then
         Case buf[1] Of
          'Y', 'y': tournament := True;
          'N', 'n': tournament := False
         End;
        If buf2 = 'tournamentgame' Then
         Begin
          Val(buf, i, j);
          If (j = 0) And (i In [0..ngames-1]) Then
           tournamentgame := i
         End;
        If buf2 = 'xshape' Then
         Case buf[1] Of
          'C', 'c': xshape := 0;
          'E', 'e': xshape := 1;
          'M', 'm': xshape := 2;
          'H', 'h': xshape := 3
         End;
        If buf2 = 'styleblocks' Then
         Case buf[1] Of
          'N', 'n': styleblocks := 1;
          'C', 'c': styleblocks := 2;
          'P', 'p': styleblocks := 3;
          'R', 'r': styleblocks := 4
         End
       End
     End;
   Close(fconfig);

   If ParamCount > 0 Then
    Begin
     buf := Copy(ParamStr(1), 1, 1);
     Case buf[1] Of
      'C', 'c': display := color;
      'M', 'm': display := mono;
      'P', 'p': display := plasma
     End
    End;

   If RegisterBGIdriver(@EGAVGADriver) < 0 Then
    abortgraphics;

   If RegisterBGIfont(@SansSerifFontProc) < 0 Then
    abortgraphics;
   If RegisterBGIfont(@SmallFontProc) < 0 Then
    abortgraphics;

   For i := 1 To nshapes Do
    For j := 1 To shapesiz Do
     Begin
      xshapetab[i, 0, j, 1] :=  pixelsperblock*shapetab[i, j, 1];
      yshapetab[i, 0, j, 1] :=  shapetab[i, j, 1];
      xshapetab[i, 0, j, 2] :=  pixelsperblock*shapetab[i, j, 2];
      yshapetab[i, 0, j, 2] :=  shapetab[i, j, 2];
      xshapetab[i, 1, j, 1] :=  pixelsperblock*shapetab[i, j, 2];
      yshapetab[i, 1, j, 1] :=  shapetab[i, j, 2];
      xshapetab[i, 1, j, 2] := -pixelsperblock*shapetab[i, j, 1];
      yshapetab[i, 1, j, 2] := -shapetab[i, j, 1];
      xshapetab[i, 2, j, 1] := -pixelsperblock*shapetab[i, j, 1];
      yshapetab[i, 2, j, 1] := -shapetab[i, j, 1];
      xshapetab[i, 2, j, 2] := -pixelsperblock*shapetab[i, j, 2];
      yshapetab[i, 2, j, 2] := -shapetab[i, j, 2];
      xshapetab[i, 3, j, 1] := -pixelsperblock*shapetab[i, j, 2];
      yshapetab[i, 3, j, 1] := -shapetab[i, j, 2];
      xshapetab[i, 3, j, 2] :=  pixelsperblock*shapetab[i, j, 1];
      yshapetab[i, 3, j, 2] :=  shapetab[i, j, 1]
     End;

   For i := 1 To ncolors Do
    shapecolors[i] := shapecolortab[display, i];

   colornormal := mesgcolortab[display, normal];
   colorhigh   := mesgcolortab[display, high];

   FillChar(hiscore, SizeOf(hiscore), 0);
   Assign(fhiscore, hiscorename);
   Reset(fhiscore);
   i := 1;
   If IOResult = 0 Then
    Begin
     While (i <= nhiscores) And (Not Eof(fhiscore)) Do
      Begin
       Read(fhiscore, hiscore[i]);
       Inc(i)
      End;
     Close(fhiscore)
    End;

   dographics;
   SetVisualPage(page);
   page := 1-page;
   SetActivePage(page);

   isiz := ImageSize(colmin+1, rowmin, colmax-1, rowmin+pixelsperblock);
   GetMem(emptyrow, isiz);
   GetImage(colmin+1, rowmin, colmax-1, rowmin+pixelsperblock, emptyrow^);

 { isiz := ImageSize(0, 0, pixelsperblock, pixelsperblock);
   SetColor(colorhigh);
   SetFillPattern(filltab[2], colornormal);
   Bar(0, 0, pixelsperblock, pixelsperblock Shr 1);
   GetMem(shadows, isiz);
   GetImage(0, 0, pixelsperblock, pixelsperblock Shr 1, shadows^);
   PutImage(0, 0, shadows^, XORPut); }

   isiz := ImageSize(0, 0, pixelsperblock, pixelsperblock);
   SetColor(colornormal);
   SetFillPattern(filltab[1], colornormal);
   Bar(1, 1, pixelsperblock-1, pixelsperblock-1);
   SetColor(Black);
   Rectangle(3, 3, pixelsperblock-3, pixelsperblock-3);
   Line(1, 1, 3, 3);
   Line(1, pixelsperblock-1, 3, pixelsperblock-3);
   Line(pixelsperblock-1, 1, pixelsperblock-3, 3);
   Line(pixelsperblock-1, pixelsperblock-1, pixelsperblock-3,
        pixelsperblock-3);
   For i := 1 To ncolors Do
    For j := 1 To nstyles Do
     Begin
      SetFillPattern(filltab[j], shapecolors[i]);
      Bar(4, 4, pixelsperblock-4, pixelsperblock-4);
      GetMem(xstyletabs[1, i, j], isiz);
      GetImage(0, 0, pixelsperblock, pixelsperblock, xstyletabs[1, i, j]^)
     End;
   For i := 1 To ncolors Do
    For j := 1 To nstyles Do
     Begin
      SetFillPattern(filltab[Random(nstyles)+1],
                     shapecolors[Random(ncolors)+1]);
      Bar(4, 4, 7, 7);
      SetFillPattern(filltab[Random(nstyles)+1],
                     shapecolors[Random(ncolors)+1]);
      Bar(7, 4, 10, 7);
      SetFillPattern(filltab[Random(nstyles)+1],
                     shapecolors[Random(ncolors)+1]);
      Bar(4, 7, 7, 10);
      SetFillPattern(filltab[Random(nstyles)+1],
                     shapecolors[Random(ncolors)+1]);
      Bar(7, 7, 10, 10);
      GetMem(xstyletabs[3, i, j], isiz);
      GetImage(0, 0, pixelsperblock, pixelsperblock, xstyletabs[3, i, j]^)
     End;

   SetFillPattern(filltab[2], colornormal);
   Bar(4, 4, pixelsperblock-4, pixelsperblock-4);
   GetMem(filler, isiz);
   GetImage(0, 0, pixelsperblock, pixelsperblock, filler^);
   PutImage(0, 0, filler^, XORPut);

   For i := 1 To ncolors Do
    Begin
     SetColor(shapecolors[i]);
     For j := 1 To nstyles Do
      Begin
       SetFillPattern(filltab[j], shapecolors[i]);
       Rectangle(1, 1, pixelsperblock-1, pixelsperblock-1);
       Bar(3, 3, pixelsperblock-3, pixelsperblock-3);
       GetMem(xstyletabs[2, i, j], isiz);
       GetImage(0, 0, pixelsperblock, pixelsperblock, xstyletabs[2, i, j]^)
      End
    End;

   SetColor(colornormal);
   SetFillPattern(filltab[3], colornormal);
   Bar(1, 1, pixelsperblock-1, pixelsperblock-1);
   GetMem(curtain[true], isiz);
   GetImage(0, 0, pixelsperblock, pixelsperblock, curtain[true]^);

   SetFillPattern(filltab[4], colornormal);
   Bar(1, 1, pixelsperblock-1, pixelsperblock-1);
   GetMem(curtain[false], isiz);
   GetImage(0, 0, pixelsperblock, pixelsperblock, curtain[false]^);
   PutImage(0, 0, curtain[false]^, XORPut);

   For i := 1 To ncolors Do
    For j := 1 To nstyles Do
     xstyletabs[4, i, j] := xstyletabs[Random(nstyletabs-1)+1,
                                       Random(ncolors)+1,
                                       Random(nstyles)+1]
  End; {-init-}


 Procedure drawtitle;

  Const
   titlesiz       = 95;
   titletab       : Array [1..titlesiz, 1..2] Of integer =
                    (( 75,  57), ( 75,  71), ( 75, 85), ( 75, 99),
                      ( 75, 113), ( 75, 127), ( 75, 141),
                     ( 89,  57), ( 89, 99), ( 89, 141),
                     (103,  57), (103, 99), (103, 141),
                     (117,  57), (117, 99), (117, 141),
                     (131,  57), (131, 141),

                     (159,  71), (159, 85), (159, 99), (159, 113),
                      (159, 127),
                     (173,  57), (173, 141),
                     (187,  57), (187, 141),
                     (201,  57), (201, 99), (201, 141),
                     (215,  71), (215, 99), (215, 113), (215, 127),

                     (243,  71), (243, 85), (243, 99), (243, 113),
                      (243, 127), (243, 141),
                     (257,  57), (257, 99),
                     (271,  57), (271, 99),
                     (285,  57), (285, 99),
                     (299,  71), (299, 85), (299, 99), (299, 113),
                      (299, 127), (299, 141),

                     (327,  57), (327, 141),
                     (341,  57), (341, 141),
                     (355,  57), (355,  71), (355, 85), (355, 99),
                      (355, 113), (355, 127), (355, 141),
                     (369,  57), (369, 141),
                     (383,  57), (383, 141),

                     (411,  57), (411,  71), (411, 85), (411, 99),
                      (411, 113), (411, 127), (411, 141),
                     (425,  71),
                     (439, 85),
                     (453, 99),
                     (467,  57), (467,  71), (467, 85), (467, 99),
                      (467, 113), (467, 127), (467, 141),

                     (495,  57),
                     (509,  57),
                     (523,  57), (523,  71), (523, 85), (523, 99),
                      (523, 113), (523, 127), (523, 141),
                     (537,  57),
                     (551,  57));

  Var
   test           : Array [1..titlesiz] Of boolean;
   ch             : char;
   i, j, c, s     : integer;
   x, y1, y2      : integer;
   p              : pointer;

  Begin {-drawtitle-}
   FillChar(test, SizeOf(test), 0);

   If styleblocks = 0 Then
    styleblocks := Random(nstyletabs-1)+1;
   s := 1;

   For i := 1 To titlesiz Do
    Begin
     Repeat
      j := Random(titlesiz)+1
     Until Not test[j];
     c := Random(ncolors)+1;
     If styleblocks = 3 Then
      s := Random(nstyles)+1;
     x := titletab[j, 1];
     If KeyPressed Then
      y1 := titletab[j, 2]
     Else
      Begin
       y1 := 0;
       y2 := dropinc
      End;
     p := xstyletabs[styleblocks, c, s];
     PutImage(x, y1, p^, XORPut);
     SetVisualPage(page);
     page := 1-page;
     SetActivePage(page);

     While (Not KeyPressed) And (y2 < titletab[j, 2]) Do
      Begin
       PutImage(x, y2, p^, XORPut);
       Delay(dropdelay);
       SetVisualPage(page);
       page := 1-page;
       SetActivePage(page);
       PutImage(x, y1, p^, XORPut);
       y1 := y2;
       Inc(y2, dropinc)
      End;

     PutImage(x, titletab[j, 2], p^, XORPut);
     SetVisualPage(page);
     page := 1-page;
     SetActivePage(page);

     PutImage(x, y1, p^, XORPut);
     PutImage(x, titletab[j, 2], p^, XORPut);
     test[j] := True
    End;
   While KeyPressed Do
    ch := ReadKey;

   SetTextJustify(CenterText, TopText);
   SetColor(colorhigh);
   SetTextStyle(SansSerifFont, HorizDir, 4);
   OutTextXY(320, 10, 'Welcome to version '+version+' of');
   OutTextXY(320, 165, copyright);

   SetTextStyle(SmallFont, HorizDir, 4);
   OutTextXY(320, 215,
'This program is free software; you can redistribute it under the terms of '+
'the GNU General Public License,');
   OutTextXY(320, 227,
'Version 1, as published by the Free Software Foundation.  This program is '+
'distributed in the hope that it');
   OutTextXY(320, 239,
'will be useful, but without any warranty whatsoever, without even the '+
'implied warranties of merchantability or');
   OutTextXY(320, 251,
'fitness for a particular purpose.  See the enclosed GNU General Public '+
'License for more details, or write to:');
   OutTextXY(320, 263,
'Free Software Foundation, Inc., 675 Massachusetts Avenue, Cambridge, '+
'Massachusetts 02139');
   OutTextXY(160, 329, 'Internet:  erc@{mars,irss,inis}.njit.edu');

   SetColor(colornormal);
   OutTextXY(160, 281, 'To obtain the complete source code for this');
   OutTextXY(160, 293, 'particular version, call either T. McDermet''s');
   OutTextXY(160, 305, 'The Odyssey at 201/984-6574 or J. Looker''s');
   OutTextXY(160, 317, 'Bandersnatch at 201/766-3801');

   OutTextXY(480, 281, 'This program requires an IBM PC-AT compatible');
   OutTextXY(480, 293, '(286s or 386s are strongly recommended) with an');
   OutTextXY(480, 305, 'IBM EGA with 256K RAM or equivalent.  VGA cards');
   OutTextXY(480, 317, 'have been rumored to work, but this has not been');
   OutTextXY(480, 329, 'tested (or witnessed) by the author.');

   SetVisualPage(page);
   page := 1-page;
   SetActivePage(page);
   ClearDevice;

   Repeat Until KeyPressed;
   Repeat
    ch := ReadKey
   Until Not KeyPressed
  End; {-drawtitle-}


 Procedure initgame;

  Var
   i, j           : integer;

  Procedure getoptions;

   Const
    noptions      = 8;

    optiontitles  : Array [1..noptions] Of String [22] =
                    ('Tournament Game',
                     'Tournament Game Number',
                     'Initial Level',
                     'Initial Height',
                     'Show Next',
                     'Show Shadow',
                     'Extended Shapes',
                     'Block Style');

    optionytab    : Array [1..noptions] Of integer =
                    (86, 114, 142, 170, 198, 226, 254, 282);

   Var
    done          : boolean;
    o             : byte;
    ch            : char;

   Procedure drawoptions;

    Var
     i            : integer;

    Begin {-drawoptions-}
     SetTextJustify(CenterText, TopText);
     SetColor(colorhigh);
     SetTextStyle(SansSerifFont, HorizDir, 4);
     OutTextXY(320, 5, id+' '+version);

     SetColor(colornormal);
     SetTextStyle(DefaultFont, HorizDir, 1);
     OutTextXY(320, 40, 'Options');
     OutTextXY(320, 330,
      'Press J for up, K to rotate, L for left, and the Space Bar when done.');

     SetFillStyle(SolidFill, colornormal);
     placewindow(150, 60, 490, 312);

     SetTextJustify(LeftText, TopText);
     For i := 1 To noptions Do
      OutTextXY(200, optionytab[i]+2, optiontitles[i])
    End; {-drawoptions-}

   Procedure showflag(f : boolean;
                      y : integer);
    Begin
     If f Then
      OutTextXY(440, optionytab[y], 'Yes')
     Else
      OutTextXY(440, optionytab[y], 'No')
    End; {-showflag-}

   Procedure showoption(o : byte);
    Begin
     Case o Of
      1: showflag(tournament, 1);
      2: Begin
          Str(tournamentgame, buf);
          OutTextXY(440, optionytab[2], buf)
         End;
      3: Begin
          Str(level, buf);
          OutTextXY(440, optionytab[3], buf)
         End;
      4: Begin
          If height > maxheight Then
           Begin
            Str(height-maxheight, buf);
            buf := 'Hidden '+buf
           End
          Else
           Str(height, buf);
          OutTextXY(440, optionytab[4], buf)
         End;
      5: showflag(shownext, 5);
      6: showflag(showshadow, 6);
      7: OutTextXY(440, optionytab[7], xshapetitles[xshape+1]);
      8: OutTextXY(440, optionytab[8], styleblocktitles[styleblocks])
     End
    End; {-showoptions-}

   Procedure rotateopt(o : byte);
    Begin
     SetTextJustify(RightText, TopText);
     SetTextStyle(SmallFont, HorizDir, 4);
     SetColor(Black);
     showoption(o);
     Case o Of
      1: tournament     := Not tournament;
      2: tournamentgame := (tournamentgame+1) Mod ngames;
      3: level          := (level Mod maxlevel)+1;
      4: height         := (height+1) Mod ((maxheight Shl 1)+1);
      5: shownext       := Not shownext;
      6: showshadow     := False;
      7: xshape         := (xshape+1) Mod xshapelevels;
      8: styleblocks    := (styleblocks Mod nstyletabs)+1
     End;
     SetColor(colorhigh);
     showoption(o)
    End; {-rotateopt-}

   Begin {-getoptions-}
    drawoptions;
    SetTextJustify(RightText, TopText);
    SetTextStyle(SmallFont, HorizDir, 4);
    SetColor(colorhigh);
    For o := 1 To noptions Do
     showoption(o);
    SetVisualPage(page);

    done := False;
    o    := 1;
    Repeat
     SetTextJustify(LeftText, TopText);
     SetTextStyle(DefaultFont, HorizDir, 1);
     SetColor(colorhigh);
     OutTextXY(200, optionytab[o]+2, optiontitles[o]);

     Repeat Until KeyPressed;
     ch := ReadKey;
     Case ch Of
                     #27: Begin
                           done   := True;
                           endrun := True
                          End;
                     #32: done := True;
                'J', 'j': Begin
                           SetColor(colornormal);
                           OutTextXY(200, optionytab[o]+2, optiontitles[o]);
                           If o < 2 Then
                            o := noptions
                           Else
                            Dec(o)
                          End;
      'K', 'k', 'I', 'i': rotateopt(o);
                'L', 'l': Begin
                           SetColor(colornormal);
                           OutTextXY(200, optionytab[o]+2, optiontitles[o]);
                           If o > noptions-1 Then
                            o := 1
                           Else
                            Inc(o)
                          End
     End
    Until done;

    page := 1-page;
    SetActivePage(page);
    ClearDevice;
   End; {-getoptions-}

  Procedure fillfield(h : byte);

   Var
    i, j          : integer;
    k             : byte;

   Begin {-fillfield-}
    For i := blockrows DownTo blockrows-(h-1) Do
     Begin
      k := Random(filladd)+fillbase;
      For j := 1 To k Do
       field[i, Random(blockcols)+1] := True
     End
   End; {-fillfield-}

  Begin {-initgame-}
   getoptions;

   FillChar(field, SizeOf(field)-blockcols, 0);
   FillChar(field[xblockrows, 1], blockcols, 1);
 { FillChar(fieldshadows, SizeOf(fieldshadows), 0); }

   If tournament Then
    RandSeed := tournamentgame;

   If height <> 0 Then
    Begin
     If height > maxheight Then
      Begin
       fillfield(height-maxheight);
       bonus := (height-maxheight)+bonushidden
      End
     Else
      Begin
       fillfield(height);
       bonus := height
      End
    End
   Else
    bonus := 0;
   If Not shownext Then
    Inc(bonus, bonusnext);
   If Not showshadow Then
    Inc(bonus, bonusshadow);

   rowsclear := 0;
   score     := 0;

   Case xshape Of
    0: shapemap := xshapeclassic;
    1: shapemap := xshapeeasy;
    2: shapemap := xshapemedium;
    3: shapemap := xshapehard
   End;

   Move(xstyletabs[styleblocks], styletab, SizeOf(styletab))
  End; {-initgame-}

 Procedure drawscreen;

  Procedure drawfieldwin;

   Var
    rowmaxpel      : integer;
    colminpel      : integer;
    colmaxpel      : integer;
    i              : integer;

   Begin {-drawfieldwin-}
    rowmaxpel := rowmax+pixelsperblock-2;
    colminpel := colmin-pixelsperblock;
    colmaxpel := colmax+pixelsperblock;

    SetColor(colornormal);
    SetFillPattern(filltab[2], colornormal);
    Bar(colminpel, rowmin, colmin, rowmaxpel);
    Bar(colmin, rowmax, colmax, rowmaxpel);
    Bar(colmax, rowmin, colmaxpel, rowmaxpel);
    Line(colminpel, rowmin, colminpel, rowmaxpel);
    Line(colmin, rowmin, colmin, rowmax);
    Line(colmax, rowmin, colmax, rowmax);
    Line(colmaxpel, rowmin, colmaxpel, rowmaxpel);
    Line(colminpel, rowmin, colmin, rowmin);
    Line(colmin, rowmax, colmax, rowmax);
    Line(colmax, rowmin, colmaxpel, rowmin);
    Line(colminpel, rowmaxpel, colmaxpel, rowmaxpel);
   End; {-drawfieldwin-}

  Procedure drawnextwin;
   Begin
    SetColor(colornormal);
    SetFillStyle(SolidFill, colornormal);
    placewindow(35, 16, 201, 126);

    SetTextStyle(DefaultFont, HorizDir, 1);
    OutTextXY(102, 114, 'Next')
   End;

  Procedure drawscorewin;
   Begin
    SetColor(colornormal);
    SetFillStyle(SolidFill, colornormal);
    placewindow(439, 16, 605, 126);

    SetColor(colorhigh);
    SetTextStyle(SansSerifFont, HorizDir, 4);
    SetTextJustify(CenterText, TopText);
    OutTextXY(522, 24, id);

    SetColor(colornormal);
    SetTextStyle(SmallFont, HorizDir, 4);
    OutTextXY(522, 60, copr);

    SetTextStyle(DefaultFont, HorizDir, 1);
    SetTextJustify(LeftText, TopText);
    OutTextXY(466, 74, 'Score:');
    OutTextXY(466, 86, 'Value:');
    OutTextXY(466, 98, 'Level:');
    OutTextXY(466, 110, ' Rows:');
   End; {-drawscorewin-}

  Procedure drawhelpwin;
   Begin
    SetColor(colornormal);
    SetFillStyle(SolidFill, colornormal);
    placewindow(35, 224, 201, 334);
    placewindow(439, 224, 605, 334);

    SetColor(colorhigh);
    SetTextStyle(DefaultFont, HorizDir, 1);
    OutTextXY(58, 245, 'J');
    OutTextXY(58, 257, 'I');
    OutTextXY(58, 269, 'K');
    OutTextXY(58, 281, 'L');
    OutTextXY(58, 293, 'Sp');
    OutTextXY(58, 305, 'Esc');
    OutTextXY(462, 245, 'B');
    OutTextXY(462, 257, 'N');
    OutTextXY(462, 269, 'S');
    OutTextXY(462, 281, 'V');
    OutTextXY(462, 293, 'X');

    SetColor(colornormal);
    SetTextStyle(SmallFont, HorizDir, 4);
    OutTextXY(90, 243, 'move left');
    OutTextXY(90, 255, 'rotate left');
    OutTextXY(90, 267, 'rotate right');
    OutTextXY(90, 279, 'move right');
    OutTextXY(90, 291, 'drop');
    OutTextXY(90, 303, 'pause/quit');
    OutTextXY(494, 243, 'block style');
    OutTextXY(494, 255, 'show next');
    OutTextXY(494, 267, 'show shadow');
    OutTextXY(494, 279, 'change level');
    OutTextXY(494, 291, 'extended shapes');
   End; {-drawhelpwin-}

  Procedure refill;

   Var
    i, j          : integer;

   Begin {-refill-}
    For i := blockrows DownTo blockrows-(height-1) Do
     For j := 1 To blockcols Do
      If field[i, j] Then
       PutImage(colmin+(pixelsperblock*(j-1))+1,
                rowmin+(pixelsperblock*(i-1)), filler^, XORPut)
   End; {-refill-}

  Begin {-drawscreen-}
   ClearDevice;
   drawfieldwin;
   drawnextwin;
   drawscorewin;
   drawhelpwin;
   If height In [1..maxheight] Then
    refill;

   SetVisualPage(page);
   page := 1-page;
   SetActivePage(page);

   ClearDevice;
   drawfieldwin;
   drawnextwin;
   drawscorewin;
   drawhelpwin;
   If height In [1..maxheight] Then
    refill;
  End; {-drawscreen-}

 Procedure play;

  Var
   dropped        : boolean;
   endgame        : boolean;
   shape          : byte;
   orient         : byte;
   row, col       : byte;
   color          : byte;
   style          : byte;
   ch             : char;
   t, tdelay      : longint;

   nextshape      : byte;
   nextcolor      : byte;
   nextstyle      : byte;

   xsize          : byte;
   xvalue         : integer;

   oldscore       : longint;
   oldxvalue      : integer;
   oldlevel       : byte;
   oldxshape      : byte;
   oldrowsclear   : byte;

   i, j           : integer;
   r, c           : byte;

  Procedure scrolldown(rclr  : byte;
                       var r : rinfotype);

   Var
    rz            : Array [1..clearlimit] Of integer;
    i, j, s       : integer;
    p             : pointer;

   Begin {-scrolldown-}
    For i := 1 To rclr Do
     rz[i] := pixelsperblock*(r[i]-1);

    s := ImageSize(colmin+1, rowmin, colmax-1, rz[rclr]);
    GetMem(p, s);

    For i := 1 To rclr Do
     Begin
      GetImage(colmin+1, rowmin, colmax-1, rz[i], p^);
      PutImage(colmin+1, rowmin, emptyrow^, NormalPut);
      PutImage(colmin+1, rowmin+pixelsperblock, p^, NormalPut);
      SetVisualPage(page);
      page := 1-page;
      SetActivePage(page);
      PutImage(colmin+1, rowmin, emptyrow^, NormalPut);
      PutImage(colmin+1, rowmin+pixelsperblock, p^, NormalPut)
     End;

    FreeMem(p, s)
   End; {-scrolldown-}

  Procedure drawshape;

   Var
    i             : integer;
    x, y, x1, y1  : integer;
    p             : pointer;

   Begin {-drawshape-}
  { If showshadow Then
     FillChar(fieldshadows, SizeOf(fieldshadows), 0); }
    x := colmin+(pixelsperblock*(col-1))+1;
    y := rowmin+(pixelsperblock*(row-1));
    p := styletab[color, style];

    PutImage(x, y, p^, XORPut);
  { If showshadow Then
     Begin
      PutImage(x, rowmax+1, shadows^, XORPut);
      fieldshadows[col] := True
     End; }
    For i := 1 To xsize Do
     Begin
      x1 := x+xshapetab[shape, orient, i, 2];
      y1 := y+xshapetab[shape, orient, i, 1];
      If (y1 >= rowmin) Then
       PutImage(x1, y1, p^, XORPut);
    { If showshadow And Not fieldshadows[col+yshapetab[shape, orient, i, 2]]
      Then
       Begin
        PutImage(x1, rowmax+1, shadows^, XORPut);
        fieldshadows[col+yshapetab[shape, orient, i, 2]] := True
       End }
     End
   End; {-drawshape-}

  Procedure dispscore;
   Begin
    If oldscore <> score Then
     Begin
      SetColor(Black);
      Str(oldscore, buf);
      OutTextXY(522, 72, buf);
      SetColor(colorhigh);
      Str(score, buf);
      OutTextXY(522, 72, buf)
     End;
    If oldxvalue <> xvalue Then
     Begin
      SetColor(Black);
      Str(oldxvalue, buf);
      OutTextXY(522, 84, buf);
      SetColor(colorhigh);
      Str(xvalue, buf);
      OutTextXY(522, 84, buf)
     End;
    If (oldlevel <> level) Or (oldxshape <> xshape) Then
     Begin
      SetColor(Black);
      Str(oldlevel, buf);
      buf := buf+' '+xshapetitles[oldxshape+1];
      OutTextXY(522, 96, buf);
      SetColor(colorhigh);
      Str(level, buf);
      buf := buf+' '+xshapetitles[xshape+1];
      OutTextXY(522, 96, buf)
     End;
    If oldrowsclear <> rowsclear Then
     Begin
      SetColor(Black);
      Str(oldrowsclear, buf);
      OutTextXY(522, 108, buf);
      SetColor(colorhigh);
      Str(rowsclear, buf);
      OutTextXY(522, 108, buf)
     End
   End; {-dispscore-}

  Function chk : boolean;

   Var
    f             : boolean;
    x, y, r       : shortint;
    i             : integer;

   Begin {-chk-}
    r := row+1;

    f := field[r, col];
    For i := 1 To xsize Do
     Begin
      y := r+yshapetab[shape, orient, i, 1];
      x := col+yshapetab[shape, orient, i, 2];
      If ((y >= 1) And (y <= xblockrows)) And ((x >= 1) And (x <= blockcols))
      Then
       f := f Or field[y, x]
     End;

    chk := f
   End; {-chk-}

  Procedure chkmv(c : shortint);

   Var
    f1, f2        : boolean;
    x, y          : shortint;
    i             : integer;
    xcol          : shortint;

   Begin {-chkmv-}
    Inc(c, col);

    f1 := (c >= 1) And (c <= blockcols);
    If f1 Then
     f2 := field[row, c]
    Else
     f2 := True;
    For i := 1 To xsize Do
     Begin
      x  := c+yshapetab[shape, orient, i, 2];
      y  := row+yshapetab[shape, orient, i, 1];
      f1 := f1 And ((x >= 1) And (x <= blockcols));
      If f1 And ((y >= 1) And (y <= blockrows)) Then
       f2 := f2 Or field[y, x]
     End;

    If f1 And (Not f2) Then
     Begin
      xcol := col;
      col := c;
      drawshape;
      SetVisualPage(page);
      page := 1-page;
      SetActivePage(page);
      col := xcol;
      drawshape;
      col := c
     End
   End; {-chkmv-}

  Procedure chkrot(o : byte);

   Var
    f1, f2     : boolean;
    xorient    : byte;
    x, y       : shortint;
    i          : integer;
    f          : Text;

   Begin {-chkrot-}
    f1 := True;
    f2 := False;

    For i := 1 To xsize Do
     Begin
      y  := row+yshapetab[shape, o, i, 1];
      x  := col+yshapetab[shape, o, i, 2];
      f1 := f1 And ((x >= 1) And (x <= blockcols)) And
                   (y <= blockrows);
      If f1 And (y >= 1) Then
       f2 := f2 Or field[y, x]
     End;

    If f1 And (Not f2) Then
     Begin
      xorient := orient;
      orient := o;
      drawshape;
      SetVisualPage(page);
      page := 1-page;
      SetActivePage(page);
      orient := xorient;
      drawshape;
      orient := o
     End
   End; {-chkrot-}

  Procedure dropshape;

   Var
    oldrow, xrow  : byte;

   Begin {-dropshape-}
    oldrow := row;

    While Not chk Do
     Inc(row);
    drawshape;
    SetVisualPage(page);
    page := 1-page;
    SetActivePage(page);
    xrow := row;
    row := oldrow;
    drawshape;
    row := xrow;

    Inc(score, level*(row-oldrow)+bonus);
    dropped := True
   End; {-dropshape-}

  Procedure chkrows;

   Var
    rows       : byte;
    r          : byte;
    rinfo      : rinfotype;

   Function chkrow(r : byte) : boolean;

    Var
     f         : boolean;
     i, j      : integer;

    Begin {-chkrow-}
     f := False;
     If r < xblockrows Then
      Begin
       f := field[r, 1];
       i := 2;
       While f And (i <= blockcols) Do
        Begin
         f := f And field[r, i];
         Inc(i)
        End;

       If f Then
        Begin
         Inc(rowsclear);
         If (level < maxlevel) And (rowsclear = ((level+1)*rowsperlevel)) Then
          Begin
           Inc(level);
           tdelay := timedelaytab[level]
          End;
         Move(field[0, 1], field[1, 1], blockcols*r);
         Inc(score, level*bonusrowclear+bonus)
        End
      End;
     chkrow := f
    End; {-chkrow-}

   Begin {-chkrows-}
    rows := 0;
    For r := row-2 To row+2 Do
     If chkrow(r) Then
      Begin
       Inc(rows);
       rinfo[rows] := r
      End;

    If rows > 0 Then
     Begin
      scrolldown(rows, rinfo);
      If rows > 1 Then
       Inc(score, level*((rows-1)*bonusmultclear)+bonus)
     End
   End; {-chkrows-}

  Procedure gameover;

   Var
    i, x, y, p    : integer;
    f             : boolean;

   Begin {-gameover-}
    f := True;
    For y := 1 To blockrows Do
     For p := 1 To 2 Do
      Begin
       For x := 1 To blockcols Do
        Begin
         If Not field[y, x] Then
           PutImage(colmin+(pixelsperblock*(x-1))+1,
                   rowmin+(pixelsperblock*(y-1)),
                   curtain[f]^, NormalPut);
         f := Not f
        End;
       SetVisualPage(page);
       page := 1-page;
       SetActivePage(page);
       If Not KeyPressed Then
        Delay(dropdelay)
      End;

    PutImage(colmin+1, rowmin, emptyrow^, NormalPut);
    SetColor(colorhigh);
    SetTextStyle(DefaultFont, HorizDir, 1);
    SetTextJustify(CenterText, TopText);
    OutTextXY(320, rowmin+4, 'Game Over');

    i := 1;
    Repeat
     SetVisualPage(page);
     page := 1-page;
     SetActivePage(page);
     Delay(i*dropdelay);
     Inc(i)
    Until (i > 25) Or (Not Odd(i) And KeyPressed);

    While KeyPressed Do
     ch := ReadKey
   End; {-gameover-}

  Begin {-play-}
   endgame   := False;
   nextshape := Random(shapemap)+1;
   nextcolor := Random(ncolors)+1;
   nextstyle := Random(nstyles)+1;
   xvalue    := 0;
   tdelay    := timedelaytab[level];

   oldscore     := 255;
   oldlevel     := 255;
   oldxvalue    := 0;
   oldxshape    := 255;
   oldrowsclear := 255;

 { dispscore;
   SetVisualPage(page);
   page := 1-page;
   SetActivePage(page);
   dispscore;
   oldscore     := 0;
   oldlevel     := level;
   oldxvalue    := xvalue;
   oldxshape    := xshape;
   oldrowsclear := 0; }

   If shownext Then
    putshape(111, 54, nextshape, styletab[nextcolor, nextstyle]);
   SetVisualPage(page);
   page := 1-page;
   SetActivePage(page);
   If shownext Then
    putshape(111, 54, nextshape, styletab[nextcolor, nextstyle]);

   Repeat
    Inc(score, xvalue);
    shape   := nextshape;
    orient  := 0;
    row     := initrow;
    col     := initcol;
    color   := nextcolor;
    style   := nextstyle;
    dropped := False;
    xsize   := shapetab[shape, info, 1];
    xvalue  := level*shapetab[shape, info, 2]+bonus;
    nextshape := Random(shapemap)+1;
    nextcolor := Random(ncolors)+1;
    nextstyle := Random(nstyles)+1;

    drawshape;
    dispscore;
    If shownext Then
     Begin
      putshape(111, 54, shape, styletab[color, style]);
      putshape(111, 54, nextshape, styletab[nextcolor, nextstyle])
     End;
    SetVisualPage(page);
    page := 1-page;
    SetActivePage(page);
    dispscore;
    If shownext Then
     Begin
      putshape(111, 54, shape, styletab[color, style]);
      putshape(111, 54, nextshape, styletab[nextcolor, nextstyle])
     End;
    oldscore     := score;
    oldxvalue    := xvalue;
    oldlevel     := level;
    oldxshape    := xshape;
    oldrowsclear := rowsclear;

    t := gettimer+tdelay;
    Repeat Until (gettimer > t);
    While KeyPressed Do
     ch := ReadKey;

    If chk Then
     endgame := True
    Else
     Begin
      Repeat
       Inc(row);
       drawshape;
       SetVisualPage(page);
       page := 1-page;
       SetActivePage(page);
       Dec(row);
       drawshape;
       Inc(row);

       t := gettimer+tdelay;
       Repeat
        Repeat Until KeyPressed Or (gettimer > t);
        If KeyPressed Then
         Begin
          ch := ReadKey;
          Case ch Of
                #27: Begin
                      Repeat Until KeyPressed;
                      ch := ReadKey;
                      If ch = #27 Then
                       Begin
                        dropshape;
                        endgame := True
                       End
                     End;
                #32: dropshape;
           'B', 'b': Begin
                      i := styleblocks;
                      If shownext Then
                       putshape(111, 54, nextshape,
                                styletab[nextcolor, nextstyle]);
                      styleblocks := (styleblocks Mod nstyletabs)+1;
                      Move(xstyletabs[styleblocks], styletab,
                           SizeOf(styletab));
                      drawshape;
                      If shownext Then
                       putshape(111, 54, nextshape,
                                styletab[nextcolor, nextstyle]);
                      SetVisualPage(page);
                      page := 1-page;
                      SetActivePage(page);
                      Move(xstyletabs[i], styletab,
                           SizeOf(styletab));
                      drawshape;
                      If shownext Then
                       putshape(111, 54, nextshape,
                                styletab[nextcolor, nextstyle]);
                      Move(xstyletabs[styleblocks], styletab,
                           SizeOf(styletab));
                      If shownext Then
                       putshape(111, 54, nextshape,
                                styletab[nextcolor, nextstyle]);
                      While KeyPressed Do
                       ch := ReadKey
                     End;
           'I', 'i': chkrot((norients+orient) Mod (norients+1));
           'J', 'j': chkmv(left);
           'K', 'k': chkrot((orient+1) Mod (norients+1));
           'L', 'l': chkmv(right);
           'N', 'n': Begin
                      shownext := Not shownext;
                      If shownext Then
                       Dec(bonus, bonusnext)
                      Else
                       Inc(bonus, bonusnext);
                      putshape(111, 54, nextshape,
                               styletab[nextcolor, nextstyle]);
                      drawshape;
                      SetVisualPage(page);
                      page := 1-page;
                      SetActivePage(page);
                      putshape(111, 54, nextshape,
                               styletab[nextcolor, nextstyle]);
                      drawshape;
                      While KeyPressed Do
                       ch := ReadKey
                     End;
           'S', 's': Begin
                      showshadow := Not showshadow;
                      drawshape;
                      SetVisualPage(page);
                      page := 1-page;
                      SetActivePage(page);
                      showshadow := Not showshadow;
                      drawshape;
                      showshadow := Not showshadow;
                      If showshadow Then
                       Dec(bonus, bonusshadow)
                      Else
                       Inc(bonus, bonusshadow);
                      While KeyPressed Do
                       ch := ReadKey
                     End;
           'V', 'v': Begin
                      level := (level Mod maxlevel)+1;
                      tdelay := timedelaytab[level];
                      drawshape;
                      dispscore;
                      SetVisualPage(page);
                      page := 1-page;
                      SetActivePage(page);
                      drawshape;
                      dispscore;
                      oldlevel := level;
                      While KeyPressed Do
                       ch := ReadKey
                     End;
           'X', 'x': Begin
                      xshape := (xshape+1) Mod xshapelevels;
                      Case xshape Of
                       0: shapemap := xshapeclassic;
                       1: shapemap := xshapeeasy;
                       2: shapemap := xshapemedium;
                       3: shapemap := xshapehard
                      End;
                      drawshape;
                      dispscore;
                      SetVisualPage(page);
                      page := 1-page;
                      SetActivePage(page);
                      drawshape;
                      dispscore;
                      oldxshape := xshape;
                      While KeyPressed Do
                       ch := ReadKey
                     End
          End
         End
       Until dropped Or (gettimer > t);
      Until dropped Or chk;

      drawshape;

      field[row, col] := True;
      For i := 1 To xsize Do
       field[row+yshapetab[shape, orient, i, 1],
             col+yshapetab[shape, orient, i, 2]] := True;

      chkrows;

      t := gettimer+(tdelay Shr 1);
      Repeat Until (gettimer > t);
      While KeyPressed Do
       ch := ReadKey
     End;
   Until endgame;

   dispscore;
   SetVisualPage(page);
   page := 1-page;
   SetActivePage(page);
   dispscore;
   oldscore     := score;
   oldxvalue    := xvalue;
   oldlevel     := level;
   oldxshape    := xshape;
   oldrowsclear := rowsclear;

   While KeyPressed Do
    ch := ReadKey;
   gameover;

   Repeat Until KeyPressed;
   While KeyPressed Do
    ch := ReadKey
  End;

 Procedure postgame;

  Var
   ch             : char;
   today          : DateTime;
   i, j           : word;
   rank, x, s     : integer;

  Begin
   rank := 0;

   If rowsclear > 0 Then
    Begin
     i    := 1;
     While (i <= nhiscores) And (hiscore[i].score >= score) Do
      Inc(i);
     If i <= nhiscores Then
      Begin
       rank := i;
       For j := nhiscores-1 DownTo i Do
        hiscore[j+1] := hiscore[j];
       hiscore[i].score     := score;
       hiscore[i].level     := level;
       hiscore[i].rowsclear := rowsclear;

       GetTime(today.hour, today.min, today.sec, j);
       GetDate(today.year, today.month, today.day, j);
       Dec(today.year, 1900);
       Str(today.month:2, hiscore[i].date);
       Str(today.day:2, buf);
       hiscore[i].date := hiscore[i].date+'/'+buf;
       Str(today.year:2, buf);
       hiscore[i].date := hiscore[i].date+'/'+buf;
       fillzero(hiscore[i].date);
       Str(today.hour:2, hiscore[i].time);
       Str(today.min:2, buf);
       hiscore[i].time := hiscore[i].time+':'+buf;
       Str(today.sec:2, buf);
       hiscore[i].time := hiscore[i].time+':'+buf;
       fillzero(hiscore[i].time);
       hiscore[i].version := version;

       ClearDevice;

       SetTextJustify(CenterText, TopText);
       SetTextStyle(SansSerifFont, HorizDir, 4);
       SetColor(colorhigh);
       OutTextXY(320, 5, 'Congratulations!');

       SetTextStyle(DefaultFont, HorizDir, 1);
       SetColor(colornormal);
       OutTextXY(320, 45, 'You''ve made it into the Glorious Fifteen;');
       OutTextXY(320, 57, 'please enter your name for posterity:');

       SetColor(colornormal);
       placewindow(214, 155, 426, 195);

       SetVisualPage(page);
       page := 1-page;

       SetTextStyle(SmallFont, HorizDir, 4);
       x := 1;
       Repeat
        SetColor(colorhigh);
        OutTextXY(224+6*(x-1), 171, '_');
        Repeat Until KeyPressed;
        ch := ReadKey;
        Case ch Of
          #0: While KeyPressed Do
               ch := ReadKey;
          #8: If x > 1 Then
               Begin
                SetColor(Black);
                OutTextXY(224+6*(x-1), 171, '_');
                Dec(x);
                OutTextXY(224+6*(x-1), 171, hiscore[i].name[x])
               End;
         #13: hiscore[i].name[0] := Chr(x-1);
         #27: If x > 1 Then
               Begin
                SetColor(Black);
                OutTextXY(224+6*(x-1), 171, '_');
                For s := x DownTo 1 Do
                 OutTextXY(224+6*(s-1), 171, hiscore[i].name[s]);
                x := 1
               End;
         Else If x < SizeOf(bufstr) Then
               Begin
                SetColor(Black);
                OutTextXY(224+6*(x-1), 171, '_');
                SetColor(colorhigh);
                OutTextXY(224+6*(x-1), 171, ch);
                hiscore[i].name[x] := ch;
                Inc(x)
               End
        End
       Until (ch = #13) or (x > SizeOf(bufstr))
      End
    End;

   SetActivePage(page);
   ClearDevice;

   SetTextStyle(SansSerifFont, HorizDir, 4);
   SetTextJustify(CenterText, TopText);
   SetColor(colorhigh);
   OutTextXY(320, 5, 'The Glorious Fifteen');

   SetColor(colornormal);
   SetFillStyle(SolidFill, colornormal);
   placewindow(16, 50, 615, 256);

   SetTextStyle(DefaultFont, HorizDir, 1);
   SetTextJustify(LeftText, TopText);
   SetColor(colorhigh);
   OutTextXY(24, 60, 'Rank  Score  Level Rows   Date     Time   Name');

   SetColor(colornormal);
   SetTextStyle(SmallFont, HorizDir, 4);
   For i := 1 To nhiscores Do
    Begin
     If rank = i Then
      SetColor(colorhigh);
     SetTextJustify(CenterText, TopText);
     Str(i:2, buf);
     fillzero(buf);
     OutTextXY(40, 72+12*(i-1), buf);
     If hiscore[i].score <> 0 Then
      Begin
       Str(hiscore[i].score:7, buf);
       fillzero(buf);
       OutTextXY(92, 72+12*(i-1), buf);
       Str(hiscore[i].level:2, buf);
       fillzero(buf);
       OutTextXY(148, 72+12*(i-1), buf);
       Str(hiscore[i].rowsclear:2, buf);
       fillzero(buf);
       OutTextXY(192, 72+12*(i-1), buf);
       OutTextXY(248, 72+12*(i-1), hiscore[i].date);
       OutTextXY(320, 72+12*(i-1), hiscore[i].time);
       SetTextJustify(LeftText, TopText);
       OutTextXY(360, 72+12*(i-1), hiscore[i].name);
       OutTextXY(563, 72+12*(i-1), hiscore[i].version)
      End;
     If rank = i Then
      SetColor(colornormal)
    End;

   SetTextStyle(DefaultFont, HorizDir, 1);
   SetTextJustify(CenterText, TopText);
   SetColor(colornormal);
   OutTextXY(320, 300, 'Press Y to try again or N to exit.');

   SetVisualPage(page);
   page := 1-page;
   SetActivePage(page);
   ClearDevice;

   Repeat
    Repeat Until KeyPressed;
    ch := ReadKey;
   Until (ch In ['N', 'Y', 'n', 'y']);

   endrun := ch In ['N', 'n']
  End;

{ 12345678901234567890123456789012345678901234567890123456789012345678901234
  rank  score  level rows   date     time   name'
   00  0000000   00  0000 00/00/00 00:00:00 12345678901234567890123456789012
}

 Procedure cleanup;

  Var
   i              : integer;

  Procedure configflag(f : boolean);
   Begin
    If f Then
     WriteLn(fconfig, 'Yes')
    Else
     WriteLn(fconfig, 'No')
   End; {-configflag-}

  Begin {-cleanup-}
   dotext;

   Assign(fhiscore, hiscorename);
   Rewrite(fhiscore);

   i := 1;
   While (i <= nhiscores) And (hiscore[i].score > 0) Do
    Begin
     Write(fhiscore, hiscore[i]);
     Inc(i)
    End;
   Close(fhiscore);

   Assign(fconfig, configname);
   Rewrite(fconfig);
   WriteLn(fconfig, '# ', id, '':1, version, ' configuration file');
   WriteLn(fconfig, '# ', copyright);
   Write(fconfig, 'display=');
   Case display Of
    color : WriteLn(fconfig, 'Color');
    mono  : WriteLn(fconfig, 'Mono');
    plasma: WriteLn(fconfig, 'Plasma')
   End;
   WriteLn(fconfig, 'height=', height);
   WriteLn(fconfig, 'level=', level);
   Write(fconfig, 'shownext=');
   configflag(shownext);
   Write(fconfig, 'showshadow=');
   configflag(showshadow);
   WriteLn(fconfig, 'styleblocks=', styleblocktitles[styleblocks]);
   Write(fconfig, 'tournament=');
   configflag(tournament);
   WriteLn(fconfig, 'tournamentgame=', tournamentgame);
   WriteLn(fconfig, 'xshape=', xshapetitles[xshape+1]);
   Close(fconfig)
  End; {-cleanup-}

 Begin
  init;
  drawtitle;
  Repeat
   initgame;
   If Not endrun Then
    Begin
     drawscreen;
     play;
     postgame
    End;
  Until endrun;
  cleanup
 End.

